/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; fill-column: 100 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include <sal/config.h>

#include <o3tl/unit_conversion.hxx>

// Just some static asserts

namespace
{
constexpr double const_abs(double f) { return f >= 0 ? f : -f; }
constexpr bool eq(double a, double b) { return const_abs(a - b) < 1E-6; }
bool eq(sal_Int64, double) = delete;
}

// testing floating-point conversion

static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm100, o3tl::Length::mm100), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm10, o3tl::Length::mm100), 10));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm, o3tl::Length::mm100), 100));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::cm, o3tl::Length::mm100), 1000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::m, o3tl::Length::mm100), 100000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::km, o3tl::Length::mm100), 100E6));
static_assert(eq(o3tl::convert(360.0, o3tl::Length::emu, o3tl::Length::mm100), 1));
static_assert(eq(o3tl::convert(72.0, o3tl::Length::twip, o3tl::Length::mm100), 127));
static_assert(eq(o3tl::convert(18.0, o3tl::Length::pt, o3tl::Length::mm100), 635));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::pc, o3tl::Length::mm100), 1270));
static_assert(eq(o3tl::convert(50.0, o3tl::Length::in1000, o3tl::Length::mm100), 127));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in100, o3tl::Length::mm100), 127));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in10, o3tl::Length::mm100), 254));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::mm100), 2540));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::mm100), 30480));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::mm100), 160934400));
static_assert(eq(o3tl::convert(144.0, o3tl::Length::master, o3tl::Length::mm100), 635));
static_assert(eq(o3tl::convert(24.0, o3tl::Length::px, o3tl::Length::mm100), 635));
static_assert(eq(o3tl::convert(12.0, o3tl::Length::ch, o3tl::Length::mm100), 4445));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::line, o3tl::Length::mm100), 1651));

static_assert(eq(o3tl::convert(10.0, o3tl::Length::mm100, o3tl::Length::mm10), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm10, o3tl::Length::mm10), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm, o3tl::Length::mm10), 10));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::cm, o3tl::Length::mm10), 100));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::m, o3tl::Length::mm10), 10000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::km, o3tl::Length::mm10), 10E6));
static_assert(eq(o3tl::convert(3600.0, o3tl::Length::emu, o3tl::Length::mm10), 1));
static_assert(eq(o3tl::convert(720.0, o3tl::Length::twip, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(36.0, o3tl::Length::pt, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::pc, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(500.0, o3tl::Length::in1000, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(50.0, o3tl::Length::in100, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in10, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::mm10), 254));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::mm10), 3048));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::mm10), 16093440));
static_assert(eq(o3tl::convert(288.0, o3tl::Length::master, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(48.0, o3tl::Length::px, o3tl::Length::mm10), 127));
static_assert(eq(o3tl::convert(24.0, o3tl::Length::ch, o3tl::Length::mm10), 889));
static_assert(eq(o3tl::convert(30.0, o3tl::Length::line, o3tl::Length::mm10), 1651));

static_assert(eq(o3tl::convert(100.0, o3tl::Length::mm100, o3tl::Length::mm), 1));
static_assert(eq(o3tl::convert(10.0, o3tl::Length::mm10, o3tl::Length::mm), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm, o3tl::Length::mm), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::cm, o3tl::Length::mm), 10));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::m, o3tl::Length::mm), 1000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::km, o3tl::Length::mm), 1E6));
static_assert(eq(o3tl::convert(36000.0, o3tl::Length::emu, o3tl::Length::mm), 1));
static_assert(eq(o3tl::convert(7200.0, o3tl::Length::twip, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(360.0, o3tl::Length::pt, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(30.0, o3tl::Length::pc, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(5000.0, o3tl::Length::in1000, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(500.0, o3tl::Length::in100, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(50.0, o3tl::Length::in10, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::ft, o3tl::Length::mm), 1524));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::mm), 1609344));
static_assert(eq(o3tl::convert(2880.0, o3tl::Length::master, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(480.0, o3tl::Length::px, o3tl::Length::mm), 127));
static_assert(eq(o3tl::convert(240.0, o3tl::Length::ch, o3tl::Length::mm), 889));
static_assert(eq(o3tl::convert(300.0, o3tl::Length::line, o3tl::Length::mm), 1651));

static_assert(eq(o3tl::convert(1000.0, o3tl::Length::mm100, o3tl::Length::cm), 1));
static_assert(eq(o3tl::convert(100.0, o3tl::Length::mm10, o3tl::Length::cm), 1));
static_assert(eq(o3tl::convert(10.0, o3tl::Length::mm, o3tl::Length::cm), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::cm, o3tl::Length::cm), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::m, o3tl::Length::cm), 100));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::km, o3tl::Length::cm), 100000));
static_assert(eq(o3tl::convert(360000.0, o3tl::Length::emu, o3tl::Length::cm), 1));
static_assert(eq(o3tl::convert(72000.0, o3tl::Length::twip, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(3600.0, o3tl::Length::pt, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(300.0, o3tl::Length::pc, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(50000.0, o3tl::Length::in1000, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(5000.0, o3tl::Length::in100, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(500.0, o3tl::Length::in10, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(50.0, o3tl::Length::in, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(25.0, o3tl::Length::ft, o3tl::Length::cm), 762));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::mi, o3tl::Length::cm), 804672));
static_assert(eq(o3tl::convert(28800.0, o3tl::Length::master, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(4800.0, o3tl::Length::px, o3tl::Length::cm), 127));
static_assert(eq(o3tl::convert(2400.0, o3tl::Length::ch, o3tl::Length::cm), 889));
static_assert(eq(o3tl::convert(3000.0, o3tl::Length::line, o3tl::Length::cm), 1651));

static_assert(eq(o3tl::convert(100000.0, o3tl::Length::mm100, o3tl::Length::m), 1));
static_assert(eq(o3tl::convert(10000.0, o3tl::Length::mm10, o3tl::Length::m), 1));
static_assert(eq(o3tl::convert(1000.0, o3tl::Length::mm, o3tl::Length::m), 1));
static_assert(eq(o3tl::convert(100.0, o3tl::Length::cm, o3tl::Length::m), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::m, o3tl::Length::m), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::km, o3tl::Length::m), 1000));
static_assert(eq(o3tl::convert(36E6, o3tl::Length::emu, o3tl::Length::m), 1));
static_assert(eq(o3tl::convert(7200000.0, o3tl::Length::twip, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(360000.0, o3tl::Length::pt, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(30000.0, o3tl::Length::pc, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(5000000.0, o3tl::Length::in1000, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(500000.0, o3tl::Length::in100, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(50000.0, o3tl::Length::in10, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(5000.0, o3tl::Length::in, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(1250.0, o3tl::Length::ft, o3tl::Length::m), 381));
static_assert(eq(o3tl::convert(125.0, o3tl::Length::mi, o3tl::Length::m), 201168));
static_assert(eq(o3tl::convert(2880000.0, o3tl::Length::master, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(480000.0, o3tl::Length::px, o3tl::Length::m), 127));
static_assert(eq(o3tl::convert(240000.0, o3tl::Length::ch, o3tl::Length::m), 889));
static_assert(eq(o3tl::convert(300000.0, o3tl::Length::line, o3tl::Length::m), 1651));

static_assert(eq(o3tl::convert(100E6, o3tl::Length::mm100, o3tl::Length::km), 1));
static_assert(eq(o3tl::convert(10E6, o3tl::Length::mm10, o3tl::Length::km), 1));
static_assert(eq(o3tl::convert(1E6, o3tl::Length::mm, o3tl::Length::km), 1));
static_assert(eq(o3tl::convert(100000.0, o3tl::Length::cm, o3tl::Length::km), 1));
static_assert(eq(o3tl::convert(1000.0, o3tl::Length::m, o3tl::Length::km), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::km, o3tl::Length::km), 1));
static_assert(eq(o3tl::convert(36E9, o3tl::Length::emu, o3tl::Length::km), 1));
static_assert(eq(o3tl::convert(7200E6, o3tl::Length::twip, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(360E6, o3tl::Length::pt, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(30E6, o3tl::Length::pc, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(5E9, o3tl::Length::in1000, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(500E6, o3tl::Length::in100, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(50E6, o3tl::Length::in10, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(5E6, o3tl::Length::in, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(1250000.0, o3tl::Length::ft, o3tl::Length::km), 381));
static_assert(eq(o3tl::convert(15625.0, o3tl::Length::mi, o3tl::Length::km), 25146));
static_assert(eq(o3tl::convert(2880E6, o3tl::Length::master, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(480E6, o3tl::Length::px, o3tl::Length::km), 127));
static_assert(eq(o3tl::convert(240E6, o3tl::Length::ch, o3tl::Length::km), 889));
static_assert(eq(o3tl::convert(300E6, o3tl::Length::line, o3tl::Length::km), 1651));

static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm100, o3tl::Length::emu), 360));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm10, o3tl::Length::emu), 3600));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mm, o3tl::Length::emu), 36000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::cm, o3tl::Length::emu), 360000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::m, o3tl::Length::emu), 36E6));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::km, o3tl::Length::emu), 36E9));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::emu, o3tl::Length::emu), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::twip, o3tl::Length::emu), 635));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pt, o3tl::Length::emu), 12700));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pc, o3tl::Length::emu), 152400));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in1000, o3tl::Length::emu), 4572));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in100, o3tl::Length::emu), 9144));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in10, o3tl::Length::emu), 91440));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::emu), 914400));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::emu), 10972800));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::emu), 57936384000));
static_assert(eq(o3tl::convert(2.0, o3tl::Length::master, o3tl::Length::emu), 3175));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::px, o3tl::Length::emu), 9525));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ch, o3tl::Length::emu), 133350));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::line, o3tl::Length::emu), 198120));

static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm100, o3tl::Length::twip), 72));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::twip), 720));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::twip), 7200));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::twip), 72000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::twip), 7200000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::twip), 7200E6));
static_assert(eq(o3tl::convert(635.0, o3tl::Length::emu, o3tl::Length::twip), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::twip, o3tl::Length::twip), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pt, o3tl::Length::twip), 20));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pc, o3tl::Length::twip), 240));
static_assert(eq(o3tl::convert(25.0, o3tl::Length::in1000, o3tl::Length::twip), 36));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in100, o3tl::Length::twip), 72));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in10, o3tl::Length::twip), 144));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::twip), 1440));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::twip), 17280));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::twip), 91238400));
static_assert(eq(o3tl::convert(2.0, o3tl::Length::master, o3tl::Length::twip), 5));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::px, o3tl::Length::twip), 15));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ch, o3tl::Length::twip), 210));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::line, o3tl::Length::twip), 312));

static_assert(eq(o3tl::convert(635.0, o3tl::Length::mm100, o3tl::Length::pt), 18));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::pt), 36));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::pt), 360));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::pt), 3600));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::pt), 360000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::pt), 360E6));
static_assert(eq(o3tl::convert(12700.0, o3tl::Length::emu, o3tl::Length::pt), 1));
static_assert(eq(o3tl::convert(20.0, o3tl::Length::twip, o3tl::Length::pt), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pt, o3tl::Length::pt), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pc, o3tl::Length::pt), 12));
static_assert(eq(o3tl::convert(125.0, o3tl::Length::in1000, o3tl::Length::pt), 9));
static_assert(eq(o3tl::convert(25.0, o3tl::Length::in100, o3tl::Length::pt), 18));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in10, o3tl::Length::pt), 36));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::pt), 72));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::pt), 864));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::pt), 4561920));
static_assert(eq(o3tl::convert(8.0, o3tl::Length::master, o3tl::Length::pt), 1));
static_assert(eq(o3tl::convert(4.0, o3tl::Length::px, o3tl::Length::pt), 3));
static_assert(eq(o3tl::convert(2.0, o3tl::Length::ch, o3tl::Length::pt), 21));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::line, o3tl::Length::pt), 78));

static_assert(eq(o3tl::convert(1270.0, o3tl::Length::mm100, o3tl::Length::pc), 3));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::pc), 3));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::pc), 30));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::pc), 300));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::pc), 30000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::pc), 30E6));
static_assert(eq(o3tl::convert(152400.0, o3tl::Length::emu, o3tl::Length::pc), 1));
static_assert(eq(o3tl::convert(240.0, o3tl::Length::twip, o3tl::Length::pc), 1));
static_assert(eq(o3tl::convert(12.0, o3tl::Length::pt, o3tl::Length::pc), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pc, o3tl::Length::pc), 1));
static_assert(eq(o3tl::convert(500.0, o3tl::Length::in1000, o3tl::Length::pc), 3));
static_assert(eq(o3tl::convert(50.0, o3tl::Length::in100, o3tl::Length::pc), 3));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in10, o3tl::Length::pc), 3));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::pc), 6));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::pc), 72));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::pc), 380160));
static_assert(eq(o3tl::convert(96.0, o3tl::Length::master, o3tl::Length::pc), 1));
static_assert(eq(o3tl::convert(16.0, o3tl::Length::px, o3tl::Length::pc), 1));
static_assert(eq(o3tl::convert(8.0, o3tl::Length::ch, o3tl::Length::pc), 7));
static_assert(eq(o3tl::convert(10.0, o3tl::Length::line, o3tl::Length::pc), 13));

static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm100, o3tl::Length::in1000), 50));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::in1000), 500));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::in1000), 5000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::in1000), 50000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::in1000), 5E6));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::in1000), 5E9));
static_assert(eq(o3tl::convert(4572.0, o3tl::Length::emu, o3tl::Length::in1000), 5));
static_assert(eq(o3tl::convert(36.0, o3tl::Length::twip, o3tl::Length::in1000), 25));
static_assert(eq(o3tl::convert(9.0, o3tl::Length::pt, o3tl::Length::in1000), 125));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::pc, o3tl::Length::in1000), 500));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in1000, o3tl::Length::in1000), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in100, o3tl::Length::in1000), 10));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in10, o3tl::Length::in1000), 100));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::in1000), 1000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::in1000), 12000));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::in1000), 63360000));
static_assert(eq(o3tl::convert(72.0, o3tl::Length::master, o3tl::Length::in1000), 125));
static_assert(eq(o3tl::convert(12.0, o3tl::Length::px, o3tl::Length::in1000), 125));
static_assert(eq(o3tl::convert(6.0, o3tl::Length::ch, o3tl::Length::in1000), 875));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::line, o3tl::Length::in1000), 650));

static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm100, o3tl::Length::in100), 5));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::in100), 50));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::in100), 500));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::in100), 5000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::in100), 500000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::in100), 500E6));
static_assert(eq(o3tl::convert(9144.0, o3tl::Length::emu, o3tl::Length::in100), 1));
static_assert(eq(o3tl::convert(72.0, o3tl::Length::twip, o3tl::Length::in100), 5));
static_assert(eq(o3tl::convert(18.0, o3tl::Length::pt, o3tl::Length::in100), 25));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::pc, o3tl::Length::in100), 50));
static_assert(eq(o3tl::convert(10.0, o3tl::Length::in1000, o3tl::Length::in100), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in100, o3tl::Length::in100), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in10, o3tl::Length::in100), 10));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::in100), 100));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::in100), 1200));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::in100), 6336000));
static_assert(eq(o3tl::convert(144.0, o3tl::Length::master, o3tl::Length::in100), 25));
static_assert(eq(o3tl::convert(24.0, o3tl::Length::px, o3tl::Length::in100), 25));
static_assert(eq(o3tl::convert(12.0, o3tl::Length::ch, o3tl::Length::in100), 175));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::line, o3tl::Length::in100), 65));

static_assert(eq(o3tl::convert(254.0, o3tl::Length::mm100, o3tl::Length::in10), 1));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::in10), 5));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::in10), 50));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::in10), 500));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::in10), 50000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::in10), 50E6));
static_assert(eq(o3tl::convert(91440.0, o3tl::Length::emu, o3tl::Length::in10), 1));
static_assert(eq(o3tl::convert(144.0, o3tl::Length::twip, o3tl::Length::in10), 1));
static_assert(eq(o3tl::convert(36.0, o3tl::Length::pt, o3tl::Length::in10), 5));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::pc, o3tl::Length::in10), 5));
static_assert(eq(o3tl::convert(100.0, o3tl::Length::in1000, o3tl::Length::in10), 1));
static_assert(eq(o3tl::convert(10.0, o3tl::Length::in100, o3tl::Length::in10), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in10, o3tl::Length::in10), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::in10), 10));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::in10), 120));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::in10), 633600));
static_assert(eq(o3tl::convert(288.0, o3tl::Length::master, o3tl::Length::in10), 5));
static_assert(eq(o3tl::convert(48.0, o3tl::Length::px, o3tl::Length::in10), 5));
static_assert(eq(o3tl::convert(24.0, o3tl::Length::ch, o3tl::Length::in10), 35));
static_assert(eq(o3tl::convert(6.0, o3tl::Length::line, o3tl::Length::in10), 13));

static_assert(eq(o3tl::convert(2540.0, o3tl::Length::mm100, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(254.0, o3tl::Length::mm10, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::in), 5));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::in), 50));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::in), 5000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::in), 5E6));
static_assert(eq(o3tl::convert(914400.0, o3tl::Length::emu, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(1440.0, o3tl::Length::twip, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(72.0, o3tl::Length::pt, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(6.0, o3tl::Length::pc, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(1000.0, o3tl::Length::in1000, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(100.0, o3tl::Length::in100, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(10.0, o3tl::Length::in10, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::in), 12));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::in), 63360));
static_assert(eq(o3tl::convert(576.0, o3tl::Length::master, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(96.0, o3tl::Length::px, o3tl::Length::in), 1));
static_assert(eq(o3tl::convert(48.0, o3tl::Length::ch, o3tl::Length::in), 7));
static_assert(eq(o3tl::convert(60.0, o3tl::Length::line, o3tl::Length::in), 13));

static_assert(eq(o3tl::convert(30480.0, o3tl::Length::mm100, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(3048.0, o3tl::Length::mm10, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(1524.0, o3tl::Length::mm, o3tl::Length::ft), 5));
static_assert(eq(o3tl::convert(762.0, o3tl::Length::cm, o3tl::Length::ft), 25));
static_assert(eq(o3tl::convert(381.0, o3tl::Length::m, o3tl::Length::ft), 1250));
static_assert(eq(o3tl::convert(381.0, o3tl::Length::km, o3tl::Length::ft), 1250000));
static_assert(eq(o3tl::convert(10972800.0, o3tl::Length::emu, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(17280.0, o3tl::Length::twip, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(864.0, o3tl::Length::pt, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(72.0, o3tl::Length::pc, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(12000.0, o3tl::Length::in1000, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(1200.0, o3tl::Length::in100, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(120.0, o3tl::Length::in10, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(12.0, o3tl::Length::in, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::ft), 5280));
static_assert(eq(o3tl::convert(6912.0, o3tl::Length::master, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(1152.0, o3tl::Length::px, o3tl::Length::ft), 1));
static_assert(eq(o3tl::convert(576.0, o3tl::Length::ch, o3tl::Length::ft), 7));
static_assert(eq(o3tl::convert(720.0, o3tl::Length::line, o3tl::Length::ft), 13));

static_assert(eq(o3tl::convert(160934400.0, o3tl::Length::mm100, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(16093440.0, o3tl::Length::mm10, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(1609344.0, o3tl::Length::mm, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(804672.0, o3tl::Length::cm, o3tl::Length::mi), 5));
static_assert(eq(o3tl::convert(201168.0, o3tl::Length::m, o3tl::Length::mi), 125));
static_assert(eq(o3tl::convert(25146.0, o3tl::Length::km, o3tl::Length::mi), 15625));
static_assert(eq(o3tl::convert(57936384000.0, o3tl::Length::emu, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(91238400.0, o3tl::Length::twip, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(4561920.0, o3tl::Length::pt, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(380160.0, o3tl::Length::pc, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(63360000.0, o3tl::Length::in1000, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(6336000.0, o3tl::Length::in100, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(633600.0, o3tl::Length::in10, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(63360.0, o3tl::Length::in, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(5280.0, o3tl::Length::ft, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(36495360.0, o3tl::Length::master, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(6082560.0, o3tl::Length::px, o3tl::Length::mi), 1));
static_assert(eq(o3tl::convert(3041280.0, o3tl::Length::ch, o3tl::Length::mi), 7));
static_assert(eq(o3tl::convert(3801600.0, o3tl::Length::line, o3tl::Length::mi), 13));

static_assert(eq(o3tl::convert(635.0, o3tl::Length::mm100, o3tl::Length::master), 144));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::master), 288));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::master), 2880));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::master), 28800));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::master), 2880000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::master), 2880E6));
static_assert(eq(o3tl::convert(3175.0, o3tl::Length::emu, o3tl::Length::master), 2));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::twip, o3tl::Length::master), 2));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pt, o3tl::Length::master), 8));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pc, o3tl::Length::master), 96));
static_assert(eq(o3tl::convert(125.0, o3tl::Length::in1000, o3tl::Length::master), 72));
static_assert(eq(o3tl::convert(25.0, o3tl::Length::in100, o3tl::Length::master), 144));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in10, o3tl::Length::master), 288));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::master), 576));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::master), 6912));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::master), 36495360));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::master, o3tl::Length::master), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::px, o3tl::Length::master), 6));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ch, o3tl::Length::master), 84));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::line, o3tl::Length::master), 624));

static_assert(eq(o3tl::convert(635.0, o3tl::Length::mm100, o3tl::Length::px), 24));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm10, o3tl::Length::px), 48));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::mm, o3tl::Length::px), 480));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::cm, o3tl::Length::px), 4800));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::m, o3tl::Length::px), 480000));
static_assert(eq(o3tl::convert(127.0, o3tl::Length::km, o3tl::Length::px), 480E6));
static_assert(eq(o3tl::convert(9525.0, o3tl::Length::emu, o3tl::Length::px), 1));
static_assert(eq(o3tl::convert(15.0, o3tl::Length::twip, o3tl::Length::px), 1));
static_assert(eq(o3tl::convert(3.0, o3tl::Length::pt, o3tl::Length::px), 4));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::pc, o3tl::Length::px), 16));
static_assert(eq(o3tl::convert(125.0, o3tl::Length::in1000, o3tl::Length::px), 12));
static_assert(eq(o3tl::convert(25.0, o3tl::Length::in100, o3tl::Length::px), 24));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::in10, o3tl::Length::px), 48));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::in, o3tl::Length::px), 96));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ft, o3tl::Length::px), 1152));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::mi, o3tl::Length::px), 6082560));
static_assert(eq(o3tl::convert(6.0, o3tl::Length::master, o3tl::Length::px), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::px, o3tl::Length::px), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ch, o3tl::Length::px), 14));
static_assert(eq(o3tl::convert(5.0, o3tl::Length::line, o3tl::Length::px), 104));

static_assert(eq(o3tl::convert(4445.0, o3tl::Length::mm100, o3tl::Length::ch), 12));
static_assert(eq(o3tl::convert(889.0, o3tl::Length::mm10, o3tl::Length::ch), 24));
static_assert(eq(o3tl::convert(889.0, o3tl::Length::mm, o3tl::Length::ch), 240));
static_assert(eq(o3tl::convert(889.0, o3tl::Length::cm, o3tl::Length::ch), 2400));
static_assert(eq(o3tl::convert(889.0, o3tl::Length::m, o3tl::Length::ch), 240000));
static_assert(eq(o3tl::convert(889.0, o3tl::Length::km, o3tl::Length::ch), 240E6));
static_assert(eq(o3tl::convert(133350.0, o3tl::Length::emu, o3tl::Length::ch), 1));
static_assert(eq(o3tl::convert(210.0, o3tl::Length::twip, o3tl::Length::ch), 1));
static_assert(eq(o3tl::convert(21.0, o3tl::Length::pt, o3tl::Length::ch), 2));
static_assert(eq(o3tl::convert(7.0, o3tl::Length::pc, o3tl::Length::ch), 8));
static_assert(eq(o3tl::convert(875.0, o3tl::Length::in1000, o3tl::Length::ch), 6));
static_assert(eq(o3tl::convert(175.0, o3tl::Length::in100, o3tl::Length::ch), 12));
static_assert(eq(o3tl::convert(35.0, o3tl::Length::in10, o3tl::Length::ch), 24));
static_assert(eq(o3tl::convert(7.0, o3tl::Length::in, o3tl::Length::ch), 48));
static_assert(eq(o3tl::convert(7.0, o3tl::Length::ft, o3tl::Length::ch), 576));
static_assert(eq(o3tl::convert(7.0, o3tl::Length::mi, o3tl::Length::ch), 3041280));
static_assert(eq(o3tl::convert(84.0, o3tl::Length::master, o3tl::Length::ch), 1));
static_assert(eq(o3tl::convert(14.0, o3tl::Length::px, o3tl::Length::ch), 1));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::ch, o3tl::Length::ch), 1));
static_assert(eq(o3tl::convert(35.0, o3tl::Length::line, o3tl::Length::ch), 52));

static_assert(eq(o3tl::convert(1651.0, o3tl::Length::mm100, o3tl::Length::line), 3));
static_assert(eq(o3tl::convert(1651.0, o3tl::Length::mm10, o3tl::Length::line), 30));
static_assert(eq(o3tl::convert(1651.0, o3tl::Length::mm, o3tl::Length::line), 300));
static_assert(eq(o3tl::convert(1651.0, o3tl::Length::cm, o3tl::Length::line), 3000));
static_assert(eq(o3tl::convert(1651.0, o3tl::Length::m, o3tl::Length::line), 300000));
static_assert(eq(o3tl::convert(1651.0, o3tl::Length::km, o3tl::Length::line), 300E6));
static_assert(eq(o3tl::convert(198120.0, o3tl::Length::emu, o3tl::Length::line), 1));
static_assert(eq(o3tl::convert(312.0, o3tl::Length::twip, o3tl::Length::line), 1));
static_assert(eq(o3tl::convert(78.0, o3tl::Length::pt, o3tl::Length::line), 5));
static_assert(eq(o3tl::convert(13.0, o3tl::Length::pc, o3tl::Length::line), 10));
static_assert(eq(o3tl::convert(650.0, o3tl::Length::in1000, o3tl::Length::line), 3));
static_assert(eq(o3tl::convert(65.0, o3tl::Length::in100, o3tl::Length::line), 3));
static_assert(eq(o3tl::convert(13.0, o3tl::Length::in10, o3tl::Length::line), 6));
static_assert(eq(o3tl::convert(13.0, o3tl::Length::in, o3tl::Length::line), 60));
static_assert(eq(o3tl::convert(13.0, o3tl::Length::ft, o3tl::Length::line), 720));
static_assert(eq(o3tl::convert(13.0, o3tl::Length::mi, o3tl::Length::line), 3801600));
static_assert(eq(o3tl::convert(624.0, o3tl::Length::master, o3tl::Length::line), 5));
static_assert(eq(o3tl::convert(104.0, o3tl::Length::px, o3tl::Length::line), 5));
static_assert(eq(o3tl::convert(52.0, o3tl::Length::ch, o3tl::Length::line), 35));
static_assert(eq(o3tl::convert(1.0, o3tl::Length::line, o3tl::Length::line), 1));

// testing integral conversion

static_assert(o3tl::convert(100, o3tl::Length::mm100, o3tl::Length::mm100) == 100);
static_assert(o3tl::convert(100, o3tl::Length::mm10, o3tl::Length::mm100) == 1000);
static_assert(o3tl::convert(100, o3tl::Length::mm, o3tl::Length::mm100) == 10000);
static_assert(o3tl::convert(100, o3tl::Length::cm, o3tl::Length::mm100) == 100000);
static_assert(o3tl::convert(100, o3tl::Length::m, o3tl::Length::mm100) == 10000000);
static_assert(o3tl::convert(100, o3tl::Length::km, o3tl::Length::mm100) == 10000000000);
static_assert(o3tl::convert(36000, o3tl::Length::emu, o3tl::Length::mm100) == 100);
static_assert(o3tl::convert(7200, o3tl::Length::twip, o3tl::Length::mm100) == 12700);
static_assert(o3tl::convert(1800, o3tl::Length::pt, o3tl::Length::mm100) == 63500);
static_assert(o3tl::convert(300, o3tl::Length::pc, o3tl::Length::mm100) == 127000);
static_assert(o3tl::convert(5000, o3tl::Length::in1000, o3tl::Length::mm100) == 12700);
static_assert(o3tl::convert(500, o3tl::Length::in100, o3tl::Length::mm100) == 12700);
static_assert(o3tl::convert(100, o3tl::Length::in10, o3tl::Length::mm100) == 25400);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::mm100) == 254000);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::mm100) == 3048000);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::mm100) == 16093440000);
static_assert(o3tl::convert(14400, o3tl::Length::master, o3tl::Length::mm100) == 63500);
static_assert(o3tl::convert(2400, o3tl::Length::px, o3tl::Length::mm100) == 63500);
static_assert(o3tl::convert(1200, o3tl::Length::ch, o3tl::Length::mm100) == 444500);
static_assert(o3tl::convert(300, o3tl::Length::line, o3tl::Length::mm100) == 165100);

static_assert(o3tl::convert(1000, o3tl::Length::mm100, o3tl::Length::mm10) == 100);
static_assert(o3tl::convert(100, o3tl::Length::mm10, o3tl::Length::mm10) == 100);
static_assert(o3tl::convert(100, o3tl::Length::mm, o3tl::Length::mm10) == 1000);
static_assert(o3tl::convert(100, o3tl::Length::cm, o3tl::Length::mm10) == 10000);
static_assert(o3tl::convert(100, o3tl::Length::m, o3tl::Length::mm10) == 1000000);
static_assert(o3tl::convert(100, o3tl::Length::km, o3tl::Length::mm10) == 1000000000);
static_assert(o3tl::convert(360000, o3tl::Length::emu, o3tl::Length::mm10) == 100);
static_assert(o3tl::convert(72000, o3tl::Length::twip, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(3600, o3tl::Length::pt, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(300, o3tl::Length::pc, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(50000, o3tl::Length::in1000, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(5000, o3tl::Length::in100, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(500, o3tl::Length::in10, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::mm10) == 25400);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::mm10) == 304800);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::mm10) == 1609344000);
static_assert(o3tl::convert(28800, o3tl::Length::master, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(4800, o3tl::Length::px, o3tl::Length::mm10) == 12700);
static_assert(o3tl::convert(2400, o3tl::Length::ch, o3tl::Length::mm10) == 88900);
static_assert(o3tl::convert(3000, o3tl::Length::line, o3tl::Length::mm10) == 165100);

static_assert(o3tl::convert(10000, o3tl::Length::mm100, o3tl::Length::mm) == 100);
static_assert(o3tl::convert(1000, o3tl::Length::mm10, o3tl::Length::mm) == 100);
static_assert(o3tl::convert(100, o3tl::Length::mm, o3tl::Length::mm) == 100);
static_assert(o3tl::convert(100, o3tl::Length::cm, o3tl::Length::mm) == 1000);
static_assert(o3tl::convert(100, o3tl::Length::m, o3tl::Length::mm) == 100000);
static_assert(o3tl::convert(100, o3tl::Length::km, o3tl::Length::mm) == 100000000);
static_assert(o3tl::convert(3600000, o3tl::Length::emu, o3tl::Length::mm) == 100);
static_assert(o3tl::convert(720000, o3tl::Length::twip, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(36000, o3tl::Length::pt, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(3000, o3tl::Length::pc, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(500000, o3tl::Length::in1000, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(50000, o3tl::Length::in100, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(5000, o3tl::Length::in10, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(500, o3tl::Length::in, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(500, o3tl::Length::ft, o3tl::Length::mm) == 152400);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::mm) == 160934400);
static_assert(o3tl::convert(288000, o3tl::Length::master, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(48000, o3tl::Length::px, o3tl::Length::mm) == 12700);
static_assert(o3tl::convert(24000, o3tl::Length::ch, o3tl::Length::mm) == 88900);
static_assert(o3tl::convert(30000, o3tl::Length::line, o3tl::Length::mm) == 165100);

static_assert(o3tl::convert(100000, o3tl::Length::mm100, o3tl::Length::cm) == 100);
static_assert(o3tl::convert(10000, o3tl::Length::mm10, o3tl::Length::cm) == 100);
static_assert(o3tl::convert(1000, o3tl::Length::mm, o3tl::Length::cm) == 100);
static_assert(o3tl::convert(100, o3tl::Length::cm, o3tl::Length::cm) == 100);
static_assert(o3tl::convert(100, o3tl::Length::m, o3tl::Length::cm) == 10000);
static_assert(o3tl::convert(100, o3tl::Length::km, o3tl::Length::cm) == 10000000);
static_assert(o3tl::convert(36000000, o3tl::Length::emu, o3tl::Length::cm) == 100);
static_assert(o3tl::convert(7200000, o3tl::Length::twip, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(360000, o3tl::Length::pt, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(30000, o3tl::Length::pc, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(5000000, o3tl::Length::in1000, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(500000, o3tl::Length::in100, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(50000, o3tl::Length::in10, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(5000, o3tl::Length::in, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(2500, o3tl::Length::ft, o3tl::Length::cm) == 76200);
static_assert(o3tl::convert(500, o3tl::Length::mi, o3tl::Length::cm) == 80467200);
static_assert(o3tl::convert(2880000, o3tl::Length::master, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(480000, o3tl::Length::px, o3tl::Length::cm) == 12700);
static_assert(o3tl::convert(240000, o3tl::Length::ch, o3tl::Length::cm) == 88900);
static_assert(o3tl::convert(300000, o3tl::Length::line, o3tl::Length::cm) == 165100);

static_assert(o3tl::convert(10000000, o3tl::Length::mm100, o3tl::Length::m) == 100);
static_assert(o3tl::convert(1000000, o3tl::Length::mm10, o3tl::Length::m) == 100);
static_assert(o3tl::convert(100000, o3tl::Length::mm, o3tl::Length::m) == 100);
static_assert(o3tl::convert(10000, o3tl::Length::cm, o3tl::Length::m) == 100);
static_assert(o3tl::convert(100, o3tl::Length::m, o3tl::Length::m) == 100);
static_assert(o3tl::convert(100, o3tl::Length::km, o3tl::Length::m) == 100000);
static_assert(o3tl::convert(3600000000, o3tl::Length::emu, o3tl::Length::m) == 100);
static_assert(o3tl::convert(720000000, o3tl::Length::twip, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(36000000, o3tl::Length::pt, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(3000000, o3tl::Length::pc, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(500000000, o3tl::Length::in1000, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(50000000, o3tl::Length::in100, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(5000000, o3tl::Length::in10, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(500000, o3tl::Length::in, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(125000, o3tl::Length::ft, o3tl::Length::m) == 38100);
static_assert(o3tl::convert(12500, o3tl::Length::mi, o3tl::Length::m) == 20116800);
static_assert(o3tl::convert(288000000, o3tl::Length::master, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(48000000, o3tl::Length::px, o3tl::Length::m) == 12700);
static_assert(o3tl::convert(24000000, o3tl::Length::ch, o3tl::Length::m) == 88900);
static_assert(o3tl::convert(30000000, o3tl::Length::line, o3tl::Length::m) == 165100);

static_assert(o3tl::convert(10000000000, o3tl::Length::mm100, o3tl::Length::km) == 100);
static_assert(o3tl::convert(1000000000, o3tl::Length::mm10, o3tl::Length::km) == 100);
static_assert(o3tl::convert(100000000, o3tl::Length::mm, o3tl::Length::km) == 100);
static_assert(o3tl::convert(10000000, o3tl::Length::cm, o3tl::Length::km) == 100);
static_assert(o3tl::convert(100000, o3tl::Length::m, o3tl::Length::km) == 100);
static_assert(o3tl::convert(100, o3tl::Length::km, o3tl::Length::km) == 100);
static_assert(o3tl::convert(3600000000000, o3tl::Length::emu, o3tl::Length::km) == 100);
static_assert(o3tl::convert(720000000000, o3tl::Length::twip, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(36000000000, o3tl::Length::pt, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(3000000000, o3tl::Length::pc, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(500000000000, o3tl::Length::in1000, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(50000000000, o3tl::Length::in100, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(5000000000, o3tl::Length::in10, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(500000000, o3tl::Length::in, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(125000000, o3tl::Length::ft, o3tl::Length::km) == 38100);
static_assert(o3tl::convert(1562500, o3tl::Length::mi, o3tl::Length::km) == 2514600);
static_assert(o3tl::convert(288000000000, o3tl::Length::master, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(48000000000, o3tl::Length::px, o3tl::Length::km) == 12700);
static_assert(o3tl::convert(24000000000, o3tl::Length::ch, o3tl::Length::km) == 88900);
static_assert(o3tl::convert(30000000000, o3tl::Length::line, o3tl::Length::km) == 165100);

static_assert(o3tl::convert(100, o3tl::Length::mm100, o3tl::Length::emu) == 36000);
static_assert(o3tl::convert(100, o3tl::Length::mm10, o3tl::Length::emu) == 360000);
static_assert(o3tl::convert(100, o3tl::Length::mm, o3tl::Length::emu) == 3600000);
static_assert(o3tl::convert(100, o3tl::Length::cm, o3tl::Length::emu) == 36000000);
static_assert(o3tl::convert(100, o3tl::Length::m, o3tl::Length::emu) == 3600000000);
static_assert(o3tl::convert(100, o3tl::Length::km, o3tl::Length::emu) == 3600000000000);
static_assert(o3tl::convert(100, o3tl::Length::emu, o3tl::Length::emu) == 100);
static_assert(o3tl::convert(100, o3tl::Length::twip, o3tl::Length::emu) == 63500);
static_assert(o3tl::convert(100, o3tl::Length::pt, o3tl::Length::emu) == 1270000);
static_assert(o3tl::convert(100, o3tl::Length::pc, o3tl::Length::emu) == 15240000);
static_assert(o3tl::convert(500, o3tl::Length::in1000, o3tl::Length::emu) == 457200);
static_assert(o3tl::convert(100, o3tl::Length::in100, o3tl::Length::emu) == 914400);
static_assert(o3tl::convert(100, o3tl::Length::in10, o3tl::Length::emu) == 9144000);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::emu) == 91440000);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::emu) == 1097280000);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::emu) == 5793638400000);
static_assert(o3tl::convert(200, o3tl::Length::master, o3tl::Length::emu) == 317500);
static_assert(o3tl::convert(100, o3tl::Length::px, o3tl::Length::emu) == 952500);
static_assert(o3tl::convert(100, o3tl::Length::ch, o3tl::Length::emu) == 13335000);
static_assert(o3tl::convert(100, o3tl::Length::line, o3tl::Length::emu) == 19812000);

static_assert(o3tl::convert(12700, o3tl::Length::mm100, o3tl::Length::twip) == 7200);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::twip) == 72000);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::twip) == 720000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::twip) == 7200000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::twip) == 720000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::twip) == 720000000000);
static_assert(o3tl::convert(63500, o3tl::Length::emu, o3tl::Length::twip) == 100);
static_assert(o3tl::convert(100, o3tl::Length::twip, o3tl::Length::twip) == 100);
static_assert(o3tl::convert(100, o3tl::Length::pt, o3tl::Length::twip) == 2000);
static_assert(o3tl::convert(100, o3tl::Length::pc, o3tl::Length::twip) == 24000);
static_assert(o3tl::convert(2500, o3tl::Length::in1000, o3tl::Length::twip) == 3600);
static_assert(o3tl::convert(500, o3tl::Length::in100, o3tl::Length::twip) == 7200);
static_assert(o3tl::convert(100, o3tl::Length::in10, o3tl::Length::twip) == 14400);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::twip) == 144000);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::twip) == 1728000);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::twip) == 9123840000);
static_assert(o3tl::convert(200, o3tl::Length::master, o3tl::Length::twip) == 500);
static_assert(o3tl::convert(100, o3tl::Length::px, o3tl::Length::twip) == 1500);
static_assert(o3tl::convert(100, o3tl::Length::ch, o3tl::Length::twip) == 21000);
static_assert(o3tl::convert(100, o3tl::Length::line, o3tl::Length::twip) == 31200);

static_assert(o3tl::convert(63500, o3tl::Length::mm100, o3tl::Length::pt) == 1800);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::pt) == 3600);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::pt) == 36000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::pt) == 360000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::pt) == 36000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::pt) == 36000000000);
static_assert(o3tl::convert(1270000, o3tl::Length::emu, o3tl::Length::pt) == 100);
static_assert(o3tl::convert(2000, o3tl::Length::twip, o3tl::Length::pt) == 100);
static_assert(o3tl::convert(100, o3tl::Length::pt, o3tl::Length::pt) == 100);
static_assert(o3tl::convert(100, o3tl::Length::pc, o3tl::Length::pt) == 1200);
static_assert(o3tl::convert(12500, o3tl::Length::in1000, o3tl::Length::pt) == 900);
static_assert(o3tl::convert(2500, o3tl::Length::in100, o3tl::Length::pt) == 1800);
static_assert(o3tl::convert(500, o3tl::Length::in10, o3tl::Length::pt) == 3600);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::pt) == 7200);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::pt) == 86400);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::pt) == 456192000);
static_assert(o3tl::convert(800, o3tl::Length::master, o3tl::Length::pt) == 100);
static_assert(o3tl::convert(400, o3tl::Length::px, o3tl::Length::pt) == 300);
static_assert(o3tl::convert(200, o3tl::Length::ch, o3tl::Length::pt) == 2100);
static_assert(o3tl::convert(500, o3tl::Length::line, o3tl::Length::pt) == 7800);

static_assert(o3tl::convert(127000, o3tl::Length::mm100, o3tl::Length::pc) == 300);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::pc) == 300);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::pc) == 3000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::pc) == 30000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::pc) == 3000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::pc) == 3000000000);
static_assert(o3tl::convert(15240000, o3tl::Length::emu, o3tl::Length::pc) == 100);
static_assert(o3tl::convert(24000, o3tl::Length::twip, o3tl::Length::pc) == 100);
static_assert(o3tl::convert(1200, o3tl::Length::pt, o3tl::Length::pc) == 100);
static_assert(o3tl::convert(100, o3tl::Length::pc, o3tl::Length::pc) == 100);
static_assert(o3tl::convert(50000, o3tl::Length::in1000, o3tl::Length::pc) == 300);
static_assert(o3tl::convert(5000, o3tl::Length::in100, o3tl::Length::pc) == 300);
static_assert(o3tl::convert(500, o3tl::Length::in10, o3tl::Length::pc) == 300);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::pc) == 600);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::pc) == 7200);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::pc) == 38016000);
static_assert(o3tl::convert(9600, o3tl::Length::master, o3tl::Length::pc) == 100);
static_assert(o3tl::convert(1600, o3tl::Length::px, o3tl::Length::pc) == 100);
static_assert(o3tl::convert(800, o3tl::Length::ch, o3tl::Length::pc) == 700);
static_assert(o3tl::convert(1000, o3tl::Length::line, o3tl::Length::pc) == 1300);

static_assert(o3tl::convert(12700, o3tl::Length::mm100, o3tl::Length::in1000) == 5000);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::in1000) == 50000);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::in1000) == 500000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::in1000) == 5000000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::in1000) == 500000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::in1000) == 500000000000);
static_assert(o3tl::convert(457200, o3tl::Length::emu, o3tl::Length::in1000) == 500);
static_assert(o3tl::convert(3600, o3tl::Length::twip, o3tl::Length::in1000) == 2500);
static_assert(o3tl::convert(900, o3tl::Length::pt, o3tl::Length::in1000) == 12500);
static_assert(o3tl::convert(300, o3tl::Length::pc, o3tl::Length::in1000) == 50000);
static_assert(o3tl::convert(100, o3tl::Length::in1000, o3tl::Length::in1000) == 100);
static_assert(o3tl::convert(100, o3tl::Length::in100, o3tl::Length::in1000) == 1000);
static_assert(o3tl::convert(100, o3tl::Length::in10, o3tl::Length::in1000) == 10000);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::in1000) == 100000);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::in1000) == 1200000);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::in1000) == 6336000000);
static_assert(o3tl::convert(7200, o3tl::Length::master, o3tl::Length::in1000) == 12500);
static_assert(o3tl::convert(1200, o3tl::Length::px, o3tl::Length::in1000) == 12500);
static_assert(o3tl::convert(600, o3tl::Length::ch, o3tl::Length::in1000) == 87500);
static_assert(o3tl::convert(300, o3tl::Length::line, o3tl::Length::in1000) == 65000);

static_assert(o3tl::convert(12700, o3tl::Length::mm100, o3tl::Length::in100) == 500);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::in100) == 5000);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::in100) == 50000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::in100) == 500000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::in100) == 50000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::in100) == 50000000000);
static_assert(o3tl::convert(914400, o3tl::Length::emu, o3tl::Length::in100) == 100);
static_assert(o3tl::convert(7200, o3tl::Length::twip, o3tl::Length::in100) == 500);
static_assert(o3tl::convert(1800, o3tl::Length::pt, o3tl::Length::in100) == 2500);
static_assert(o3tl::convert(300, o3tl::Length::pc, o3tl::Length::in100) == 5000);
static_assert(o3tl::convert(1000, o3tl::Length::in1000, o3tl::Length::in100) == 100);
static_assert(o3tl::convert(100, o3tl::Length::in100, o3tl::Length::in100) == 100);
static_assert(o3tl::convert(100, o3tl::Length::in10, o3tl::Length::in100) == 1000);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::in100) == 10000);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::in100) == 120000);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::in100) == 633600000);
static_assert(o3tl::convert(14400, o3tl::Length::master, o3tl::Length::in100) == 2500);
static_assert(o3tl::convert(2400, o3tl::Length::px, o3tl::Length::in100) == 2500);
static_assert(o3tl::convert(1200, o3tl::Length::ch, o3tl::Length::in100) == 17500);
static_assert(o3tl::convert(300, o3tl::Length::line, o3tl::Length::in100) == 6500);

static_assert(o3tl::convert(25400, o3tl::Length::mm100, o3tl::Length::in10) == 100);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::in10) == 500);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::in10) == 5000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::in10) == 50000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::in10) == 5000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::in10) == 5000000000);
static_assert(o3tl::convert(9144000, o3tl::Length::emu, o3tl::Length::in10) == 100);
static_assert(o3tl::convert(14400, o3tl::Length::twip, o3tl::Length::in10) == 100);
static_assert(o3tl::convert(3600, o3tl::Length::pt, o3tl::Length::in10) == 500);
static_assert(o3tl::convert(300, o3tl::Length::pc, o3tl::Length::in10) == 500);
static_assert(o3tl::convert(10000, o3tl::Length::in1000, o3tl::Length::in10) == 100);
static_assert(o3tl::convert(1000, o3tl::Length::in100, o3tl::Length::in10) == 100);
static_assert(o3tl::convert(100, o3tl::Length::in10, o3tl::Length::in10) == 100);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::in10) == 1000);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::in10) == 12000);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::in10) == 63360000);
static_assert(o3tl::convert(28800, o3tl::Length::master, o3tl::Length::in10) == 500);
static_assert(o3tl::convert(4800, o3tl::Length::px, o3tl::Length::in10) == 500);
static_assert(o3tl::convert(2400, o3tl::Length::ch, o3tl::Length::in10) == 3500);
static_assert(o3tl::convert(600, o3tl::Length::line, o3tl::Length::in10) == 1300);

static_assert(o3tl::convert(254000, o3tl::Length::mm100, o3tl::Length::in) == 100);
static_assert(o3tl::convert(25400, o3tl::Length::mm10, o3tl::Length::in) == 100);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::in) == 500);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::in) == 5000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::in) == 500000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::in) == 500000000);
static_assert(o3tl::convert(91440000, o3tl::Length::emu, o3tl::Length::in) == 100);
static_assert(o3tl::convert(144000, o3tl::Length::twip, o3tl::Length::in) == 100);
static_assert(o3tl::convert(7200, o3tl::Length::pt, o3tl::Length::in) == 100);
static_assert(o3tl::convert(600, o3tl::Length::pc, o3tl::Length::in) == 100);
static_assert(o3tl::convert(100000, o3tl::Length::in1000, o3tl::Length::in) == 100);
static_assert(o3tl::convert(10000, o3tl::Length::in100, o3tl::Length::in) == 100);
static_assert(o3tl::convert(1000, o3tl::Length::in10, o3tl::Length::in) == 100);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::in) == 100);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::in) == 1200);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::in) == 6336000);
static_assert(o3tl::convert(57600, o3tl::Length::master, o3tl::Length::in) == 100);
static_assert(o3tl::convert(9600, o3tl::Length::px, o3tl::Length::in) == 100);
static_assert(o3tl::convert(4800, o3tl::Length::ch, o3tl::Length::in) == 700);
static_assert(o3tl::convert(6000, o3tl::Length::line, o3tl::Length::in) == 1300);

static_assert(o3tl::convert(3048000, o3tl::Length::mm100, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(304800, o3tl::Length::mm10, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(152400, o3tl::Length::mm, o3tl::Length::ft) == 500);
static_assert(o3tl::convert(76200, o3tl::Length::cm, o3tl::Length::ft) == 2500);
static_assert(o3tl::convert(38100, o3tl::Length::m, o3tl::Length::ft) == 125000);
static_assert(o3tl::convert(38100, o3tl::Length::km, o3tl::Length::ft) == 125000000);
static_assert(o3tl::convert(1097280000, o3tl::Length::emu, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(1728000, o3tl::Length::twip, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(86400, o3tl::Length::pt, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(7200, o3tl::Length::pc, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(1200000, o3tl::Length::in1000, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(120000, o3tl::Length::in100, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(12000, o3tl::Length::in10, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(1200, o3tl::Length::in, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::ft) == 528000);
static_assert(o3tl::convert(691200, o3tl::Length::master, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(115200, o3tl::Length::px, o3tl::Length::ft) == 100);
static_assert(o3tl::convert(57600, o3tl::Length::ch, o3tl::Length::ft) == 700);
static_assert(o3tl::convert(72000, o3tl::Length::line, o3tl::Length::ft) == 1300);

static_assert(o3tl::convert(16093440000, o3tl::Length::mm100, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(1609344000, o3tl::Length::mm10, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(160934400, o3tl::Length::mm, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(80467200, o3tl::Length::cm, o3tl::Length::mi) == 500);
static_assert(o3tl::convert(20116800, o3tl::Length::m, o3tl::Length::mi) == 12500);
static_assert(o3tl::convert(2514600, o3tl::Length::km, o3tl::Length::mi) == 1562500);
static_assert(o3tl::convert(5793638400000, o3tl::Length::emu, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(9123840000, o3tl::Length::twip, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(456192000, o3tl::Length::pt, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(38016000, o3tl::Length::pc, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(6336000000, o3tl::Length::in1000, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(633600000, o3tl::Length::in100, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(63360000, o3tl::Length::in10, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(6336000, o3tl::Length::in, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(528000, o3tl::Length::ft, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(3649536000, o3tl::Length::master, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(608256000, o3tl::Length::px, o3tl::Length::mi) == 100);
static_assert(o3tl::convert(304128000, o3tl::Length::ch, o3tl::Length::mi) == 700);
static_assert(o3tl::convert(380160000, o3tl::Length::line, o3tl::Length::mi) == 1300);

static_assert(o3tl::convert(63500, o3tl::Length::mm100, o3tl::Length::master) == 14400);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::master) == 28800);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::master) == 288000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::master) == 2880000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::master) == 288000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::master) == 288000000000);
static_assert(o3tl::convert(317500, o3tl::Length::emu, o3tl::Length::master) == 200);
static_assert(o3tl::convert(500, o3tl::Length::twip, o3tl::Length::master) == 200);
static_assert(o3tl::convert(100, o3tl::Length::pt, o3tl::Length::master) == 800);
static_assert(o3tl::convert(100, o3tl::Length::pc, o3tl::Length::master) == 9600);
static_assert(o3tl::convert(12500, o3tl::Length::in1000, o3tl::Length::master) == 7200);
static_assert(o3tl::convert(2500, o3tl::Length::in100, o3tl::Length::master) == 14400);
static_assert(o3tl::convert(500, o3tl::Length::in10, o3tl::Length::master) == 28800);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::master) == 57600);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::master) == 691200);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::master) == 3649536000);
static_assert(o3tl::convert(100, o3tl::Length::master, o3tl::Length::master) == 100);
static_assert(o3tl::convert(100, o3tl::Length::px, o3tl::Length::master) == 600);
static_assert(o3tl::convert(100, o3tl::Length::ch, o3tl::Length::master) == 8400);
static_assert(o3tl::convert(500, o3tl::Length::line, o3tl::Length::master) == 62400);

static_assert(o3tl::convert(63500, o3tl::Length::mm100, o3tl::Length::px) == 2400);
static_assert(o3tl::convert(12700, o3tl::Length::mm10, o3tl::Length::px) == 4800);
static_assert(o3tl::convert(12700, o3tl::Length::mm, o3tl::Length::px) == 48000);
static_assert(o3tl::convert(12700, o3tl::Length::cm, o3tl::Length::px) == 480000);
static_assert(o3tl::convert(12700, o3tl::Length::m, o3tl::Length::px) == 48000000);
static_assert(o3tl::convert(12700, o3tl::Length::km, o3tl::Length::px) == 48000000000);
static_assert(o3tl::convert(952500, o3tl::Length::emu, o3tl::Length::px) == 100);
static_assert(o3tl::convert(1500, o3tl::Length::twip, o3tl::Length::px) == 100);
static_assert(o3tl::convert(300, o3tl::Length::pt, o3tl::Length::px) == 400);
static_assert(o3tl::convert(100, o3tl::Length::pc, o3tl::Length::px) == 1600);
static_assert(o3tl::convert(12500, o3tl::Length::in1000, o3tl::Length::px) == 1200);
static_assert(o3tl::convert(2500, o3tl::Length::in100, o3tl::Length::px) == 2400);
static_assert(o3tl::convert(500, o3tl::Length::in10, o3tl::Length::px) == 4800);
static_assert(o3tl::convert(100, o3tl::Length::in, o3tl::Length::px) == 9600);
static_assert(o3tl::convert(100, o3tl::Length::ft, o3tl::Length::px) == 115200);
static_assert(o3tl::convert(100, o3tl::Length::mi, o3tl::Length::px) == 608256000);
static_assert(o3tl::convert(600, o3tl::Length::master, o3tl::Length::px) == 100);
static_assert(o3tl::convert(100, o3tl::Length::px, o3tl::Length::px) == 100);
static_assert(o3tl::convert(100, o3tl::Length::ch, o3tl::Length::px) == 1400);
static_assert(o3tl::convert(500, o3tl::Length::line, o3tl::Length::px) == 10400);

static_assert(o3tl::convert(444500, o3tl::Length::mm100, o3tl::Length::ch) == 1200);
static_assert(o3tl::convert(88900, o3tl::Length::mm10, o3tl::Length::ch) == 2400);
static_assert(o3tl::convert(88900, o3tl::Length::mm, o3tl::Length::ch) == 24000);
static_assert(o3tl::convert(88900, o3tl::Length::cm, o3tl::Length::ch) == 240000);
static_assert(o3tl::convert(88900, o3tl::Length::m, o3tl::Length::ch) == 24000000);
static_assert(o3tl::convert(88900, o3tl::Length::km, o3tl::Length::ch) == 24000000000);
static_assert(o3tl::convert(13335000, o3tl::Length::emu, o3tl::Length::ch) == 100);
static_assert(o3tl::convert(21000, o3tl::Length::twip, o3tl::Length::ch) == 100);
static_assert(o3tl::convert(2100, o3tl::Length::pt, o3tl::Length::ch) == 200);
static_assert(o3tl::convert(700, o3tl::Length::pc, o3tl::Length::ch) == 800);
static_assert(o3tl::convert(87500, o3tl::Length::in1000, o3tl::Length::ch) == 600);
static_assert(o3tl::convert(17500, o3tl::Length::in100, o3tl::Length::ch) == 1200);
static_assert(o3tl::convert(3500, o3tl::Length::in10, o3tl::Length::ch) == 2400);
static_assert(o3tl::convert(700, o3tl::Length::in, o3tl::Length::ch) == 4800);
static_assert(o3tl::convert(700, o3tl::Length::ft, o3tl::Length::ch) == 57600);
static_assert(o3tl::convert(700, o3tl::Length::mi, o3tl::Length::ch) == 304128000);
static_assert(o3tl::convert(8400, o3tl::Length::master, o3tl::Length::ch) == 100);
static_assert(o3tl::convert(1400, o3tl::Length::px, o3tl::Length::ch) == 100);
static_assert(o3tl::convert(100, o3tl::Length::ch, o3tl::Length::ch) == 100);
static_assert(o3tl::convert(3500, o3tl::Length::line, o3tl::Length::ch) == 5200);

static_assert(o3tl::convert(165100, o3tl::Length::mm100, o3tl::Length::line) == 300);
static_assert(o3tl::convert(165100, o3tl::Length::mm10, o3tl::Length::line) == 3000);
static_assert(o3tl::convert(165100, o3tl::Length::mm, o3tl::Length::line) == 30000);
static_assert(o3tl::convert(165100, o3tl::Length::cm, o3tl::Length::line) == 300000);
static_assert(o3tl::convert(165100, o3tl::Length::m, o3tl::Length::line) == 30000000);
static_assert(o3tl::convert(165100, o3tl::Length::km, o3tl::Length::line) == 30000000000);
static_assert(o3tl::convert(19812000, o3tl::Length::emu, o3tl::Length::line) == 100);
static_assert(o3tl::convert(31200, o3tl::Length::twip, o3tl::Length::line) == 100);
static_assert(o3tl::convert(7800, o3tl::Length::pt, o3tl::Length::line) == 500);
static_assert(o3tl::convert(1300, o3tl::Length::pc, o3tl::Length::line) == 1000);
static_assert(o3tl::convert(65000, o3tl::Length::in1000, o3tl::Length::line) == 300);
static_assert(o3tl::convert(6500, o3tl::Length::in100, o3tl::Length::line) == 300);
static_assert(o3tl::convert(1300, o3tl::Length::in10, o3tl::Length::line) == 600);
static_assert(o3tl::convert(1300, o3tl::Length::in, o3tl::Length::line) == 6000);
static_assert(o3tl::convert(1300, o3tl::Length::ft, o3tl::Length::line) == 72000);
static_assert(o3tl::convert(1300, o3tl::Length::mi, o3tl::Length::line) == 380160000);
static_assert(o3tl::convert(62400, o3tl::Length::master, o3tl::Length::line) == 500);
static_assert(o3tl::convert(10400, o3tl::Length::px, o3tl::Length::line) == 500);
static_assert(o3tl::convert(5200, o3tl::Length::ch, o3tl::Length::line) == 3500);
static_assert(o3tl::convert(100, o3tl::Length::line, o3tl::Length::line) == 100);

// Integral rounding

static_assert(o3tl::convert(49, o3tl::Length::mm100, o3tl::Length::mm) == 0);
static_assert(o3tl::convert(50, o3tl::Length::mm100, o3tl::Length::mm) == 1);

// Conversions used in the code - to make sure they produce the expected and unchanged result

static_assert(o3tl::toTwips(25, o3tl::Length::in100) == 1440 / 4);
static_assert(o3tl::toTwips(15, o3tl::Length::in100) == 216);
// the following twip value used to the constant for 20mm
static_assert(o3tl::toTwips(20, o3tl::Length::mm) == 1134);
// 847 100thmm used to represent 24pt
static_assert(o3tl::convert(24, o3tl::Length::pt, o3tl::Length::mm100) == 847);

static_assert(o3tl::convertSaturate(SAL_MAX_INT64, 72, 127) == 5228998320106644552); // no overflow

/* vim:set shiftwidth=4 softtabstop=4 expandtab cinoptions=b1,g0,N-s cinkeys+=0=break: */
