"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
/*********************************************************************
 * Copyright (c) 2025 Renesas Electronics Corporation and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
const path = require("path");
const os = require("os");
const chai_1 = require("chai");
const utils_1 = require("./utils");
// This mock adapter is overriding the getBreakpointOptions method.
const adapter = 'integration-tests/mocks/debugAdapters/dynamicBreakpointOptions.js';
const argHardwareBreakpointTrue = '--hardware-breakpoint-true';
const argHardwareBreakpointFalse = '--hardware-breakpoint-false';
const argBreakpointModeHardware = ['--breakpoint-mode', 'hardware'];
const argBreakpointModeSoftware = ['--breakpoint-mode', 'software'];
const program = 'count';
const startDebugClientWithArgs = (test, ...args) => __awaiter(void 0, void 0, void 0, function* () {
    const dc = yield (0, utils_1.standardBeforeEach)(adapter, args);
    yield dc.launchRequest((0, utils_1.fillDefaults)(test, {
        program: path.join(utils_1.testProgramsDir, 'count'),
    }));
    return dc;
});
const sendBreakpointRequest = (dc, options) => {
    return dc.setBreakpointsRequest({
        source: {
            name: `${program}.c`,
            path: path.join(utils_1.testProgramsDir, `${program}.c`),
        },
        breakpoints: [
            Object.assign({ column: 1, line: 4 }, (options !== null && options !== void 0 ? options : {})),
        ],
    });
};
const expectBreakpoint = (dc, response, breakpointMode) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    (0, chai_1.expect)(response.body.breakpoints.length).eq(1);
    (0, chai_1.expect)(response.body.breakpoints[0].verified).eq(true);
    (0, chai_1.expect)(response.body.breakpoints[0].message).eq(undefined);
    yield dc.configurationDoneRequest();
    let isCorrect;
    let outputs;
    while (!isCorrect) {
        // Cover the case of getting event in Linux environment.
        // If cannot get correct event, program timeout and test case failed.
        outputs = yield dc.waitForEvent('output');
        isCorrect = outputs.body.output.includes('breakpoint-modified');
    }
    (0, chai_1.expect)(outputs === null || outputs === void 0 ? void 0 : outputs.body.output).includes(`type="${breakpointMode === 'hardware' ? 'hw breakpoint' : 'breakpoint'}"`);
    const stoppedOutput = yield dc.waitForEvent('stopped');
    (0, chai_1.expect)((_a = stoppedOutput.body) === null || _a === void 0 ? void 0 : _a.reason).eq('breakpoint');
});
// Hardware breakpoint tests are not working, so skipped for Windows.
// For further information, please check the discussion here:
// https://github.com/eclipse-cdt-cloud/cdt-gdb-adapter/pull/350
const skipHardwareBreakpointTest = os.platform() === 'win32';
describe('breakpoint mode', () => __awaiter(void 0, void 0, void 0, function* () {
    let dc;
    afterEach(() => __awaiter(void 0, void 0, void 0, function* () {
        yield dc.stop();
    }));
    describe('with no override', () => __awaiter(void 0, void 0, void 0, function* () {
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield startDebugClientWithArgs(this.currentTest);
            });
        });
        it('insert breakpoint as hardware breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (skipHardwareBreakpointTest) {
                    this.skip();
                }
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'hardware',
                });
                yield expectBreakpoint(dc, response, 'hardware');
            });
        });
        it('insert breakpoint as software breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'software',
                });
                yield expectBreakpoint(dc, response, 'software');
            });
        });
    }));
    describe('with options hardware is true', () => __awaiter(void 0, void 0, void 0, function* () {
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield startDebugClientWithArgs(this.currentTest, argHardwareBreakpointTrue);
            });
        });
        it('when no mode - insert breakpoint as hardware breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (skipHardwareBreakpointTest) {
                    this.skip();
                }
                const response = yield sendBreakpointRequest(dc);
                yield expectBreakpoint(dc, response, 'hardware');
            });
        });
        it('when mode is hardware - insert breakpoint as hardware breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (skipHardwareBreakpointTest) {
                    this.skip();
                }
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'hardware',
                });
                yield expectBreakpoint(dc, response, 'hardware');
            });
        });
        it('when mode is software - insert breakpoint as software breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'software',
                });
                yield expectBreakpoint(dc, response, 'software');
            });
        });
    }));
    describe('with options hardware is false', () => __awaiter(void 0, void 0, void 0, function* () {
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield startDebugClientWithArgs(this.currentTest, argHardwareBreakpointFalse);
            });
        });
        it('when no mode - insert breakpoint as software breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                const response = yield sendBreakpointRequest(dc);
                yield expectBreakpoint(dc, response, 'software');
            });
        });
        it('when mode is hardware - insert breakpoint as hardware breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (skipHardwareBreakpointTest) {
                    this.skip();
                }
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'hardware',
                });
                yield expectBreakpoint(dc, response, 'hardware');
            });
        });
        it('when mode is software - insert breakpoint as software breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'software',
                });
                yield expectBreakpoint(dc, response, 'software');
            });
        });
    }));
    describe('with options mode overriden to hardware', () => __awaiter(void 0, void 0, void 0, function* () {
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield startDebugClientWithArgs(this.currentTest, ...argBreakpointModeHardware);
            });
        });
        it('with no mode in request - insert breakpoint as hardware breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (skipHardwareBreakpointTest) {
                    this.skip();
                }
                const response = yield sendBreakpointRequest(dc);
                yield expectBreakpoint(dc, response, 'hardware');
            });
        });
        it('with mode is hardware in request - insert breakpoint as hardware breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (skipHardwareBreakpointTest) {
                    this.skip();
                }
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'hardware',
                });
                yield expectBreakpoint(dc, response, 'hardware');
            });
        });
        it('with mode is software in request - still insert breakpoint as hardware breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (skipHardwareBreakpointTest) {
                    this.skip();
                }
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'software',
                });
                yield expectBreakpoint(dc, response, 'hardware');
            });
        });
    }));
    describe('with options mode overriden to software', () => __awaiter(void 0, void 0, void 0, function* () {
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield startDebugClientWithArgs(this.currentTest, ...argBreakpointModeSoftware);
            });
        });
        it('with no mode in request - insert breakpoint as software breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                const response = yield sendBreakpointRequest(dc);
                yield expectBreakpoint(dc, response, 'software');
            });
        });
        it('with mode is hardware in request - still insert breakpoint as software breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'hardware',
                });
                yield expectBreakpoint(dc, response, 'software');
            });
        });
        it('with mode is software in request - insert breakpoint as software breakpoint', function () {
            return __awaiter(this, void 0, void 0, function* () {
                const response = yield sendBreakpointRequest(dc, {
                    mode: 'software',
                });
                yield expectBreakpoint(dc, response, 'software');
            });
        });
    }));
}));
//# sourceMappingURL=breakpointModes.spec.js.map