"use strict";
/*********************************************************************
 * Copyright (c) 2019 Arm and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const chai_1 = require("chai");
const path_1 = require("path");
const utils_1 = require("./utils");
const os = require("os");
describe('function breakpoints', function () {
    return __awaiter(this, void 0, void 0, function* () {
        let dc;
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield (0, utils_1.standardBeforeEach)();
                yield dc.launchRequest((0, utils_1.fillDefaults)(this.currentTest, {
                    program: (0, path_1.join)(utils_1.testProgramsDir, 'functions'),
                }));
            });
        });
        afterEach(() => __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        }));
        it('set type of function breakpoint', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp = yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'main',
                    },
                ],
            });
            (0, chai_1.expect)(bpResp.body.breakpoints.length).eq(1);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).eq(true);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].message).eq(undefined);
            yield dc.configurationDoneRequest();
            let isCorrect;
            let outputs;
            while (!isCorrect) {
                // Cover the case of getting event in Linux environment.
                // If cannot get correct event, program timeout and test case failed.
                outputs = yield dc.waitForEvent('output');
                isCorrect = outputs.body.output.includes('breakpoint-modified');
            }
            let substring;
            if (utils_1.hardwareBreakpoint) {
                substring = 'type="hw breakpoint"';
            }
            else {
                substring = 'type="breakpoint"';
            }
            (0, chai_1.expect)(outputs === null || outputs === void 0 ? void 0 : outputs.body.output).includes(substring);
        }));
        it('hits the main function breakpoint', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp = yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'main',
                    },
                ],
            });
            (0, chai_1.expect)(bpResp.body.breakpoints.length).eq(1);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).eq(true);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].message).eq(undefined);
            yield dc.configurationDoneRequest();
            yield dc.assertStoppedLocation('function breakpoint', { line: 14 });
        }));
        it('hits the sub function breakpoint', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'sub',
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.assertStoppedLocation('function breakpoint', { line: 10 });
        }));
        it('can set and hit the sub function breakpoint while program is running', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (os.platform() === 'win32' && (!utils_1.isRemoteTest || !utils_1.gdbAsync)) {
                    // win32 host can only pause remote + mi-async targets
                    this.skip();
                }
                const bpResp1 = yield dc.setFunctionBreakpointsRequest({
                    breakpoints: [
                        {
                            name: 'main',
                        },
                    ],
                });
                (0, chai_1.expect)(bpResp1.body.breakpoints.length).to.eq(1);
                yield dc.configurationDoneRequest();
                yield dc.waitForEvent('stopped');
                const scope = yield (0, utils_1.getScopes)(dc);
                yield dc.continueRequest({ threadId: scope.thread.id });
                // start listening for stopped events before we issue the
                // setBreakpointsRequest to ensure we don't get extra
                // stopped events
                const stoppedEventWaitor = dc.waitForEvent('stopped');
                const bpResp2 = yield dc.setFunctionBreakpointsRequest({
                    breakpoints: [
                        {
                            name: 'sub',
                        },
                    ],
                });
                (0, chai_1.expect)(bpResp2.body.breakpoints.length).to.eq(1);
                yield dc.assertStoppedLocation('function breakpoint', {
                    line: 10,
                    path: /functions.c$/,
                });
                const stoppedEvent = yield stoppedEventWaitor;
                (0, chai_1.expect)(stoppedEvent).to.have.property('body');
                (0, chai_1.expect)(stoppedEvent.body).to.have.property('reason', 'function breakpoint');
            });
        });
        it('handles <MULTIPLE> responses (e.g. multiple static functions with same name)', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'staticfunc1',
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.assertStoppedLocation('function breakpoint', {
                line: 3,
                path: /functions.c$/,
            });
            const scope = yield (0, utils_1.getScopes)(dc);
            yield dc.continue({ threadId: scope.thread.id }, 'function breakpoint', { line: 2, path: /functions_other.c$/ });
        }));
        it('handles <MULTIPLE> function changes', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'staticfunc1',
                    },
                ],
            });
            yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'staticfunc2',
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.assertStoppedLocation('function breakpoint', {
                line: 6,
                path: /functions.c$/,
            });
            const scope = yield (0, utils_1.getScopes)(dc);
            yield dc.continue({ threadId: scope.thread.id }, 'function breakpoint', { line: 5, path: /functions_other.c$/ });
        }));
        it('handles <MULTIPLE> mixed with line breakpoints', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'staticfunc1',
                    },
                ],
            });
            // This a regression test as this did not lead to an error back to
            // the user, but did mean that the adapter was trying to do:
            //  -break-delete 1.1 1.2
            // which gets a warning back from GDB:
            //  warning: bad breakpoint number at or near '0'
            let logOutput = '';
            dc.on('output', (e) => {
                if (e.body.category === 'log') {
                    logOutput += e.body.output;
                }
            });
            yield dc.setBreakpointsRequest({
                source: {
                    name: 'functions.c',
                    path: path.join(utils_1.testProgramsDir, 'functions.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 14,
                    },
                ],
            });
            (0, chai_1.expect)(logOutput).does.not.contain('warning');
            yield dc.configurationDoneRequest();
            yield dc.assertStoppedLocation('breakpoint', {
                line: 14,
                path: /functions.c$/,
            });
        }));
        it('fails gracefully on unknown function', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp = yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'mainBAD',
                    },
                ],
            });
            (0, chai_1.expect)(bpResp.body.breakpoints.length).eq(1);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).eq(false);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].message).not.eq(undefined);
        }));
        it('maintains breakpoint order when modifying function breakpoints', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp1 = yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'main',
                    },
                ],
            });
            const bpResp2 = yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'sub',
                    },
                    {
                        name: 'main',
                    },
                ],
            });
            // Unlike with line breakpoints, function breakpoints don't
            // really report back anything other than the ID that can
            // be used to check order is maintained
            (0, chai_1.expect)(bpResp2.body.breakpoints[1].id).eq(bpResp1.body.breakpoints[0].id);
            (0, chai_1.expect)(bpResp2.body.breakpoints[0].id).not.eq(bpResp1.body.breakpoints[0].id);
        }));
        it('deletes breakpoints in gdb when removed in IDE', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'sub',
                    },
                    {
                        name: 'main',
                    },
                ],
            });
            yield dc.setFunctionBreakpointsRequest({
                breakpoints: [
                    {
                        name: 'sub',
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.assertStoppedLocation('function breakpoint', { line: 10 });
        }));
    });
});
//# sourceMappingURL=functionBreakpoints.spec.js.map