/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.java.editor.options;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JCheckBox;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.netbeans.api.editor.EditorRegistry;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.modules.parsing.api.indexing.IndexingManager;
import org.openide.util.Exceptions;

public class InlineHintsPanel extends javax.swing.JPanel {

    public static final String JAVA_INLINE_HINT_PARAMETER_NAME = "javaInlineHintParameterName"; //NOI18N
    public static final String JAVA_INLINE_HINT_CHAINED_TYPES = "javaInlineHintChainedTypes"; //NOI18N
    public static final String JAVA_INLINE_HINT_VAR_TYPE = "javaInlineHintVarType"; //NOI18N

    private static final Map<String, Boolean> DEFAULT_VALUES;

    static {
        Map<String, Boolean> defaultValuesBuilder = new HashMap<>();
        defaultValuesBuilder.put(JAVA_INLINE_HINT_PARAMETER_NAME, true);
        defaultValuesBuilder.put(JAVA_INLINE_HINT_CHAINED_TYPES, false);
        defaultValuesBuilder.put(JAVA_INLINE_HINT_VAR_TYPE, false);
        DEFAULT_VALUES = Collections.unmodifiableMap(defaultValuesBuilder);
    }

    private List<JCheckBox> boxes;
    private InlineHintsOptionsPanelController controller;
    private boolean changed = false;

    /**
     * Creates new form InlineHintsPanel
     */
    public InlineHintsPanel(InlineHintsOptionsPanelController controller) {
        initComponents();
//        if( "Windows".equals(UIManager.getLookAndFeel().getID()) ) //NOI18N
//            setOpaque( false );
        fillBoxes();
        addListeners();
        load(controller);
    }

    public void load(InlineHintsOptionsPanelController controller) {
        this.controller = controller;

        Preferences node = InlineHintsSettings.getCurrentNode();

        for (JCheckBox box : boxes) {
            box.setSelected(node.getBoolean(box.getActionCommand(), DEFAULT_VALUES.get(box.getActionCommand())));
        }

        changed = false;
    }

    public void store() {
        Preferences node = InlineHintsSettings.getCurrentNode();

        for (javax.swing.JCheckBox box : boxes) {
            boolean value = box.isSelected();
            boolean original = node.getBoolean(box.getActionCommand(),
                    DEFAULT_VALUES.get(box.getActionCommand()));

            if (value != original) {
                node.putBoolean(box.getActionCommand(), value);
            }
        }
        try {
            node.flush();
        } catch (BackingStoreException ex) {
            Exceptions.printStackTrace(ex);
        }
        changed = false;

        EditorRegistry
                .componentList()
                .stream()
                .map(jtc -> jtc.getDocument())
                .filter(doc -> doc != null)
                .map(doc -> NbEditorUtilities.getFileObject(doc))
                .filter(fo -> fo != null)
                .forEach(fo -> IndexingManager.getDefault().refreshAllIndices(fo));
    }

    public boolean changed() {
        return changed;
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        javaInlineHintParameterNameCB = new javax.swing.JCheckBox();
        javaInlineHintChainedTypesCB = new javax.swing.JCheckBox();
        javaInlineHintVarTypeCB = new javax.swing.JCheckBox();
        filler1 = new javax.swing.Box.Filler(new java.awt.Dimension(0, 0), new java.awt.Dimension(0, 0), new java.awt.Dimension(32767, 32767));

        setBorder(javax.swing.BorderFactory.createEmptyBorder(8, 8, 8, 8));
        setLayout(new java.awt.GridBagLayout());

        org.openide.awt.Mnemonics.setLocalizedText(javaInlineHintParameterNameCB, org.openide.util.NbBundle.getMessage(InlineHintsPanel.class, "InlineHintsPanel.javaInlineHintParameterNameCB.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
        add(javaInlineHintParameterNameCB, gridBagConstraints);

        org.openide.awt.Mnemonics.setLocalizedText(javaInlineHintChainedTypesCB, org.openide.util.NbBundle.getMessage(InlineHintsPanel.class, "InlineHintsPanel.javaInlineHintChainedTypesCB.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
        add(javaInlineHintChainedTypesCB, gridBagConstraints);

        org.openide.awt.Mnemonics.setLocalizedText(javaInlineHintVarTypeCB, org.openide.util.NbBundle.getMessage(InlineHintsPanel.class, "InlineHintsPanel.javaInlineHintVarTypeCB.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
        add(javaInlineHintVarTypeCB, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(filler1, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.Box.Filler filler1;
    private javax.swing.JCheckBox javaInlineHintChainedTypesCB;
    private javax.swing.JCheckBox javaInlineHintParameterNameCB;
    private javax.swing.JCheckBox javaInlineHintVarTypeCB;
    // End of variables declaration//GEN-END:variables
    // End of variables declaration

    private void fillBoxes() {
        boxes = new ArrayList<JCheckBox>();
        boxes.add(javaInlineHintParameterNameCB);
        boxes.add(javaInlineHintChainedTypesCB);
        boxes.add(javaInlineHintVarTypeCB);

        javaInlineHintParameterNameCB.setActionCommand(JAVA_INLINE_HINT_PARAMETER_NAME);
        javaInlineHintChainedTypesCB.setActionCommand(JAVA_INLINE_HINT_CHAINED_TYPES);
        javaInlineHintVarTypeCB.setActionCommand(JAVA_INLINE_HINT_VAR_TYPE);
    }

    private void addListeners() {
        ChangeListener cl = new CheckChangeListener();

        for (JCheckBox box : boxes) {
            box.addChangeListener(cl);
        }

    }

    private void fireChanged() {
        Preferences node = InlineHintsSettings.getCurrentNode();
        for (JCheckBox box : boxes) {
            if (node.getBoolean(box.getActionCommand(), DEFAULT_VALUES.get(box.getActionCommand())) != box.isSelected()) {
                changed = true;
                return;
            }
        }
        changed = false;
    }

    private class CheckChangeListener implements ChangeListener {

        @Override
        public void stateChanged(ChangeEvent evt) {
            fireChanged();
        }
    }

}
