/**
 * @packageDocumentation
 * @module nbformat
 */
import { PartialJSONObject } from '@lumino/coreutils';
/**
 * The earliest major version of the notebook format we support.
 */
export declare const MAJOR_VERSION: number;
/**
 * The earliest minor version of the notebook format we support.
 */
export declare const MINOR_VERSION: number;
/**
 * The kernelspec metadata.
 */
export interface IKernelspecMetadata extends PartialJSONObject {
    name: string;
    display_name: string;
}
/**
 * The language info metadata
 */
export interface ILanguageInfoMetadata extends PartialJSONObject {
    name: string;
    codemirror_mode?: string | PartialJSONObject;
    file_extension?: string;
    mimetype?: string;
    pygments_lexer?: string;
}
/**
 * The default metadata for the notebook.
 */
export interface INotebookMetadata extends PartialJSONObject {
    kernelspec?: IKernelspecMetadata;
    language_info?: ILanguageInfoMetadata;
    orig_nbformat?: number;
}
/**
 * The notebook content.
 */
export interface INotebookContent extends PartialJSONObject {
    metadata: INotebookMetadata;
    nbformat_minor: number;
    nbformat: number;
    cells: ICell[];
}
/**
 * A multiline string.
 */
export declare type MultilineString = string | string[];
/**
 * A mime-type keyed dictionary of data.
 */
export interface IMimeBundle extends PartialJSONObject {
    [key: string]: MultilineString | PartialJSONObject;
}
/**
 * Media attachments (e.g. inline images).
 */
export interface IAttachments {
    [key: string]: IMimeBundle | undefined;
}
/**
 * The code cell's prompt number. Will be null if the cell has not been run.
 */
export declare type ExecutionCount = number | null;
/**
 * Cell output metadata.
 */
export declare type OutputMetadata = PartialJSONObject;
/**
 * Validate a mime type/value pair.
 *
 * @param type - The mimetype name.
 *
 * @param value - The value associated with the type.
 *
 * @returns Whether the type/value pair are valid.
 */
export declare function validateMimeValue(type: string, value: MultilineString | PartialJSONObject): boolean;
/**
 * A type which describes the type of cell.
 */
export declare type CellType = 'code' | 'markdown' | 'raw';
/**
 * The Jupyter metadata namespace.
 */
export interface IBaseCellJupyterMetadata extends PartialJSONObject {
    /**
     * Whether the source is hidden.
     */
    source_hidden: boolean;
}
/**
 * Cell-level metadata.
 */
export interface IBaseCellMetadata extends PartialJSONObject {
    /**
     * Whether the cell is trusted.
     *
     * #### Notes
     * This is not strictly part of the nbformat spec, but it is added by
     * the contents manager.
     *
     * See https://jupyter-server.readthedocs.io/en/latest/operators/security.html.
     */
    trusted: boolean;
    /**
     * The cell's name. If present, must be a non-empty string.
     */
    name: string;
    /**
     * The Jupyter metadata namespace
     */
    jupyter: Partial<IBaseCellJupyterMetadata>;
    /**
     * The cell's tags. Tags must be unique, and must not contain commas.
     */
    tags: string[];
}
/**
 * The base cell interface.
 */
export interface IBaseCell extends PartialJSONObject {
    /**
     * String identifying the type of cell.
     */
    cell_type: string;
    /**
     * Contents of the cell, represented as an array of lines.
     */
    source: MultilineString;
    /**
     * Cell-level metadata.
     */
    metadata: Partial<ICellMetadata>;
}
/**
 * Metadata for the raw cell.
 */
export interface IRawCellMetadata extends IBaseCellMetadata {
    /**
     * Raw cell metadata format for nbconvert.
     */
    format: string;
}
/**
 * A raw cell.
 */
export interface IRawCell extends IBaseCell {
    /**
     * A string field representing the identifier of this particular cell.
     *
     * Notebook format 4.4 requires no id field, but format 4.5 requires an id
     * field. We need to handle both cases, so we make id optional here.
     */
    id?: string;
    /**
     * String identifying the type of cell.
     */
    cell_type: 'raw';
    /**
     * Cell-level metadata.
     */
    metadata: Partial<IRawCellMetadata>;
    /**
     * Cell attachments.
     */
    attachments?: IAttachments;
}
/**
 * A markdown cell.
 */
export interface IMarkdownCell extends IBaseCell {
    /**
     * A string field representing the identifier of this particular cell.
     *
     * Notebook format 4.4 requires no id field, but format 4.5 requires an id
     * field. We need to handle both cases, so we make id optional here.
     */
    id?: string;
    /**
     * String identifying the type of cell.
     */
    cell_type: 'markdown';
    /**
     * Cell attachments.
     */
    attachments?: IAttachments;
}
/**
 * The Jupyter metadata namespace for code cells.
 */
export interface ICodeCellJupyterMetadata extends IBaseCellJupyterMetadata {
    /**
     * Whether the outputs are hidden. See https://github.com/jupyter/nbformat/issues/137.
     */
    outputs_hidden: boolean;
}
/**
 * Metadata for a code cell.
 */
export interface ICodeCellMetadata extends IBaseCellMetadata {
    /**
     * Whether the cell is collapsed/expanded.
     */
    collapsed: boolean;
    /**
     * The Jupyter metadata namespace
     */
    jupyter: Partial<ICodeCellJupyterMetadata>;
    /**
     * Whether the cell's output is scrolled, unscrolled, or autoscrolled.
     */
    scrolled: boolean | 'auto';
}
/**
 * A code cell.
 */
export interface ICodeCell extends IBaseCell {
    /**
     * A string field representing the identifier of this particular cell.
     *
     * Notebook format 4.4 requires no id field, but format 4.5 requires an id
     * field. We need to handle both cases, so we make id optional here.
     */
    id?: string;
    /**
     * String identifying the type of cell.
     */
    cell_type: 'code';
    /**
     * Cell-level metadata.
     */
    metadata: Partial<ICodeCellMetadata>;
    /**
     * Execution, display, or stream outputs.
     */
    outputs: IOutput[];
    /**
     * The code cell's prompt number. Will be null if the cell has not been run.
     */
    execution_count: ExecutionCount;
}
/**
 * An unrecognized cell.
 */
export interface IUnrecognizedCell extends IBaseCell {
}
/**
 * A cell union type.
 */
export declare type ICell = IRawCell | IMarkdownCell | ICodeCell | IUnrecognizedCell;
/**
 * Test whether a cell is a raw cell.
 */
export declare function isRaw(cell: ICell): cell is IRawCell;
/**
 * Test whether a cell is a markdown cell.
 */
export declare function isMarkdown(cell: ICell): cell is IMarkdownCell;
/**
 * Test whether a cell is a code cell.
 */
export declare function isCode(cell: ICell): cell is ICodeCell;
/**
 * A union metadata type.
 */
export declare type ICellMetadata = IBaseCellMetadata | IRawCellMetadata | ICodeCellMetadata;
/**
 * The valid output types.
 */
export declare type OutputType = 'execute_result' | 'display_data' | 'stream' | 'error' | 'update_display_data';
/**
 * The base output type.
 */
export interface IBaseOutput extends PartialJSONObject {
    /**
     * Type of cell output.
     */
    output_type: string;
}
/**
 * Result of executing a code cell.
 */
export interface IExecuteResult extends IBaseOutput {
    /**
     * Type of cell output.
     */
    output_type: 'execute_result';
    /**
     * A result's prompt number.
     */
    execution_count: ExecutionCount;
    /**
     * A mime-type keyed dictionary of data.
     */
    data: IMimeBundle;
    /**
     * Cell output metadata.
     */
    metadata: OutputMetadata;
}
/**
 * Data displayed as a result of code cell execution.
 */
export interface IDisplayData extends IBaseOutput {
    /**
     * Type of cell output.
     */
    output_type: 'display_data';
    /**
     * A mime-type keyed dictionary of data.
     */
    data: IMimeBundle;
    /**
     * Cell output metadata.
     */
    metadata: OutputMetadata;
}
/**
 * Data displayed as an update to existing display data.
 */
export interface IDisplayUpdate extends IBaseOutput {
    /**
     * Type of cell output.
     */
    output_type: 'update_display_data';
    /**
     * A mime-type keyed dictionary of data.
     */
    data: IMimeBundle;
    /**
     * Cell output metadata.
     */
    metadata: OutputMetadata;
}
/**
 * Stream output from a code cell.
 */
export interface IStream extends IBaseOutput {
    /**
     * Type of cell output.
     */
    output_type: 'stream';
    /**
     * The name of the stream.
     */
    name: StreamType;
    /**
     * The stream's text output.
     */
    text: MultilineString;
}
/**
 * An alias for a stream type.
 */
export declare type StreamType = 'stdout' | 'stderr';
/**
 * Output of an error that occurred during code cell execution.
 */
export interface IError extends IBaseOutput {
    /**
     * Type of cell output.
     */
    output_type: 'error';
    /**
     * The name of the error.
     */
    ename: string;
    /**
     * The value, or message, of the error.
     */
    evalue: string;
    /**
     * The error's traceback.
     */
    traceback: string[];
}
/**
 * Unrecognized output.
 */
export interface IUnrecognizedOutput extends IBaseOutput {
}
/**
 * Test whether an output is an execute result.
 */
export declare function isExecuteResult(output: IOutput): output is IExecuteResult;
/**
 * Test whether an output is from display data.
 */
export declare function isDisplayData(output: IOutput): output is IDisplayData;
/**
 * Test whether an output is from updated display data.
 */
export declare function isDisplayUpdate(output: IOutput): output is IDisplayUpdate;
/**
 * Test whether an output is from a stream.
 */
export declare function isStream(output: IOutput): output is IStream;
/**
 * Test whether an output is an error.
 */
export declare function isError(output: IOutput): output is IError;
/**
 * An output union type.
 */
export declare type IOutput = IUnrecognizedOutput | IExecuteResult | IDisplayData | IStream | IError;
