#!/bin/bash
set -o verbose
set -o errexit

# This script fetches and creates a copy of sources for snappy
# snappy uses CMake and this script will invoke CMake to generate a
# config.h for "posix", Linux, and Windows
#
# To get the sources for Snappy, run this script as follows:
#
# 1. Run on Darwin
# 2. Run on Linux
# 3. Run on s390x Linux
# 4. Run on Windows via Cygwin
#
# For s390x CMake is already installed on the ZAP dev machines, you'll
# need to add it to the $PATH before running this script with:
#
# export PATH="/opt/cmake/bin:$PATH"
#
# For Windows you will need CMake installed. If using an
# Evergreen spawn host it is not installed by default but, you can
# easily install it with the following command. (this works in
# cygwin):
#
# choco install cmake
#
# You will also need to add it to the $PATH with the following:
#
# export PATH="/cygdrive/c/Program Files/CMake/bin/:$PATH"


VERSION=1.1.7
NAME=snappy
TARBALL=$NAME-$VERSION.tar.gz
TARBALL_DIR=$NAME-$VERSION
TARBALL_DEST_DIR=$NAME-$VERSION
TEMP_DIR=/tmp/temp-$NAME-$VERSION
DEST_DIR=`git rev-parse --show-toplevel`/src/third_party/$NAME-$VERSION
UNAME=`uname | tr A-Z a-z`

if [ "$UNAME" == "linux" && "$(uname -m)" == "s390x" ]; then
    TARGET_UNAME=linux_s390x
elif [ "$UNAME" == "linux" ]
    TARGET_UNAME=linux
elif [[ "$UNAME" == "cygwin"* ]]; then
    TARGET_UNAME=windows
else
    TARGET_UNAME=posix
fi

echo TARGET_UNAME: $TARGET_UNAME

if [ ! -f $TARBALL ]; then
    echo "Get tarball"
    curl -L -o $NAME-$VERSION.tar.gz https://github.com/google/$NAME/archive/$VERSION.tar.gz
fi

echo $TARBALL
tar -zxvf $TARBALL

rm -rf $TEMP_DIR
mv $TARBALL_DIR $TEMP_DIR
mkdir $DEST_DIR || true

cd $TEMP_DIR
cp $TEMP_DIR/* $DEST_DIR || true
rm -f $DEST_DIR/Makefile* $DEST_DIR/config* $DEST_DIR/*sh
rm -f $DEST_DIR/compile* $DEST_DIR/depcomp $DEST_DIR/libtool
rm -f $DEST_DIR/test-driver $DEST_DIR/*.m4 $DEST_DIR/missing

echo "Generating Config.h and other files"
cmake $TEMP_DIR

# Copy over the platform independent generated files
cp $TEMP_DIR/snappy-stubs-public.h $DEST_DIR

# Copy over config.h
mkdir $DEST_DIR/build_$TARGET_UNAME
cp $TEMP_DIR/config.h $DEST_DIR/build_$TARGET_UNAME

# Change the snappy-stubs-public.h to use the defined variables
# instead of hardcoded values generated by CMake
#
# Changes lines like:
#
#    #if !0  // !HAVE_SYS_UIO_H
#    #if 1  // HAVE_STDINT_H
#
# To:
#
#    #if !HAVE_SYS_UIO_H
#    #if HAVE_STDINT_H
if [ "$TARGET_UNAME" = "posix" ]; then
    sed -i '' 's/if !\{0,1\}[0-9]  \/\/ \(!\{0,1\}HAVE_.*\)/if \1/' snappy-stubs-public.h
else
    sed -i 's/if !\{0,1\}[0-9]  \/\/ \(!\{0,1\}HAVE_.*\)/if \1/' snappy-stubs-public.h
fi

echo "Done"
