/*
 * Copyright (C) 2014 Guitarix project MOD project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */


////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_common.h"      // faust support and denormal protection (SSE)
#include "gx_aclipper.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "gx_resampler.h"
#include "aclipper.cc"      // dsp class generated by faust -> dsp2cc
#include "gx_bypass.cc"     // bypass ramping

////////////////////////////// PLUG-IN CLASS ///////////////////////////

namespace aclipper {

class Gx_aclipper_
{
private:
  GxBypass    bp;
  float*      bypass;
  // pointer to buffer
  float*      output;
  float*      input;
  // pointer to dsp class
  PluginLV2*  aclipper;
  // private functions
  inline void run_dsp_(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);
  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  Gx_aclipper_();
  ~Gx_aclipper_();
};

// constructor
Gx_aclipper_::Gx_aclipper_() :
  bp(),
  bypass(0),
  output(NULL),
  input(NULL),
  aclipper(aclipper::plugin()) {};

// destructor
Gx_aclipper_::~Gx_aclipper_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (aclipper->activate_plugin !=0)
    aclipper->activate_plugin(false, aclipper);
  // delete DSP class
  aclipper->delete_instance(aclipper);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_aclipper_::init_dsp_(uint32_t rate)
{
  AVOIDDENORMALS(); // init the SSE denormal protection
  bp.init_bypass(rate);
  aclipper->set_samplerate(rate, aclipper); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_aclipper_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
    case BYPASS: 
      bypass = static_cast<float*>(data); // , 0.0, 0.0, 1.0, 1.0 
      break;
    default:
      break;
    }
}

void Gx_aclipper_::activate_f()
{
  // allocate the internal DSP mem
  if (aclipper->activate_plugin !=0)
    aclipper->activate_plugin(true, aclipper);
}

void Gx_aclipper_::clean_up()
{
  // delete the internal DSP mem
  if (aclipper->activate_plugin !=0)
    aclipper->activate_plugin(false, aclipper);
}

void Gx_aclipper_::deactivate_f()
{
  // delete the internal DSP mem
  if (aclipper->activate_plugin !=0)
    aclipper->activate_plugin(false, aclipper);
}

void Gx_aclipper_::run_dsp_(uint32_t n_samples)
{
  if(n_samples<1) return;
  FAUSTFLOAT buf[n_samples];
  // do inplace processing at default
  if (output != input)
    memcpy(output, input, n_samples*sizeof(float));
  // check if bypass is pressed
  if (!bp.pre_check_bypass(bypass, buf, input, n_samples))
    aclipper->mono_audio(static_cast<int>(n_samples), input, output, aclipper);
  bp.post_check_bypass(buf, output, n_samples);
}

void Gx_aclipper_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  aclipper->connect_ports(port,  data, aclipper);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Handle 
Gx_aclipper_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_aclipper_ *self = new Gx_aclipper_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_((uint32_t)rate);

  return (LV2_Handle)self;
}

void Gx_aclipper_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_aclipper_*>(instance)->connect_all__ports(port, data);
}

void Gx_aclipper_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_aclipper_*>(instance)->activate_f();
}

void Gx_aclipper_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  static_cast<Gx_aclipper_*>(instance)->run_dsp_(n_samples);
}

void Gx_aclipper_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_aclipper_*>(instance)->deactivate_f();
}

void Gx_aclipper_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_aclipper_* self = static_cast<Gx_aclipper_*>(instance);
  self->clean_up();
  delete self;
}

const LV2_Descriptor Gx_aclipper_::descriptor =
{
  GXPLUGIN_URI "#_aclipper_",
  Gx_aclipper_::instantiate,
  Gx_aclipper_::connect_port,
  Gx_aclipper_::activate,
  Gx_aclipper_::run,
  Gx_aclipper_::deactivate,
  Gx_aclipper_::cleanup,
  NULL
};


} // end namespace aclipper

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &aclipper::Gx_aclipper_::descriptor;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
