///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/// Banded level set routines 
///
/// Authors: Lara Aborm, Jocelyn Etienne, Pierre Saramito
///
#include "rheolef/banded_level_set.h"
#include "rheolef.h"
#include "rheolef/tiny_element.h"
#include "rheolef/ublas-invert.h"
namespace rheolef {

using namespace std;

// --------------------------------------------------------------------------------
// gestion de la precision
// --------------------------------------------------------------------------------
static Float tol = 100*numeric_limits<Float>::epsilon();
//static Float tol = 0;
void set_tol (const Float& tol1) { tol = tol1; }
bool is_zero (const Float& x) { return fabs(x) <= tol; }
bool have_same_sign (const Float& x, const Float& y) {
   return !is_zero(x) && !is_zero(y) && x*y > 0;
}
bool have_opposite_sign (const Float& x, const Float& y) {
   return !is_zero(x) && !is_zero(y) && x*y < 0;
}
// --------------------------------------------------------------------------------
// 2D: fonctions locales : sur un seul element triangle
// --------------------------------------------------------------------------------
// appele lors du 1er passage qui liste les elements de la bande cas de dimension 2
bool belongs_to_band_t (const ublas::vector<Float>& f) {
    if (have_same_sign(f[0],f[1]) && have_same_sign(f[0],f[2])) return false;
    // on rejette le triangle dans tous les sommets de meme signe :
    if (is_zero(f[0]) && is_zero(f[1]) && is_zero(f[2])) return false;
    // on rejette les triangles dont un sommet :
    if (is_zero(f[0]) && have_same_sign(f[1],f[2])) return false;
    if (is_zero(f[1]) && have_same_sign(f[0],f[2])) return false;
    if (is_zero(f[2]) && have_same_sign(f[0],f[1])) return false;
    return true;
}
// apellee lors du calcul des matrices M_K et A_K pour K dans la bande cas de dimension 2
size_t
isolated_vertex_t (const ublas::vector<Float>& f) {
    /* on retourne le sommet isole a chaque fois */
    if (have_same_sign    (f[0],f[1]) && have_opposite_sign(f[0],f[2]))  return 2;
    if (have_opposite_sign(f[0],f[1]) && have_same_sign    (f[0],f[2]))  return 1;
    if (have_opposite_sign(f[0],f[1]) && have_opposite_sign(f[0],f[2]))  return 0;
    if (is_zero(f[0])                 && have_opposite_sign(f[1],f[2]))  return 1; /* on peut retourner 2 de meme*/

    if (is_zero(f[1]) && have_opposite_sign(f[0],f[2]))  return 0; /* on peut retourner 2 */
    if (is_zero(f[2]) && have_opposite_sign(f[0],f[1]))  return 0; /* on peut retourner 1 */
    if (is_zero(f[0]) && is_zero(f[1]) && ! is_zero(f[2])) return 2;
    if (is_zero(f[0]) && is_zero(f[2]) && ! is_zero(f[1])) return 1;
    return 0; /* f1 == 0 et f2 == 0 et f0 != 0 */
}
void
subcompute_matrix_t (const ublas::vector<point>& x ,const ublas::vector<Float>& f,
		     ublas::vector<size_t>& j, point& a, point& b, Float& S) {
    j.resize (3);
    j[0] = isolated_vertex_t (f);
    j[1] = (j[0]+1) % 3;
    j[2] = (j[1]+1) % 3;
    // edge {j1,j2} has normal oriented as grad(f), in f>0 direction:
    if (! is_zero(f[j[0]]) && f[j[0]] < 0) swap (j[1], j[2]);
    Float theta_1= f[j[1]]/(f[j[1]]-f[j[0]]);
    Float theta_2= f[j[2]]/(f[j[2]]-f[j[0]]);
    // calcul des coordonnes d'intersection
    a = theta_1*x[j[0]]+(1-theta_1)*x[j[1]];
    b = theta_2*x[j[0]]+(1-theta_2)*x[j[2]];
    S = sqrt(pow(a[0]-b[0],2)+pow(a[1]-b[1],2));
    if (is_zero(f[j[1]]) && is_zero(f[j[2]])) {
        S /= 2;
    }
}
void
inv_piola_t (const ublas::vector<point>& x, tensor& inv_B) {
    /* calcul de la matrice B qui nous mene du triangle de reference au triangle ordinaire*/
    tensor B;
    B(0,0) = x[1][0]-x[0][0];
    B(0,1) = x[2][0]-x[0][0];
    B(1,0) = x[1][1]-x[0][1];
    B(1,1) = x[2][1]-x[0][1];
 
    /*calcul de l'inverse de la matrice B*/
    inv_B = inv(B, 2);
}
void projection_s (point& v, const point& n) {
    Float a = dot(v,n);
    v = v - a*n;
}
// definition des fonctions de base sur le triangle de reference en dimension 2
Float hat_a0 (const point& x) {
    return 1 - x[0] - x[1];
}
Float hat_a1 (const point& x) {
    return x[0];
}
Float hat_a2 (const point& x) {
    return x[1];
}
Float hat_a (size_t i, const point& x) {
  switch (i) {
    case 0 :  return 1 - x[0] - x[1];
    case 1 :  return x[0];
    default : return x[1];
  }
}
Float hat_T (size_t i, const point& x) {
  switch (i) {
    case 0 :  return 1 - x[0] - x[1] -x[2];
    case 1 :  return x[0];
    case 2 :  return x[1];
    default : return x[2];
  }
}
// --------------------------------------------------------------------------------
// 3D: fonctions locales : sur un seul element tetraedre
// --------------------------------------------------------------------------------

// appele lors du 1er passage qui liste les elements de la bande cas de dimension 3
bool
belongs_to_band_T (const ublas::vector<Float>& f) {
    if (have_same_sign(f[0],f[1]) && have_same_sign(f[0],f[2]) && have_same_sign(f[2],f[3])) return false;
    // cas ou 4 points s'annulent en dimension 3 est degenere
    if (is_zero(f[0]) && is_zero(f[1]) && is_zero(f[2]) && is_zero(f[3])) return false;

    if (is_zero(f[0]) && have_same_sign(f[1],f[2]) && have_same_sign(f[1],f[3])) return false;
    if (is_zero(f[1]) && have_same_sign(f[0],f[2]) && have_same_sign(f[0],f[3])) return false;
    if (is_zero(f[2]) && have_same_sign(f[0],f[1]) && have_same_sign(f[0],f[3])) return false;
    if (is_zero(f[3]) && have_same_sign(f[0],f[1]) && have_same_sign(f[0],f[2])) return false;
    // cas ou phi s'annule sur 2 sommets et garde le meme signe sur les 2 autres sommets est exclu
    if (is_zero(f[0]) && is_zero(f[1]) && have_same_sign(f[2],f[3])) return false;
    if (is_zero(f[0]) && is_zero(f[2]) && have_same_sign(f[1],f[3])) return false;
    if (is_zero(f[0]) && is_zero(f[3]) && have_same_sign(f[1],f[2])) return false;
    if (is_zero(f[1]) && is_zero(f[2]) && have_same_sign(f[0],f[3])) return false;
    if (is_zero(f[1]) && is_zero(f[3]) && have_same_sign(f[0],f[2])) return false;
    if (is_zero(f[2]) && is_zero(f[3]) && have_same_sign(f[1],f[0])) return false;
    return true;
}
// definition des fonctions de base sur le tetraedre de reference en dimension 3
Float hat_a0_T(const point& p) {
    Float z = 1-p[0]-p[1]-p[2];
    return z;
}
Float hat_a1_T(const point& p) {
    Float z = p[0];
    return z;
}
Float hat_a2_T(const point& p) {
    Float z = p[1];
    return z;
}
Float hat_a3_T(const point& p) {
    Float z = p[2];
    return z;
}
// apellee lors du calcul des matrices M_K et A_K pour T dans la bande cas de dimension 
bool
intersection_is_quadrilateral_T (const ublas::vector<Float>& f, quadruplet& q) {
    if (have_same_sign(f[0],f[1]) && have_opposite_sign(f[0],f[2]) && have_same_sign(f[2],f[3])) {
	if (f[0] > 0) q = quadruplet(0,1, 2,3);
	else          q = quadruplet(2,3, 0,1);
        return true;
    }
    if (have_opposite_sign(f[0],f[1]) && have_same_sign(f[0],f[2]) && have_opposite_sign(f[2],f[3])) {
        if (f[0] < 0) q = quadruplet(0,2, 1,3);
        else          q = quadruplet(1,3, 0,2);
        return true;
    }
    if (have_opposite_sign(f[0],f[1]) && have_opposite_sign(f[0],f[2]) && have_opposite_sign(f[2],f[3])) {
        if (f[0] > 0) q = quadruplet(0,3, 1,2);
        else          q = quadruplet(1,2, 0,3);
        return true;
    }
    return false;
}
// cas d'une intersection triangle:
size_t
isolated_vertex_T (const ublas::vector<Float>& f) {
    // cas ou l'intersection est un triangle
    if (have_opposite_sign(f[0],f[1]) && have_opposite_sign(f[0],f[2]) && have_same_sign    (f[2],f[3])) return 0;
    if (have_same_sign    (f[0],f[1]) && have_opposite_sign(f[0],f[2]) && have_opposite_sign(f[2],f[3])) return 2;
    if (have_same_sign    (f[0],f[1]) && have_same_sign    (f[0],f[2]) && have_opposite_sign(f[2],f[3])) return 3;
    // cas ou phi s'annule sur un sommet et change de signe sur les 2 autres sommets
    if (have_opposite_sign(f[0],f[1]) && have_same_sign(f[0],f[2]) && have_same_sign(f[2],f[3])) return 1;
    if (is_zero(f[0]) && have_same_sign    (f[1],f[2]) && have_opposite_sign(f[1],f[3])) return 3;
    if (is_zero(f[0]) && have_opposite_sign(f[1],f[2]) && have_same_sign    (f[1],f[3])) return 2;
    if (is_zero(f[0]) && have_opposite_sign(f[1],f[2]) && have_opposite_sign(f[1],f[3])) return 1;
    if (is_zero(f[1]) && have_opposite_sign(f[0],f[2]) && have_same_sign    (f[0],f[3])) return 2;
    if (is_zero(f[1]) && have_same_sign    (f[0],f[2]) && have_opposite_sign(f[0],f[3])) return 3;
    if (is_zero(f[1]) && have_opposite_sign(f[0],f[2]) && have_opposite_sign(f[0],f[3])) return 0;
    if (is_zero(f[2]) && have_opposite_sign(f[0],f[1]) && have_same_sign    (f[0],f[3])) return 1;
    if (is_zero(f[2]) && have_same_sign    (f[0],f[1]) && have_opposite_sign(f[0],f[3])) return 3;
    if (is_zero(f[2]) && have_opposite_sign(f[0],f[1]) && have_opposite_sign(f[0],f[3])) return 0;
    if (is_zero(f[3]) && have_opposite_sign(f[0],f[1]) && have_same_sign    (f[0],f[2])) return 1;
    if (is_zero(f[3]) && have_same_sign    (f[0],f[1]) && have_opposite_sign(f[0],f[2])) return 2;
    if (is_zero(f[3]) && have_opposite_sign(f[0],f[1]) && have_opposite_sign(f[0],f[2])) return 0;
    // cas ou phi s'annule en 2 sommets et change de signe sur les 2 autres sommets
    if (is_zero(f[0]) && is_zero(f[1]) && have_opposite_sign(f[2],f[3])) return 2; // ou 3
    if (is_zero(f[0]) && is_zero(f[2]) && have_opposite_sign(f[1],f[3])) return 1;
    if (is_zero(f[0]) && is_zero(f[3]) && have_opposite_sign(f[1],f[2])) return 1;
    if (is_zero(f[1]) && is_zero(f[2]) && have_opposite_sign(f[0],f[3])) return 0;
    if (is_zero(f[1]) && is_zero(f[3]) && have_opposite_sign(f[0],f[2])) return 0;
    if (is_zero(f[2]) && is_zero(f[3]) && have_opposite_sign(f[0],f[1])) return 0;

    // le triangle d'intersection est la face du tetradre ou phi s'annule sur les 3 sommets
    if (is_zero(f[0]) && is_zero(f[1]) && is_zero(f[2]) && ! is_zero(f[3])) return 3;
    if (is_zero(f[0]) && is_zero(f[1]) && is_zero(f[3]) && ! is_zero(f[2])) return 2;
    if (is_zero(f[1]) && is_zero(f[2]) && is_zero(f[3]) && ! is_zero(f[0])) return 0;
    return 1;
}
void
subcompute_matrix_triangle_T (const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::vector<size_t>& j,
                              point& a, point& b, point& c, Float& aire) {
    j.resize(4);
    j[0] = isolated_vertex_T (f);
    j[1] = (j[0]+1) % 4;
    j[2] = (j[1]+1) % 4;
    j[3] = (j[2]+1) % 4;
    // orient
    if (! is_zero(f[j[0]]) && ((f[j[0]] > 0 && j[0] % 2 == 0) || (f[j[0]] < 0 && j[0] % 2 == 1))) 
        swap (j[1], j[2]);
    Float theta_1 = f[j[1]]/(f[j[1]]-f[j[0]]);
    Float theta_2 = f[j[2]]/(f[j[2]]-f[j[0]]);
    Float theta_3 = f[j[3]]/(f[j[3]]-f[j[0]]);

    /* calcul des coordonnees d'intersection */
    a = theta_1*x[j[0]]+(1-theta_1)*x[j[1]];
    b = theta_2*x[j[0]]+(1-theta_2)*x[j[2]];
    c = theta_3*x[j[0]]+(1-theta_3)*x[j[3]];
    aire = 0.5* norm (vect( b-a, c-a));
    if (is_zero(f[j[1]]) && is_zero(f[j[2]]) && is_zero(f[j[3]])) {
        aire /= 2;
    }
}
void
inv_piola_T (const ublas::vector<point>& x, tensor& inv_B)
{
    tensor B;
    B(0,0) = x[1][0]-x[0][0];
    B(0,1) = x[2][0]-x[0][0];
    B(0,2) = x[3][0]-x[0][0];

    B(1,0) = x[1][1]-x[0][1];
    B(1,1) = x[2][1]-x[0][1];
    B(1,2) = x[3][1]-x[0][1];

    B(2,0) = x[1][2]-x[0][2];
    B(2,1) = x[2][2]-x[0][2];
    B(2,2) = x[3][2]-x[0][2];

    //calcul de l'inverse de la matrice B
    inv_B = inv(B);
}
void
subcompute_matrix_quadrilateral_T (const ublas::vector<point>& x, const ublas::vector<Float>& f, const quadruplet& q,
                                   point& a, point& b, point& c, point& d, Float& aire_Q) {
    // intersection:
    // a = segment {x(q0) x(q2)} inter {phi=0}
    // b = segment {x(q1) x(q2)} inter {phi=0}
    // d = segment {x(q1) x(q3)} inter {phi=0}
    // c = segment {x(q0) x(q3)} inter {phi=0}
    // quadrilatere abdc = triangle(abd) union triangle(adc)
    Float theta_1 = f[q[2]]/(f[q[2]]-f[q[0]]);
    Float theta_2 = f[q[2]]/(f[q[2]]-f[q[1]]);
    Float theta_3 = f[q[3]]/(f[q[3]]-f[q[0]]);
    Float theta_4 = f[q[3]]/(f[q[3]]-f[q[1]]);
    /* calcul des coordonnees d'intersection */
    a = theta_1*x[q[0]]+(1-theta_1)*x[q[2]];
    b = theta_2*x[q[1]]+(1-theta_2)*x[q[2]];
    c = theta_3*x[q[0]]+(1-theta_3)*x[q[3]];
    d = theta_4*x[q[1]]+(1-theta_4)*x[q[3]];
    aire_Q = 0.5*norm(vect(a-c,a-b)) + 0.5*norm(vect(d-c,d-b));
}

//--------------------------------------------------------------------
//  Assembly of elementary matrices
//--------------------------------------------------------------------
void
compute_matrix_a_t (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& ak) {
    point a, b;
    Float S;
    tensor inv_B;
    ublas::vector<size_t> j(3);
    subcompute_matrix_t (x, f, j, a, b, S);
    inv_piola_t (x, inv_B);

    // calcul des coordonnes du milieu de segment [a,b]
    point c = (a+b)*0.5;

    /*calcul de la transposee de l'inverse de la matrice B */
    tensor transposee;
    for (int i=0;i<2;i++) {
        for (int j=0;j<2;j++) {
            transposee (j,i) = inv_B (i,j);
        }
    }
    /* la definition des gradients sur le triangle de reference */
    point grad_hat_a0 (-1,-1);
    point grad_hat_a1 ( 1, 0);
    point grad_hat_a2 ( 0, 1);

    /*calcul des valeurs des gradients sur chaque sommet du triangle K */
    point grad_a0 = transposee*grad_hat_a0;
    point grad_a1 = transposee*grad_hat_a1;
    point grad_a2 = transposee*grad_hat_a2;

    point grad_f = f[0]*grad_a0 +f[1]*grad_a1 +f[2]*grad_a2; 
    point normal = grad_f/norm(grad_f);
    projection_s (grad_a0, normal);
    projection_s (grad_a1, normal);
    projection_s (grad_a2, normal);

    /* calcul de la matrice ak */
    ak(0,0)=S*dot(grad_a0,grad_a0);
    ak(0,1)=S*dot(grad_a0,grad_a1);
    ak(0,2)=S*dot(grad_a0,grad_a2);
    ak(1,0)=S*dot(grad_a1,grad_a0);
    ak(1,1)=S*dot(grad_a1,grad_a1);
    ak(1,2)=S*dot(grad_a1,grad_a2);
    ak(2,0)=S*dot(grad_a2,grad_a0);
    ak(2,1)=S*dot(grad_a2,grad_a1);
    ak(2,2)=S*dot(grad_a2,grad_a2);
}
void
compute_matrix_m_t (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {

    point a, b;
    Float S;
    tensor inv_B;
    ublas::vector<size_t> j(3);
    subcompute_matrix_t (x, f, j, a, b, S);
    inv_piola_t (x, inv_B);

    // calcul des coordonnes du milieu de segment [a,b]
    point c = (a+b)*0.5;

    // definition des coordonnes des points a, b et c dans le triangle de reference
    point hat_a = inv_B*(a-x[0]);
    point hat_b = inv_B*(b-x[0]);
    point hat_c = inv_B*(c-x[0]);
  
    // caclul des termes de la matrice de masse en utilisant la formule de simpson
    mk(0,0)          = 1./6* S* (hat_a0(hat_a)*hat_a0(hat_a)+4*hat_a0(hat_c)*hat_a0(hat_c)+hat_a0(hat_b)*hat_a0(hat_b));
    mk(0,1) = mk(1,0)= 1./6 *S* (hat_a0(hat_a)*hat_a1(hat_a)+4*hat_a0(hat_c)*hat_a1(hat_c)+hat_a0(hat_b)*hat_a1(hat_b));
    mk(0,2) = mk(2,0)= 1./6 *S* (hat_a0(hat_a)*hat_a2(hat_a)+4*hat_a0(hat_c)*hat_a2(hat_c)+hat_a0(hat_b)*hat_a2(hat_b));
    mk(1,1) =          1./6 *S* (hat_a1(hat_a)*hat_a1(hat_a)+4*hat_a1(hat_c)*hat_a1(hat_c)+hat_a1(hat_b)*hat_a1(hat_b));
    mk(1,2) = mk(2,1)= 1./6 *S* (hat_a1(hat_a)*hat_a2(hat_a)+4*hat_a1(hat_c)*hat_a2(hat_c)+hat_a1(hat_b)*hat_a2(hat_b));
    mk(2,2)=           1./6 *S* (hat_a2(hat_a)*hat_a2(hat_a)+4*hat_a2(hat_c)*hat_a2(hat_c)+hat_a2(hat_b)*hat_a2(hat_b));
}
// b(q,v) = int_S grad_s(q).v ds
void
compute_matrix_grad_s_t (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& bk) {
    point A, B;
    Float S;
    tensor inv_DF;
    ublas::vector<size_t> j(3);
    subcompute_matrix_t (x, f, j, A, B, S);
    inv_piola_t (x, inv_DF);

    // definition des coordonnes des points a, b et c dans le triangle de reference
    point hat_A = inv_DF*(A-x[0]);
    point hat_B = inv_DF*(B-x[0]);
  
    // calcul de la transposee de l'inverse de la matrice B
    tensor transposee;
    for (int i=0;i<2;i++) {
      for (int j=0;j<2;j++) {
        transposee (j,i) = inv_DF (i,j);
      }
    }
    // la definition des gradients sur le triangle de reference
    point grad_hat_a[3];
    grad_hat_a[0] = point(-1,-1);
    grad_hat_a[1] = point( 1, 0);
    grad_hat_a[2] = point( 0, 1);

    // calcul des valeurs des gradients sur chaque sommet du triangle K
    point grad_a [3];
    for (size_t i = 0; i < 3; i++) {
      grad_a[i] = transposee*grad_hat_a[i];
    }
    point grad_f = f[0]*grad_a[0] +f[1]*grad_a[1] +f[2]*grad_a[2]; 
    point normal = grad_f/norm(grad_f);
    for (size_t i = 0; i < 3; i++) {
      projection_s (grad_a[i], normal);
    }
    for (size_t k = 0; k < 2; k++) {
      for (size_t i = 0; i < 3; i++) {
        for (size_t j = 0; j < 3; j++) {
          bk(i+3*k,j) = 0.5*S*(hat_a(i,hat_A) + hat_a(i,hat_B))*grad_a[j][k];
        }
      }
    }
}
// b(v,q) = int_S div_s(v) q ds
void
compute_matrix_div_s_triangle_T (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& bk) 
{
    point A, B, C;
    Float aire;
    tensor inv_DF;
    ublas::vector<size_t> jv(4);
    subcompute_matrix_triangle_T (x, f, jv, A, B, C, aire);
    inv_piola_T (x, inv_DF);

    // definition des coordonnes des points a, b et c dans le triangle de reference
    point hat_A = inv_DF*(A-x[0]);
    point hat_B = inv_DF*(B-x[0]);
    point hat_C = inv_DF*(C-x[0]);
  
    // calcul de la transposee de l'inverse de la matrice B
    tensor transposee = trans(inv_DF);
    
    // la definition des gradients sur le triangle de reference
    point grad_hat_a[4];
    grad_hat_a[0] = point(-1,-1,-1);
    grad_hat_a[1] = point( 1, 0, 0);
    grad_hat_a[2] = point( 0, 1, 0);
    grad_hat_a[3] = point( 0, 0, 1);

    // calcul des valeurs des gradients sur chaque sommet du triangle K
    point grad_a [4];
    for (size_t i = 0; i < 4; i++) {
      grad_a[i] = transposee*grad_hat_a[i];
    }
    point grad_f = f[0]*grad_a[0] +f[1]*grad_a[1] +f[2]*grad_a[2] +f[3]*grad_a[3]; 
    point n = grad_f/norm(grad_f);
    vector<Float> div_s_a_0_0(4);
    vector<Float> div_s_0_a_0(4);
    vector<Float> div_s_0_0_a(4);
    for (size_t i = 0; i < 4; i++) {
      // div_s([a;0;0]) = (I-nxn):[grad(a)^T;0;0]
      // div_s([0;a;0]) = (I-nxn):[0;grad(a)^T;0]
      // div_s([0;0;a]) = (I-nxn):[0;0;grad(a)^T]
      div_s_a_0_0 [i] = (1-n[0]*n[0])*grad_a[i][0] + ( -n[0]*n[1])*grad_a[i][1] + (  -n[0]*n[2])*grad_a[i][2];
      div_s_0_a_0 [i] = ( -n[1]*n[0])*grad_a[i][0] + (1-n[1]*n[1])*grad_a[i][1] + (  -n[1]*n[2])*grad_a[i][2];
      div_s_0_0_a [i] = ( -n[2]*n[0])*grad_a[i][0] + ( -n[2]*n[1])*grad_a[i][1] + (1 -n[2]*n[2])*grad_a[i][2];
    }
    for (size_t i = 0; i < 4; i++) {
      for (size_t j = 0; j < 4; j++) {
        bk(i,j+4*0) = 1/3.*aire*(hat_T(i,hat_A) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_a_0_0[j];
        bk(i,j+4*1) = 1/3.*aire*(hat_T(i,hat_A) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_0_a_0[j];
        bk(i,j+4*2) = 1/3.*aire*(hat_T(i,hat_A) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_0_0_a[j];
      }
    }
}

void
compute_matrix_div_s_quadrilateral_T (const ublas::vector<point>& x, const ublas::vector<Float>& f,
             const quadruplet& q, ublas::matrix<Float>& bk) 
{
  point A, B, C, D;
    Float aire;
    tensor inv_DF;
    subcompute_matrix_quadrilateral_T (x, f, q, A, B, C, D, aire);
    inv_piola_T (x, inv_DF);

    // definition des coordonnes des points a, b et c dans le triangle de reference
    point hat_A = inv_DF*(A-x[0]);
    point hat_B = inv_DF*(B-x[0]);
    point hat_C = inv_DF*(C-x[0]);
    point hat_D = inv_DF*(D-x[0]);
    Float aire_ABC = 1./2.*norm(vect(A-B,A-C));
    Float aire_DBC = 1./2.*norm(vect(B-C,B-D));
  
    // calcul de la transposee de l'inverse de la matrice B
    tensor transposee = trans(inv_DF);
    
    // la definition des gradients sur le triangle de reference
    point grad_hat_a[4];
    grad_hat_a[0] = point(-1,-1,-1);
    grad_hat_a[1] = point( 1, 0, 0);
    grad_hat_a[2] = point( 0, 1, 0);
    grad_hat_a[3] = point( 0, 0, 1);

    // calcul des valeurs des gradients sur chaque sommet du triangle K
    point grad_a [4];
    for (size_t i = 0; i < 4; i++) {
      grad_a[i] = transposee*grad_hat_a[i];
    }
    point grad_f = f[0]*grad_a[0] +f[1]*grad_a[1] +f[2]*grad_a[2] +f[3]*grad_a[3]; 
    point n = grad_f/norm(grad_f);
    vector<Float> div_s_a_0_0(4);
    vector<Float> div_s_0_a_0(4);
    vector<Float> div_s_0_0_a(4);
    for (size_t i = 0; i < 4; i++) {
      // div_s([a;0;0]) = (I-nxn):[grad(a)^T;0;0]
      // div_s([0;a;0]) = (I-nxn):[0;grad(a)^T;0]
      // div_s([0;0;a]) = (I-nxn):[0;0;grad(a)^T]
      div_s_a_0_0 [i] = (1-n[0]*n[0])*grad_a[i][0] + ( -n[0]*n[1])*grad_a[i][1] + ( -n[0]*n[2])*grad_a[i][2];
      div_s_0_a_0 [i] = ( -n[1]*n[0])*grad_a[i][0] + (1-n[1]*n[1])*grad_a[i][1] + ( -n[1]*n[2])*grad_a[i][2];
      div_s_0_0_a [i] = ( -n[2]*n[0])*grad_a[i][0] + ( -n[2]*n[1])*grad_a[i][1] + (1-n[2]*n[2])*grad_a[i][2];
    }
    for (size_t i = 0; i < 4; i++) {
      for (size_t j = 0; j < 4; j++) {
        bk(i,j+4*0) = 1/3.*aire_ABC*(hat_T(i,hat_A) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_a_0_0[j]
                    + 1/3.*aire_DBC*(hat_T(i,hat_D) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_a_0_0[j];
        bk(i,j+4*1) = 1/3.*aire_ABC*(hat_T(i,hat_A) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_0_a_0[j]
                    + 1/3.*aire_DBC*(hat_T(i,hat_D) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_0_a_0[j];
        bk(i,j+4*2) = 1/3.*aire_ABC*(hat_T(i,hat_A) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_0_0_a[j]
                    + 1/3.*aire_DBC*(hat_T(i,hat_D) + hat_T(i,hat_B) + hat_T(i, hat_C))*div_s_0_0_a[j];
      }
    }
}

void
compute_matrix_div_s_T (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& bk) {
    quadruplet q;
    if (intersection_is_quadrilateral_T (f, q)) {
        compute_matrix_div_s_quadrilateral_T (x, f, q, bk);
    } else {
        compute_matrix_div_s_triangle_T(x, f, bk);
    }
}


void
compute_matrix_div_s_t (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& bk) {
    point A, B;
    Float S;
    tensor inv_DF;
    ublas::vector<size_t> j(3);
    subcompute_matrix_t (x, f, j, A, B, S);
    inv_piola_t (x, inv_DF);

    // definition des coordonnes des points a, b et c dans le triangle de reference
    point hat_A = inv_DF*(A-x[0]);
    point hat_B = inv_DF*(B-x[0]);
  
    // calcul de la transposee de l'inverse de la matrice B
    tensor transposee;
    for (int i=0;i<2;i++) {
      for (int j=0;j<2;j++) {
        transposee (j,i) = inv_DF (i,j);
      }
    }
    // la definition des gradients sur le triangle de reference
    point grad_hat_a[3];
    grad_hat_a[0] = point(-1,-1);
    grad_hat_a[1] = point( 1, 0);
    grad_hat_a[2] = point( 0, 1);

    // calcul des valeurs des gradients sur chaque sommet du triangle K
    point grad_a [3];
    for (size_t i = 0; i < 3; i++) {
      grad_a[i] = transposee*grad_hat_a[i];
    }
    point grad_f = f[0]*grad_a[0] +f[1]*grad_a[1] +f[2]*grad_a[2]; 
    point n = grad_f/norm(grad_f);
    point div_s_a_0;
    point div_s_0_a;
    for (size_t i = 0; i < 3; i++) {
      // div_s([a;0]) = (I-nxn):[grad(a)^T;0]
      // div_s([0;a]) = (I-nxn):[0;grad(a)^T]
      div_s_a_0 [i] = (1-n[0]*n[0])*grad_a[i][0] + ( -n[0]*n[1])*grad_a[i][1];
      div_s_0_a [i] = ( -n[1]*n[0])*grad_a[i][0] + (1-n[1]*n[1])*grad_a[i][1];
    }
    for (size_t i = 0; i < 3; i++) {
      for (size_t j = 0; j < 3; j++) {
        bk(i,j+3*0) = 0.5*S*(hat_a(i,hat_A) + hat_a(i,hat_B))*div_s_a_0[j];
        bk(i,j+3*1) = 0.5*S*(hat_a(i,hat_A) + hat_a(i,hat_B))*div_s_0_a[j];
      }
    }
}

void
compute_matrix_2D_s_D_s_T (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& bk) {
    quadruplet q;
    point A, B, C;
    Float aire;
    if (intersection_is_quadrilateral_T (f, q)) {
        point D;
        subcompute_matrix_quadrilateral_T (x, f, q, A, B, C, D, aire);
    } else {
        ublas::vector<size_t> jv(4);
        subcompute_matrix_triangle_T (x, f, jv, A, B, C, aire);
    }
    tensor inv_DF;
    inv_piola_T (x, inv_DF);

    // calcul de la transposee de l'inverse de la matrice B
    tensor transposee = trans(inv_DF);
    
    // la definition des gradients sur le triangle de reference
    point grad_hat_a[4];
    grad_hat_a[0] = point(-1,-1,-1);
    grad_hat_a[1] = point( 1, 0, 0);
    grad_hat_a[2] = point( 0, 1, 0);
    grad_hat_a[3] = point( 0, 0, 1);

    // calcul des valeurs des gradients sur chaque sommet du triangle K
    point gphi [4];
    for (size_t i = 0; i < 4; i++) {
      gphi[i] = transposee*grad_hat_a[i];
    }
    point grad_f = f[0]*gphi[0] +f[1]*gphi[1] +f[2]*gphi[2] +f[3]*gphi[3]; 
    point n = grad_f/norm(grad_f);

    /* Code generated by maxima. Source :
      g_gu[a,b]:=sum(U[j,a-1]*gphi[j][b-1],j,0,3); 
      g_gv[a,b]:=sum(V[i,a-1]*gphi[i][b-1],i,0,3); 
      u:makelist(U[mod(i,4),floor(i/4)], i, 0, 11);
      v:makelist(V[mod(i,4),floor(i/4)], i, 0, 11);
      gu:genmatrix(g_gu,3,3);
      gv:genmatrix(g_gv,3,3);
      D(gu):=1/2*(gu+transpose(gu));
      v_n:[n[0],n[1],n[2]];
      P:ident(3)-transpose(v_n).v_n;
      ddot(A,B):=list_matrix_entries(A).list_matrix_entries(B);
      dev:ddot(2*P.D(gu).P,P.D(gv).P);
      thedev:coeff(expand(l*dev),l); 
      sys:makelist(makelist(bk(i-1,j-1)=coeff(coeff(thedev, v[i]),u[j]), j, 1, 12), i, 1, 12);
      file_output_append:true;
      for i:1 thru 12 do for j:i thru 12 do stringout("2Ds_Ds.c",subst("^"=pow,ratsimp(sys[i][j])));
      for i:1 thru 12 do for j:1 thru i-1 do stringout("2Ds_Ds.c",bk(i-1,j-1)=bk(j-1,i-1));
    */


bk(0,0) = 2*pow(n[0],2)*pow(gphi[0][2],2)*pow(n[2],6)+gphi[0][2]*(4*gphi[0][0]*pow(n[0],3)+4*gphi[0][1]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[0][2],2)+2*pow(n[0],2)*pow(gphi[0][1],2)*pow(n[1],2)+2*pow(gphi[0][0],2)*pow(n[0],4)+4*gphi[0][0]*gphi[0][1]*n[1]*pow(n[0],3))*pow(n[2],4)+gphi[0][2]*(8*gphi[0][1]*pow(n[0],2)*pow(n[1],3)+8*gphi[0][0]*pow(n[0],3)*pow(n[1],2)+8*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)-16*gphi[0][0]*pow(n[0],3)+4*gphi[0][0]*n[0])*pow(n[2],3)+((2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[0][2],2)+4*pow(n[0],2)*pow(gphi[0][1],2)*pow(n[1],4)+8*gphi[0][0]*gphi[0][1]*pow(n[0],3)*pow(n[1],3)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],4))*pow(n[1],2)+pow(n[0],2)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],6)+gphi[0][1]*n[1]*(8*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+4*gphi[0][0]*n[0])-8*pow(gphi[0][0],2)*pow(n[0],4)+4*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[2],2)+(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[0][2],2)+2*pow(n[0],2)*pow(gphi[0][1],2)*pow(n[1],6)+gphi[0][2]*n[2]*(4*gphi[0][1]*pow(n[0],2)*pow(n[1],5)+4*gphi[0][0]*pow(n[0],3)*pow(n[1],4)+gphi[0][1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)*pow(n[1],3)+(8*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+4*gphi[0][0]*n[0])*pow(n[1],2)+4*gphi[0][0]*pow(n[0],7)+gphi[0][1]*n[1]*(4*pow(n[0],6)-16*pow(n[0],4)+18*pow(n[0],2)-4)-16*gphi[0][0]*pow(n[0],5)+20*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])+4*gphi[0][0]*gphi[0][1]*pow(n[0],3)*pow(n[1],5)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],4))*pow(n[1],4)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+4*gphi[0][0]*n[0])*pow(n[1],3)+((2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],6)-8*pow(gphi[0][0],2)*pow(n[0],4)+4*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],2)+(pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],8)+gphi[0][1]*n[1]*(4*gphi[0][0]*pow(n[0],7)-16*gphi[0][0]*pow(n[0],5)+20*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])-8*pow(gphi[0][0],2)*pow(n[0],6)+12*pow(gphi[0][0],2)*pow(n[0],4)-8*pow(gphi[0][0],2)*pow(n[0],2)+2*pow(gphi[0][0],2);

bk(0,1) = 2*gphi[0][2]*gphi[1][2]*pow(n[0],2)*pow(n[2],6)+(gphi[0][2]*(2*gphi[1][0]*pow(n[0],3)+2*gphi[1][1]*n[1]*pow(n[0],2))+gphi[1][2]*(2*gphi[0][0]*pow(n[0],3)+2*gphi[0][1]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[1][2]*(4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[0][1]*gphi[1][1]*pow(n[0],2)*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],4)+n[1]*(2*gphi[0][0]*gphi[1][1]*pow(n[0],3)+2*gphi[1][0]*gphi[0][1]*pow(n[0],3)))*pow(n[2],4)+(gphi[0][2]*(4*gphi[1][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[1][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0])+gphi[1][2]*(4*gphi[0][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[0][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))*pow(n[2],3)+(gphi[0][2]*gphi[1][2]*(2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[0][1]*gphi[1][1]*pow(n[0],2)*pow(n[1],4)+(4*gphi[0][0]*gphi[1][1]*pow(n[0],3)+4*gphi[1][0]*gphi[0][1]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+4*gphi[0][0]*gphi[1][0]*pow(n[0],4))*pow(n[1],2)+4*gphi[0][0]*gphi[1][0]*pow(n[0],6)+n[1]*(gphi[0][1]*(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[1][0]*pow(n[0],4)+gphi[0][1]*gphi[1][1]*pow(n[0],2)+4*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[2],2)+2*gphi[0][1]*gphi[1][1]*pow(n[0],2)*pow(n[1],6)+n[2]*(gphi[0][2]*(2*gphi[1][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[1][0]*pow(n[0],3)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],7)+gphi[1][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[1][0]*pow(n[0],5)+10*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0])+gphi[1][2]*(2*gphi[0][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[0][0]*pow(n[0],3)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],7)+gphi[0][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[0][0]*pow(n[0],5)+10*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))+(2*gphi[0][0]*gphi[1][1]*pow(n[0],3)+2*gphi[1][0]*gphi[0][1]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[0][0]*gphi[1][0]*pow(n[0],4))*pow(n[1],4)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))*pow(n[1],3)+gphi[0][2]*gphi[1][2]*(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[0][0]*gphi[1][0]*pow(n[0],6)-8*gphi[0][0]*gphi[1][0]*pow(n[0],4)+4*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],8)+n[1]*(gphi[0][1]*(2*gphi[1][0]*pow(n[0],7)-8*gphi[1][0]*pow(n[0],5)+10*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],7)-8*gphi[0][0]*pow(n[0],5)+10*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[1][0]*pow(n[0],6)+gphi[0][1]*gphi[1][1]*(pow(n[0],4)-2*pow(n[0],2)+1)+12*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+2*gphi[0][0]*gphi[1][0];

bk(0,2) = 2*gphi[0][2]*gphi[2][2]*pow(n[0],2)*pow(n[2],6)+(gphi[0][2]*(2*gphi[2][0]*pow(n[0],3)+2*gphi[2][1]*n[1]*pow(n[0],2))+gphi[2][2]*(2*gphi[0][0]*pow(n[0],3)+2*gphi[0][1]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[2][2]*(4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[0][1]*gphi[2][1]*pow(n[0],2)*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],4)+n[1]*(2*gphi[0][0]*gphi[2][1]*pow(n[0],3)+2*gphi[2][0]*gphi[0][1]*pow(n[0],3)))*pow(n[2],4)+(gphi[0][2]*(4*gphi[2][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[2][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])+gphi[2][2]*(4*gphi[0][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[0][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))*pow(n[2],3)+(gphi[0][2]*gphi[2][2]*(2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[0][1]*gphi[2][1]*pow(n[0],2)*pow(n[1],4)+(4*gphi[0][0]*gphi[2][1]*pow(n[0],3)+4*gphi[2][0]*gphi[0][1]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+4*gphi[0][0]*gphi[2][0]*pow(n[0],4))*pow(n[1],2)+4*gphi[0][0]*gphi[2][0]*pow(n[0],6)+n[1]*(gphi[0][1]*(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[2][0]*pow(n[0],4)+gphi[0][1]*gphi[2][1]*pow(n[0],2)+4*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[2],2)+2*gphi[0][1]*gphi[2][1]*pow(n[0],2)*pow(n[1],6)+n[2]*(gphi[0][2]*(2*gphi[2][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[2][0]*pow(n[0],3)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],7)+gphi[2][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[2][0]*pow(n[0],5)+10*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[0][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[0][0]*pow(n[0],3)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],7)+gphi[0][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[0][0]*pow(n[0],5)+10*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))+(2*gphi[0][0]*gphi[2][1]*pow(n[0],3)+2*gphi[2][0]*gphi[0][1]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[0][0]*gphi[2][0]*pow(n[0],4))*pow(n[1],4)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))*pow(n[1],3)+gphi[0][2]*gphi[2][2]*(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[0][0]*gphi[2][0]*pow(n[0],6)-8*gphi[0][0]*gphi[2][0]*pow(n[0],4)+4*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],8)+n[1]*(gphi[0][1]*(2*gphi[2][0]*pow(n[0],7)-8*gphi[2][0]*pow(n[0],5)+10*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],7)-8*gphi[0][0]*pow(n[0],5)+10*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[2][0]*pow(n[0],6)+gphi[0][1]*gphi[2][1]*(pow(n[0],4)-2*pow(n[0],2)+1)+12*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+2*gphi[0][0]*gphi[2][0];

bk(0,3) = 2*gphi[0][2]*gphi[3][2]*pow(n[0],2)*pow(n[2],6)+(gphi[0][2]*(2*gphi[3][0]*pow(n[0],3)+2*gphi[3][1]*n[1]*pow(n[0],2))+gphi[3][2]*(2*gphi[0][0]*pow(n[0],3)+2*gphi[0][1]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[3][2]*(4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[0][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],4)+n[1]*(2*gphi[0][0]*gphi[3][1]*pow(n[0],3)+2*gphi[3][0]*gphi[0][1]*pow(n[0],3)))*pow(n[2],4)+(gphi[0][2]*(4*gphi[3][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[3][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[0][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[0][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))*pow(n[2],3)+(gphi[0][2]*gphi[3][2]*(2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[0][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],4)+(4*gphi[0][0]*gphi[3][1]*pow(n[0],3)+4*gphi[3][0]*gphi[0][1]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+4*gphi[0][0]*gphi[3][0]*pow(n[0],4))*pow(n[1],2)+4*gphi[0][0]*gphi[3][0]*pow(n[0],6)+n[1]*(gphi[0][1]*(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[3][0]*pow(n[0],4)+gphi[0][1]*gphi[3][1]*pow(n[0],2)+4*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[2],2)+2*gphi[0][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],6)+n[2]*(gphi[0][2]*(2*gphi[3][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[3][0]*pow(n[0],3)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],7)+gphi[3][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[3][0]*pow(n[0],5)+10*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[0][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[0][0]*pow(n[0],3)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],7)+gphi[0][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[0][0]*pow(n[0],5)+10*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))+(2*gphi[0][0]*gphi[3][1]*pow(n[0],3)+2*gphi[3][0]*gphi[0][1]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[0][0]*gphi[3][0]*pow(n[0],4))*pow(n[1],4)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+2*gphi[0][0]*n[0]))*pow(n[1],3)+gphi[0][2]*gphi[3][2]*(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[0][0]*gphi[3][0]*pow(n[0],6)-8*gphi[0][0]*gphi[3][0]*pow(n[0],4)+4*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],8)+n[1]*(gphi[0][1]*(2*gphi[3][0]*pow(n[0],7)-8*gphi[3][0]*pow(n[0],5)+10*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],7)-8*gphi[0][0]*pow(n[0],5)+10*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[3][0]*pow(n[0],6)+gphi[0][1]*gphi[3][1]*(pow(n[0],4)-2*pow(n[0],2)+1)+12*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+2*gphi[0][0]*gphi[3][0];

bk(0,4) = 2*n[0]*n[1]*pow(gphi[0][2],2)*pow(n[2],6)+gphi[0][2]*(4*n[0]*gphi[0][1]*pow(n[1],2)+4*gphi[0][0]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))*pow(gphi[0][2],2)+2*n[0]*pow(gphi[0][1],2)*pow(n[1],3)+4*gphi[0][0]*gphi[0][1]*pow(n[0],2)*pow(n[1],2)+2*n[1]*pow(gphi[0][0],2)*pow(n[0],3))*pow(n[2],4)+gphi[0][2]*(8*n[0]*gphi[0][1]*pow(n[1],4)+8*gphi[0][0]*pow(n[0],2)*pow(n[1],3)+gphi[0][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],2)+n[1]*(8*gphi[0][0]*pow(n[0],4)-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])+n[0]*gphi[0][1])*pow(n[2],3)+((2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))*pow(gphi[0][2],2)+4*n[0]*pow(gphi[0][1],2)*pow(n[1],5)+8*gphi[0][0]*gphi[0][1]*pow(n[0],2)*pow(n[1],4)+((4*pow(n[0],3)-8*n[0])*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],3))*pow(n[1],3)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],4)-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],2)+n[1]*(2*n[0]*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],5)-8*pow(gphi[0][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[0][0],2))+gphi[0][0]*gphi[0][1]*pow(n[0],2))*pow(n[2],2)+(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))*pow(gphi[0][2],2)+2*n[0]*pow(gphi[0][1],2)*pow(n[1],7)+gphi[0][2]*n[2]*(4*n[0]*gphi[0][1]*pow(n[1],6)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],5)+gphi[0][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],4)+(8*gphi[0][0]*pow(n[0],4)-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],6)-16*gphi[0][0]*pow(n[0],4)+17*gphi[0][0]*pow(n[0],2)-2*gphi[0][0])+gphi[0][1]*(pow(n[0],3)-2*n[0]))+4*gphi[0][0]*gphi[0][1]*pow(n[0],2)*pow(n[1],6)+((4*pow(n[0],3)-8*n[0])*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],3))*pow(n[1],5)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],4)-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],4)+((2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],5)-8*pow(gphi[0][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[0][0],2))*pow(n[1],3)+gphi[0][1]*(4*gphi[0][0]*pow(n[0],6)-16*gphi[0][0]*pow(n[0],4)+18*gphi[0][0]*pow(n[0],2)-2*gphi[0][0])*pow(n[1],2)+n[1]*((2*pow(n[0],3)-4*n[0])*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],7)-8*pow(gphi[0][0],2)*pow(n[0],5)+10*pow(gphi[0][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[0][0],2))+gphi[0][1]*(gphi[0][0]*pow(n[0],4)-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(0,5) = 2*n[0]*n[1]*gphi[0][2]*gphi[1][2]*pow(n[2],6)+(gphi[0][2]*(2*n[0]*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[1]*pow(n[0],2))+gphi[1][2]*(2*n[0]*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[1][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],3)+(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[0][2]*(4*n[0]*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))+gphi[1][2]*(4*n[0]*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))+n[0]*gphi[0][1]))*pow(n[2],3)+(gphi[0][2]*gphi[1][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],5)+(4*gphi[0][0]*gphi[1][1]*pow(n[0],2)+4*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[1][0]*pow(n[0],5)-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+2*n[0]*gphi[0][1]*gphi[1][1]+2*gphi[0][0]*gphi[1][0]*n[0])+gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],7)+n[2]*(gphi[0][2]*(2*n[0]*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+9*gphi[1][0]*pow(n[0],2)-2*gphi[1][0]))+gphi[1][2]*(2*n[0]*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2))+gphi[0][1]*(pow(n[0],3)-2*n[0])))+(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],4)+gphi[0][2]*gphi[1][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],5)-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+2*gphi[0][0]*gphi[1][0]*n[0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+10*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[1][0]*pow(n[0],7)-8*gphi[0][0]*gphi[1][0]*pow(n[0],5)+gphi[0][1]*gphi[1][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[0][0]*gphi[1][0]*pow(n[0],3)-4*gphi[0][0]*gphi[1][0]*n[0])+gphi[0][1]*(gphi[1][0]*pow(n[0],4)-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(0,6) = 2*n[0]*n[1]*gphi[0][2]*gphi[2][2]*pow(n[2],6)+(gphi[0][2]*(2*n[0]*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[1]*pow(n[0],2))+gphi[2][2]*(2*n[0]*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[2][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[0][2]*(4*n[0]*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0]))+gphi[2][2]*(4*n[0]*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))+n[0]*gphi[0][1]))*pow(n[2],3)+(gphi[0][2]*gphi[2][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[0][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[2][0]*pow(n[0],5)-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+2*n[0]*gphi[0][1]*gphi[2][1]+2*gphi[0][0]*gphi[2][0]*n[0])+gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],7)+n[2]*(gphi[0][2]*(2*n[0]*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+9*gphi[2][0]*pow(n[0],2)-2*gphi[2][0]))+gphi[2][2]*(2*n[0]*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2))+gphi[0][1]*(pow(n[0],3)-2*n[0])))+(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],4)+gphi[0][2]*gphi[2][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],5)-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+2*gphi[0][0]*gphi[2][0]*n[0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+10*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[2][0]*pow(n[0],7)-8*gphi[0][0]*gphi[2][0]*pow(n[0],5)+gphi[0][1]*gphi[2][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[0][0]*gphi[2][0]*pow(n[0],3)-4*gphi[0][0]*gphi[2][0]*n[0])+gphi[0][1]*(gphi[2][0]*pow(n[0],4)-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(0,7) = 2*n[0]*n[1]*gphi[0][2]*gphi[3][2]*pow(n[2],6)+(gphi[0][2]*(2*n[0]*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[1]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[3][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[0][2]*(4*n[0]*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0]))+gphi[3][2]*(4*n[0]*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))+n[0]*gphi[0][1]))*pow(n[2],3)+(gphi[0][2]*gphi[3][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[0][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[3][0]*pow(n[0],5)-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+2*n[0]*gphi[0][1]*gphi[3][1]+2*gphi[0][0]*gphi[3][0]*n[0])+gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],7)+n[2]*(gphi[0][2]*(2*n[0]*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+9*gphi[3][0]*pow(n[0],2)-2*gphi[3][0]))+gphi[3][2]*(2*n[0]*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2))+gphi[0][1]*(pow(n[0],3)-2*n[0])))+(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],4)+gphi[0][2]*gphi[3][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],5)-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+2*gphi[0][0]*gphi[3][0]*n[0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+10*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[3][0]*pow(n[0],7)-8*gphi[0][0]*gphi[3][0]*pow(n[0],5)+gphi[0][1]*gphi[3][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[0][0]*gphi[3][0]*pow(n[0],3)-4*gphi[0][0]*gphi[3][0]*n[0])+gphi[0][1]*(gphi[3][0]*pow(n[0],4)-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(0,8) = 2*n[0]*pow(gphi[0][2],2)*pow(n[2],7)+gphi[0][2]*(4*gphi[0][0]*pow(n[0],2)+4*n[0]*gphi[0][1]*n[1])*pow(n[2],6)+((4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])*pow(gphi[0][2],2)+2*n[0]*pow(gphi[0][1],2)*pow(n[1],2)+2*pow(gphi[0][0],2)*pow(n[0],3)+4*gphi[0][0]*gphi[0][1]*n[1]*pow(n[0],2))*pow(n[2],5)+gphi[0][2]*(8*n[0]*gphi[0][1]*pow(n[1],3)+8*gphi[0][0]*pow(n[0],2)*pow(n[1],2)+8*gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(8*pow(n[0],3)-16*n[0])-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[2],4)+((2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[0][2],2)+4*n[0]*pow(gphi[0][1],2)*pow(n[1],4)+8*gphi[0][0]*gphi[0][1]*pow(n[0],2)*pow(n[1],3)+((4*pow(n[0],3)-8*n[0])*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],3))*pow(n[1],2)+n[0]*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],5)+gphi[0][1]*n[1]*(8*gphi[0][0]*pow(n[0],4)-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])-8*pow(gphi[0][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[0][0],2))*pow(n[2],3)+gphi[0][2]*(4*n[0]*gphi[0][1]*pow(n[1],5)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],4)+gphi[0][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],3)+(8*gphi[0][0]*pow(n[0],4)-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],2)+4*gphi[0][0]*pow(n[0],6)+gphi[0][1]*n[1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])-16*gphi[0][0]*pow(n[0],4)+18*gphi[0][0]*pow(n[0],2)-2*gphi[0][0])*pow(n[2],2)+n[2]*((2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])*pow(gphi[0][2],2)+2*n[0]*pow(gphi[0][1],2)*pow(n[1],6)+4*gphi[0][0]*gphi[0][1]*pow(n[0],2)*pow(n[1],5)+((4*pow(n[0],3)-8*n[0])*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],3))*pow(n[1],4)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],4)-16*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],3)+((2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],5)-8*pow(gphi[0][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[0][0],2))*pow(n[1],2)+(pow(n[0],3)-2*n[0])*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],7)+gphi[0][1]*n[1]*(4*gphi[0][0]*pow(n[0],6)-16*gphi[0][0]*pow(n[0],4)+17*gphi[0][0]*pow(n[0],2)-2*gphi[0][0])-8*pow(gphi[0][0],2)*pow(n[0],5)+10*pow(gphi[0][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[0][0],2))+gphi[0][2]*(n[0]*gphi[0][1]*pow(n[1],3)+gphi[0][0]*pow(n[0],2)*pow(n[1],2)+gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(0,9) = 2*n[0]*gphi[0][2]*gphi[1][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[1][0]*pow(n[0],2)+2*n[0]*gphi[1][1]*n[1])+gphi[1][2]*(2*gphi[0][0]*pow(n[0],2)+2*n[0]*gphi[0][1]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[1][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],3)+n[1]*(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*(4*n[0]*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])+gphi[1][2]*(4*n[0]*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[0][0]*pow(n[0],2)))*pow(n[2],4)+(gphi[0][2]*gphi[1][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],4)+(4*gphi[0][0]*gphi[1][1]*pow(n[0],2)+4*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[0][0]*gphi[1][0]*pow(n[0],5)+n[1]*(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+n[0]*gphi[0][1]*gphi[1][1]+2*gphi[0][0]*gphi[1][0]*n[0])*pow(n[2],3)+(gphi[0][2]*(2*n[0]*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],6)+gphi[1][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[1][0]*pow(n[0],4)+10*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])+gphi[1][2]*(2*n[0]*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*pow(n[0],6)+gphi[0][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))*pow(n[2],2)+n[2]*(2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],6)+(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],4)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],3)+gphi[0][2]*gphi[1][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],5)-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+2*gphi[0][0]*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],7)+n[1]*(gphi[0][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+9*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))-8*gphi[0][0]*gphi[1][0]*pow(n[0],5)+gphi[0][1]*gphi[1][1]*(pow(n[0],3)-2*n[0])+10*gphi[0][0]*gphi[1][0]*pow(n[0],3)-4*gphi[0][0]*gphi[1][0]*n[0])+gphi[0][2]*(n[0]*gphi[1][1]*pow(n[1],3)+gphi[1][0]*pow(n[0],2)*pow(n[1],2)+gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(0,10) = 2*n[0]*gphi[0][2]*gphi[2][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[2][0]*pow(n[0],2)+2*n[0]*gphi[2][1]*n[1])+gphi[2][2]*(2*gphi[0][0]*pow(n[0],2)+2*n[0]*gphi[0][1]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[2][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],3)+n[1]*(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*(4*n[0]*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][2]*(4*n[0]*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[0][0]*pow(n[0],2)))*pow(n[2],4)+(gphi[0][2]*gphi[2][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],4)+(4*gphi[0][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[0][0]*gphi[2][0]*pow(n[0],5)+n[1]*(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+n[0]*gphi[0][1]*gphi[2][1]+2*gphi[0][0]*gphi[2][0]*n[0])*pow(n[2],3)+(gphi[0][2]*(2*n[0]*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],6)+gphi[2][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[2][0]*pow(n[0],4)+10*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])+gphi[2][2]*(2*n[0]*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*pow(n[0],6)+gphi[0][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))*pow(n[2],2)+n[2]*(2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],6)+(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],4)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],3)+gphi[0][2]*gphi[2][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],5)-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+2*gphi[0][0]*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],7)+n[1]*(gphi[0][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+9*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))-8*gphi[0][0]*gphi[2][0]*pow(n[0],5)+gphi[0][1]*gphi[2][1]*(pow(n[0],3)-2*n[0])+10*gphi[0][0]*gphi[2][0]*pow(n[0],3)-4*gphi[0][0]*gphi[2][0]*n[0])+gphi[0][2]*(n[0]*gphi[2][1]*pow(n[1],3)+gphi[2][0]*pow(n[0],2)*pow(n[1],2)+gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(0,11) = 2*n[0]*gphi[0][2]*gphi[3][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[3][0]*pow(n[0],2)+2*n[0]*gphi[3][1]*n[1])+gphi[3][2]*(2*gphi[0][0]*pow(n[0],2)+2*n[0]*gphi[0][1]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[3][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],3)+n[1]*(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*(4*n[0]*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][2]*(4*n[0]*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[0][0]*pow(n[0],2)))*pow(n[2],4)+(gphi[0][2]*gphi[3][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[0][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[0][0]*gphi[3][0]*pow(n[0],5)+n[1]*(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+n[0]*gphi[0][1]*gphi[3][1]+2*gphi[0][0]*gphi[3][0]*n[0])*pow(n[2],3)+(gphi[0][2]*(2*n[0]*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],6)+gphi[3][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[3][0]*pow(n[0],4)+10*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][2]*(2*n[0]*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*pow(n[0],6)+gphi[0][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))*pow(n[2],2)+n[2]*(2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],4)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)))*pow(n[1],3)+gphi[0][2]*gphi[3][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],5)-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+2*gphi[0][0]*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],7)+n[1]*(gphi[0][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+9*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+8*gphi[0][0]*pow(n[0],2)))-8*gphi[0][0]*gphi[3][0]*pow(n[0],5)+gphi[0][1]*gphi[3][1]*(pow(n[0],3)-2*n[0])+10*gphi[0][0]*gphi[3][0]*pow(n[0],3)-4*gphi[0][0]*gphi[3][0]*n[0])+gphi[0][2]*(n[0]*gphi[3][1]*pow(n[1],3)+gphi[3][0]*pow(n[0],2)*pow(n[1],2)+gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(1,1) = 2*pow(n[0],2)*pow(gphi[1][2],2)*pow(n[2],6)+gphi[1][2]*(4*gphi[1][0]*pow(n[0],3)+4*gphi[1][1]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[1][2],2)+2*pow(n[0],2)*pow(gphi[1][1],2)*pow(n[1],2)+2*pow(gphi[1][0],2)*pow(n[0],4)+4*gphi[1][0]*gphi[1][1]*n[1]*pow(n[0],3))*pow(n[2],4)+gphi[1][2]*(8*gphi[1][1]*pow(n[0],2)*pow(n[1],3)+8*gphi[1][0]*pow(n[0],3)*pow(n[1],2)+8*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)-16*gphi[1][0]*pow(n[0],3)+4*gphi[1][0]*n[0])*pow(n[2],3)+((2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[1][2],2)+4*pow(n[0],2)*pow(gphi[1][1],2)*pow(n[1],4)+8*gphi[1][0]*gphi[1][1]*pow(n[0],3)*pow(n[1],3)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],4))*pow(n[1],2)+pow(n[0],2)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],6)+gphi[1][1]*n[1]*(8*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+4*gphi[1][0]*n[0])-8*pow(gphi[1][0],2)*pow(n[0],4)+4*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[2],2)+(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[1][2],2)+2*pow(n[0],2)*pow(gphi[1][1],2)*pow(n[1],6)+gphi[1][2]*n[2]*(4*gphi[1][1]*pow(n[0],2)*pow(n[1],5)+4*gphi[1][0]*pow(n[0],3)*pow(n[1],4)+gphi[1][1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)*pow(n[1],3)+(8*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+4*gphi[1][0]*n[0])*pow(n[1],2)+4*gphi[1][0]*pow(n[0],7)+gphi[1][1]*n[1]*(4*pow(n[0],6)-16*pow(n[0],4)+18*pow(n[0],2)-4)-16*gphi[1][0]*pow(n[0],5)+20*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+4*gphi[1][0]*gphi[1][1]*pow(n[0],3)*pow(n[1],5)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],4))*pow(n[1],4)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+4*gphi[1][0]*n[0])*pow(n[1],3)+((2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],6)-8*pow(gphi[1][0],2)*pow(n[0],4)+4*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],2)+(pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],8)+gphi[1][1]*n[1]*(4*gphi[1][0]*pow(n[0],7)-16*gphi[1][0]*pow(n[0],5)+20*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])-8*pow(gphi[1][0],2)*pow(n[0],6)+12*pow(gphi[1][0],2)*pow(n[0],4)-8*pow(gphi[1][0],2)*pow(n[0],2)+2*pow(gphi[1][0],2);

bk(1,2) = 2*gphi[1][2]*gphi[2][2]*pow(n[0],2)*pow(n[2],6)+(gphi[1][2]*(2*gphi[2][0]*pow(n[0],3)+2*gphi[2][1]*n[1]*pow(n[0],2))+gphi[2][2]*(2*gphi[1][0]*pow(n[0],3)+2*gphi[1][1]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*gphi[2][2]*(4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[1][1]*gphi[2][1]*pow(n[0],2)*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],4)+n[1]*(2*gphi[1][0]*gphi[2][1]*pow(n[0],3)+2*gphi[2][0]*gphi[1][1]*pow(n[0],3)))*pow(n[2],4)+(gphi[1][2]*(4*gphi[2][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[2][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])+gphi[2][2]*(4*gphi[1][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[1][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0]))*pow(n[2],3)+(gphi[1][2]*gphi[2][2]*(2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[1][1]*gphi[2][1]*pow(n[0],2)*pow(n[1],4)+(4*gphi[1][0]*gphi[2][1]*pow(n[0],3)+4*gphi[2][0]*gphi[1][1]*pow(n[0],3))*pow(n[1],3)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+4*gphi[1][0]*gphi[2][0]*pow(n[0],4))*pow(n[1],2)+4*gphi[1][0]*gphi[2][0]*pow(n[0],6)+n[1]*(gphi[1][1]*(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[2][0]*pow(n[0],4)+gphi[1][1]*gphi[2][1]*pow(n[0],2)+4*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[2],2)+2*gphi[1][1]*gphi[2][1]*pow(n[0],2)*pow(n[1],6)+n[2]*(gphi[1][2]*(2*gphi[2][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[2][0]*pow(n[0],3)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],7)+gphi[2][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[2][0]*pow(n[0],5)+10*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[1][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[1][0]*pow(n[0],3)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],7)+gphi[1][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[1][0]*pow(n[0],5)+10*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0]))+(2*gphi[1][0]*gphi[2][1]*pow(n[0],3)+2*gphi[2][0]*gphi[1][1]*pow(n[0],3))*pow(n[1],5)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[1][0]*gphi[2][0]*pow(n[0],4))*pow(n[1],4)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0]))*pow(n[1],3)+gphi[1][2]*gphi[2][2]*(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[1][0]*gphi[2][0]*pow(n[0],6)-8*gphi[1][0]*gphi[2][0]*pow(n[0],4)+4*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],8)+n[1]*(gphi[1][1]*(2*gphi[2][0]*pow(n[0],7)-8*gphi[2][0]*pow(n[0],5)+10*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],7)-8*gphi[1][0]*pow(n[0],5)+10*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[2][0]*pow(n[0],6)+gphi[1][1]*gphi[2][1]*(pow(n[0],4)-2*pow(n[0],2)+1)+12*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+2*gphi[1][0]*gphi[2][0];

bk(1,3) = 2*gphi[1][2]*gphi[3][2]*pow(n[0],2)*pow(n[2],6)+(gphi[1][2]*(2*gphi[3][0]*pow(n[0],3)+2*gphi[3][1]*n[1]*pow(n[0],2))+gphi[3][2]*(2*gphi[1][0]*pow(n[0],3)+2*gphi[1][1]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*gphi[3][2]*(4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[1][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],4)+n[1]*(2*gphi[1][0]*gphi[3][1]*pow(n[0],3)+2*gphi[3][0]*gphi[1][1]*pow(n[0],3)))*pow(n[2],4)+(gphi[1][2]*(4*gphi[3][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[3][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[1][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[1][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0]))*pow(n[2],3)+(gphi[1][2]*gphi[3][2]*(2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[1][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],4)+(4*gphi[1][0]*gphi[3][1]*pow(n[0],3)+4*gphi[3][0]*gphi[1][1]*pow(n[0],3))*pow(n[1],3)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+4*gphi[1][0]*gphi[3][0]*pow(n[0],4))*pow(n[1],2)+4*gphi[1][0]*gphi[3][0]*pow(n[0],6)+n[1]*(gphi[1][1]*(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[3][0]*pow(n[0],4)+gphi[1][1]*gphi[3][1]*pow(n[0],2)+4*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[2],2)+2*gphi[1][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],6)+n[2]*(gphi[1][2]*(2*gphi[3][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[3][0]*pow(n[0],3)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],7)+gphi[3][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[3][0]*pow(n[0],5)+10*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[1][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[1][0]*pow(n[0],3)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],7)+gphi[1][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[1][0]*pow(n[0],5)+10*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0]))+(2*gphi[1][0]*gphi[3][1]*pow(n[0],3)+2*gphi[3][0]*gphi[1][1]*pow(n[0],3))*pow(n[1],5)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[1][0]*gphi[3][0]*pow(n[0],4))*pow(n[1],4)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+2*gphi[1][0]*n[0]))*pow(n[1],3)+gphi[1][2]*gphi[3][2]*(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[1][0]*gphi[3][0]*pow(n[0],6)-8*gphi[1][0]*gphi[3][0]*pow(n[0],4)+4*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],8)+n[1]*(gphi[1][1]*(2*gphi[3][0]*pow(n[0],7)-8*gphi[3][0]*pow(n[0],5)+10*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],7)-8*gphi[1][0]*pow(n[0],5)+10*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[3][0]*pow(n[0],6)+gphi[1][1]*gphi[3][1]*(pow(n[0],4)-2*pow(n[0],2)+1)+12*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+2*gphi[1][0]*gphi[3][0];

bk(1,4) = 2*n[0]*n[1]*gphi[0][2]*gphi[1][2]*pow(n[2],6)+(gphi[0][2]*(2*n[0]*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[1]*pow(n[0],2))+gphi[1][2]*(2*n[0]*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[1][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],3)+(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[0][2]*(4*n[0]*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))+n[0]*gphi[1][1])+gphi[1][2]*(4*n[0]*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])))*pow(n[2],3)+(gphi[0][2]*gphi[1][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],5)+(4*gphi[0][0]*gphi[1][1]*pow(n[0],2)+4*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[1][0]*pow(n[0],5)-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+2*n[0]*gphi[0][1]*gphi[1][1]+2*gphi[0][0]*gphi[1][0]*n[0])+gphi[0][0]*gphi[1][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],7)+n[2]*(gphi[0][2]*(2*n[0]*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(pow(n[0],3)-2*n[0]))+gphi[1][2]*(2*n[0]*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+9*gphi[0][0]*pow(n[0],2)-2*gphi[0][0])))+(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],4)+gphi[0][2]*gphi[1][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],5)-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+2*gphi[0][0]*gphi[1][0]*n[0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+10*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[1][0]*pow(n[0],7)-8*gphi[0][0]*gphi[1][0]*pow(n[0],5)+gphi[0][1]*gphi[1][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[0][0]*gphi[1][0]*pow(n[0],3)-4*gphi[0][0]*gphi[1][0]*n[0])+gphi[1][1]*(gphi[0][0]*pow(n[0],4)-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(1,5) = 2*n[0]*n[1]*pow(gphi[1][2],2)*pow(n[2],6)+gphi[1][2]*(4*n[0]*gphi[1][1]*pow(n[1],2)+4*gphi[1][0]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))*pow(gphi[1][2],2)+2*n[0]*pow(gphi[1][1],2)*pow(n[1],3)+4*gphi[1][0]*gphi[1][1]*pow(n[0],2)*pow(n[1],2)+2*n[1]*pow(gphi[1][0],2)*pow(n[0],3))*pow(n[2],4)+gphi[1][2]*(8*n[0]*gphi[1][1]*pow(n[1],4)+8*gphi[1][0]*pow(n[0],2)*pow(n[1],3)+gphi[1][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],2)+n[1]*(8*gphi[1][0]*pow(n[0],4)-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])+n[0]*gphi[1][1])*pow(n[2],3)+((2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))*pow(gphi[1][2],2)+4*n[0]*pow(gphi[1][1],2)*pow(n[1],5)+8*gphi[1][0]*gphi[1][1]*pow(n[0],2)*pow(n[1],4)+((4*pow(n[0],3)-8*n[0])*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],3))*pow(n[1],3)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],4)-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],2)+n[1]*(2*n[0]*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],5)-8*pow(gphi[1][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[1][0],2))+gphi[1][0]*gphi[1][1]*pow(n[0],2))*pow(n[2],2)+(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))*pow(gphi[1][2],2)+2*n[0]*pow(gphi[1][1],2)*pow(n[1],7)+gphi[1][2]*n[2]*(4*n[0]*gphi[1][1]*pow(n[1],6)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],5)+gphi[1][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],4)+(8*gphi[1][0]*pow(n[0],4)-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],6)-16*gphi[1][0]*pow(n[0],4)+17*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])+gphi[1][1]*(pow(n[0],3)-2*n[0]))+4*gphi[1][0]*gphi[1][1]*pow(n[0],2)*pow(n[1],6)+((4*pow(n[0],3)-8*n[0])*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],3))*pow(n[1],5)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],4)-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],4)+((2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],5)-8*pow(gphi[1][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[1][0],2))*pow(n[1],3)+gphi[1][1]*(4*gphi[1][0]*pow(n[0],6)-16*gphi[1][0]*pow(n[0],4)+18*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])*pow(n[1],2)+n[1]*((2*pow(n[0],3)-4*n[0])*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],7)-8*pow(gphi[1][0],2)*pow(n[0],5)+10*pow(gphi[1][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[1][0],2))+gphi[1][1]*(gphi[1][0]*pow(n[0],4)-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(1,6) = 2*n[0]*n[1]*gphi[1][2]*gphi[2][2]*pow(n[2],6)+(gphi[1][2]*(2*n[0]*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[1]*pow(n[0],2))+gphi[2][2]*(2*n[0]*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*gphi[2][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[1][2]*(4*n[0]*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0]))+gphi[2][2]*(4*n[0]*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))+n[0]*gphi[1][1]))*pow(n[2],3)+(gphi[1][2]*gphi[2][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[1][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],3)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[2][0]*pow(n[0],5)-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+2*n[0]*gphi[1][1]*gphi[2][1]+2*gphi[1][0]*gphi[2][0]*n[0])+gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],7)+n[2]*(gphi[1][2]*(2*n[0]*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+9*gphi[2][0]*pow(n[0],2)-2*gphi[2][0]))+gphi[2][2]*(2*n[0]*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(pow(n[0],3)-2*n[0])))+(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],6)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],5)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))*pow(n[1],4)+gphi[1][2]*gphi[2][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],5)-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+2*gphi[1][0]*gphi[2][0]*n[0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+10*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[2][0]*pow(n[0],7)-8*gphi[1][0]*gphi[2][0]*pow(n[0],5)+gphi[1][1]*gphi[2][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[1][0]*gphi[2][0]*pow(n[0],3)-4*gphi[1][0]*gphi[2][0]*n[0])+gphi[1][1]*(gphi[2][0]*pow(n[0],4)-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(1,7) = 2*n[0]*n[1]*gphi[1][2]*gphi[3][2]*pow(n[2],6)+(gphi[1][2]*(2*n[0]*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[1]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*gphi[3][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[1][2]*(4*n[0]*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0]))+gphi[3][2]*(4*n[0]*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))+n[0]*gphi[1][1]))*pow(n[2],3)+(gphi[1][2]*gphi[3][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[1][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],3)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[3][0]*pow(n[0],5)-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+2*n[0]*gphi[1][1]*gphi[3][1]+2*gphi[1][0]*gphi[3][0]*n[0])+gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],7)+n[2]*(gphi[1][2]*(2*n[0]*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+9*gphi[3][0]*pow(n[0],2)-2*gphi[3][0]))+gphi[3][2]*(2*n[0]*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(pow(n[0],3)-2*n[0])))+(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],6)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],5)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))*pow(n[1],4)+gphi[1][2]*gphi[3][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],5)-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+2*gphi[1][0]*gphi[3][0]*n[0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+10*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[3][0]*pow(n[0],7)-8*gphi[1][0]*gphi[3][0]*pow(n[0],5)+gphi[1][1]*gphi[3][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[1][0]*gphi[3][0]*pow(n[0],3)-4*gphi[1][0]*gphi[3][0]*n[0])+gphi[1][1]*(gphi[3][0]*pow(n[0],4)-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(1,8) = 2*n[0]*gphi[0][2]*gphi[1][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[1][0]*pow(n[0],2)+2*n[0]*gphi[1][1]*n[1])+gphi[1][2]*(2*gphi[0][0]*pow(n[0],2)+2*n[0]*gphi[0][1]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[1][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],3)+n[1]*(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*(4*n[0]*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[1][0]*pow(n[0],2))+gphi[1][2]*(4*n[0]*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[2],4)+(gphi[0][2]*gphi[1][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],4)+(4*gphi[0][0]*gphi[1][1]*pow(n[0],2)+4*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[0][0]*gphi[1][0]*pow(n[0],5)+n[1]*(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+n[0]*gphi[0][1]*gphi[1][1]+2*gphi[0][0]*gphi[1][0]*n[0])*pow(n[2],3)+(gphi[0][2]*(2*n[0]*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*pow(n[0],6)+gphi[1][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2))+gphi[1][2]*(2*n[0]*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],6)+gphi[0][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[0][0]*pow(n[0],4)+10*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))*pow(n[2],2)+n[2]*(2*n[0]*gphi[0][1]*gphi[1][1]*pow(n[1],6)+(2*gphi[0][0]*gphi[1][1]*pow(n[0],2)+2*gphi[1][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[1][0]*pow(n[0],3))*pow(n[1],4)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],3)+gphi[0][2]*gphi[1][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[0][0]*gphi[1][0]*pow(n[0],5)-8*gphi[0][0]*gphi[1][0]*pow(n[0],3)+2*gphi[0][0]*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],7)+n[1]*(gphi[0][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2))+gphi[1][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+9*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))-8*gphi[0][0]*gphi[1][0]*pow(n[0],5)+gphi[0][1]*gphi[1][1]*(pow(n[0],3)-2*n[0])+10*gphi[0][0]*gphi[1][0]*pow(n[0],3)-4*gphi[0][0]*gphi[1][0]*n[0])+gphi[1][2]*(n[0]*gphi[0][1]*pow(n[1],3)+gphi[0][0]*pow(n[0],2)*pow(n[1],2)+gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(1,9) = 2*n[0]*pow(gphi[1][2],2)*pow(n[2],7)+gphi[1][2]*(4*gphi[1][0]*pow(n[0],2)+4*n[0]*gphi[1][1]*n[1])*pow(n[2],6)+((4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])*pow(gphi[1][2],2)+2*n[0]*pow(gphi[1][1],2)*pow(n[1],2)+2*pow(gphi[1][0],2)*pow(n[0],3)+4*gphi[1][0]*gphi[1][1]*n[1]*pow(n[0],2))*pow(n[2],5)+gphi[1][2]*(8*n[0]*gphi[1][1]*pow(n[1],3)+8*gphi[1][0]*pow(n[0],2)*pow(n[1],2)+8*gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(8*pow(n[0],3)-16*n[0])-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[2],4)+((2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[1][2],2)+4*n[0]*pow(gphi[1][1],2)*pow(n[1],4)+8*gphi[1][0]*gphi[1][1]*pow(n[0],2)*pow(n[1],3)+((4*pow(n[0],3)-8*n[0])*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],3))*pow(n[1],2)+n[0]*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],5)+gphi[1][1]*n[1]*(8*gphi[1][0]*pow(n[0],4)-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])-8*pow(gphi[1][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[1][0],2))*pow(n[2],3)+gphi[1][2]*(4*n[0]*gphi[1][1]*pow(n[1],5)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],4)+gphi[1][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],3)+(8*gphi[1][0]*pow(n[0],4)-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],2)+4*gphi[1][0]*pow(n[0],6)+gphi[1][1]*n[1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])-16*gphi[1][0]*pow(n[0],4)+18*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])*pow(n[2],2)+n[2]*((2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])*pow(gphi[1][2],2)+2*n[0]*pow(gphi[1][1],2)*pow(n[1],6)+4*gphi[1][0]*gphi[1][1]*pow(n[0],2)*pow(n[1],5)+((4*pow(n[0],3)-8*n[0])*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],3))*pow(n[1],4)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],4)-16*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],3)+((2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],5)-8*pow(gphi[1][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[1][0],2))*pow(n[1],2)+(pow(n[0],3)-2*n[0])*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],7)+gphi[1][1]*n[1]*(4*gphi[1][0]*pow(n[0],6)-16*gphi[1][0]*pow(n[0],4)+17*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])-8*pow(gphi[1][0],2)*pow(n[0],5)+10*pow(gphi[1][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[1][0],2))+gphi[1][2]*(n[0]*gphi[1][1]*pow(n[1],3)+gphi[1][0]*pow(n[0],2)*pow(n[1],2)+gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(1,10) = 2*n[0]*gphi[1][2]*gphi[2][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[2][0]*pow(n[0],2)+2*n[0]*gphi[2][1]*n[1])+gphi[2][2]*(2*gphi[1][0]*pow(n[0],2)+2*n[0]*gphi[1][1]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[2][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],3)+n[1]*(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*(4*n[0]*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][2]*(4*n[0]*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[1][0]*pow(n[0],2)))*pow(n[2],4)+(gphi[1][2]*gphi[2][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],4)+(4*gphi[1][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[1][0]*gphi[2][0]*pow(n[0],5)+n[1]*(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+n[0]*gphi[1][1]*gphi[2][1]+2*gphi[1][0]*gphi[2][0]*n[0])*pow(n[2],3)+(gphi[1][2]*(2*n[0]*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],6)+gphi[2][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[2][0]*pow(n[0],4)+10*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])+gphi[2][2]*(2*n[0]*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*pow(n[0],6)+gphi[1][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2)))*pow(n[2],2)+n[2]*(2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],6)+(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],4)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))*pow(n[1],3)+gphi[1][2]*gphi[2][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],5)-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+2*gphi[1][0]*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],7)+n[1]*(gphi[1][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+9*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2)))-8*gphi[1][0]*gphi[2][0]*pow(n[0],5)+gphi[1][1]*gphi[2][1]*(pow(n[0],3)-2*n[0])+10*gphi[1][0]*gphi[2][0]*pow(n[0],3)-4*gphi[1][0]*gphi[2][0]*n[0])+gphi[1][2]*(n[0]*gphi[2][1]*pow(n[1],3)+gphi[2][0]*pow(n[0],2)*pow(n[1],2)+gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(1,11) = 2*n[0]*gphi[1][2]*gphi[3][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[3][0]*pow(n[0],2)+2*n[0]*gphi[3][1]*n[1])+gphi[3][2]*(2*gphi[1][0]*pow(n[0],2)+2*n[0]*gphi[1][1]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[3][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],3)+n[1]*(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*(4*n[0]*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][2]*(4*n[0]*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[1][0]*pow(n[0],2)))*pow(n[2],4)+(gphi[1][2]*gphi[3][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[1][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[1][0]*gphi[3][0]*pow(n[0],5)+n[1]*(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+n[0]*gphi[1][1]*gphi[3][1]+2*gphi[1][0]*gphi[3][0]*n[0])*pow(n[2],3)+(gphi[1][2]*(2*n[0]*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],6)+gphi[3][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[3][0]*pow(n[0],4)+10*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][2]*(2*n[0]*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*pow(n[0],6)+gphi[1][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2)))*pow(n[2],2)+n[2]*(2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],4)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)))*pow(n[1],3)+gphi[1][2]*gphi[3][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],5)-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+2*gphi[1][0]*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],7)+n[1]*(gphi[1][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+9*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+8*gphi[1][0]*pow(n[0],2)))-8*gphi[1][0]*gphi[3][0]*pow(n[0],5)+gphi[1][1]*gphi[3][1]*(pow(n[0],3)-2*n[0])+10*gphi[1][0]*gphi[3][0]*pow(n[0],3)-4*gphi[1][0]*gphi[3][0]*n[0])+gphi[1][2]*(n[0]*gphi[3][1]*pow(n[1],3)+gphi[3][0]*pow(n[0],2)*pow(n[1],2)+gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(2,2) = 2*pow(n[0],2)*pow(gphi[2][2],2)*pow(n[2],6)+gphi[2][2]*(4*gphi[2][0]*pow(n[0],3)+4*gphi[2][1]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[2][2],2)+2*pow(n[0],2)*pow(gphi[2][1],2)*pow(n[1],2)+2*pow(gphi[2][0],2)*pow(n[0],4)+4*gphi[2][0]*gphi[2][1]*n[1]*pow(n[0],3))*pow(n[2],4)+gphi[2][2]*(8*gphi[2][1]*pow(n[0],2)*pow(n[1],3)+8*gphi[2][0]*pow(n[0],3)*pow(n[1],2)+8*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)-16*gphi[2][0]*pow(n[0],3)+4*gphi[2][0]*n[0])*pow(n[2],3)+((2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[2][2],2)+4*pow(n[0],2)*pow(gphi[2][1],2)*pow(n[1],4)+8*gphi[2][0]*gphi[2][1]*pow(n[0],3)*pow(n[1],3)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],4))*pow(n[1],2)+pow(n[0],2)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],6)+gphi[2][1]*n[1]*(8*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+4*gphi[2][0]*n[0])-8*pow(gphi[2][0],2)*pow(n[0],4)+4*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[2],2)+(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[2][2],2)+2*pow(n[0],2)*pow(gphi[2][1],2)*pow(n[1],6)+gphi[2][2]*n[2]*(4*gphi[2][1]*pow(n[0],2)*pow(n[1],5)+4*gphi[2][0]*pow(n[0],3)*pow(n[1],4)+gphi[2][1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)*pow(n[1],3)+(8*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+4*gphi[2][0]*n[0])*pow(n[1],2)+4*gphi[2][0]*pow(n[0],7)+gphi[2][1]*n[1]*(4*pow(n[0],6)-16*pow(n[0],4)+18*pow(n[0],2)-4)-16*gphi[2][0]*pow(n[0],5)+20*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+4*gphi[2][0]*gphi[2][1]*pow(n[0],3)*pow(n[1],5)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],4))*pow(n[1],4)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+4*gphi[2][0]*n[0])*pow(n[1],3)+((2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],6)-8*pow(gphi[2][0],2)*pow(n[0],4)+4*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],2)+(pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],8)+gphi[2][1]*n[1]*(4*gphi[2][0]*pow(n[0],7)-16*gphi[2][0]*pow(n[0],5)+20*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])-8*pow(gphi[2][0],2)*pow(n[0],6)+12*pow(gphi[2][0],2)*pow(n[0],4)-8*pow(gphi[2][0],2)*pow(n[0],2)+2*pow(gphi[2][0],2);

bk(2,3) = 2*gphi[2][2]*gphi[3][2]*pow(n[0],2)*pow(n[2],6)+(gphi[2][2]*(2*gphi[3][0]*pow(n[0],3)+2*gphi[3][1]*n[1]*pow(n[0],2))+gphi[3][2]*(2*gphi[2][0]*pow(n[0],3)+2*gphi[2][1]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[2][2]*gphi[3][2]*(4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[2][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],4)+n[1]*(2*gphi[2][0]*gphi[3][1]*pow(n[0],3)+2*gphi[3][0]*gphi[2][1]*pow(n[0],3)))*pow(n[2],4)+(gphi[2][2]*(4*gphi[3][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[3][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[2][1]*pow(n[0],2)*pow(n[1],3)+4*gphi[2][0]*pow(n[0],3)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0]))*pow(n[2],3)+(gphi[2][2]*gphi[3][2]*(2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[2][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],4)+(4*gphi[2][0]*gphi[3][1]*pow(n[0],3)+4*gphi[3][0]*gphi[2][1]*pow(n[0],3))*pow(n[1],3)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+4*gphi[2][0]*gphi[3][0]*pow(n[0],4))*pow(n[1],2)+4*gphi[2][0]*gphi[3][0]*pow(n[0],6)+n[1]*(gphi[2][1]*(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0]))-8*gphi[2][0]*gphi[3][0]*pow(n[0],4)+gphi[2][1]*gphi[3][1]*pow(n[0],2)+4*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[2],2)+2*gphi[2][1]*gphi[3][1]*pow(n[0],2)*pow(n[1],6)+n[2]*(gphi[2][2]*(2*gphi[3][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[3][0]*pow(n[0],3)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],7)+gphi[3][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[3][0]*pow(n[0],5)+10*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[2][1]*pow(n[0],2)*pow(n[1],5)+2*gphi[2][0]*pow(n[0],3)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],7)+gphi[2][1]*n[1]*(2*pow(n[0],6)-8*pow(n[0],4)+9*pow(n[0],2)-2)-8*gphi[2][0]*pow(n[0],5)+10*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0]))+(2*gphi[2][0]*gphi[3][1]*pow(n[0],3)+2*gphi[3][0]*gphi[2][1]*pow(n[0],3))*pow(n[1],5)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],4)-8*pow(n[0],2)+1)+2*gphi[2][0]*gphi[3][0]*pow(n[0],4))*pow(n[1],4)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+2*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+2*gphi[2][0]*n[0]))*pow(n[1],3)+gphi[2][2]*gphi[3][2]*(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)+4*gphi[2][0]*gphi[3][0]*pow(n[0],6)-8*gphi[2][0]*gphi[3][0]*pow(n[0],4)+4*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],8)+n[1]*(gphi[2][1]*(2*gphi[3][0]*pow(n[0],7)-8*gphi[3][0]*pow(n[0],5)+10*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],7)-8*gphi[2][0]*pow(n[0],5)+10*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0]))-8*gphi[2][0]*gphi[3][0]*pow(n[0],6)+gphi[2][1]*gphi[3][1]*(pow(n[0],4)-2*pow(n[0],2)+1)+12*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+2*gphi[2][0]*gphi[3][0];

bk(2,4) = 2*n[0]*n[1]*gphi[0][2]*gphi[2][2]*pow(n[2],6)+(gphi[0][2]*(2*n[0]*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[1]*pow(n[0],2))+gphi[2][2]*(2*n[0]*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[2][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[0][2]*(4*n[0]*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+n[0]*gphi[2][1])+gphi[2][2]*(4*n[0]*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])))*pow(n[2],3)+(gphi[0][2]*gphi[2][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[0][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[2][0]*pow(n[0],5)-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+2*n[0]*gphi[0][1]*gphi[2][1]+2*gphi[0][0]*gphi[2][0]*n[0])+gphi[0][0]*gphi[2][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],7)+n[2]*(gphi[0][2]*(2*n[0]*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(pow(n[0],3)-2*n[0]))+gphi[2][2]*(2*n[0]*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+9*gphi[0][0]*pow(n[0],2)-2*gphi[0][0])))+(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],4)+gphi[0][2]*gphi[2][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],5)-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+2*gphi[0][0]*gphi[2][0]*n[0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+10*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[2][0]*pow(n[0],7)-8*gphi[0][0]*gphi[2][0]*pow(n[0],5)+gphi[0][1]*gphi[2][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[0][0]*gphi[2][0]*pow(n[0],3)-4*gphi[0][0]*gphi[2][0]*n[0])+gphi[2][1]*(gphi[0][0]*pow(n[0],4)-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(2,5) = 2*n[0]*n[1]*gphi[1][2]*gphi[2][2]*pow(n[2],6)+(gphi[1][2]*(2*n[0]*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[1]*pow(n[0],2))+gphi[2][2]*(2*n[0]*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*gphi[2][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[1][2]*(4*n[0]*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+n[0]*gphi[2][1])+gphi[2][2]*(4*n[0]*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])))*pow(n[2],3)+(gphi[1][2]*gphi[2][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[1][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],3)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[2][0]*pow(n[0],5)-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+2*n[0]*gphi[1][1]*gphi[2][1]+2*gphi[1][0]*gphi[2][0]*n[0])+gphi[1][0]*gphi[2][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],7)+n[2]*(gphi[1][2]*(2*n[0]*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(pow(n[0],3)-2*n[0]))+gphi[2][2]*(2*n[0]*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+9*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])))+(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],6)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],5)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[1],4)+gphi[1][2]*gphi[2][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],5)-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+2*gphi[1][0]*gphi[2][0]*n[0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+10*gphi[1][0]*pow(n[0],2)-2*gphi[1][0]))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[2][0]*pow(n[0],7)-8*gphi[1][0]*gphi[2][0]*pow(n[0],5)+gphi[1][1]*gphi[2][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[1][0]*gphi[2][0]*pow(n[0],3)-4*gphi[1][0]*gphi[2][0]*n[0])+gphi[2][1]*(gphi[1][0]*pow(n[0],4)-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(2,6) = 2*n[0]*n[1]*pow(gphi[2][2],2)*pow(n[2],6)+gphi[2][2]*(4*n[0]*gphi[2][1]*pow(n[1],2)+4*gphi[2][0]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))*pow(gphi[2][2],2)+2*n[0]*pow(gphi[2][1],2)*pow(n[1],3)+4*gphi[2][0]*gphi[2][1]*pow(n[0],2)*pow(n[1],2)+2*n[1]*pow(gphi[2][0],2)*pow(n[0],3))*pow(n[2],4)+gphi[2][2]*(8*n[0]*gphi[2][1]*pow(n[1],4)+8*gphi[2][0]*pow(n[0],2)*pow(n[1],3)+gphi[2][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],2)+n[1]*(8*gphi[2][0]*pow(n[0],4)-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])+n[0]*gphi[2][1])*pow(n[2],3)+((2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))*pow(gphi[2][2],2)+4*n[0]*pow(gphi[2][1],2)*pow(n[1],5)+8*gphi[2][0]*gphi[2][1]*pow(n[0],2)*pow(n[1],4)+((4*pow(n[0],3)-8*n[0])*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],3))*pow(n[1],3)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],4)-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],2)+n[1]*(2*n[0]*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],5)-8*pow(gphi[2][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[2][0],2))+gphi[2][0]*gphi[2][1]*pow(n[0],2))*pow(n[2],2)+(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))*pow(gphi[2][2],2)+2*n[0]*pow(gphi[2][1],2)*pow(n[1],7)+gphi[2][2]*n[2]*(4*n[0]*gphi[2][1]*pow(n[1],6)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],5)+gphi[2][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],4)+(8*gphi[2][0]*pow(n[0],4)-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],6)-16*gphi[2][0]*pow(n[0],4)+17*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])+gphi[2][1]*(pow(n[0],3)-2*n[0]))+4*gphi[2][0]*gphi[2][1]*pow(n[0],2)*pow(n[1],6)+((4*pow(n[0],3)-8*n[0])*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],3))*pow(n[1],5)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],4)-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],4)+((2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],5)-8*pow(gphi[2][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[2][0],2))*pow(n[1],3)+gphi[2][1]*(4*gphi[2][0]*pow(n[0],6)-16*gphi[2][0]*pow(n[0],4)+18*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])*pow(n[1],2)+n[1]*((2*pow(n[0],3)-4*n[0])*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],7)-8*pow(gphi[2][0],2)*pow(n[0],5)+10*pow(gphi[2][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[2][0],2))+gphi[2][1]*(gphi[2][0]*pow(n[0],4)-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(2,7) = 2*n[0]*n[1]*gphi[2][2]*gphi[3][2]*pow(n[2],6)+(gphi[2][2]*(2*n[0]*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[1]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[2][2]*gphi[3][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[2][2]*(4*n[0]*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0]))+gphi[3][2]*(4*n[0]*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+n[0]*gphi[2][1]))*pow(n[2],3)+(gphi[2][2]*gphi[3][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[2][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],4)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],3)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(4*gphi[2][0]*gphi[3][0]*pow(n[0],5)-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+2*n[0]*gphi[2][1]*gphi[3][1]+2*gphi[2][0]*gphi[3][0]*n[0])+gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],7)+n[2]*(gphi[2][2]*(2*n[0]*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+9*gphi[3][0]*pow(n[0],2)-2*gphi[3][0]))+gphi[3][2]*(2*n[0]*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(pow(n[0],3)-2*n[0])))+(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],6)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],5)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)))*pow(n[1],4)+gphi[2][2]*gphi[3][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],5)-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+2*gphi[2][0]*gphi[3][0]*n[0])*pow(n[1],3)+(gphi[2][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+10*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2)))*pow(n[1],2)+n[1]*(2*gphi[2][0]*gphi[3][0]*pow(n[0],7)-8*gphi[2][0]*gphi[3][0]*pow(n[0],5)+gphi[2][1]*gphi[3][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[2][0]*gphi[3][0]*pow(n[0],3)-4*gphi[2][0]*gphi[3][0]*n[0])+gphi[2][1]*(gphi[3][0]*pow(n[0],4)-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(2,8) = 2*n[0]*gphi[0][2]*gphi[2][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[2][0]*pow(n[0],2)+2*n[0]*gphi[2][1]*n[1])+gphi[2][2]*(2*gphi[0][0]*pow(n[0],2)+2*n[0]*gphi[0][1]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[2][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],3)+n[1]*(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*(4*n[0]*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[2][0]*pow(n[0],2))+gphi[2][2]*(4*n[0]*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[2],4)+(gphi[0][2]*gphi[2][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],4)+(4*gphi[0][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[0][0]*gphi[2][0]*pow(n[0],5)+n[1]*(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+n[0]*gphi[0][1]*gphi[2][1]+2*gphi[0][0]*gphi[2][0]*n[0])*pow(n[2],3)+(gphi[0][2]*(2*n[0]*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[2][0]*pow(n[0],6)+gphi[2][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][2]*(2*n[0]*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],6)+gphi[0][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[0][0]*pow(n[0],4)+10*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))*pow(n[2],2)+n[2]*(2*n[0]*gphi[0][1]*gphi[2][1]*pow(n[1],6)+(2*gphi[0][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],4)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],3)+gphi[0][2]*gphi[2][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[0][0]*gphi[2][0]*pow(n[0],5)-8*gphi[0][0]*gphi[2][0]*pow(n[0],3)+2*gphi[0][0]*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],7)+n[1]*(gphi[0][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+9*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))-8*gphi[0][0]*gphi[2][0]*pow(n[0],5)+gphi[0][1]*gphi[2][1]*(pow(n[0],3)-2*n[0])+10*gphi[0][0]*gphi[2][0]*pow(n[0],3)-4*gphi[0][0]*gphi[2][0]*n[0])+gphi[2][2]*(n[0]*gphi[0][1]*pow(n[1],3)+gphi[0][0]*pow(n[0],2)*pow(n[1],2)+gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(2,9) = 2*n[0]*gphi[1][2]*gphi[2][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[2][0]*pow(n[0],2)+2*n[0]*gphi[2][1]*n[1])+gphi[2][2]*(2*gphi[1][0]*pow(n[0],2)+2*n[0]*gphi[1][1]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[2][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],3)+n[1]*(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*(4*n[0]*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[2][0]*pow(n[0],2))+gphi[2][2]*(4*n[0]*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[2],4)+(gphi[1][2]*gphi[2][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],4)+(4*gphi[1][0]*gphi[2][1]*pow(n[0],2)+4*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[1][0]*gphi[2][0]*pow(n[0],5)+n[1]*(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+n[0]*gphi[1][1]*gphi[2][1]+2*gphi[1][0]*gphi[2][0]*n[0])*pow(n[2],3)+(gphi[1][2]*(2*n[0]*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[2][0]*pow(n[0],6)+gphi[2][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][2]*(2*n[0]*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],6)+gphi[1][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[1][0]*pow(n[0],4)+10*gphi[1][0]*pow(n[0],2)-2*gphi[1][0]))*pow(n[2],2)+n[2]*(2*n[0]*gphi[1][1]*gphi[2][1]*pow(n[1],6)+(2*gphi[1][0]*gphi[2][1]*pow(n[0],2)+2*gphi[2][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[2][0]*pow(n[0],3))*pow(n[1],4)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[1],3)+gphi[1][2]*gphi[2][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[1][0]*gphi[2][0]*pow(n[0],5)-8*gphi[1][0]*gphi[2][0]*pow(n[0],3)+2*gphi[1][0]*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],7)+n[1]*(gphi[1][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2))+gphi[2][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+9*gphi[1][0]*pow(n[0],2)-2*gphi[1][0]))-8*gphi[1][0]*gphi[2][0]*pow(n[0],5)+gphi[1][1]*gphi[2][1]*(pow(n[0],3)-2*n[0])+10*gphi[1][0]*gphi[2][0]*pow(n[0],3)-4*gphi[1][0]*gphi[2][0]*n[0])+gphi[2][2]*(n[0]*gphi[1][1]*pow(n[1],3)+gphi[1][0]*pow(n[0],2)*pow(n[1],2)+gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(2,10) = 2*n[0]*pow(gphi[2][2],2)*pow(n[2],7)+gphi[2][2]*(4*gphi[2][0]*pow(n[0],2)+4*n[0]*gphi[2][1]*n[1])*pow(n[2],6)+((4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])*pow(gphi[2][2],2)+2*n[0]*pow(gphi[2][1],2)*pow(n[1],2)+2*pow(gphi[2][0],2)*pow(n[0],3)+4*gphi[2][0]*gphi[2][1]*n[1]*pow(n[0],2))*pow(n[2],5)+gphi[2][2]*(8*n[0]*gphi[2][1]*pow(n[1],3)+8*gphi[2][0]*pow(n[0],2)*pow(n[1],2)+8*gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(8*pow(n[0],3)-16*n[0])-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[2],4)+((2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[2][2],2)+4*n[0]*pow(gphi[2][1],2)*pow(n[1],4)+8*gphi[2][0]*gphi[2][1]*pow(n[0],2)*pow(n[1],3)+((4*pow(n[0],3)-8*n[0])*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],3))*pow(n[1],2)+n[0]*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],5)+gphi[2][1]*n[1]*(8*gphi[2][0]*pow(n[0],4)-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])-8*pow(gphi[2][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[2][0],2))*pow(n[2],3)+gphi[2][2]*(4*n[0]*gphi[2][1]*pow(n[1],5)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],4)+gphi[2][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],3)+(8*gphi[2][0]*pow(n[0],4)-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],2)+4*gphi[2][0]*pow(n[0],6)+gphi[2][1]*n[1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])-16*gphi[2][0]*pow(n[0],4)+18*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])*pow(n[2],2)+n[2]*((2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])*pow(gphi[2][2],2)+2*n[0]*pow(gphi[2][1],2)*pow(n[1],6)+4*gphi[2][0]*gphi[2][1]*pow(n[0],2)*pow(n[1],5)+((4*pow(n[0],3)-8*n[0])*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],3))*pow(n[1],4)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],4)-16*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],3)+((2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],5)-8*pow(gphi[2][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[2][0],2))*pow(n[1],2)+(pow(n[0],3)-2*n[0])*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],7)+gphi[2][1]*n[1]*(4*gphi[2][0]*pow(n[0],6)-16*gphi[2][0]*pow(n[0],4)+17*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])-8*pow(gphi[2][0],2)*pow(n[0],5)+10*pow(gphi[2][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[2][0],2))+gphi[2][2]*(n[0]*gphi[2][1]*pow(n[1],3)+gphi[2][0]*pow(n[0],2)*pow(n[1],2)+gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(2,11) = 2*n[0]*gphi[2][2]*gphi[3][2]*pow(n[2],7)+(gphi[2][2]*(2*gphi[3][0]*pow(n[0],2)+2*n[0]*gphi[3][1]*n[1])+gphi[3][2]*(2*gphi[2][0]*pow(n[0],2)+2*n[0]*gphi[2][1]*n[1]))*pow(n[2],6)+(gphi[2][2]*gphi[3][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],3)+n[1]*(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[2][2]*(4*n[0]*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][2]*(4*n[0]*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[2][0]*pow(n[0],2)))*pow(n[2],4)+(gphi[2][2]*gphi[3][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[2][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],3)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[2][0]*gphi[3][0]*pow(n[0],5)+n[1]*(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)))-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+n[0]*gphi[2][1]*gphi[3][1]+2*gphi[2][0]*gphi[3][0]*n[0])*pow(n[2],3)+(gphi[2][2]*(2*n[0]*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],6)+gphi[3][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[3][0]*pow(n[0],4)+10*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][2]*(2*n[0]*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[2][0]*pow(n[0],6)+gphi[2][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2)))*pow(n[2],2)+n[2]*(2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],5)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],4)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2)+gphi[3][0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)))*pow(n[1],3)+gphi[2][2]*gphi[3][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],5)-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+2*gphi[2][0]*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],7)+n[1]*(gphi[2][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+9*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+8*gphi[2][0]*pow(n[0],2)))-8*gphi[2][0]*gphi[3][0]*pow(n[0],5)+gphi[2][1]*gphi[3][1]*(pow(n[0],3)-2*n[0])+10*gphi[2][0]*gphi[3][0]*pow(n[0],3)-4*gphi[2][0]*gphi[3][0]*n[0])+gphi[2][2]*(n[0]*gphi[3][1]*pow(n[1],3)+gphi[3][0]*pow(n[0],2)*pow(n[1],2)+gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(3,3) = 2*pow(n[0],2)*pow(gphi[3][2],2)*pow(n[2],6)+gphi[3][2]*(4*gphi[3][0]*pow(n[0],3)+4*gphi[3][1]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*pow(n[0],2)*pow(n[1],2)+4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[3][2],2)+2*pow(n[0],2)*pow(gphi[3][1],2)*pow(n[1],2)+2*pow(gphi[3][0],2)*pow(n[0],4)+4*gphi[3][0]*gphi[3][1]*n[1]*pow(n[0],3))*pow(n[2],4)+gphi[3][2]*(8*gphi[3][1]*pow(n[0],2)*pow(n[1],3)+8*gphi[3][0]*pow(n[0],3)*pow(n[1],2)+8*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)-16*gphi[3][0]*pow(n[0],3)+4*gphi[3][0]*n[0])*pow(n[2],3)+((2*pow(n[0],2)*pow(n[1],4)+(4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(n[1],2)+2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[3][2],2)+4*pow(n[0],2)*pow(gphi[3][1],2)*pow(n[1],4)+8*gphi[3][0]*gphi[3][1]*pow(n[0],3)*pow(n[1],3)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],4))*pow(n[1],2)+pow(n[0],2)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],6)+gphi[3][1]*n[1]*(8*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+4*gphi[3][0]*n[0])-8*pow(gphi[3][0],2)*pow(n[0],4)+4*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[2],2)+(pow(n[0],2)*pow(n[1],2)+pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[3][2],2)+2*pow(n[0],2)*pow(gphi[3][1],2)*pow(n[1],6)+gphi[3][2]*n[2]*(4*gphi[3][1]*pow(n[0],2)*pow(n[1],5)+4*gphi[3][0]*pow(n[0],3)*pow(n[1],4)+gphi[3][1]*(8*pow(n[0],4)-16*pow(n[0],2)+2)*pow(n[1],3)+(8*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+4*gphi[3][0]*n[0])*pow(n[1],2)+4*gphi[3][0]*pow(n[0],7)+gphi[3][1]*n[1]*(4*pow(n[0],6)-16*pow(n[0],4)+18*pow(n[0],2)-4)-16*gphi[3][0]*pow(n[0],5)+20*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+4*gphi[3][0]*gphi[3][1]*pow(n[0],3)*pow(n[1],5)+((4*pow(n[0],4)-8*pow(n[0],2)+1)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],4))*pow(n[1],4)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+4*gphi[3][0]*n[0])*pow(n[1],3)+((2*pow(n[0],6)-8*pow(n[0],4)+10*pow(n[0],2)-2)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],6)-8*pow(gphi[3][0],2)*pow(n[0],4)+4*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],2)+(pow(n[0],4)-2*pow(n[0],2)+1)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],8)+gphi[3][1]*n[1]*(4*gphi[3][0]*pow(n[0],7)-16*gphi[3][0]*pow(n[0],5)+20*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])-8*pow(gphi[3][0],2)*pow(n[0],6)+12*pow(gphi[3][0],2)*pow(n[0],4)-8*pow(gphi[3][0],2)*pow(n[0],2)+2*pow(gphi[3][0],2);

bk(3,4) = 2*n[0]*n[1]*gphi[0][2]*gphi[3][2]*pow(n[2],6)+(gphi[0][2]*(2*n[0]*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[1]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*gphi[3][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[0][2]*(4*n[0]*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+n[0]*gphi[3][1])+gphi[3][2]*(4*n[0]*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])))*pow(n[2],3)+(gphi[0][2]*gphi[3][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[0][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],3)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[3][0]*pow(n[0],5)-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+2*n[0]*gphi[0][1]*gphi[3][1]+2*gphi[0][0]*gphi[3][0]*n[0])+gphi[0][0]*gphi[3][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],7)+n[2]*(gphi[0][2]*(2*n[0]*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(pow(n[0],3)-2*n[0]))+gphi[3][2]*(2*n[0]*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+9*gphi[0][0]*pow(n[0],2)-2*gphi[0][0])))+(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],5)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],4)+gphi[0][2]*gphi[3][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],5)-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+2*gphi[0][0]*gphi[3][0]*n[0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+10*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[3][0]*pow(n[0],7)-8*gphi[0][0]*gphi[3][0]*pow(n[0],5)+gphi[0][1]*gphi[3][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[0][0]*gphi[3][0]*pow(n[0],3)-4*gphi[0][0]*gphi[3][0]*n[0])+gphi[3][1]*(gphi[0][0]*pow(n[0],4)-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(3,5) = 2*n[0]*n[1]*gphi[1][2]*gphi[3][2]*pow(n[2],6)+(gphi[1][2]*(2*n[0]*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[1]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*gphi[3][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[1][2]*(4*n[0]*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+n[0]*gphi[3][1])+gphi[3][2]*(4*n[0]*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])))*pow(n[2],3)+(gphi[1][2]*gphi[3][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[1][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],3)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[3][0]*pow(n[0],5)-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+2*n[0]*gphi[1][1]*gphi[3][1]+2*gphi[1][0]*gphi[3][0]*n[0])+gphi[1][0]*gphi[3][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],7)+n[2]*(gphi[1][2]*(2*n[0]*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(pow(n[0],3)-2*n[0]))+gphi[3][2]*(2*n[0]*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+9*gphi[1][0]*pow(n[0],2)-2*gphi[1][0])))+(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],6)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],5)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[1],4)+gphi[1][2]*gphi[3][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],5)-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+2*gphi[1][0]*gphi[3][0]*n[0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+10*gphi[1][0]*pow(n[0],2)-2*gphi[1][0]))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[3][0]*pow(n[0],7)-8*gphi[1][0]*gphi[3][0]*pow(n[0],5)+gphi[1][1]*gphi[3][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[1][0]*gphi[3][0]*pow(n[0],3)-4*gphi[1][0]*gphi[3][0]*n[0])+gphi[3][1]*(gphi[1][0]*pow(n[0],4)-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(3,6) = 2*n[0]*n[1]*gphi[2][2]*gphi[3][2]*pow(n[2],6)+(gphi[2][2]*(2*n[0]*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[1]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[1]*pow(n[0],2)))*pow(n[2],5)+(gphi[2][2]*gphi[3][2]*(4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))+2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*n[1]*pow(n[0],3))*pow(n[2],4)+(gphi[2][2]*(4*n[0]*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+n[0]*gphi[3][1])+gphi[3][2]*(4*n[0]*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])))*pow(n[2],3)+(gphi[2][2]*gphi[3][2]*(2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))+4*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[2][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],4)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],3)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0]))*pow(n[1],2)+n[1]*(4*gphi[2][0]*gphi[3][0]*pow(n[0],5)-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+2*n[0]*gphi[2][1]*gphi[3][1]+2*gphi[2][0]*gphi[3][0]*n[0])+gphi[2][0]*gphi[3][1]*pow(n[0],2))*pow(n[2],2)+2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],7)+n[2]*(gphi[2][2]*(2*n[0]*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(pow(n[0],3)-2*n[0]))+gphi[3][2]*(2*n[0]*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+9*gphi[2][0]*pow(n[0],2)-2*gphi[2][0])))+(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],6)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],5)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0]))*pow(n[1],4)+gphi[2][2]*gphi[3][2]*(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],5)-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+2*gphi[2][0]*gphi[3][0]*n[0])*pow(n[1],3)+(gphi[2][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+10*gphi[2][0]*pow(n[0],2)-2*gphi[2][0]))*pow(n[1],2)+n[1]*(2*gphi[2][0]*gphi[3][0]*pow(n[0],7)-8*gphi[2][0]*gphi[3][0]*pow(n[0],5)+gphi[2][1]*gphi[3][1]*(2*pow(n[0],3)-4*n[0])+10*gphi[2][0]*gphi[3][0]*pow(n[0],3)-4*gphi[2][0]*gphi[3][0]*n[0])+gphi[3][1]*(gphi[2][0]*pow(n[0],4)-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(3,7) = 2*n[0]*n[1]*pow(gphi[3][2],2)*pow(n[2],6)+gphi[3][2]*(4*n[0]*gphi[3][1]*pow(n[1],2)+4*gphi[3][0]*n[1]*pow(n[0],2))*pow(n[2],5)+((4*n[0]*pow(n[1],3)+n[1]*(4*pow(n[0],3)-8*n[0]))*pow(gphi[3][2],2)+2*n[0]*pow(gphi[3][1],2)*pow(n[1],3)+4*gphi[3][0]*gphi[3][1]*pow(n[0],2)*pow(n[1],2)+2*n[1]*pow(gphi[3][0],2)*pow(n[0],3))*pow(n[2],4)+gphi[3][2]*(8*n[0]*gphi[3][1]*pow(n[1],4)+8*gphi[3][0]*pow(n[0],2)*pow(n[1],3)+gphi[3][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],2)+n[1]*(8*gphi[3][0]*pow(n[0],4)-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])+n[0]*gphi[3][1])*pow(n[2],3)+((2*n[0]*pow(n[1],5)+(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0]))*pow(gphi[3][2],2)+4*n[0]*pow(gphi[3][1],2)*pow(n[1],5)+8*gphi[3][0]*gphi[3][1]*pow(n[0],2)*pow(n[1],4)+((4*pow(n[0],3)-8*n[0])*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],3))*pow(n[1],3)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],4)-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],2)+n[1]*(2*n[0]*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],5)-8*pow(gphi[3][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[3][0],2))+gphi[3][0]*gphi[3][1]*pow(n[0],2))*pow(n[2],2)+(n[0]*pow(n[1],3)+n[1]*(pow(n[0],3)-2*n[0]))*pow(gphi[3][2],2)+2*n[0]*pow(gphi[3][1],2)*pow(n[1],7)+gphi[3][2]*n[2]*(4*n[0]*gphi[3][1]*pow(n[1],6)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],5)+gphi[3][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],4)+(8*gphi[3][0]*pow(n[0],4)-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],6)-16*gphi[3][0]*pow(n[0],4)+17*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])+gphi[3][1]*(pow(n[0],3)-2*n[0]))+4*gphi[3][0]*gphi[3][1]*pow(n[0],2)*pow(n[1],6)+((4*pow(n[0],3)-8*n[0])*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],3))*pow(n[1],5)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],4)-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],4)+((2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],5)-8*pow(gphi[3][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[3][0],2))*pow(n[1],3)+gphi[3][1]*(4*gphi[3][0]*pow(n[0],6)-16*gphi[3][0]*pow(n[0],4)+18*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])*pow(n[1],2)+n[1]*((2*pow(n[0],3)-4*n[0])*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],7)-8*pow(gphi[3][0],2)*pow(n[0],5)+10*pow(gphi[3][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[3][0],2))+gphi[3][1]*(gphi[3][0]*pow(n[0],4)-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(3,8) = 2*n[0]*gphi[0][2]*gphi[3][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[3][0]*pow(n[0],2)+2*n[0]*gphi[3][1]*n[1])+gphi[3][2]*(2*gphi[0][0]*pow(n[0],2)+2*n[0]*gphi[0][1]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[3][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],3)+n[1]*(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[0][2]*(4*n[0]*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[3][0]*pow(n[0],2))+gphi[3][2]*(4*n[0]*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[2],4)+(gphi[0][2]*gphi[3][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[0][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[0][0]*gphi[3][0]*pow(n[0],5)+n[1]*(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+n[0]*gphi[0][1]*gphi[3][1]+2*gphi[0][0]*gphi[3][0]*n[0])*pow(n[2],3)+(gphi[0][2]*(2*n[0]*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[3][0]*pow(n[0],6)+gphi[3][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*pow(n[0],2)*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],6)+gphi[0][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[0][0]*pow(n[0],4)+10*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))*pow(n[2],2)+n[2]*(2*n[0]*gphi[0][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[0][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[0][1]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[0][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],4)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[0][0]*pow(n[0],4)-8*gphi[0][0]*pow(n[0],2)+gphi[0][0]))*pow(n[1],3)+gphi[0][2]*gphi[3][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[0][0]*gphi[3][0]*pow(n[0],5)-8*gphi[0][0]*gphi[3][0]*pow(n[0],3)+2*gphi[0][0]*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],7)+n[1]*(gphi[0][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(2*gphi[0][0]*pow(n[0],6)-8*gphi[0][0]*pow(n[0],4)+9*gphi[0][0]*pow(n[0],2)-2*gphi[0][0]))-8*gphi[0][0]*gphi[3][0]*pow(n[0],5)+gphi[0][1]*gphi[3][1]*(pow(n[0],3)-2*n[0])+10*gphi[0][0]*gphi[3][0]*pow(n[0],3)-4*gphi[0][0]*gphi[3][0]*n[0])+gphi[3][2]*(n[0]*gphi[0][1]*pow(n[1],3)+gphi[0][0]*pow(n[0],2)*pow(n[1],2)+gphi[0][0]*pow(n[0],4)+gphi[0][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[0][0]*pow(n[0],2)+gphi[0][0]);

bk(3,9) = 2*n[0]*gphi[1][2]*gphi[3][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[3][0]*pow(n[0],2)+2*n[0]*gphi[3][1]*n[1])+gphi[3][2]*(2*gphi[1][0]*pow(n[0],2)+2*n[0]*gphi[1][1]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[3][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],3)+n[1]*(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[1][2]*(4*n[0]*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[3][0]*pow(n[0],2))+gphi[3][2]*(4*n[0]*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[2],4)+(gphi[1][2]*gphi[3][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[1][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[1][0]*gphi[3][0]*pow(n[0],5)+n[1]*(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+n[0]*gphi[1][1]*gphi[3][1]+2*gphi[1][0]*gphi[3][0]*n[0])*pow(n[2],3)+(gphi[1][2]*(2*n[0]*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[3][0]*pow(n[0],6)+gphi[3][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*pow(n[0],2)*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],6)+gphi[1][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[1][0]*pow(n[0],4)+10*gphi[1][0]*pow(n[0],2)-2*gphi[1][0]))*pow(n[2],2)+n[2]*(2*n[0]*gphi[1][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[1][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[1][1]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[1][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],4)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[1][0]*pow(n[0],4)-8*gphi[1][0]*pow(n[0],2)+gphi[1][0]))*pow(n[1],3)+gphi[1][2]*gphi[3][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[1][0]*gphi[3][0]*pow(n[0],5)-8*gphi[1][0]*gphi[3][0]*pow(n[0],3)+2*gphi[1][0]*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],7)+n[1]*(gphi[1][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(2*gphi[1][0]*pow(n[0],6)-8*gphi[1][0]*pow(n[0],4)+9*gphi[1][0]*pow(n[0],2)-2*gphi[1][0]))-8*gphi[1][0]*gphi[3][0]*pow(n[0],5)+gphi[1][1]*gphi[3][1]*(pow(n[0],3)-2*n[0])+10*gphi[1][0]*gphi[3][0]*pow(n[0],3)-4*gphi[1][0]*gphi[3][0]*n[0])+gphi[3][2]*(n[0]*gphi[1][1]*pow(n[1],3)+gphi[1][0]*pow(n[0],2)*pow(n[1],2)+gphi[1][0]*pow(n[0],4)+gphi[1][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[1][0]*pow(n[0],2)+gphi[1][0]);

bk(3,10) = 2*n[0]*gphi[2][2]*gphi[3][2]*pow(n[2],7)+(gphi[2][2]*(2*gphi[3][0]*pow(n[0],2)+2*n[0]*gphi[3][1]*n[1])+gphi[3][2]*(2*gphi[2][0]*pow(n[0],2)+2*n[0]*gphi[2][1]*n[1]))*pow(n[2],6)+(gphi[2][2]*gphi[3][2]*(4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])+2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],3)+n[1]*(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2)))*pow(n[2],5)+(gphi[2][2]*(4*n[0]*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[3][0]*pow(n[0],2))+gphi[3][2]*(4*n[0]*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*pow(n[0],2)*pow(n[1],2)+4*gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(4*pow(n[0],3)-8*n[0])-8*gphi[2][0]*pow(n[0],2)+gphi[2][0]))*pow(n[2],4)+(gphi[2][2]*gphi[3][2]*(2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])+4*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[2][0]*gphi[3][1]*pow(n[0],2)+4*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],3)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],2)+4*gphi[2][0]*gphi[3][0]*pow(n[0],5)+n[1]*(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0]))-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+n[0]*gphi[2][1]*gphi[3][1]+2*gphi[2][0]*gphi[3][0]*n[0])*pow(n[2],3)+(gphi[2][2]*(2*n[0]*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*pow(n[0],2)*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+2*gphi[3][0]*pow(n[0],6)+gphi[3][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+8*n[0])-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][2]*(2*n[0]*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*pow(n[0],2)*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],3)-8*n[0])*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],6)+gphi[2][1]*n[1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])-8*gphi[2][0]*pow(n[0],4)+10*gphi[2][0]*pow(n[0],2)-2*gphi[2][0]))*pow(n[2],2)+n[2]*(2*n[0]*gphi[2][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[2][0]*gphi[3][1]*pow(n[0],2)+2*gphi[3][0]*gphi[2][1]*pow(n[0],2))*pow(n[1],5)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],3)-8*n[0])+2*gphi[2][0]*gphi[3][0]*pow(n[0],3))*pow(n[1],4)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],4)-8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(4*gphi[2][0]*pow(n[0],4)-8*gphi[2][0]*pow(n[0],2)+gphi[2][0]))*pow(n[1],3)+gphi[2][2]*gphi[3][2]*(2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])+4*gphi[2][0]*gphi[3][0]*pow(n[0],5)-8*gphi[2][0]*gphi[3][0]*pow(n[0],3)+2*gphi[2][0]*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],7)+n[1]*(gphi[2][1]*(2*gphi[3][0]*pow(n[0],6)-8*gphi[3][0]*pow(n[0],4)+8*gphi[3][0]*pow(n[0],2))+gphi[3][1]*(2*gphi[2][0]*pow(n[0],6)-8*gphi[2][0]*pow(n[0],4)+9*gphi[2][0]*pow(n[0],2)-2*gphi[2][0]))-8*gphi[2][0]*gphi[3][0]*pow(n[0],5)+gphi[2][1]*gphi[3][1]*(pow(n[0],3)-2*n[0])+10*gphi[2][0]*gphi[3][0]*pow(n[0],3)-4*gphi[2][0]*gphi[3][0]*n[0])+gphi[3][2]*(n[0]*gphi[2][1]*pow(n[1],3)+gphi[2][0]*pow(n[0],2)*pow(n[1],2)+gphi[2][0]*pow(n[0],4)+gphi[2][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[2][0]*pow(n[0],2)+gphi[2][0]);

bk(3,11) = 2*n[0]*pow(gphi[3][2],2)*pow(n[2],7)+gphi[3][2]*(4*gphi[3][0]*pow(n[0],2)+4*n[0]*gphi[3][1]*n[1])*pow(n[2],6)+((4*n[0]*pow(n[1],2)+4*pow(n[0],3)-8*n[0])*pow(gphi[3][2],2)+2*n[0]*pow(gphi[3][1],2)*pow(n[1],2)+2*pow(gphi[3][0],2)*pow(n[0],3)+4*gphi[3][0]*gphi[3][1]*n[1]*pow(n[0],2))*pow(n[2],5)+gphi[3][2]*(8*n[0]*gphi[3][1]*pow(n[1],3)+8*gphi[3][0]*pow(n[0],2)*pow(n[1],2)+8*gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(8*pow(n[0],3)-16*n[0])-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[2],4)+((2*n[0]*pow(n[1],4)+(4*pow(n[0],3)-8*n[0])*pow(n[1],2)+2*pow(n[0],5)-8*pow(n[0],3)+10*n[0])*pow(gphi[3][2],2)+4*n[0]*pow(gphi[3][1],2)*pow(n[1],4)+8*gphi[3][0]*gphi[3][1]*pow(n[0],2)*pow(n[1],3)+((4*pow(n[0],3)-8*n[0])*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],3))*pow(n[1],2)+n[0]*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],5)+gphi[3][1]*n[1]*(8*gphi[3][0]*pow(n[0],4)-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])-8*pow(gphi[3][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[3][0],2))*pow(n[2],3)+gphi[3][2]*(4*n[0]*gphi[3][1]*pow(n[1],5)+4*gphi[3][0]*pow(n[0],2)*pow(n[1],4)+gphi[3][1]*(8*pow(n[0],3)-16*n[0])*pow(n[1],3)+(8*gphi[3][0]*pow(n[0],4)-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],2)+4*gphi[3][0]*pow(n[0],6)+gphi[3][1]*n[1]*(4*pow(n[0],5)-16*pow(n[0],3)+17*n[0])-16*gphi[3][0]*pow(n[0],4)+18*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])*pow(n[2],2)+n[2]*((2*n[0]*pow(n[1],2)+2*pow(n[0],3)-4*n[0])*pow(gphi[3][2],2)+2*n[0]*pow(gphi[3][1],2)*pow(n[1],6)+4*gphi[3][0]*gphi[3][1]*pow(n[0],2)*pow(n[1],5)+((4*pow(n[0],3)-8*n[0])*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],3))*pow(n[1],4)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],4)-16*gphi[3][0]*pow(n[0],2)+gphi[3][0])*pow(n[1],3)+((2*pow(n[0],5)-8*pow(n[0],3)+9*n[0])*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],5)-8*pow(gphi[3][0],2)*pow(n[0],3)+2*n[0]*pow(gphi[3][0],2))*pow(n[1],2)+(pow(n[0],3)-2*n[0])*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],7)+gphi[3][1]*n[1]*(4*gphi[3][0]*pow(n[0],6)-16*gphi[3][0]*pow(n[0],4)+17*gphi[3][0]*pow(n[0],2)-2*gphi[3][0])-8*pow(gphi[3][0],2)*pow(n[0],5)+10*pow(gphi[3][0],2)*pow(n[0],3)-4*n[0]*pow(gphi[3][0],2))+gphi[3][2]*(n[0]*gphi[3][1]*pow(n[1],3)+gphi[3][0]*pow(n[0],2)*pow(n[1],2)+gphi[3][0]*pow(n[0],4)+gphi[3][1]*n[1]*(pow(n[0],3)-2*n[0])-2*gphi[3][0]*pow(n[0],2)+gphi[3][0]);

bk(4,4) = 2*pow(n[1],2)*pow(gphi[0][2],2)*pow(n[2],6)+gphi[0][2]*(4*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*n[0]*pow(n[1],2))*pow(n[2],5)+((4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)*pow(gphi[0][2],2)+2*pow(gphi[0][1],2)*pow(n[1],4)+4*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],3)+2*pow(gphi[0][0],2)*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+gphi[0][2]*(8*gphi[0][1]*pow(n[1],5)+8*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],2)+4*gphi[0][1]*n[1]+2*gphi[0][0]*n[0])*pow(n[2],3)+((2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)*pow(gphi[0][2],2)+4*pow(gphi[0][1],2)*pow(n[1],6)+8*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],3)+(4*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],4)-8*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2))*pow(n[1],2)+pow(gphi[0][0],2)*pow(n[0],2)+4*gphi[0][0]*n[0]*gphi[0][1]*n[1])*pow(n[2],2)+(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)*pow(gphi[0][2],2)+2*pow(gphi[0][1],2)*pow(n[1],8)+gphi[0][2]*n[2]*(4*gphi[0][1]*pow(n[1],7)+4*gphi[0][0]*n[0]*pow(n[1],6)+gphi[0][1]*(8*pow(n[0],2)-16)*pow(n[1],5)+(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+18*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(4*pow(n[0],2)-8)-4*gphi[0][0]*n[0])+4*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],7)+((4*pow(n[0],2)-8)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],6)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],5)+((2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],4)-8*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2))*pow(n[1],4)+gphi[0][1]*(4*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+20*gphi[0][0]*n[0])*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],6)-8*pow(gphi[0][0],2)*pow(n[0],4)+10*pow(gphi[0][0],2)*pow(n[0],2)-2*pow(gphi[0][0],2))*pow(n[1],2)+2*pow(gphi[0][1],2)+pow(gphi[0][0],2)*pow(n[0],4)+gphi[0][1]*n[1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])-2*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2);

bk(4,5) = 2*gphi[0][2]*gphi[1][2]*pow(n[1],2)*pow(n[2],6)+(gphi[0][2]*(2*gphi[1][1]*pow(n[1],3)+2*gphi[1][0]*n[0]*pow(n[1],2))+gphi[1][2]*(2*gphi[0][1]*pow(n[1],3)+2*gphi[0][0]*n[0]*pow(n[1],2)))*pow(n[2],5)+(gphi[0][2]*gphi[1][2]*(4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)+2*gphi[0][1]*gphi[1][1]*pow(n[1],4)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],3)+2*gphi[0][0]*gphi[1][0]*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+(gphi[0][2]*(4*gphi[1][1]*pow(n[1],5)+4*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][1]*n[1]+gphi[1][0]*n[0])+gphi[1][2]*(4*gphi[0][1]*pow(n[1],5)+4*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][1]*n[1]+gphi[0][0]*n[0]))*pow(n[2],3)+(gphi[0][2]*gphi[1][2]*(2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)+4*gphi[0][1]*gphi[1][1]*pow(n[1],6)+(4*gphi[0][0]*n[0]*gphi[1][1]+4*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],5)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],3)+(4*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+4*gphi[0][1]*gphi[1][1]+gphi[0][0]*gphi[1][0])*pow(n[1],2)+gphi[0][0]*gphi[1][0]*pow(n[0],2)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*n[1])*pow(n[2],2)+2*gphi[0][1]*gphi[1][1]*pow(n[1],8)+n[2]*(gphi[0][2]*(2*gphi[1][1]*pow(n[1],7)+2*gphi[1][0]*n[0]*pow(n[1],6)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],4)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])*pow(n[1],2)+gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[1][0]*n[0])+gphi[1][2]*(2*gphi[0][1]*pow(n[1],7)+2*gphi[0][0]*n[0]*pow(n[1],6)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],4)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0])*pow(n[1],2)+gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[0][0]*n[0]))+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],7)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],5)+gphi[0][2]*gphi[1][2]*(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+gphi[0][0]*gphi[1][0])*pow(n[1],4)+(gphi[0][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+10*gphi[1][0]*n[0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+10*gphi[0][0]*n[0]))*pow(n[1],3)+(2*gphi[0][0]*gphi[1][0]*pow(n[0],6)-8*gphi[0][0]*gphi[1][0]*pow(n[0],4)+gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+10*gphi[0][0]*gphi[1][0]*pow(n[0],2)-2*gphi[0][0]*gphi[1][0])*pow(n[1],2)+gphi[0][0]*gphi[1][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(2*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))-2*gphi[0][0]*gphi[1][0]*pow(n[0],2)+2*gphi[0][1]*gphi[1][1]+gphi[0][0]*gphi[1][0];

bk(4,6) = 2*gphi[0][2]*gphi[2][2]*pow(n[1],2)*pow(n[2],6)+(gphi[0][2]*(2*gphi[2][1]*pow(n[1],3)+2*gphi[2][0]*n[0]*pow(n[1],2))+gphi[2][2]*(2*gphi[0][1]*pow(n[1],3)+2*gphi[0][0]*n[0]*pow(n[1],2)))*pow(n[2],5)+(gphi[0][2]*gphi[2][2]*(4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)+2*gphi[0][1]*gphi[2][1]*pow(n[1],4)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],3)+2*gphi[0][0]*gphi[2][0]*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+(gphi[0][2]*(4*gphi[2][1]*pow(n[1],5)+4*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][1]*n[1]+gphi[2][0]*n[0])+gphi[2][2]*(4*gphi[0][1]*pow(n[1],5)+4*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][1]*n[1]+gphi[0][0]*n[0]))*pow(n[2],3)+(gphi[0][2]*gphi[2][2]*(2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)+4*gphi[0][1]*gphi[2][1]*pow(n[1],6)+(4*gphi[0][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],5)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],3)+(4*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+4*gphi[0][1]*gphi[2][1]+gphi[0][0]*gphi[2][0])*pow(n[1],2)+gphi[0][0]*gphi[2][0]*pow(n[0],2)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*n[1])*pow(n[2],2)+2*gphi[0][1]*gphi[2][1]*pow(n[1],8)+n[2]*(gphi[0][2]*(2*gphi[2][1]*pow(n[1],7)+2*gphi[2][0]*n[0]*pow(n[1],6)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],4)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])*pow(n[1],2)+gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[0][1]*pow(n[1],7)+2*gphi[0][0]*n[0]*pow(n[1],6)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],4)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0])*pow(n[1],2)+gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[0][0]*n[0]))+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],7)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],5)+gphi[0][2]*gphi[2][2]*(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+gphi[0][0]*gphi[2][0])*pow(n[1],4)+(gphi[0][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+10*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+10*gphi[0][0]*n[0]))*pow(n[1],3)+(2*gphi[0][0]*gphi[2][0]*pow(n[0],6)-8*gphi[0][0]*gphi[2][0]*pow(n[0],4)+gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+10*gphi[0][0]*gphi[2][0]*pow(n[0],2)-2*gphi[0][0]*gphi[2][0])*pow(n[1],2)+gphi[0][0]*gphi[2][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(2*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))-2*gphi[0][0]*gphi[2][0]*pow(n[0],2)+2*gphi[0][1]*gphi[2][1]+gphi[0][0]*gphi[2][0];

bk(4,7) = 2*gphi[0][2]*gphi[3][2]*pow(n[1],2)*pow(n[2],6)+(gphi[0][2]*(2*gphi[3][1]*pow(n[1],3)+2*gphi[3][0]*n[0]*pow(n[1],2))+gphi[3][2]*(2*gphi[0][1]*pow(n[1],3)+2*gphi[0][0]*n[0]*pow(n[1],2)))*pow(n[2],5)+(gphi[0][2]*gphi[3][2]*(4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)+2*gphi[0][1]*gphi[3][1]*pow(n[1],4)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],3)+2*gphi[0][0]*gphi[3][0]*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+(gphi[0][2]*(4*gphi[3][1]*pow(n[1],5)+4*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][1]*n[1]+gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[0][1]*pow(n[1],5)+4*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][1]*n[1]+gphi[0][0]*n[0]))*pow(n[2],3)+(gphi[0][2]*gphi[3][2]*(2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)+4*gphi[0][1]*gphi[3][1]*pow(n[1],6)+(4*gphi[0][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],5)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],3)+(4*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+4*gphi[0][1]*gphi[3][1]+gphi[0][0]*gphi[3][0])*pow(n[1],2)+gphi[0][0]*gphi[3][0]*pow(n[0],2)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*n[1])*pow(n[2],2)+2*gphi[0][1]*gphi[3][1]*pow(n[1],8)+n[2]*(gphi[0][2]*(2*gphi[3][1]*pow(n[1],7)+2*gphi[3][0]*n[0]*pow(n[1],6)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],4)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])*pow(n[1],2)+gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[0][1]*pow(n[1],7)+2*gphi[0][0]*n[0]*pow(n[1],6)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],4)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0])*pow(n[1],2)+gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[0][0]*n[0]))+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],7)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],6)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],5)+gphi[0][2]*gphi[3][2]*(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+gphi[0][0]*gphi[3][0])*pow(n[1],4)+(gphi[0][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+10*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+10*gphi[0][0]*n[0]))*pow(n[1],3)+(2*gphi[0][0]*gphi[3][0]*pow(n[0],6)-8*gphi[0][0]*gphi[3][0]*pow(n[0],4)+gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+10*gphi[0][0]*gphi[3][0]*pow(n[0],2)-2*gphi[0][0]*gphi[3][0])*pow(n[1],2)+gphi[0][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(2*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0]))-2*gphi[0][0]*gphi[3][0]*pow(n[0],2)+2*gphi[0][1]*gphi[3][1]+gphi[0][0]*gphi[3][0];

bk(4,8) = 2*n[1]*pow(gphi[0][2],2)*pow(n[2],7)+gphi[0][2]*(4*gphi[0][1]*pow(n[1],2)+4*gphi[0][0]*n[0]*n[1])*pow(n[2],6)+((4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))*pow(gphi[0][2],2)+2*pow(gphi[0][1],2)*pow(n[1],3)+4*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],2)+2*n[1]*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[2],5)+gphi[0][2]*(8*gphi[0][1]*pow(n[1],4)+8*gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(8*pow(n[0],2)-16)*pow(n[1],2)+n[1]*(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])+gphi[0][1])*pow(n[2],4)+((2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))*pow(gphi[0][2],2)+4*pow(gphi[0][1],2)*pow(n[1],5)+8*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],4)+((4*pow(n[0],2)-8)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],3)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],2)+n[1]*(2*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],4)-8*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2))+gphi[0][0]*n[0]*gphi[0][1])*pow(n[2],3)+gphi[0][2]*(4*gphi[0][1]*pow(n[1],6)+4*gphi[0][0]*n[0]*pow(n[1],5)+gphi[0][1]*(8*pow(n[0],2)-16)*pow(n[1],4)+(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],4)-16*pow(n[0],2)+18)*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+17*gphi[0][0]*n[0])+gphi[0][1]*(pow(n[0],2)-2))*pow(n[2],2)+n[2]*((2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))*pow(gphi[0][2],2)+2*pow(gphi[0][1],2)*pow(n[1],7)+4*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],6)+((4*pow(n[0],2)-8)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],5)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],4)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],4)-8*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2))*pow(n[1],3)+gphi[0][1]*(4*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+17*gphi[0][0]*n[0])*pow(n[1],2)+n[1]*((2*pow(n[0],2)-4)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],6)-8*pow(gphi[0][0],2)*pow(n[0],4)+9*pow(gphi[0][0],2)*pow(n[0],2)-2*pow(gphi[0][0],2))+gphi[0][1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0]))+gphi[0][2]*(gphi[0][1]*pow(n[1],4)+gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0])+gphi[0][1]);

bk(4,9) = 2*n[1]*gphi[0][2]*gphi[1][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[0]*n[1])+gphi[1][2]*(2*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[1][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[0][1]*gphi[1][1]*pow(n[1],3)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[0][2]*(4*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1])+gphi[1][2]*(4*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])))*pow(n[2],4)+(gphi[0][2]*gphi[1][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[0][1]*gphi[1][1]*pow(n[1],5)+(4*gphi[0][0]*n[0]*gphi[1][1]+4*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],4)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+2*gphi[0][1]*gphi[1][1]+gphi[0][0]*gphi[1][0])+gphi[0][0]*n[0]*gphi[1][1])*pow(n[2],3)+(gphi[0][2]*(2*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*n[0]*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])+gphi[1][1]*(pow(n[0],2)-2))+gphi[1][2]*(2*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*n[0]*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+8*gphi[0][0]*n[0])))*pow(n[2],2)+n[2]*(2*gphi[0][1]*gphi[1][1]*pow(n[1],7)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],6)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],4)+gphi[0][2]*gphi[1][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+gphi[0][0]*gphi[1][0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+8*gphi[1][0]*n[0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[1][0]*pow(n[0],6)-8*gphi[0][0]*gphi[1][0]*pow(n[0],4)+gphi[0][1]*gphi[1][1]*(2*pow(n[0],2)-4)+9*gphi[0][0]*gphi[1][0]*pow(n[0],2)-2*gphi[0][0]*gphi[1][0])+gphi[1][1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0]))+gphi[0][2]*(gphi[1][1]*pow(n[1],4)+gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0])+gphi[1][1]);

bk(4,10) = 2*n[1]*gphi[0][2]*gphi[2][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[0]*n[1])+gphi[2][2]*(2*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[2][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[0][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[0][2]*(4*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1])+gphi[2][2]*(4*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])))*pow(n[2],4)+(gphi[0][2]*gphi[2][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[0][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[0][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],4)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+2*gphi[0][1]*gphi[2][1]+gphi[0][0]*gphi[2][0])+gphi[0][0]*n[0]*gphi[2][1])*pow(n[2],3)+(gphi[0][2]*(2*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*n[0]*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])+gphi[2][1]*(pow(n[0],2)-2))+gphi[2][2]*(2*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*n[0]*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+8*gphi[0][0]*n[0])))*pow(n[2],2)+n[2]*(2*gphi[0][1]*gphi[2][1]*pow(n[1],7)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],6)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],4)+gphi[0][2]*gphi[2][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+gphi[0][0]*gphi[2][0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+8*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[2][0]*pow(n[0],6)-8*gphi[0][0]*gphi[2][0]*pow(n[0],4)+gphi[0][1]*gphi[2][1]*(2*pow(n[0],2)-4)+9*gphi[0][0]*gphi[2][0]*pow(n[0],2)-2*gphi[0][0]*gphi[2][0])+gphi[2][1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0]))+gphi[0][2]*(gphi[2][1]*pow(n[1],4)+gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0])+gphi[2][1]);

bk(4,11) = 2*n[1]*gphi[0][2]*gphi[3][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[0]*n[1])+gphi[3][2]*(2*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[3][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[0][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[0][2]*(4*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1])+gphi[3][2]*(4*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])))*pow(n[2],4)+(gphi[0][2]*gphi[3][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[0][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[0][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],4)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+2*gphi[0][1]*gphi[3][1]+gphi[0][0]*gphi[3][0])+gphi[0][0]*n[0]*gphi[3][1])*pow(n[2],3)+(gphi[0][2]*(2*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*n[0]*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(pow(n[0],2)-2))+gphi[3][2]*(2*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*n[0]*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+8*gphi[0][0]*n[0])))*pow(n[2],2)+n[2]*(2*gphi[0][1]*gphi[3][1]*pow(n[1],7)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],6)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],4)+gphi[0][2]*gphi[3][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+gphi[0][0]*gphi[3][0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+8*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[3][0]*pow(n[0],6)-8*gphi[0][0]*gphi[3][0]*pow(n[0],4)+gphi[0][1]*gphi[3][1]*(2*pow(n[0],2)-4)+9*gphi[0][0]*gphi[3][0]*pow(n[0],2)-2*gphi[0][0]*gphi[3][0])+gphi[3][1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0]))+gphi[0][2]*(gphi[3][1]*pow(n[1],4)+gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0])+gphi[3][1]);

bk(5,5) = 2*pow(n[1],2)*pow(gphi[1][2],2)*pow(n[2],6)+gphi[1][2]*(4*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*n[0]*pow(n[1],2))*pow(n[2],5)+((4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)*pow(gphi[1][2],2)+2*pow(gphi[1][1],2)*pow(n[1],4)+4*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],3)+2*pow(gphi[1][0],2)*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+gphi[1][2]*(8*gphi[1][1]*pow(n[1],5)+8*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],2)+4*gphi[1][1]*n[1]+2*gphi[1][0]*n[0])*pow(n[2],3)+((2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)*pow(gphi[1][2],2)+4*pow(gphi[1][1],2)*pow(n[1],6)+8*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],3)+(4*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],4)-8*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2))*pow(n[1],2)+pow(gphi[1][0],2)*pow(n[0],2)+4*gphi[1][0]*n[0]*gphi[1][1]*n[1])*pow(n[2],2)+(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)*pow(gphi[1][2],2)+2*pow(gphi[1][1],2)*pow(n[1],8)+gphi[1][2]*n[2]*(4*gphi[1][1]*pow(n[1],7)+4*gphi[1][0]*n[0]*pow(n[1],6)+gphi[1][1]*(8*pow(n[0],2)-16)*pow(n[1],5)+(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+18*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(4*pow(n[0],2)-8)-4*gphi[1][0]*n[0])+4*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],7)+((4*pow(n[0],2)-8)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],6)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],5)+((2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],4)-8*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2))*pow(n[1],4)+gphi[1][1]*(4*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+20*gphi[1][0]*n[0])*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],6)-8*pow(gphi[1][0],2)*pow(n[0],4)+10*pow(gphi[1][0],2)*pow(n[0],2)-2*pow(gphi[1][0],2))*pow(n[1],2)+2*pow(gphi[1][1],2)+pow(gphi[1][0],2)*pow(n[0],4)+gphi[1][1]*n[1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])-2*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2);

bk(5,6) = 2*gphi[1][2]*gphi[2][2]*pow(n[1],2)*pow(n[2],6)+(gphi[1][2]*(2*gphi[2][1]*pow(n[1],3)+2*gphi[2][0]*n[0]*pow(n[1],2))+gphi[2][2]*(2*gphi[1][1]*pow(n[1],3)+2*gphi[1][0]*n[0]*pow(n[1],2)))*pow(n[2],5)+(gphi[1][2]*gphi[2][2]*(4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)+2*gphi[1][1]*gphi[2][1]*pow(n[1],4)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],3)+2*gphi[1][0]*gphi[2][0]*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+(gphi[1][2]*(4*gphi[2][1]*pow(n[1],5)+4*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][1]*n[1]+gphi[2][0]*n[0])+gphi[2][2]*(4*gphi[1][1]*pow(n[1],5)+4*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][1]*n[1]+gphi[1][0]*n[0]))*pow(n[2],3)+(gphi[1][2]*gphi[2][2]*(2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)+4*gphi[1][1]*gphi[2][1]*pow(n[1],6)+(4*gphi[1][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],5)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],3)+(4*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+4*gphi[1][1]*gphi[2][1]+gphi[1][0]*gphi[2][0])*pow(n[1],2)+gphi[1][0]*gphi[2][0]*pow(n[0],2)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*n[1])*pow(n[2],2)+2*gphi[1][1]*gphi[2][1]*pow(n[1],8)+n[2]*(gphi[1][2]*(2*gphi[2][1]*pow(n[1],7)+2*gphi[2][0]*n[0]*pow(n[1],6)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],4)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])*pow(n[1],2)+gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[1][1]*pow(n[1],7)+2*gphi[1][0]*n[0]*pow(n[1],6)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],4)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])*pow(n[1],2)+gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[1][0]*n[0]))+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],7)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],6)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],5)+gphi[1][2]*gphi[2][2]*(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+gphi[1][0]*gphi[2][0])*pow(n[1],4)+(gphi[1][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+10*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+10*gphi[1][0]*n[0]))*pow(n[1],3)+(2*gphi[1][0]*gphi[2][0]*pow(n[0],6)-8*gphi[1][0]*gphi[2][0]*pow(n[0],4)+gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+10*gphi[1][0]*gphi[2][0]*pow(n[0],2)-2*gphi[1][0]*gphi[2][0])*pow(n[1],2)+gphi[1][0]*gphi[2][0]*pow(n[0],4)+n[1]*(gphi[1][1]*(2*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0]))-2*gphi[1][0]*gphi[2][0]*pow(n[0],2)+2*gphi[1][1]*gphi[2][1]+gphi[1][0]*gphi[2][0];

bk(5,7) = 2*gphi[1][2]*gphi[3][2]*pow(n[1],2)*pow(n[2],6)+(gphi[1][2]*(2*gphi[3][1]*pow(n[1],3)+2*gphi[3][0]*n[0]*pow(n[1],2))+gphi[3][2]*(2*gphi[1][1]*pow(n[1],3)+2*gphi[1][0]*n[0]*pow(n[1],2)))*pow(n[2],5)+(gphi[1][2]*gphi[3][2]*(4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)+2*gphi[1][1]*gphi[3][1]*pow(n[1],4)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],3)+2*gphi[1][0]*gphi[3][0]*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+(gphi[1][2]*(4*gphi[3][1]*pow(n[1],5)+4*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][1]*n[1]+gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[1][1]*pow(n[1],5)+4*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][1]*n[1]+gphi[1][0]*n[0]))*pow(n[2],3)+(gphi[1][2]*gphi[3][2]*(2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)+4*gphi[1][1]*gphi[3][1]*pow(n[1],6)+(4*gphi[1][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],5)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],3)+(4*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+4*gphi[1][1]*gphi[3][1]+gphi[1][0]*gphi[3][0])*pow(n[1],2)+gphi[1][0]*gphi[3][0]*pow(n[0],2)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*n[1])*pow(n[2],2)+2*gphi[1][1]*gphi[3][1]*pow(n[1],8)+n[2]*(gphi[1][2]*(2*gphi[3][1]*pow(n[1],7)+2*gphi[3][0]*n[0]*pow(n[1],6)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],4)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])*pow(n[1],2)+gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[1][1]*pow(n[1],7)+2*gphi[1][0]*n[0]*pow(n[1],6)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],4)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])*pow(n[1],2)+gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[1][0]*n[0]))+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],7)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],6)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],5)+gphi[1][2]*gphi[3][2]*(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+gphi[1][0]*gphi[3][0])*pow(n[1],4)+(gphi[1][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+10*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+10*gphi[1][0]*n[0]))*pow(n[1],3)+(2*gphi[1][0]*gphi[3][0]*pow(n[0],6)-8*gphi[1][0]*gphi[3][0]*pow(n[0],4)+gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+10*gphi[1][0]*gphi[3][0]*pow(n[0],2)-2*gphi[1][0]*gphi[3][0])*pow(n[1],2)+gphi[1][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[1][1]*(2*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0]))-2*gphi[1][0]*gphi[3][0]*pow(n[0],2)+2*gphi[1][1]*gphi[3][1]+gphi[1][0]*gphi[3][0];

bk(5,8) = 2*n[1]*gphi[0][2]*gphi[1][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[0]*n[1])+gphi[1][2]*(2*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[1][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[0][1]*gphi[1][1]*pow(n[1],3)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[0][2]*(4*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))+gphi[1][2]*(4*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])+gphi[0][1]))*pow(n[2],4)+(gphi[0][2]*gphi[1][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[0][1]*gphi[1][1]*pow(n[1],5)+(4*gphi[0][0]*n[0]*gphi[1][1]+4*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],4)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+2*gphi[0][1]*gphi[1][1]+gphi[0][0]*gphi[1][0])+gphi[1][0]*n[0]*gphi[0][1])*pow(n[2],3)+(gphi[0][2]*(2*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*n[0]*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+8*gphi[1][0]*n[0]))+gphi[1][2]*(2*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*n[0]*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0])+gphi[0][1]*(pow(n[0],2)-2)))*pow(n[2],2)+n[2]*(2*gphi[0][1]*gphi[1][1]*pow(n[1],7)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],6)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],4)+gphi[0][2]*gphi[1][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+gphi[0][0]*gphi[1][0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[1][0]*pow(n[0],6)-8*gphi[0][0]*gphi[1][0]*pow(n[0],4)+gphi[0][1]*gphi[1][1]*(2*pow(n[0],2)-4)+9*gphi[0][0]*gphi[1][0]*pow(n[0],2)-2*gphi[0][0]*gphi[1][0])+gphi[0][1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0]))+gphi[1][2]*(gphi[0][1]*pow(n[1],4)+gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0])+gphi[0][1]);

bk(5,9) = 2*n[1]*pow(gphi[1][2],2)*pow(n[2],7)+gphi[1][2]*(4*gphi[1][1]*pow(n[1],2)+4*gphi[1][0]*n[0]*n[1])*pow(n[2],6)+((4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))*pow(gphi[1][2],2)+2*pow(gphi[1][1],2)*pow(n[1],3)+4*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],2)+2*n[1]*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[2],5)+gphi[1][2]*(8*gphi[1][1]*pow(n[1],4)+8*gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(8*pow(n[0],2)-16)*pow(n[1],2)+n[1]*(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])+gphi[1][1])*pow(n[2],4)+((2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))*pow(gphi[1][2],2)+4*pow(gphi[1][1],2)*pow(n[1],5)+8*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],4)+((4*pow(n[0],2)-8)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],3)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],2)+n[1]*(2*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],4)-8*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2))+gphi[1][0]*n[0]*gphi[1][1])*pow(n[2],3)+gphi[1][2]*(4*gphi[1][1]*pow(n[1],6)+4*gphi[1][0]*n[0]*pow(n[1],5)+gphi[1][1]*(8*pow(n[0],2)-16)*pow(n[1],4)+(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],4)-16*pow(n[0],2)+18)*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+17*gphi[1][0]*n[0])+gphi[1][1]*(pow(n[0],2)-2))*pow(n[2],2)+n[2]*((2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))*pow(gphi[1][2],2)+2*pow(gphi[1][1],2)*pow(n[1],7)+4*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],6)+((4*pow(n[0],2)-8)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],5)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],4)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],4)-8*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2))*pow(n[1],3)+gphi[1][1]*(4*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+17*gphi[1][0]*n[0])*pow(n[1],2)+n[1]*((2*pow(n[0],2)-4)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],6)-8*pow(gphi[1][0],2)*pow(n[0],4)+9*pow(gphi[1][0],2)*pow(n[0],2)-2*pow(gphi[1][0],2))+gphi[1][1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0]))+gphi[1][2]*(gphi[1][1]*pow(n[1],4)+gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0])+gphi[1][1]);

bk(5,10) = 2*n[1]*gphi[1][2]*gphi[2][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[0]*n[1])+gphi[2][2]*(2*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[2][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[1][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[1][2]*(4*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1])+gphi[2][2]*(4*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])))*pow(n[2],4)+(gphi[1][2]*gphi[2][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[1][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[1][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],4)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+2*gphi[1][1]*gphi[2][1]+gphi[1][0]*gphi[2][0])+gphi[1][0]*n[0]*gphi[2][1])*pow(n[2],3)+(gphi[1][2]*(2*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*n[0]*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])+gphi[2][1]*(pow(n[0],2)-2))+gphi[2][2]*(2*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*n[0]*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+8*gphi[1][0]*n[0])))*pow(n[2],2)+n[2]*(2*gphi[1][1]*gphi[2][1]*pow(n[1],7)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],6)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],4)+gphi[1][2]*gphi[2][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+gphi[1][0]*gphi[2][0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+8*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[2][0]*pow(n[0],6)-8*gphi[1][0]*gphi[2][0]*pow(n[0],4)+gphi[1][1]*gphi[2][1]*(2*pow(n[0],2)-4)+9*gphi[1][0]*gphi[2][0]*pow(n[0],2)-2*gphi[1][0]*gphi[2][0])+gphi[2][1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0]))+gphi[1][2]*(gphi[2][1]*pow(n[1],4)+gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0])+gphi[2][1]);

bk(5,11) = 2*n[1]*gphi[1][2]*gphi[3][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[0]*n[1])+gphi[3][2]*(2*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[3][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[1][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[1][2]*(4*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1])+gphi[3][2]*(4*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])))*pow(n[2],4)+(gphi[1][2]*gphi[3][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[1][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[1][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],4)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+2*gphi[1][1]*gphi[3][1]+gphi[1][0]*gphi[3][0])+gphi[1][0]*n[0]*gphi[3][1])*pow(n[2],3)+(gphi[1][2]*(2*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*n[0]*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(pow(n[0],2)-2))+gphi[3][2]*(2*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*n[0]*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+8*gphi[1][0]*n[0])))*pow(n[2],2)+n[2]*(2*gphi[1][1]*gphi[3][1]*pow(n[1],7)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],6)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],4)+gphi[1][2]*gphi[3][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+gphi[1][0]*gphi[3][0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+8*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[3][0]*pow(n[0],6)-8*gphi[1][0]*gphi[3][0]*pow(n[0],4)+gphi[1][1]*gphi[3][1]*(2*pow(n[0],2)-4)+9*gphi[1][0]*gphi[3][0]*pow(n[0],2)-2*gphi[1][0]*gphi[3][0])+gphi[3][1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0]))+gphi[1][2]*(gphi[3][1]*pow(n[1],4)+gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0])+gphi[3][1]);

bk(6,6) = 2*pow(n[1],2)*pow(gphi[2][2],2)*pow(n[2],6)+gphi[2][2]*(4*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*n[0]*pow(n[1],2))*pow(n[2],5)+((4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)*pow(gphi[2][2],2)+2*pow(gphi[2][1],2)*pow(n[1],4)+4*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],3)+2*pow(gphi[2][0],2)*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+gphi[2][2]*(8*gphi[2][1]*pow(n[1],5)+8*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],2)+4*gphi[2][1]*n[1]+2*gphi[2][0]*n[0])*pow(n[2],3)+((2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)*pow(gphi[2][2],2)+4*pow(gphi[2][1],2)*pow(n[1],6)+8*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],3)+(4*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],4)-8*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2))*pow(n[1],2)+pow(gphi[2][0],2)*pow(n[0],2)+4*gphi[2][0]*n[0]*gphi[2][1]*n[1])*pow(n[2],2)+(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)*pow(gphi[2][2],2)+2*pow(gphi[2][1],2)*pow(n[1],8)+gphi[2][2]*n[2]*(4*gphi[2][1]*pow(n[1],7)+4*gphi[2][0]*n[0]*pow(n[1],6)+gphi[2][1]*(8*pow(n[0],2)-16)*pow(n[1],5)+(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+18*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(4*pow(n[0],2)-8)-4*gphi[2][0]*n[0])+4*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],7)+((4*pow(n[0],2)-8)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],6)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],5)+((2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],4)-8*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2))*pow(n[1],4)+gphi[2][1]*(4*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+20*gphi[2][0]*n[0])*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],6)-8*pow(gphi[2][0],2)*pow(n[0],4)+10*pow(gphi[2][0],2)*pow(n[0],2)-2*pow(gphi[2][0],2))*pow(n[1],2)+2*pow(gphi[2][1],2)+pow(gphi[2][0],2)*pow(n[0],4)+gphi[2][1]*n[1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])-2*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2);

bk(6,7) = 2*gphi[2][2]*gphi[3][2]*pow(n[1],2)*pow(n[2],6)+(gphi[2][2]*(2*gphi[3][1]*pow(n[1],3)+2*gphi[3][0]*n[0]*pow(n[1],2))+gphi[3][2]*(2*gphi[2][1]*pow(n[1],3)+2*gphi[2][0]*n[0]*pow(n[1],2)))*pow(n[2],5)+(gphi[2][2]*gphi[3][2]*(4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)+2*gphi[2][1]*gphi[3][1]*pow(n[1],4)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],3)+2*gphi[2][0]*gphi[3][0]*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+(gphi[2][2]*(4*gphi[3][1]*pow(n[1],5)+4*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][1]*n[1]+gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[2][1]*pow(n[1],5)+4*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][1]*n[1]+gphi[2][0]*n[0]))*pow(n[2],3)+(gphi[2][2]*gphi[3][2]*(2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)+4*gphi[2][1]*gphi[3][1]*pow(n[1],6)+(4*gphi[2][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],5)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],4)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))*pow(n[1],3)+(4*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+4*gphi[2][1]*gphi[3][1]+gphi[2][0]*gphi[3][0])*pow(n[1],2)+gphi[2][0]*gphi[3][0]*pow(n[0],2)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*n[1])*pow(n[2],2)+2*gphi[2][1]*gphi[3][1]*pow(n[1],8)+n[2]*(gphi[2][2]*(2*gphi[3][1]*pow(n[1],7)+2*gphi[3][0]*n[0]*pow(n[1],6)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],4)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])*pow(n[1],2)+gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[2][1]*pow(n[1],7)+2*gphi[2][0]*n[0]*pow(n[1],6)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],5)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],4)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],3)+(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])*pow(n[1],2)+gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(2*pow(n[0],2)-4)-2*gphi[2][0]*n[0]))+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],7)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],6)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))*pow(n[1],5)+gphi[2][2]*gphi[3][2]*(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+gphi[2][0]*gphi[3][0])*pow(n[1],4)+(gphi[2][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+10*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+10*gphi[2][0]*n[0]))*pow(n[1],3)+(2*gphi[2][0]*gphi[3][0]*pow(n[0],6)-8*gphi[2][0]*gphi[3][0]*pow(n[0],4)+gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+10*gphi[2][0]*gphi[3][0]*pow(n[0],2)-2*gphi[2][0]*gphi[3][0])*pow(n[1],2)+gphi[2][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[2][1]*(2*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0]))-2*gphi[2][0]*gphi[3][0]*pow(n[0],2)+2*gphi[2][1]*gphi[3][1]+gphi[2][0]*gphi[3][0];

bk(6,8) = 2*n[1]*gphi[0][2]*gphi[2][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[0]*n[1])+gphi[2][2]*(2*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[2][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[0][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[0][2]*(4*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))+gphi[2][2]*(4*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])+gphi[0][1]))*pow(n[2],4)+(gphi[0][2]*gphi[2][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[0][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[0][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],4)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+2*gphi[0][1]*gphi[2][1]+gphi[0][0]*gphi[2][0])+gphi[2][0]*n[0]*gphi[0][1])*pow(n[2],3)+(gphi[0][2]*(2*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*n[0]*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+8*gphi[2][0]*n[0]))+gphi[2][2]*(2*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*n[0]*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0])+gphi[0][1]*(pow(n[0],2)-2)))*pow(n[2],2)+n[2]*(2*gphi[0][1]*gphi[2][1]*pow(n[1],7)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],6)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],4)+gphi[0][2]*gphi[2][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+gphi[0][0]*gphi[2][0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[2][0]*pow(n[0],6)-8*gphi[0][0]*gphi[2][0]*pow(n[0],4)+gphi[0][1]*gphi[2][1]*(2*pow(n[0],2)-4)+9*gphi[0][0]*gphi[2][0]*pow(n[0],2)-2*gphi[0][0]*gphi[2][0])+gphi[0][1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0]))+gphi[2][2]*(gphi[0][1]*pow(n[1],4)+gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0])+gphi[0][1]);

bk(6,9) = 2*n[1]*gphi[1][2]*gphi[2][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[0]*n[1])+gphi[2][2]*(2*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[2][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[1][1]*gphi[2][1]*pow(n[1],3)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[1][2]*(4*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))+gphi[2][2]*(4*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]))*pow(n[2],4)+(gphi[1][2]*gphi[2][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[1][1]*gphi[2][1]*pow(n[1],5)+(4*gphi[1][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],4)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+2*gphi[1][1]*gphi[2][1]+gphi[1][0]*gphi[2][0])+gphi[2][0]*n[0]*gphi[1][1])*pow(n[2],3)+(gphi[1][2]*(2*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*n[0]*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+8*gphi[2][0]*n[0]))+gphi[2][2]*(2*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*n[0]*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])+gphi[1][1]*(pow(n[0],2)-2)))*pow(n[2],2)+n[2]*(2*gphi[1][1]*gphi[2][1]*pow(n[1],7)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],6)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],4)+gphi[1][2]*gphi[2][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+gphi[1][0]*gphi[2][0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+8*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[2][0]*pow(n[0],6)-8*gphi[1][0]*gphi[2][0]*pow(n[0],4)+gphi[1][1]*gphi[2][1]*(2*pow(n[0],2)-4)+9*gphi[1][0]*gphi[2][0]*pow(n[0],2)-2*gphi[1][0]*gphi[2][0])+gphi[1][1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0]))+gphi[2][2]*(gphi[1][1]*pow(n[1],4)+gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0])+gphi[1][1]);

bk(6,10) = 2*n[1]*pow(gphi[2][2],2)*pow(n[2],7)+gphi[2][2]*(4*gphi[2][1]*pow(n[1],2)+4*gphi[2][0]*n[0]*n[1])*pow(n[2],6)+((4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))*pow(gphi[2][2],2)+2*pow(gphi[2][1],2)*pow(n[1],3)+4*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],2)+2*n[1]*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[2],5)+gphi[2][2]*(8*gphi[2][1]*pow(n[1],4)+8*gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(8*pow(n[0],2)-16)*pow(n[1],2)+n[1]*(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])+gphi[2][1])*pow(n[2],4)+((2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))*pow(gphi[2][2],2)+4*pow(gphi[2][1],2)*pow(n[1],5)+8*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],4)+((4*pow(n[0],2)-8)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],3)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],2)+n[1]*(2*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],4)-8*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2))+gphi[2][0]*n[0]*gphi[2][1])*pow(n[2],3)+gphi[2][2]*(4*gphi[2][1]*pow(n[1],6)+4*gphi[2][0]*n[0]*pow(n[1],5)+gphi[2][1]*(8*pow(n[0],2)-16)*pow(n[1],4)+(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],4)-16*pow(n[0],2)+18)*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+17*gphi[2][0]*n[0])+gphi[2][1]*(pow(n[0],2)-2))*pow(n[2],2)+n[2]*((2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))*pow(gphi[2][2],2)+2*pow(gphi[2][1],2)*pow(n[1],7)+4*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],6)+((4*pow(n[0],2)-8)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],5)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],4)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],4)-8*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2))*pow(n[1],3)+gphi[2][1]*(4*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+17*gphi[2][0]*n[0])*pow(n[1],2)+n[1]*((2*pow(n[0],2)-4)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],6)-8*pow(gphi[2][0],2)*pow(n[0],4)+9*pow(gphi[2][0],2)*pow(n[0],2)-2*pow(gphi[2][0],2))+gphi[2][1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0]))+gphi[2][2]*(gphi[2][1]*pow(n[1],4)+gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0])+gphi[2][1]);

bk(6,11) = 2*n[1]*gphi[2][2]*gphi[3][2]*pow(n[2],7)+(gphi[2][2]*(2*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[0]*n[1])+gphi[3][2]*(2*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[2][2]*gphi[3][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[2][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[2][2]*(4*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1])+gphi[3][2]*(4*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])))*pow(n[2],4)+(gphi[2][2]*gphi[3][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[2][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[2][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],4)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+2*gphi[2][1]*gphi[3][1]+gphi[2][0]*gphi[3][0])+gphi[2][0]*n[0]*gphi[3][1])*pow(n[2],3)+(gphi[2][2]*(2*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*n[0]*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(pow(n[0],2)-2))+gphi[3][2]*(2*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*n[0]*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+8*gphi[2][0]*n[0])))*pow(n[2],2)+n[2]*(2*gphi[2][1]*gphi[3][1]*pow(n[1],7)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],6)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],5)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))*pow(n[1],4)+gphi[2][2]*gphi[3][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+gphi[2][0]*gphi[3][0])*pow(n[1],3)+(gphi[2][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+8*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[2][0]*gphi[3][0]*pow(n[0],6)-8*gphi[2][0]*gphi[3][0]*pow(n[0],4)+gphi[2][1]*gphi[3][1]*(2*pow(n[0],2)-4)+9*gphi[2][0]*gphi[3][0]*pow(n[0],2)-2*gphi[2][0]*gphi[3][0])+gphi[3][1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0]))+gphi[2][2]*(gphi[3][1]*pow(n[1],4)+gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0])+gphi[3][1]);

bk(7,7) = 2*pow(n[1],2)*pow(gphi[3][2],2)*pow(n[2],6)+gphi[3][2]*(4*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*n[0]*pow(n[1],2))*pow(n[2],5)+((4*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+1)*pow(gphi[3][2],2)+2*pow(gphi[3][1],2)*pow(n[1],4)+4*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],3)+2*pow(gphi[3][0],2)*pow(n[0],2)*pow(n[1],2))*pow(n[2],4)+gphi[3][2]*(8*gphi[3][1]*pow(n[1],5)+8*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],2)+4*gphi[3][1]*n[1]+2*gphi[3][0]*n[0])*pow(n[2],3)+((2*pow(n[1],6)+(4*pow(n[0],2)-8)*pow(n[1],4)+(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+pow(n[0],2)-2)*pow(gphi[3][2],2)+4*pow(gphi[3][1],2)*pow(n[1],6)+8*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],3)+(4*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],4)-8*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2))*pow(n[1],2)+pow(gphi[3][0],2)*pow(n[0],2)+4*gphi[3][0]*n[0]*gphi[3][1]*n[1])*pow(n[2],2)+(pow(n[1],4)+(pow(n[0],2)-2)*pow(n[1],2)+1)*pow(gphi[3][2],2)+2*pow(gphi[3][1],2)*pow(n[1],8)+gphi[3][2]*n[2]*(4*gphi[3][1]*pow(n[1],7)+4*gphi[3][0]*n[0]*pow(n[1],6)+gphi[3][1]*(8*pow(n[0],2)-16)*pow(n[1],5)+(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+18*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(4*pow(n[0],2)-8)-4*gphi[3][0]*n[0])+4*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],7)+((4*pow(n[0],2)-8)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],6)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],5)+((2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],4)-8*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2))*pow(n[1],4)+gphi[3][1]*(4*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+20*gphi[3][0]*n[0])*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],6)-8*pow(gphi[3][0],2)*pow(n[0],4)+10*pow(gphi[3][0],2)*pow(n[0],2)-2*pow(gphi[3][0],2))*pow(n[1],2)+2*pow(gphi[3][1],2)+pow(gphi[3][0],2)*pow(n[0],4)+gphi[3][1]*n[1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])-2*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2);

bk(7,8) = 2*n[1]*gphi[0][2]*gphi[3][2]*pow(n[2],7)+(gphi[0][2]*(2*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[0]*n[1])+gphi[3][2]*(2*gphi[0][1]*pow(n[1],2)+2*gphi[0][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[0][2]*gphi[3][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[0][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[0][2]*(4*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0]))+gphi[3][2]*(4*gphi[0][1]*pow(n[1],4)+4*gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])+gphi[0][1]))*pow(n[2],4)+(gphi[0][2]*gphi[3][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[0][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[0][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],4)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+2*gphi[0][1]*gphi[3][1]+gphi[0][0]*gphi[3][0])+gphi[3][0]*n[0]*gphi[0][1])*pow(n[2],3)+(gphi[0][2]*(2*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*n[0]*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+8*gphi[3][0]*n[0]))+gphi[3][2]*(2*gphi[0][1]*pow(n[1],6)+2*gphi[0][0]*n[0]*pow(n[1],5)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],3)+gphi[0][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0])+gphi[0][1]*(pow(n[0],2)-2)))*pow(n[2],2)+n[2]*(2*gphi[0][1]*gphi[3][1]*pow(n[1],7)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],6)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],5)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],4)+gphi[0][2]*gphi[3][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+gphi[0][0]*gphi[3][0])*pow(n[1],3)+(gphi[0][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+8*gphi[0][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[0][0]*gphi[3][0]*pow(n[0],6)-8*gphi[0][0]*gphi[3][0]*pow(n[0],4)+gphi[0][1]*gphi[3][1]*(2*pow(n[0],2)-4)+9*gphi[0][0]*gphi[3][0]*pow(n[0],2)-2*gphi[0][0]*gphi[3][0])+gphi[0][1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0]))+gphi[3][2]*(gphi[0][1]*pow(n[1],4)+gphi[0][0]*n[0]*pow(n[1],3)+gphi[0][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0])+gphi[0][1]);

bk(7,9) = 2*n[1]*gphi[1][2]*gphi[3][2]*pow(n[2],7)+(gphi[1][2]*(2*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[0]*n[1])+gphi[3][2]*(2*gphi[1][1]*pow(n[1],2)+2*gphi[1][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[1][2]*gphi[3][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[1][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[1][2]*(4*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0]))+gphi[3][2]*(4*gphi[1][1]*pow(n[1],4)+4*gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]))*pow(n[2],4)+(gphi[1][2]*gphi[3][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[1][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[1][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],4)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+2*gphi[1][1]*gphi[3][1]+gphi[1][0]*gphi[3][0])+gphi[3][0]*n[0]*gphi[1][1])*pow(n[2],3)+(gphi[1][2]*(2*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*n[0]*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+8*gphi[3][0]*n[0]))+gphi[3][2]*(2*gphi[1][1]*pow(n[1],6)+2*gphi[1][0]*n[0]*pow(n[1],5)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],3)+gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])+gphi[1][1]*(pow(n[0],2)-2)))*pow(n[2],2)+n[2]*(2*gphi[1][1]*gphi[3][1]*pow(n[1],7)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],6)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],5)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],4)+gphi[1][2]*gphi[3][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+gphi[1][0]*gphi[3][0])*pow(n[1],3)+(gphi[1][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+8*gphi[1][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[1][0]*gphi[3][0]*pow(n[0],6)-8*gphi[1][0]*gphi[3][0]*pow(n[0],4)+gphi[1][1]*gphi[3][1]*(2*pow(n[0],2)-4)+9*gphi[1][0]*gphi[3][0]*pow(n[0],2)-2*gphi[1][0]*gphi[3][0])+gphi[1][1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0]))+gphi[3][2]*(gphi[1][1]*pow(n[1],4)+gphi[1][0]*n[0]*pow(n[1],3)+gphi[1][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0])+gphi[1][1]);

bk(7,10) = 2*n[1]*gphi[2][2]*gphi[3][2]*pow(n[2],7)+(gphi[2][2]*(2*gphi[3][1]*pow(n[1],2)+2*gphi[3][0]*n[0]*n[1])+gphi[3][2]*(2*gphi[2][1]*pow(n[1],2)+2*gphi[2][0]*n[0]*n[1]))*pow(n[2],6)+(gphi[2][2]*gphi[3][2]*(4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))+2*gphi[2][1]*gphi[3][1]*pow(n[1],3)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*n[1]*pow(n[0],2))*pow(n[2],5)+(gphi[2][2]*(4*gphi[3][1]*pow(n[1],4)+4*gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0]))+gphi[3][2]*(4*gphi[2][1]*pow(n[1],4)+4*gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],2)+n[1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]))*pow(n[2],4)+(gphi[2][2]*gphi[3][2]*(2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))+4*gphi[2][1]*gphi[3][1]*pow(n[1],5)+(4*gphi[2][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],4)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],3)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))*pow(n[1],2)+n[1]*(4*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+2*gphi[2][1]*gphi[3][1]+gphi[2][0]*gphi[3][0])+gphi[3][0]*n[0]*gphi[2][1])*pow(n[2],3)+(gphi[2][2]*(2*gphi[3][1]*pow(n[1],6)+2*gphi[3][0]*n[0]*pow(n[1],5)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],3)+gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+8)*pow(n[1],2)+n[1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+8*gphi[3][0]*n[0]))+gphi[3][2]*(2*gphi[2][1]*pow(n[1],6)+2*gphi[2][0]*n[0]*pow(n[1],5)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],4)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],3)+gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(n[1],2)+n[1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])+gphi[2][1]*(pow(n[0],2)-2)))*pow(n[2],2)+n[2]*(2*gphi[2][1]*gphi[3][1]*pow(n[1],7)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],6)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],5)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))*pow(n[1],4)+gphi[2][2]*gphi[3][2]*(2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+gphi[2][0]*gphi[3][0])*pow(n[1],3)+(gphi[2][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+8*gphi[2][0]*n[0]))*pow(n[1],2)+n[1]*(2*gphi[2][0]*gphi[3][0]*pow(n[0],6)-8*gphi[2][0]*gphi[3][0]*pow(n[0],4)+gphi[2][1]*gphi[3][1]*(2*pow(n[0],2)-4)+9*gphi[2][0]*gphi[3][0]*pow(n[0],2)-2*gphi[2][0]*gphi[3][0])+gphi[2][1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0]))+gphi[3][2]*(gphi[2][1]*pow(n[1],4)+gphi[2][0]*n[0]*pow(n[1],3)+gphi[2][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0])+gphi[2][1]);

bk(7,11) = 2*n[1]*pow(gphi[3][2],2)*pow(n[2],7)+gphi[3][2]*(4*gphi[3][1]*pow(n[1],2)+4*gphi[3][0]*n[0]*n[1])*pow(n[2],6)+((4*pow(n[1],3)+n[1]*(4*pow(n[0],2)-8))*pow(gphi[3][2],2)+2*pow(gphi[3][1],2)*pow(n[1],3)+4*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],2)+2*n[1]*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[2],5)+gphi[3][2]*(8*gphi[3][1]*pow(n[1],4)+8*gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(8*pow(n[0],2)-16)*pow(n[1],2)+n[1]*(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])+gphi[3][1])*pow(n[2],4)+((2*pow(n[1],5)+(4*pow(n[0],2)-8)*pow(n[1],3)+n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10))*pow(gphi[3][2],2)+4*pow(gphi[3][1],2)*pow(n[1],5)+8*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],4)+((4*pow(n[0],2)-8)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],3)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],2)+n[1]*(2*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],4)-8*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2))+gphi[3][0]*n[0]*gphi[3][1])*pow(n[2],3)+gphi[3][2]*(4*gphi[3][1]*pow(n[1],6)+4*gphi[3][0]*n[0]*pow(n[1],5)+gphi[3][1]*(8*pow(n[0],2)-16)*pow(n[1],4)+(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],3)+gphi[3][1]*(4*pow(n[0],4)-16*pow(n[0],2)+18)*pow(n[1],2)+n[1]*(4*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+17*gphi[3][0]*n[0])+gphi[3][1]*(pow(n[0],2)-2))*pow(n[2],2)+n[2]*((2*pow(n[1],3)+n[1]*(2*pow(n[0],2)-4))*pow(gphi[3][2],2)+2*pow(gphi[3][1],2)*pow(n[1],7)+4*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],6)+((4*pow(n[0],2)-8)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],5)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],4)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],4)-8*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2))*pow(n[1],3)+gphi[3][1]*(4*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+17*gphi[3][0]*n[0])*pow(n[1],2)+n[1]*((2*pow(n[0],2)-4)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],6)-8*pow(gphi[3][0],2)*pow(n[0],4)+9*pow(gphi[3][0],2)*pow(n[0],2)-2*pow(gphi[3][0],2))+gphi[3][1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0]))+gphi[3][2]*(gphi[3][1]*pow(n[1],4)+gphi[3][0]*n[0]*pow(n[1],3)+gphi[3][1]*(pow(n[0],2)-2)*pow(n[1],2)+n[1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0])+gphi[3][1]);

bk(8,8) = 2*pow(gphi[0][2],2)*pow(n[2],8)+(4*gphi[0][1]*n[1]+4*gphi[0][0]*n[0])*gphi[0][2]*pow(n[2],7)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[0][2],2)+2*pow(gphi[0][1],2)*pow(n[1],2)+2*pow(gphi[0][0],2)*pow(n[0],2)+4*gphi[0][0]*n[0]*gphi[0][1]*n[1])*pow(n[2],6)+gphi[0][2]*(8*gphi[0][1]*pow(n[1],3)+8*gphi[0][0]*n[0]*pow(n[1],2)+8*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(8*pow(n[0],2)-16)-16*gphi[0][0]*n[0])*pow(n[2],5)+((2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[0][2],2)+4*pow(gphi[0][1],2)*pow(n[1],4)+8*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],4)+gphi[0][1]*n[1]*(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])-8*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2))*pow(n[2],4)+gphi[0][2]*(4*gphi[0][1]*pow(n[1],5)+4*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],2)+4*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)-16*gphi[0][0]*pow(n[0],3)+20*gphi[0][0]*n[0])*pow(n[2],3)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[0][2],2)+2*pow(gphi[0][1],2)*pow(n[1],6)+4*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[0][1]*(8*gphi[0][0]*pow(n[0],3)-16*gphi[0][0]*n[0])*pow(n[1],3)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[0][1],2)+4*pow(gphi[0][0],2)*pow(n[0],4)-8*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2))*pow(n[1],2)+(pow(n[0],2)-2)*pow(gphi[0][1],2)+2*pow(gphi[0][0],2)*pow(n[0],6)+gphi[0][1]*n[1]*(4*gphi[0][0]*pow(n[0],5)-16*gphi[0][0]*pow(n[0],3)+18*gphi[0][0]*n[0])-8*pow(gphi[0][0],2)*pow(n[0],4)+10*pow(gphi[0][0],2)*pow(n[0],2)-2*pow(gphi[0][0],2))*pow(n[2],2)+2*pow(gphi[0][2],2)+pow(gphi[0][1],2)*pow(n[1],4)+gphi[0][2]*n[2]*(4*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*n[0]*pow(n[1],2)+4*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[0][0]*n[0])+2*gphi[0][0]*n[0]*gphi[0][1]*pow(n[1],3)+((pow(n[0],2)-2)*pow(gphi[0][1],2)+pow(gphi[0][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[0][1],2)+pow(gphi[0][0],2)*pow(n[0],4)+gphi[0][1]*n[1]*(2*gphi[0][0]*pow(n[0],3)-4*gphi[0][0]*n[0])-2*pow(gphi[0][0],2)*pow(n[0],2)+pow(gphi[0][0],2);

bk(8,9) = 2*gphi[0][2]*gphi[1][2]*pow(n[2],8)+((2*gphi[0][1]*n[1]+2*gphi[0][0]*n[0])*gphi[1][2]+(2*gphi[1][1]*n[1]+2*gphi[1][0]*n[0])*gphi[0][2])*pow(n[2],7)+(gphi[0][2]*gphi[1][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+2*gphi[0][1]*gphi[1][1]*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],2)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*n[1])*pow(n[2],6)+(gphi[0][2]*(4*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*n[0]*pow(n[1],2)+4*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[1][0]*n[0])+gphi[1][2]*(4*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*n[0]*pow(n[1],2)+4*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[0][0]*n[0]))*pow(n[2],5)+(gphi[0][2]*gphi[1][2]*(2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[0][1]*gphi[1][1]*pow(n[1],4)+(4*gphi[0][0]*n[0]*gphi[1][1]+4*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],3)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],2)+4*gphi[0][0]*gphi[1][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+gphi[0][1]*gphi[1][1]+gphi[0][0]*gphi[1][0])*pow(n[2],4)+(gphi[0][2]*(2*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[1][0]*pow(n[0],3)+10*gphi[1][0]*n[0])+gphi[1][2]*(2*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[0][0]*pow(n[0],3)+10*gphi[0][0]*n[0]))*pow(n[2],3)+(2*gphi[0][1]*gphi[1][1]*pow(n[1],6)+(2*gphi[0][0]*n[0]*gphi[1][1]+2*gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],5)+(gphi[0][1]*gphi[1][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[1][0]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])+gphi[1][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],3)+gphi[0][2]*gphi[1][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+(gphi[0][1]*gphi[1][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[1][0]*pow(n[0],4)-8*gphi[0][0]*gphi[1][0]*pow(n[0],2)+gphi[0][0]*gphi[1][0])*pow(n[1],2)+2*gphi[0][0]*gphi[1][0]*pow(n[0],6)+n[1]*(gphi[0][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0])+gphi[1][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[1][0]*pow(n[0],4)+10*gphi[0][0]*gphi[1][0]*pow(n[0],2)+gphi[0][1]*gphi[1][1]*(pow(n[0],2)-2)-2*gphi[0][0]*gphi[1][0])*pow(n[2],2)+gphi[0][1]*gphi[1][1]*pow(n[1],4)+n[2]*(gphi[0][2]*(2*gphi[1][1]*pow(n[1],3)+2*gphi[1][0]*n[0]*pow(n[1],2)+2*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[1][0]*n[0])+gphi[1][2]*(2*gphi[0][1]*pow(n[1],3)+2*gphi[0][0]*n[0]*pow(n[1],2)+2*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[0][0]*n[0]))+(gphi[0][0]*n[0]*gphi[1][1]+gphi[1][0]*n[0]*gphi[0][1])*pow(n[1],3)+(gphi[0][0]*gphi[1][0]*pow(n[0],2)+gphi[0][1]*gphi[1][1]*(pow(n[0],2)-2))*pow(n[1],2)+gphi[0][0]*gphi[1][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0])+gphi[1][1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0]))-2*gphi[0][0]*gphi[1][0]*pow(n[0],2)+2*gphi[0][2]*gphi[1][2]+gphi[0][1]*gphi[1][1]+gphi[0][0]*gphi[1][0];

bk(8,10) = 2*gphi[0][2]*gphi[2][2]*pow(n[2],8)+((2*gphi[0][1]*n[1]+2*gphi[0][0]*n[0])*gphi[2][2]+(2*gphi[2][1]*n[1]+2*gphi[2][0]*n[0])*gphi[0][2])*pow(n[2],7)+(gphi[0][2]*gphi[2][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+2*gphi[0][1]*gphi[2][1]*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],2)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*n[1])*pow(n[2],6)+(gphi[0][2]*(4*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*n[0]*pow(n[1],2)+4*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[2][0]*n[0])+gphi[2][2]*(4*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*n[0]*pow(n[1],2)+4*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[0][0]*n[0]))*pow(n[2],5)+(gphi[0][2]*gphi[2][2]*(2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[0][1]*gphi[2][1]*pow(n[1],4)+(4*gphi[0][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],3)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],2)+4*gphi[0][0]*gphi[2][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+gphi[0][1]*gphi[2][1]+gphi[0][0]*gphi[2][0])*pow(n[2],4)+(gphi[0][2]*(2*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[2][0]*pow(n[0],3)+10*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[0][0]*pow(n[0],3)+10*gphi[0][0]*n[0]))*pow(n[2],3)+(2*gphi[0][1]*gphi[2][1]*pow(n[1],6)+(2*gphi[0][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],5)+(gphi[0][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],3)+gphi[0][2]*gphi[2][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+(gphi[0][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[2][0]*pow(n[0],4)-8*gphi[0][0]*gphi[2][0]*pow(n[0],2)+gphi[0][0]*gphi[2][0])*pow(n[1],2)+2*gphi[0][0]*gphi[2][0]*pow(n[0],6)+n[1]*(gphi[0][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[2][0]*pow(n[0],4)+10*gphi[0][0]*gphi[2][0]*pow(n[0],2)+gphi[0][1]*gphi[2][1]*(pow(n[0],2)-2)-2*gphi[0][0]*gphi[2][0])*pow(n[2],2)+gphi[0][1]*gphi[2][1]*pow(n[1],4)+n[2]*(gphi[0][2]*(2*gphi[2][1]*pow(n[1],3)+2*gphi[2][0]*n[0]*pow(n[1],2)+2*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[0][1]*pow(n[1],3)+2*gphi[0][0]*n[0]*pow(n[1],2)+2*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[0][0]*n[0]))+(gphi[0][0]*n[0]*gphi[2][1]+gphi[2][0]*n[0]*gphi[0][1])*pow(n[1],3)+(gphi[0][0]*gphi[2][0]*pow(n[0],2)+gphi[0][1]*gphi[2][1]*(pow(n[0],2)-2))*pow(n[1],2)+gphi[0][0]*gphi[2][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0])+gphi[2][1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0]))-2*gphi[0][0]*gphi[2][0]*pow(n[0],2)+2*gphi[0][2]*gphi[2][2]+gphi[0][1]*gphi[2][1]+gphi[0][0]*gphi[2][0];

bk(8,11) = 2*gphi[0][2]*gphi[3][2]*pow(n[2],8)+((2*gphi[0][1]*n[1]+2*gphi[0][0]*n[0])*gphi[3][2]+(2*gphi[3][1]*n[1]+2*gphi[3][0]*n[0])*gphi[0][2])*pow(n[2],7)+(gphi[0][2]*gphi[3][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+2*gphi[0][1]*gphi[3][1]*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],2)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*n[1])*pow(n[2],6)+(gphi[0][2]*(4*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*n[0]*pow(n[1],2)+4*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[0][1]*pow(n[1],3)+4*gphi[0][0]*n[0]*pow(n[1],2)+4*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[0][0]*n[0]))*pow(n[2],5)+(gphi[0][2]*gphi[3][2]*(2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[0][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[0][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],3)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+4*gphi[0][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+gphi[0][1]*gphi[3][1]+gphi[0][0]*gphi[3][0])*pow(n[2],4)+(gphi[0][2]*(2*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[3][0]*pow(n[0],3)+10*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[0][1]*pow(n[1],5)+2*gphi[0][0]*n[0]*pow(n[1],4)+gphi[0][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0])*pow(n[1],2)+2*gphi[0][0]*pow(n[0],5)+gphi[0][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[0][0]*pow(n[0],3)+10*gphi[0][0]*n[0]))*pow(n[2],3)+(2*gphi[0][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[0][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],5)+(gphi[0][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[0][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],4)+(gphi[0][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[0][0]*pow(n[0],3)-8*gphi[0][0]*n[0]))*pow(n[1],3)+gphi[0][2]*gphi[3][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+(gphi[0][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[0][0]*gphi[3][0]*pow(n[0],4)-8*gphi[0][0]*gphi[3][0]*pow(n[0],2)+gphi[0][0]*gphi[3][0])*pow(n[1],2)+2*gphi[0][0]*gphi[3][0]*pow(n[0],6)+n[1]*(gphi[0][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[0][0]*pow(n[0],5)-8*gphi[0][0]*pow(n[0],3)+9*gphi[0][0]*n[0]))-8*gphi[0][0]*gphi[3][0]*pow(n[0],4)+10*gphi[0][0]*gphi[3][0]*pow(n[0],2)+gphi[0][1]*gphi[3][1]*(pow(n[0],2)-2)-2*gphi[0][0]*gphi[3][0])*pow(n[2],2)+gphi[0][1]*gphi[3][1]*pow(n[1],4)+n[2]*(gphi[0][2]*(2*gphi[3][1]*pow(n[1],3)+2*gphi[3][0]*n[0]*pow(n[1],2)+2*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[0][1]*pow(n[1],3)+2*gphi[0][0]*n[0]*pow(n[1],2)+2*gphi[0][0]*pow(n[0],3)+gphi[0][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[0][0]*n[0]))+(gphi[0][0]*n[0]*gphi[3][1]+gphi[3][0]*n[0]*gphi[0][1])*pow(n[1],3)+(gphi[0][0]*gphi[3][0]*pow(n[0],2)+gphi[0][1]*gphi[3][1]*(pow(n[0],2)-2))*pow(n[1],2)+gphi[0][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[0][1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0])+gphi[3][1]*(gphi[0][0]*pow(n[0],3)-2*gphi[0][0]*n[0]))-2*gphi[0][0]*gphi[3][0]*pow(n[0],2)+2*gphi[0][2]*gphi[3][2]+gphi[0][1]*gphi[3][1]+gphi[0][0]*gphi[3][0];

bk(9,9) = 2*pow(gphi[1][2],2)*pow(n[2],8)+(4*gphi[1][1]*n[1]+4*gphi[1][0]*n[0])*gphi[1][2]*pow(n[2],7)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[1][2],2)+2*pow(gphi[1][1],2)*pow(n[1],2)+2*pow(gphi[1][0],2)*pow(n[0],2)+4*gphi[1][0]*n[0]*gphi[1][1]*n[1])*pow(n[2],6)+gphi[1][2]*(8*gphi[1][1]*pow(n[1],3)+8*gphi[1][0]*n[0]*pow(n[1],2)+8*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(8*pow(n[0],2)-16)-16*gphi[1][0]*n[0])*pow(n[2],5)+((2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[1][2],2)+4*pow(gphi[1][1],2)*pow(n[1],4)+8*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],4)+gphi[1][1]*n[1]*(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])-8*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2))*pow(n[2],4)+gphi[1][2]*(4*gphi[1][1]*pow(n[1],5)+4*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],2)+4*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)-16*gphi[1][0]*pow(n[0],3)+20*gphi[1][0]*n[0])*pow(n[2],3)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[1][2],2)+2*pow(gphi[1][1],2)*pow(n[1],6)+4*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[1][1]*(8*gphi[1][0]*pow(n[0],3)-16*gphi[1][0]*n[0])*pow(n[1],3)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[1][1],2)+4*pow(gphi[1][0],2)*pow(n[0],4)-8*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2))*pow(n[1],2)+(pow(n[0],2)-2)*pow(gphi[1][1],2)+2*pow(gphi[1][0],2)*pow(n[0],6)+gphi[1][1]*n[1]*(4*gphi[1][0]*pow(n[0],5)-16*gphi[1][0]*pow(n[0],3)+18*gphi[1][0]*n[0])-8*pow(gphi[1][0],2)*pow(n[0],4)+10*pow(gphi[1][0],2)*pow(n[0],2)-2*pow(gphi[1][0],2))*pow(n[2],2)+2*pow(gphi[1][2],2)+pow(gphi[1][1],2)*pow(n[1],4)+gphi[1][2]*n[2]*(4*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*n[0]*pow(n[1],2)+4*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[1][0]*n[0])+2*gphi[1][0]*n[0]*gphi[1][1]*pow(n[1],3)+((pow(n[0],2)-2)*pow(gphi[1][1],2)+pow(gphi[1][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[1][1],2)+pow(gphi[1][0],2)*pow(n[0],4)+gphi[1][1]*n[1]*(2*gphi[1][0]*pow(n[0],3)-4*gphi[1][0]*n[0])-2*pow(gphi[1][0],2)*pow(n[0],2)+pow(gphi[1][0],2);

bk(9,10) = 2*gphi[1][2]*gphi[2][2]*pow(n[2],8)+((2*gphi[1][1]*n[1]+2*gphi[1][0]*n[0])*gphi[2][2]+(2*gphi[2][1]*n[1]+2*gphi[2][0]*n[0])*gphi[1][2])*pow(n[2],7)+(gphi[1][2]*gphi[2][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+2*gphi[1][1]*gphi[2][1]*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],2)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*n[1])*pow(n[2],6)+(gphi[1][2]*(4*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*n[0]*pow(n[1],2)+4*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[2][0]*n[0])+gphi[2][2]*(4*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*n[0]*pow(n[1],2)+4*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[1][0]*n[0]))*pow(n[2],5)+(gphi[1][2]*gphi[2][2]*(2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[1][1]*gphi[2][1]*pow(n[1],4)+(4*gphi[1][0]*n[0]*gphi[2][1]+4*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],3)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],2)+4*gphi[1][0]*gphi[2][0]*pow(n[0],4)+n[1]*(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+gphi[1][1]*gphi[2][1]+gphi[1][0]*gphi[2][0])*pow(n[2],4)+(gphi[1][2]*(2*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[2][0]*pow(n[0],3)+10*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[1][0]*pow(n[0],3)+10*gphi[1][0]*n[0]))*pow(n[2],3)+(2*gphi[1][1]*gphi[2][1]*pow(n[1],6)+(2*gphi[1][0]*n[0]*gphi[2][1]+2*gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],5)+(gphi[1][1]*gphi[2][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[2][0]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])+gphi[2][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],3)+gphi[1][2]*gphi[2][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+(gphi[1][1]*gphi[2][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[1][0]*gphi[2][0]*pow(n[0],4)-8*gphi[1][0]*gphi[2][0]*pow(n[0],2)+gphi[1][0]*gphi[2][0])*pow(n[1],2)+2*gphi[1][0]*gphi[2][0]*pow(n[0],6)+n[1]*(gphi[1][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0])+gphi[2][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[2][0]*pow(n[0],4)+10*gphi[1][0]*gphi[2][0]*pow(n[0],2)+gphi[1][1]*gphi[2][1]*(pow(n[0],2)-2)-2*gphi[1][0]*gphi[2][0])*pow(n[2],2)+gphi[1][1]*gphi[2][1]*pow(n[1],4)+n[2]*(gphi[1][2]*(2*gphi[2][1]*pow(n[1],3)+2*gphi[2][0]*n[0]*pow(n[1],2)+2*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[2][0]*n[0])+gphi[2][2]*(2*gphi[1][1]*pow(n[1],3)+2*gphi[1][0]*n[0]*pow(n[1],2)+2*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[1][0]*n[0]))+(gphi[1][0]*n[0]*gphi[2][1]+gphi[2][0]*n[0]*gphi[1][1])*pow(n[1],3)+(gphi[1][0]*gphi[2][0]*pow(n[0],2)+gphi[1][1]*gphi[2][1]*(pow(n[0],2)-2))*pow(n[1],2)+gphi[1][0]*gphi[2][0]*pow(n[0],4)+n[1]*(gphi[1][1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0])+gphi[2][1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0]))-2*gphi[1][0]*gphi[2][0]*pow(n[0],2)+2*gphi[1][2]*gphi[2][2]+gphi[1][1]*gphi[2][1]+gphi[1][0]*gphi[2][0];

bk(9,11) = 2*gphi[1][2]*gphi[3][2]*pow(n[2],8)+((2*gphi[1][1]*n[1]+2*gphi[1][0]*n[0])*gphi[3][2]+(2*gphi[3][1]*n[1]+2*gphi[3][0]*n[0])*gphi[1][2])*pow(n[2],7)+(gphi[1][2]*gphi[3][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+2*gphi[1][1]*gphi[3][1]*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],2)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*n[1])*pow(n[2],6)+(gphi[1][2]*(4*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*n[0]*pow(n[1],2)+4*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[1][1]*pow(n[1],3)+4*gphi[1][0]*n[0]*pow(n[1],2)+4*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[1][0]*n[0]))*pow(n[2],5)+(gphi[1][2]*gphi[3][2]*(2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[1][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[1][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],3)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+4*gphi[1][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+gphi[1][1]*gphi[3][1]+gphi[1][0]*gphi[3][0])*pow(n[2],4)+(gphi[1][2]*(2*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[3][0]*pow(n[0],3)+10*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[1][1]*pow(n[1],5)+2*gphi[1][0]*n[0]*pow(n[1],4)+gphi[1][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0])*pow(n[1],2)+2*gphi[1][0]*pow(n[0],5)+gphi[1][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[1][0]*pow(n[0],3)+10*gphi[1][0]*n[0]))*pow(n[2],3)+(2*gphi[1][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[1][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],5)+(gphi[1][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[1][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],4)+(gphi[1][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[1][0]*pow(n[0],3)-8*gphi[1][0]*n[0]))*pow(n[1],3)+gphi[1][2]*gphi[3][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+(gphi[1][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[1][0]*gphi[3][0]*pow(n[0],4)-8*gphi[1][0]*gphi[3][0]*pow(n[0],2)+gphi[1][0]*gphi[3][0])*pow(n[1],2)+2*gphi[1][0]*gphi[3][0]*pow(n[0],6)+n[1]*(gphi[1][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[1][0]*pow(n[0],5)-8*gphi[1][0]*pow(n[0],3)+9*gphi[1][0]*n[0]))-8*gphi[1][0]*gphi[3][0]*pow(n[0],4)+10*gphi[1][0]*gphi[3][0]*pow(n[0],2)+gphi[1][1]*gphi[3][1]*(pow(n[0],2)-2)-2*gphi[1][0]*gphi[3][0])*pow(n[2],2)+gphi[1][1]*gphi[3][1]*pow(n[1],4)+n[2]*(gphi[1][2]*(2*gphi[3][1]*pow(n[1],3)+2*gphi[3][0]*n[0]*pow(n[1],2)+2*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[1][1]*pow(n[1],3)+2*gphi[1][0]*n[0]*pow(n[1],2)+2*gphi[1][0]*pow(n[0],3)+gphi[1][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[1][0]*n[0]))+(gphi[1][0]*n[0]*gphi[3][1]+gphi[3][0]*n[0]*gphi[1][1])*pow(n[1],3)+(gphi[1][0]*gphi[3][0]*pow(n[0],2)+gphi[1][1]*gphi[3][1]*(pow(n[0],2)-2))*pow(n[1],2)+gphi[1][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[1][1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0])+gphi[3][1]*(gphi[1][0]*pow(n[0],3)-2*gphi[1][0]*n[0]))-2*gphi[1][0]*gphi[3][0]*pow(n[0],2)+2*gphi[1][2]*gphi[3][2]+gphi[1][1]*gphi[3][1]+gphi[1][0]*gphi[3][0];

bk(10,10) = 2*pow(gphi[2][2],2)*pow(n[2],8)+(4*gphi[2][1]*n[1]+4*gphi[2][0]*n[0])*gphi[2][2]*pow(n[2],7)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[2][2],2)+2*pow(gphi[2][1],2)*pow(n[1],2)+2*pow(gphi[2][0],2)*pow(n[0],2)+4*gphi[2][0]*n[0]*gphi[2][1]*n[1])*pow(n[2],6)+gphi[2][2]*(8*gphi[2][1]*pow(n[1],3)+8*gphi[2][0]*n[0]*pow(n[1],2)+8*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(8*pow(n[0],2)-16)-16*gphi[2][0]*n[0])*pow(n[2],5)+((2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[2][2],2)+4*pow(gphi[2][1],2)*pow(n[1],4)+8*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],4)+gphi[2][1]*n[1]*(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])-8*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2))*pow(n[2],4)+gphi[2][2]*(4*gphi[2][1]*pow(n[1],5)+4*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],2)+4*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)-16*gphi[2][0]*pow(n[0],3)+20*gphi[2][0]*n[0])*pow(n[2],3)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[2][2],2)+2*pow(gphi[2][1],2)*pow(n[1],6)+4*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[2][1]*(8*gphi[2][0]*pow(n[0],3)-16*gphi[2][0]*n[0])*pow(n[1],3)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[2][1],2)+4*pow(gphi[2][0],2)*pow(n[0],4)-8*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2))*pow(n[1],2)+(pow(n[0],2)-2)*pow(gphi[2][1],2)+2*pow(gphi[2][0],2)*pow(n[0],6)+gphi[2][1]*n[1]*(4*gphi[2][0]*pow(n[0],5)-16*gphi[2][0]*pow(n[0],3)+18*gphi[2][0]*n[0])-8*pow(gphi[2][0],2)*pow(n[0],4)+10*pow(gphi[2][0],2)*pow(n[0],2)-2*pow(gphi[2][0],2))*pow(n[2],2)+2*pow(gphi[2][2],2)+pow(gphi[2][1],2)*pow(n[1],4)+gphi[2][2]*n[2]*(4*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*n[0]*pow(n[1],2)+4*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[2][0]*n[0])+2*gphi[2][0]*n[0]*gphi[2][1]*pow(n[1],3)+((pow(n[0],2)-2)*pow(gphi[2][1],2)+pow(gphi[2][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[2][1],2)+pow(gphi[2][0],2)*pow(n[0],4)+gphi[2][1]*n[1]*(2*gphi[2][0]*pow(n[0],3)-4*gphi[2][0]*n[0])-2*pow(gphi[2][0],2)*pow(n[0],2)+pow(gphi[2][0],2);

bk(10,11) = 2*gphi[2][2]*gphi[3][2]*pow(n[2],8)+((2*gphi[2][1]*n[1]+2*gphi[2][0]*n[0])*gphi[3][2]+(2*gphi[3][1]*n[1]+2*gphi[3][0]*n[0])*gphi[2][2])*pow(n[2],7)+(gphi[2][2]*gphi[3][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+2*gphi[2][1]*gphi[3][1]*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],2)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*n[1])*pow(n[2],6)+(gphi[2][2]*(4*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*n[0]*pow(n[1],2)+4*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[3][0]*n[0])+gphi[3][2]*(4*gphi[2][1]*pow(n[1],3)+4*gphi[2][0]*n[0]*pow(n[1],2)+4*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[2][0]*n[0]))*pow(n[2],5)+(gphi[2][2]*gphi[3][2]*(2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)+4*gphi[2][1]*gphi[3][1]*pow(n[1],4)+(4*gphi[2][0]*n[0]*gphi[3][1]+4*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],3)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+4*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],2)+4*gphi[2][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+gphi[2][1]*gphi[3][1]+gphi[2][0]*gphi[3][0])*pow(n[2],4)+(gphi[2][2]*(2*gphi[3][1]*pow(n[1],5)+2*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])*pow(n[1],2)+2*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[3][0]*pow(n[0],3)+10*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[2][1]*pow(n[1],5)+2*gphi[2][0]*n[0]*pow(n[1],4)+gphi[2][1]*(4*pow(n[0],2)-8)*pow(n[1],3)+(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0])*pow(n[1],2)+2*gphi[2][0]*pow(n[0],5)+gphi[2][1]*n[1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)-8*gphi[2][0]*pow(n[0],3)+10*gphi[2][0]*n[0]))*pow(n[2],3)+(2*gphi[2][1]*gphi[3][1]*pow(n[1],6)+(2*gphi[2][0]*n[0]*gphi[3][1]+2*gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],5)+(gphi[2][1]*gphi[3][1]*(4*pow(n[0],2)-8)+2*gphi[2][0]*gphi[3][0]*pow(n[0],2))*pow(n[1],4)+(gphi[2][1]*(4*gphi[3][0]*pow(n[0],3)-8*gphi[3][0]*n[0])+gphi[3][1]*(4*gphi[2][0]*pow(n[0],3)-8*gphi[2][0]*n[0]))*pow(n[1],3)+gphi[2][2]*gphi[3][2]*(4*pow(n[1],2)+4*pow(n[0],2)-8)+(gphi[2][1]*gphi[3][1]*(2*pow(n[0],4)-8*pow(n[0],2)+10)+4*gphi[2][0]*gphi[3][0]*pow(n[0],4)-8*gphi[2][0]*gphi[3][0]*pow(n[0],2)+gphi[2][0]*gphi[3][0])*pow(n[1],2)+2*gphi[2][0]*gphi[3][0]*pow(n[0],6)+n[1]*(gphi[2][1]*(2*gphi[3][0]*pow(n[0],5)-8*gphi[3][0]*pow(n[0],3)+9*gphi[3][0]*n[0])+gphi[3][1]*(2*gphi[2][0]*pow(n[0],5)-8*gphi[2][0]*pow(n[0],3)+9*gphi[2][0]*n[0]))-8*gphi[2][0]*gphi[3][0]*pow(n[0],4)+10*gphi[2][0]*gphi[3][0]*pow(n[0],2)+gphi[2][1]*gphi[3][1]*(pow(n[0],2)-2)-2*gphi[2][0]*gphi[3][0])*pow(n[2],2)+gphi[2][1]*gphi[3][1]*pow(n[1],4)+n[2]*(gphi[2][2]*(2*gphi[3][1]*pow(n[1],3)+2*gphi[3][0]*n[0]*pow(n[1],2)+2*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[3][0]*n[0])+gphi[3][2]*(2*gphi[2][1]*pow(n[1],3)+2*gphi[2][0]*n[0]*pow(n[1],2)+2*gphi[2][0]*pow(n[0],3)+gphi[2][1]*n[1]*(2*pow(n[0],2)-4)-4*gphi[2][0]*n[0]))+(gphi[2][0]*n[0]*gphi[3][1]+gphi[3][0]*n[0]*gphi[2][1])*pow(n[1],3)+(gphi[2][0]*gphi[3][0]*pow(n[0],2)+gphi[2][1]*gphi[3][1]*(pow(n[0],2)-2))*pow(n[1],2)+gphi[2][0]*gphi[3][0]*pow(n[0],4)+n[1]*(gphi[2][1]*(gphi[3][0]*pow(n[0],3)-2*gphi[3][0]*n[0])+gphi[3][1]*(gphi[2][0]*pow(n[0],3)-2*gphi[2][0]*n[0]))-2*gphi[2][0]*gphi[3][0]*pow(n[0],2)+2*gphi[2][2]*gphi[3][2]+gphi[2][1]*gphi[3][1]+gphi[2][0]*gphi[3][0];

bk(11,11) = 2*pow(gphi[3][2],2)*pow(n[2],8)+(4*gphi[3][1]*n[1]+4*gphi[3][0]*n[0])*gphi[3][2]*pow(n[2],7)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[3][2],2)+2*pow(gphi[3][1],2)*pow(n[1],2)+2*pow(gphi[3][0],2)*pow(n[0],2)+4*gphi[3][0]*n[0]*gphi[3][1]*n[1])*pow(n[2],6)+gphi[3][2]*(8*gphi[3][1]*pow(n[1],3)+8*gphi[3][0]*n[0]*pow(n[1],2)+8*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(8*pow(n[0],2)-16)-16*gphi[3][0]*n[0])*pow(n[2],5)+((2*pow(n[1],4)+(4*pow(n[0],2)-8)*pow(n[1],2)+2*pow(n[0],4)-8*pow(n[0],2)+12)*pow(gphi[3][2],2)+4*pow(gphi[3][1],2)*pow(n[1],4)+8*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],3)+((4*pow(n[0],2)-8)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],4)+gphi[3][1]*n[1]*(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])-8*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2))*pow(n[2],4)+gphi[3][2]*(4*gphi[3][1]*pow(n[1],5)+4*gphi[3][0]*n[0]*pow(n[1],4)+gphi[3][1]*(8*pow(n[0],2)-16)*pow(n[1],3)+(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],2)+4*gphi[3][0]*pow(n[0],5)+gphi[3][1]*n[1]*(4*pow(n[0],4)-16*pow(n[0],2)+20)-16*gphi[3][0]*pow(n[0],3)+20*gphi[3][0]*n[0])*pow(n[2],3)+((4*pow(n[1],2)+4*pow(n[0],2)-8)*pow(gphi[3][2],2)+2*pow(gphi[3][1],2)*pow(n[1],6)+4*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],5)+((4*pow(n[0],2)-8)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],4)+gphi[3][1]*(8*gphi[3][0]*pow(n[0],3)-16*gphi[3][0]*n[0])*pow(n[1],3)+((2*pow(n[0],4)-8*pow(n[0],2)+10)*pow(gphi[3][1],2)+4*pow(gphi[3][0],2)*pow(n[0],4)-8*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2))*pow(n[1],2)+(pow(n[0],2)-2)*pow(gphi[3][1],2)+2*pow(gphi[3][0],2)*pow(n[0],6)+gphi[3][1]*n[1]*(4*gphi[3][0]*pow(n[0],5)-16*gphi[3][0]*pow(n[0],3)+18*gphi[3][0]*n[0])-8*pow(gphi[3][0],2)*pow(n[0],4)+10*pow(gphi[3][0],2)*pow(n[0],2)-2*pow(gphi[3][0],2))*pow(n[2],2)+2*pow(gphi[3][2],2)+pow(gphi[3][1],2)*pow(n[1],4)+gphi[3][2]*n[2]*(4*gphi[3][1]*pow(n[1],3)+4*gphi[3][0]*n[0]*pow(n[1],2)+4*gphi[3][0]*pow(n[0],3)+gphi[3][1]*n[1]*(4*pow(n[0],2)-8)-8*gphi[3][0]*n[0])+2*gphi[3][0]*n[0]*gphi[3][1]*pow(n[1],3)+((pow(n[0],2)-2)*pow(gphi[3][1],2)+pow(gphi[3][0],2)*pow(n[0],2))*pow(n[1],2)+pow(gphi[3][1],2)+pow(gphi[3][0],2)*pow(n[0],4)+gphi[3][1]*n[1]*(2*gphi[3][0]*pow(n[0],3)-4*gphi[3][0]*n[0])-2*pow(gphi[3][0],2)*pow(n[0],2)+pow(gphi[3][0],2);

bk(1,0) = bk(0,1);

bk(2,0) = bk(0,2);

bk(2,1) = bk(1,2);

bk(3,0) = bk(0,3);

bk(3,1) = bk(1,3);

bk(3,2) = bk(2,3);

bk(4,0) = bk(0,4);

bk(4,1) = bk(1,4);

bk(4,2) = bk(2,4);

bk(4,3) = bk(3,4);

bk(5,0) = bk(0,5);

bk(5,1) = bk(1,5);

bk(5,2) = bk(2,5);

bk(5,3) = bk(3,5);

bk(5,4) = bk(4,5);

bk(6,0) = bk(0,6);

bk(6,1) = bk(1,6);

bk(6,2) = bk(2,6);

bk(6,3) = bk(3,6);

bk(6,4) = bk(4,6);

bk(6,5) = bk(5,6);

bk(7,0) = bk(0,7);

bk(7,1) = bk(1,7);

bk(7,2) = bk(2,7);

bk(7,3) = bk(3,7);

bk(7,4) = bk(4,7);

bk(7,5) = bk(5,7);

bk(7,6) = bk(6,7);

bk(8,0) = bk(0,8);

bk(8,1) = bk(1,8);

bk(8,2) = bk(2,8);

bk(8,3) = bk(3,8);

bk(8,4) = bk(4,8);

bk(8,5) = bk(5,8);

bk(8,6) = bk(6,8);

bk(8,7) = bk(7,8);

bk(9,0) = bk(0,9);

bk(9,1) = bk(1,9);

bk(9,2) = bk(2,9);

bk(9,3) = bk(3,9);

bk(9,4) = bk(4,9);

bk(9,5) = bk(5,9);

bk(9,6) = bk(6,9);

bk(9,7) = bk(7,9);

bk(9,8) = bk(8,9);

bk(10,0) = bk(0,10);

bk(10,1) = bk(1,10);

bk(10,2) = bk(2,10);

bk(10,3) = bk(3,10);

bk(10,4) = bk(4,10);

bk(10,5) = bk(5,10);

bk(10,6) = bk(6,10);

bk(10,7) = bk(7,10);

bk(10,8) = bk(8,10);

bk(10,9) = bk(9,10);

bk(11,0) = bk(0,11);

bk(11,1) = bk(1,11);

bk(11,2) = bk(2,11);

bk(11,3) = bk(3,11);

bk(11,4) = bk(4,11);

bk(11,5) = bk(5,11);

bk(11,6) = bk(6,11);

bk(11,7) = bk(7,11);

bk(11,8) = bk(8,11);

bk(11,9) = bk(9,11);

bk(11,10) = bk(10,11);

bk=aire*bk;

}


// extends from Gamma_h to the band beta_h: local matrix for triangle
void
compute_matrix_extension_t (const ublas::vector<point>& x ,const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {

    point a, b;
    Float S;
    tensor inv_B;
    ublas::vector<size_t> j(3);
    subcompute_matrix_t (x, f, j, a, b, S);
    inv_piola_t (x, inv_B);

    // calcul des coordonnes du milieu de segment [a,b]
    point c = (a+b)*0.5;

    // definition des coordonnes des points a, b et c dans le triangle de reference
    point hat_a = inv_B*(a-x[0]);
    point hat_b = inv_B*(b-x[0]);
    point hat_c = inv_B*(c-x[0]);
  
    // caclul des termes de la matrice de masse en utilisant la formule de simpson
    mk(0,0) = 1./6*S*(1*hat_a0(hat_a)+4*0.5*hat_a0(hat_c)                );
    mk(0,1) = 1./6*S*(                4*0.5*hat_a0(hat_c)+1*hat_a0(hat_b));
    mk(1,0) = 1./6*S*(1*hat_a1(hat_a)+4*0.5*hat_a1(hat_c)                );
    mk(1,1) = 1./6*S*(                4*0.5*hat_a1(hat_c)+1*hat_a1(hat_b));
    mk(2,0) = 1./6*S*(1*hat_a2(hat_a)+4*0.5*hat_a2(hat_c)                );
    mk(2,1) = 1./6*S*(                4*0.5*hat_a2(hat_c)+1*hat_a2(hat_b));
}

void
compute_matrix_a_triangle_T (const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& ak) {

    point a, b, c;
    Float aire;
    tensor inv_B;
    ublas::vector<size_t> j (4);
    subcompute_matrix_triangle_T (x, f, j, a, b, c, aire);
    inv_piola_T (x, inv_B);

    // calcul de la transposee de l'inverse de la matrice B
    tensor transposee = trans(inv_B);

    // la definition des gradients sur le triangle de reference
    point grad_hat_a0 (-1, -1, -1);
    point grad_hat_a1 ( 1,  0,  0);
    point grad_hat_a2 ( 0,  1,  0);
    point grad_hat_a3 ( 0,  0,  1);
    
    /* calcul des gradients sur les sommets du triange de reference */
    point grad_a0 = transposee*grad_hat_a0;
    point grad_a1 = transposee*grad_hat_a1;
    point grad_a2 = transposee*grad_hat_a2;
    point grad_a3 = transposee*grad_hat_a3;

    point normal = vect(b-a, c-a);
    normal = normal/norm(normal);

    // projection: grad_s
    projection_s (grad_a0, normal);
    projection_s (grad_a1, normal); 
    projection_s (grad_a2, normal);
    projection_s (grad_a3, normal);
    
    /* calcul de la matrice ak */
    ak(0,0) = aire*dot(grad_a0,grad_a0);
    ak(0,1) = aire*dot(grad_a0,grad_a1);
    ak(0,2) = aire*dot(grad_a0,grad_a2);
    ak(0,3) = aire*dot(grad_a0,grad_a3);

    ak(1,0) = aire*dot(grad_a1,grad_a0);
    ak(1,1) = aire*dot(grad_a1,grad_a1);
    ak(1,2) = aire*dot(grad_a1,grad_a2);
    ak(1,3) = aire*dot(grad_a1,grad_a3);

    ak(2,0) = aire*dot(grad_a2,grad_a0);
    ak(2,1) = aire*dot(grad_a2,grad_a1);
    ak(2,2) = aire*dot(grad_a2,grad_a2);
    ak(2,3) = aire*dot(grad_a2,grad_a3);

    ak(3,0) = aire*dot(grad_a3,grad_a0);
    ak(3,1) = aire*dot(grad_a3,grad_a1);
    ak(3,2) = aire*dot(grad_a3,grad_a2);
    ak(3,3) = aire*dot(grad_a3,grad_a3);
}
void
compute_matrix_m_triangle_T(const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {

    point a, b, c;
    Float aire;
    tensor inv_B;
    ublas::vector<size_t> j(4);
    subcompute_matrix_triangle_T (x, f, j, a, b, c, aire);
    inv_piola_T (x, inv_B);

    /*calcul des coordonnes d'intersection et des milieux des arretes dans le  tetraedre de reference*/
    point d = 0.5*(a+b);
    point e = 0.5*(a+c);
    point o = 0.5*(b+c);
    point v = d - x[0];
    point w = e - x[0];
    point z = o - x[0];
    point hat_d = inv_B*v;
    point hat_e = inv_B*w;
    point hat_o = inv_B*z;
    // calcul des termes de la martrice de masse mk
    mk(0,0)          = 1./3 * aire *(hat_a0_T(hat_d)*hat_a0_T(hat_d)+hat_a0_T(hat_e)*hat_a0_T(hat_e)+hat_a0_T(hat_o)*hat_a0_T(hat_o));
    mk(0,1) = mk(1,0)= 1./3 * aire *(hat_a0_T(hat_d)*hat_a1_T(hat_d)+hat_a0_T(hat_e)*hat_a1_T(hat_e)+hat_a0_T(hat_o)*hat_a1_T(hat_o));
    mk(0,2) = mk(2,0)= 1./3 * aire *(hat_a0_T(hat_d)*hat_a2_T(hat_d)+hat_a0_T(hat_e)*hat_a2_T(hat_e)+hat_a0_T(hat_o)*hat_a2_T(hat_o));
    mk(0,3) = mk(3,0)= 1./3 * aire *(hat_a0_T(hat_d)*hat_a3_T(hat_d)+hat_a0_T(hat_e)*hat_a3_T(hat_e)+hat_a0_T(hat_o)*hat_a3_T(hat_o));
    mk(1,1)          = 1./3 * aire *(hat_a1_T(hat_d)*hat_a1_T(hat_d)+hat_a1_T(hat_e)*hat_a1_T(hat_e)+hat_a1_T(hat_o)*hat_a1_T(hat_o));
    mk(1,2) = mk(2,1)= 1./3 * aire *(hat_a1_T(hat_d)*hat_a2_T(hat_d)+hat_a1_T(hat_e)*hat_a2_T(hat_e)+hat_a1_T(hat_o)*hat_a2_T(hat_o));
    mk(1,3) = mk(3,1)= 1./3 * aire *(hat_a1_T(hat_d)*hat_a3_T(hat_d)+hat_a1_T(hat_e)*hat_a3_T(hat_e)+hat_a1_T(hat_o)*hat_a3_T(hat_o));
    mk(2,2)          = 1./3 * aire *(hat_a2_T(hat_d)*hat_a2_T(hat_d)+hat_a2_T(hat_e)*hat_a2_T(hat_e)+hat_a2_T(hat_o)*hat_a2_T(hat_o));
    mk(2,3) = mk(3,2)= 1./3 * aire *(hat_a2_T(hat_d)*hat_a3_T(hat_d)+hat_a2_T(hat_e)*hat_a3_T(hat_e)+hat_a2_T(hat_o)*hat_a3_T(hat_o));
    mk(3,3) =          1./3 * aire *(hat_a3_T(hat_d)*hat_a3_T(hat_d)+hat_a3_T(hat_e)*hat_a3_T(hat_e)+hat_a3_T(hat_o)*hat_a3_T(hat_o));
}
void
compute_matrix_a_quadrilateral_T(const ublas::vector<point>& x, const ublas::vector<Float>& f,
                                 const quadruplet& q, ublas::matrix<Float>& ak) {

    point a, b, c, d;
    Float aire_Q;
    tensor inv_B;
    subcompute_matrix_quadrilateral_T (x, f, q, a, b, c, d, aire_Q);
    inv_piola_T (x, inv_B);

    // calcul de la transposee de l'inverse de la matrice B
    tensor transposee = trans(inv_B);

    //la definition des gradients sur le triangle ordinaire
    point grad_hat_a0 (-1, -1, -1);
    point grad_hat_a1 ( 1,  0,  0);
    point grad_hat_a2 ( 0,  1,  0);
    point grad_hat_a3 ( 0,  0,  1);

    /* calcul des gradients sur les sommets du triange de reference */
    point grad_a0 = transposee*grad_hat_a0;
    point grad_a1 = transposee*grad_hat_a1;
    point grad_a2 = transposee*grad_hat_a2;
    point grad_a3 = transposee*grad_hat_a3;

    point normal = vect(b-a, c-a);
    normal = normal/norm(normal);

    // projection: grad_s
    projection_s (grad_a0, normal);
    projection_s (grad_a1, normal); 
    projection_s (grad_a2, normal);
    projection_s (grad_a3, normal);

    /* calcul de la matrice ak */
    ak(0,0) = aire_Q* dot(grad_a0,grad_a0);
    ak(0,1) = aire_Q* dot(grad_a0,grad_a1);
    ak(0,2) = aire_Q* dot(grad_a0,grad_a2);
    ak(0,3) = aire_Q* dot(grad_a0,grad_a3);
    ak(1,0) = aire_Q* dot(grad_a1,grad_a0);
    ak(1,1) = aire_Q* dot(grad_a1,grad_a1);
    ak(1,2) = aire_Q* dot(grad_a1,grad_a2);
    ak(1,3) = aire_Q* dot(grad_a1,grad_a3);
    ak(2,0) = aire_Q* dot(grad_a2,grad_a0);
    ak(2,1) = aire_Q* dot(grad_a2,grad_a1);
    ak(2,2) = aire_Q* dot(grad_a2,grad_a2);
    ak(2,3) = aire_Q* dot(grad_a2,grad_a3);
    ak(3,0) = aire_Q* dot(grad_a3,grad_a0);
    ak(3,1) = aire_Q* dot(grad_a3,grad_a1);
    ak(3,2) = aire_Q* dot(grad_a3,grad_a2);
    ak(3,3) = aire_Q* dot(grad_a3,grad_a3);
}
void
compute_matrix_m_quadrilateral_T(const ublas::vector<point>& x, const ublas::vector<Float>& f,
                                 const quadruplet& q, ublas::matrix<Float>& mk) {

    point a, b, c, d;
    Float aire_Q;
    tensor inv_B;
    subcompute_matrix_quadrilateral_T (x, f, q, a, b, c, d, aire_Q);
    inv_piola_T (x, inv_B);

    /*calcul des coordonnes d'intersection et des milieux des arretes dans le  tetraedre de reference*/
    point r = 0.5*(a+b);
    point e = 0.5*(a+c);
    point o = 0.5*(b+c);
    point j = 0.5*(b+d);
    point h = 0.5*(d+c);
    
    // les coordonnees des milieux des arretes dans le tetra de reference
    point hat_r = inv_B*(r - x[0]);
    point hat_e = inv_B*(e - x[0]);
    point hat_o = inv_B*(o - x[0]);
    point hat_j = inv_B*(j - x[0]);
    point hat_h = inv_B*(h - x[0]);
    Float abc = 1./6.*norm(vect(a-b,a-c));
    Float dbc = 1./6.*norm(vect(b-c,b-d));

    mk(0,0) = abc*(hat_a0_T(hat_r)*hat_a0_T(hat_r)+hat_a0_T(hat_e)*hat_a0_T(hat_e)+hat_a0_T(hat_o)*hat_a0_T(hat_o))+
              dbc*(hat_a0_T(hat_j)*hat_a0_T(hat_j)+hat_a0_T(hat_h)*hat_a0_T(hat_h)+hat_a0_T(hat_o)*hat_a0_T(hat_o));


    mk(1,1) = abc*(hat_a1_T(hat_r)*hat_a1_T(hat_r)+hat_a1_T(hat_e)*hat_a1_T(hat_e)+hat_a1_T(hat_o)*hat_a1_T(hat_o))+
              dbc*(hat_a1_T(hat_j)*hat_a1_T(hat_j)+hat_a1_T(hat_h)*hat_a1_T(hat_h)+hat_a1_T(hat_o)*hat_a1_T(hat_o));

    mk(2,2) = abc*(hat_a2_T(hat_r)*hat_a2_T(hat_r)+hat_a2_T(hat_e)*hat_a2_T(hat_e)+hat_a2_T(hat_o)*hat_a2_T(hat_o))+
              dbc*(hat_a2_T(hat_j)*hat_a2_T(hat_j)+hat_a2_T(hat_h)*hat_a2_T(hat_h)+hat_a2_T(hat_o)*hat_a2_T(hat_o));

    mk(3,3) = abc*(hat_a3_T(hat_r)*hat_a3_T(hat_r)+hat_a3_T(hat_e)*hat_a3_T(hat_e)+hat_a3_T(hat_o)*hat_a3_T(hat_o))+
              dbc*(hat_a3_T(hat_j)*hat_a3_T(hat_j)+hat_a3_T(hat_h)*hat_a3_T(hat_h)+hat_a3_T(hat_o)*hat_a3_T(hat_o));

    mk(0,1) = mk(1,0) = abc*(hat_a0_T(hat_r)*hat_a1_T(hat_r)+hat_a0_T(hat_e)*hat_a1_T(hat_e)+hat_a0_T(hat_o)*hat_a1_T(hat_o))+
                        dbc*(hat_a0_T(hat_j)*hat_a1_T(hat_j)+hat_a0_T(hat_h)*hat_a1_T(hat_h)+hat_a0_T(hat_o)*hat_a1_T(hat_o));

    mk(0,2) = mk(2,0) = abc*(hat_a0_T(hat_r)*hat_a2_T(hat_r)+hat_a0_T(hat_e)*hat_a2_T(hat_e)+hat_a0_T(hat_o)*hat_a2_T(hat_o))+
                        dbc*(hat_a0_T(hat_j)*hat_a2_T(hat_j)+hat_a0_T(hat_h)*hat_a2_T(hat_h)+hat_a0_T(hat_o)*hat_a2_T(hat_o));

    mk(0,3) = mk(3,0) = abc*(hat_a0_T(hat_r)*hat_a3_T(hat_r)+hat_a0_T(hat_e)*hat_a3_T(hat_e)+hat_a0_T(hat_o)*hat_a3_T(hat_o))+
                        dbc*(hat_a0_T(hat_j)*hat_a3_T(hat_j)+hat_a0_T(hat_h)*hat_a3_T(hat_h)+hat_a0_T(hat_o)*hat_a3_T(hat_o));

    mk(1,2) = mk(2,1)= abc*(hat_a1_T(hat_r)*hat_a2_T(hat_r)+hat_a1_T(hat_e)*hat_a2_T(hat_e)+hat_a1_T(hat_o)*hat_a2_T(hat_o))+
                       dbc*(hat_a1_T(hat_j)*hat_a2_T(hat_j)+hat_a1_T(hat_h)*hat_a2_T(hat_h)+hat_a1_T(hat_o)*hat_a2_T(hat_o));

    mk(1,3) = mk(3,1)= abc*(hat_a1_T(hat_r)*hat_a3_T(hat_r)+hat_a1_T(hat_e)*hat_a3_T(hat_e)+hat_a1_T(hat_o)*hat_a3_T(hat_o))+
                       dbc*(hat_a1_T(hat_j)*hat_a3_T(hat_j)+hat_a1_T(hat_h)*hat_a3_T(hat_h)+hat_a1_T(hat_o)*hat_a3_T(hat_o));

    mk(2,3) = mk(3,2) = abc*(hat_a2_T(hat_r)*hat_a3_T(hat_r)+hat_a2_T(hat_e)*hat_a3_T(hat_e)+hat_a2_T(hat_o)*hat_a3_T(hat_o))+
                        dbc*(hat_a2_T(hat_j)*hat_a3_T(hat_j)+hat_a2_T(hat_h)*hat_a3_T(hat_h)+hat_a2_T(hat_o)*hat_a3_T(hat_o));

}
void
compute_matrix_a_T(const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& ak) {
    quadruplet q;
    if (intersection_is_quadrilateral_T (f, q)) {
        compute_matrix_a_quadrilateral_T (x, f, q, ak);
    } else {
        compute_matrix_a_triangle_T(x, f, ak);
    }
}
void
compute_matrix_m_T(const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {
    quadruplet q;
    if (intersection_is_quadrilateral_T (f, q)) {
        compute_matrix_m_quadrilateral_T (x, f, q, mk);
    } else {
        compute_matrix_m_triangle_T(x, f, mk);
    }
}
// ==========================================================================================
// generic matrix for (t,T) elements
// ==========================================================================================
void
compute_matrix_m (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {
  switch (K.variant()) {
    case geo_element::t:
      compute_matrix_m_t (x, f, mk);
      break;
    case geo_element::T:
      compute_matrix_m_T (x, f, mk);
      break;
    default :
      error_macro("mass/level set intersection: element not yet implemented: " << K.name());
  }
}
void
compute_matrix_a (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& ak) {
  switch (K.variant()) {
    case geo_element::t:
      compute_matrix_a_t (x, f, ak);
      break;
    case geo_element::T:
      compute_matrix_a_T (x, f, ak);
      break;
    default :
      error_macro("grad_grad_s/level set intersection: element not yet implemented: " << K.name());
  }
}
void
compute_matrix_grad_s (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {
  switch (K.variant()) {
    case geo_element::t:
      compute_matrix_grad_s_t (x, f, mk);
      break;
    default :
      error_macro("grad_s/level set intersection: element not yet implemented: " << K.name());
  }
}
void
compute_matrix_div_s (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {
  switch (K.variant()) {
    case geo_element::t:
      compute_matrix_div_s_t (x, f, mk);
      break;
    case geo_element::T:
      compute_matrix_div_s_T (x, f, mk);
      break;
    default :
      error_macro("div_s/level set intersection: element not yet implemented: " << K.name());
  }
}

void
compute_matrix_2D_s_D_s (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {
  switch (K.variant()) {
    case geo_element::T:
      compute_matrix_2D_s_D_s_T (x, f, mk);
      break;
    default :
      error_macro("grad_s/level set intersection: element not yet implemented: " << K.name());
  }
}


void
compute_matrix_extension (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk) {
  switch (K.variant()) {
    case geo_element::t:
      compute_matrix_extension_t (x, f, mk);
      break;
    default :
      error_macro("mass_extension/level set intersection: element not yet implemented: " << K.name());
  }
}
// =================================================================================
// maillage de la bande
// =================================================================================
size_t
build_vertex_connex_component (geo& band) {
  domain isolated_and_zero = band["isolated"] + band["zero"];
  // isolated is marked 0 and ic-th connex component is marked ic+1 (ic=0,1..)
  size_t not_marked = numeric_limits<size_t>::max();
  vector<size_t> mark (band.n_vertex(), not_marked);
  size_t n_marked = 0;
  for (size_t i = 0; i < isolated_and_zero.size(); i++) {
    const geo_element& P = isolated_and_zero.at(i);
    mark[P[0]] = 0;
    n_marked++;
  }
  size_t ic = 1;
  vector<set<size_t> > ball (band.n_vertex());
  const geo& const_band = band;
  build_point_to_element_sets (const_band.begin(), const_band.end(), ball.begin());
  for (size_t i = 0; i < band.n_vertex(); i++) {
    if (mark[i] != not_marked) continue;
    list<size_t> front;
    list<tiny_element> c_list;
    mark[i] = ic;
    n_marked++;
    front.push_back(i);
    tiny_element P;
    P.set_name ('p');
    P[0] = i;
    c_list.push_back(P);
    while (front.size() != 0) {
      size_t j = *(front.begin());
      front.pop_front();
      for (set<size_t>::const_iterator p = ball[j].begin(); p != ball[j].end(); p++) {
        const geo_element& K = band.element(*p);
        for (size_t kloc = 0; kloc < K.size(); kloc++) {
          size_t k = K[kloc];
          if (mark[k] != not_marked) continue;
          mark[k] = ic;
          n_marked++;
          front.push_back(k);
          P[0] = k;
          c_list.push_back(P);
        }
      }
    }
    domain c;
    c.set(c_list.begin(), c_list.size(), 0, "connected_component_"+itos(ic-1));
    band.insert_domain (c);
    ic++;
  }
  return ic-1;
}
void
banded_level_set (const field& phi, geo& geo_band, domain& dom_band) {
  const geo& Lambda = phi.get_geo();
  const space& Vh = phi.get_space();
  size_t d = Lambda.dimension();
  check_macro(phi.get_approx()=="P1", "Only P1 level set function supported");
  tiny_vector<size_t>  dofs (d+1);
  ublas::vector<Float> f    (d+1);
  vector<bool> belongs_to_band (Lambda.size(), false);
  size_t counter = 0;
  for (size_t i = 0; i < Lambda.size(); i++) {
    const geo_element& K = Lambda.element(i);
    Vh.set_dof (Lambda.begin()[K.index()], dofs);
    f.resize (dofs.size());
    for (size_t j = 0; j < K.size(); j++) {
      f[j] = phi.at(dofs[j]);
    }
    switch (K.variant()) {
      case geo_element::t:
        belongs_to_band [i] = belongs_to_band_t (f);
        break;
      case geo_element::T: {
        belongs_to_band [i] = belongs_to_band_T (f);
        break;
      }
      default :
        error_macro("level set intersection: element not yet implemented: " << K.name());
    }
    if (belongs_to_band[i]) counter++;
  }
  // create the domain = list of elements in the band:
  vector <tiny_element> table_band (counter);
  size_t j = 0;
  for (size_t i = 0; i < Lambda.size(); i++) {
    const geo_element& K = Lambda.element(i);
    if (belongs_to_band[i]) table_band[j++] = tiny_element(K);
  }
  dom_band.set(table_band.begin(), counter, Lambda.dimension(), "band");
  space Bh (phi.get_geo(), dom_band, "P1");
  field phi_band (Bh);
  // ICI ?
#ifdef TODO
  phi_band = phi[dom_band]; // TODO
#else // TODO
  phi_band.from_boundary_val (phi, 0, dom_band);
#endif // TODO
  geo_band = Bh.get_geo();
  //
  // add "zero" point domain, where phi = 0
  //
  size_t zero_counter = 0;
  for (size_t i = 0; i < geo_band.n_vertex(); i++) {
    if (is_zero (phi_band.at(i))) zero_counter++;
  }
  vector <tiny_element> table_zero (zero_counter);
  zero_counter = 0;
  for (size_t i = 0; i < geo_band.n_vertex(); i++) {
    if (! is_zero (phi_band.at(i))) continue;
    tiny_element P;
    P.set_name ('p');
    P[0] = i;
    table_zero [zero_counter++] = P;
  }
  domain dom_zero;
  dom_zero.set(table_zero.begin(), zero_counter, 0, "zero");
  geo_band.insert_domain (dom_zero);
  //
  // add "isolated" point domain, where phi(xi) != 0
  // but, for all element that contains K, 
  // and for all vertex xj of K, xj != xi, we have phi(xj) = 0
  //
  vector<bool> is_isolated (geo_band.n_vertex(), true);
  for (size_t j = 0; j < geo_band.size(); j++) {
    const geo_element& K = geo_band.element(j);
    size_t n_non_zero = 0;
    size_t k_isolated = 0;
    for (size_t k = 0; k < K.size(); k++) {
      if (! is_zero(phi_band.at(K[k]))) {
          n_non_zero++;
          k_isolated = k;
      }
    }
    for (size_t k = 0; k < K.size(); k++) {
      if (n_non_zero != 1 || k != k_isolated) {
        is_isolated [K[k]] = false;
      }
    }
  }
  list<tiny_element> list_isolated;
  for (size_t i = 0; i < geo_band.n_vertex(); i++) {
    if (!is_isolated[i]) continue;
    tiny_element P;
    P.set_name ('p');
    P[0] = i;
    list_isolated.push_back(P);
  }
  domain dom_isolated;
  dom_isolated.set(list_isolated.begin(), list_isolated.size(), 0, "isolated");
  geo_band.insert_domain (dom_isolated);

  // add isolated vertex connex components:
  build_vertex_connex_component (geo_band);
}
geo
banded_level_set (const field& phi) {
  geo g;
  domain d;
  banded_level_set(phi, g, d);
  return g;
}

// =================================================================================
// maillage de l'intersection
// =================================================================================
// TODO: move in reference_element
size_t edge_t_iloc (size_t l, size_t m) {
  static const int edge_t_iloc_table [3][3] = { 
			    {-1, 0, 2},
			    { 0,-1, 1},
			    { 2, 1,-1}};
  return size_t (edge_t_iloc_table [l][m]);
}
size_t edge_T_iloc (size_t l, size_t m) {
  static const int edge_T_iloc_table [4][4] = {
                            {-1, 0, 2, 3},
                            { 0,-1, 1, 4},
                            { 2, 1,-1, 5},
                            { 3, 4, 5,-1}};
  return size_t (edge_T_iloc_table [l][m]);
}
size_t face_T_iloc (size_t l, size_t m, size_t n) {
  static size_t face_T_iloc_table [4][4][4];
  bool static initialized = false;
  if (!initialized) {
    for (size_t i = 0; i < 4; i++) 
      for (size_t j = 0; j < 4; j++) 
        for (size_t k = 0; k < 4; k++) 
          face_T_iloc_table [i][j][k] = size_t(-1);
    reference_element hat_K (reference_element::T);
    for (size_t i_face = 0; i_face < hat_K.n_face(); i_face++) {
      size_t p[3];
      for (size_t k = 0; k < 3; k++) p[k] = hat_K.subgeo_local_vertex(2,i_face,k);
      face_T_iloc_table [p[0]][p[1]][p[2]] = i_face;
      face_T_iloc_table [p[0]][p[2]][p[1]] = i_face;
      face_T_iloc_table [p[1]][p[0]][p[2]] = i_face;
      face_T_iloc_table [p[1]][p[2]][p[0]] = i_face;
      face_T_iloc_table [p[2]][p[0]][p[1]] = i_face;
      face_T_iloc_table [p[2]][p[1]][p[0]] = i_face;
    }
  }
  return face_T_iloc_table [l][m][n];
}
// select in 2d the (1 or) 2 edges or in 3d the 3 or 4 edges that are intersected
// and count it only one time.
// this avoid the vertices merge process with an "epsilon" precision, as vtk does
geo
zero_level_set (const field& phi, std::vector<size_t>& element2face) {
    //
    // pass 1: loop on lambda: count elements in the band
    //
    const geo&   Lambda = phi.get_geo();
    const space& Vh     = phi.get_space();
    vector<bool> belongs_to_band (Lambda.size(), false);
    size_t d = Lambda.dimension();
    tiny_vector<size_t> dofs (d+1);
    ublas::vector<point> x(d+1);
    ublas::vector<Float> f(d+1);
    size_t counter = 0;
    for (size_t i = 0; i < Lambda.size(); i++) {
      const geo_element& K = Lambda.element(i);
      Vh.set_dof (Lambda.begin()[K.index()], dofs);
      f.resize (dofs.size());
      for (size_t j = 0; j < K.size(); j++) {
        f[j] = phi.at(dofs[j]);
      }
      switch (K.variant()) {
        case geo_element::t:
          belongs_to_band [i] = belongs_to_band_t (f);
          break;
        case geo_element::T: {
          belongs_to_band [i] = belongs_to_band_T (f);
          break;
        }
        default :
          error_macro("level set intersection: element not yet implemented: " << K.name());
      }
      if (belongs_to_band[i]) counter++;
    }
    element2face.resize (counter);
    //
    // pass 2: loop on lambda: build faces
    //
    typedef tiny_element element_type;
    bool triangle_3d_only = false;
    check_macro(phi.get_approx() == "P1", "Only P1 level set function supported");
    const size_t not_marked = numeric_limits<size_t>::max();
    vector<size_t> marked_vertex (Lambda.n_vertex(), not_marked);
    vector<size_t> marked_edge   (Lambda.n_edge(),   not_marked);
    vector<size_t> marked_face   (Lambda.n_face(),   not_marked);
    list<point>        vertex_intersection_list;
    list<element_type> element_intersection_list;
    ublas::vector<size_t> j(d+1);
    for (size_t i = 0, i_element_band = 0; i < Lambda.size(); i++) {
        if (! belongs_to_band [i]) continue;
        size_t i_element_face = not_marked;
        const geo_element& K = Lambda.element(i);
        Vh.set_dof (K, dofs);
	x.resize (K.size());
	f.resize (K.size());
        for (size_t k = 0; k < K.size(); k++) {
            x[k] = Lambda.vertex(K[k]);
            f[k] = phi.at(dofs[k]);
        }
        element_type S;
        switch (K.variant()) {
        case geo_element::t: {
            point a, b;
            Float length;
            subcompute_matrix_t (x, f, j, a, b, length);
	    if (is_zero(f[j[1]]) && is_zero(f[j[2]])) {
                // the full edge {j1,j2} is included in the surface mesh:
                for (size_t k = 0; k < 2; k++) {
                    size_t vertex_idx = K[j[k+1]];
	            if (marked_vertex [vertex_idx] == not_marked) {
	                marked_vertex [vertex_idx] = vertex_intersection_list.size();
                        vertex_intersection_list.push_back (Lambda.vertex(vertex_idx));
                    }
                }
	        size_t edge_iloc = edge_t_iloc (j[1], j[2]);
	        size_t edge_idx = K.edge (edge_iloc);
	        if (marked_edge [edge_idx] == not_marked) {
	            marked_edge [edge_idx] = element_intersection_list.size();
                    S.set_name('e');
                    for (size_t k = 0; k < S.size(); k++) {
                        S[k] = marked_vertex [K[j[k+1]]];
                    }
                    element_intersection_list.push_back (S);
                }
		i_element_face = marked_edge [edge_idx];
            } else {
                // create the new edge {j1,j2} by intersections:
                S.set_name('e');
		point x[2] = {a,b};
                for (size_t k = 0; k < 2; k++) {
                    if (! is_zero(f[j[k+1]]) && ! is_zero(f[j[0]])) {
                        // xk is inside edge {j0,j[k+1]} of triangle K:
                        size_t edge_iloc = edge_t_iloc (j[0], j[k+1]);
                        size_t edge_idx = K.edge (edge_iloc);
	                if (marked_edge [edge_idx] == not_marked) {
	                    marked_edge [edge_idx] = vertex_intersection_list.size();
                            vertex_intersection_list.push_back (x[k]);
                        }
	                S[k] = marked_edge [edge_idx];
                    } else { // xk is at edge boundary: a vertex of the 2d mesh
                        size_t vertex_idx = (!is_zero(f[j[0]])) ? K[j[k+1]] : K[j[0]];
	                if (marked_vertex [vertex_idx] == not_marked) {
	                    marked_vertex [vertex_idx] = vertex_intersection_list.size();
                            vertex_intersection_list.push_back (Lambda.vertex(vertex_idx));
                        }
	                S[k] = marked_vertex [vertex_idx];
                    }
                }
                // S[0] == S[1] when is_zero(f[j[0]]) but f[j[0]] != 0, i.e. precision pbs
		check_macro (S[0] != S[1], "degenerate edge intersection");
		i_element_face = element_intersection_list.size ();
                element_intersection_list.push_back (S);
            }
            break;
          }
        case geo_element::T: {
            quadruplet q;
            point a, b, c, d;
            Float aire;
            if (!intersection_is_quadrilateral_T (f, q)) {
                subcompute_matrix_triangle_T (x, f, j, a, b, c, aire);
	        if (is_zero(f[j[1]]) && is_zero(f[j[2]]) && is_zero(f[j[3]])) {
                    // the full face {j1,j2,j3} is included in the surface mesh:
                    for (size_t k = 0; k < 3; k++) {
                        size_t vertex_idx = K[j[k+1]];
	                if (marked_vertex [vertex_idx] == not_marked) {
	                    marked_vertex [vertex_idx] = vertex_intersection_list.size();
                            vertex_intersection_list.push_back (Lambda.vertex(vertex_idx));
                        }
                    }
	            size_t face_iloc = face_T_iloc (j[1], j[2], j[3]);
	            size_t face_idx = K.face (face_iloc);
	            if (marked_face [face_idx] == not_marked) {
	                marked_face [face_idx] = element_intersection_list.size();
                        S.set_name('t');
                        for (size_t k = 0; k < S.size(); k++) {
                            S[k] = marked_vertex [K[j[k+1]]];
                        }
                        element_intersection_list.push_back (S);
                    }
		    i_element_face = marked_face [face_idx];
                } else {
                    // create the new face {j1,j2,j3} by intersections:
                    S.set_name('t');
		    point x[3] = {a,b,c};
                    for (size_t k = 0; k < 3; k++) {
                        if (! is_zero(f[j[k+1]]) && ! is_zero(f[j[0]])) {
                            // xk is inside edge {j0,j[k+1]} of triangle K:
                            size_t edge_iloc = edge_T_iloc (j[0], j[k+1]);
                            size_t edge_idx = K.edge (edge_iloc);
	                    if (marked_edge [edge_idx] == not_marked) {
	                        marked_edge [edge_idx] = vertex_intersection_list.size();
                                vertex_intersection_list.push_back (x[k]);
                            }
	                    S[k] = marked_edge [edge_idx];
                        } else { // xk is at edge boundary: a vertex of the 2d mesh
                            size_t vertex_idx = (!is_zero(f[j[0]])) ? K[j[k+1]] : K[j[0]];
	                    if (marked_vertex [vertex_idx] == not_marked) {
	                        marked_vertex [vertex_idx] = vertex_intersection_list.size();
                                vertex_intersection_list.push_back (Lambda.vertex(vertex_idx));
                            }
	                    S[k] = marked_vertex [vertex_idx];
                        }
                    }
                    // S[0] == S[j] when is_zero(f[j[0]]) but f[j[0]] != 0, i.e. precision pbs
                    check_macro (S[0] != S[1] && S[1] != S[2] && S[2] != S[0], "degenerate 3d intersection");
		    i_element_face = element_intersection_list.size();
                    element_intersection_list.push_back (S);
                }
            } else {
                subcompute_matrix_quadrilateral_T (x, f, q, a, b, c, d, aire);
                {
                    // create the new quadri face by intersections:
                    S.set_name('q');
		    point  x[4] = {a,b,d,c};
		    size_t s[4] = {q[0],q[2],q[1],q[3]};
                    for (size_t k = 0; k < 4; k++) {
                        size_t k1 = (k+1) % 4;
                        if (! is_zero(f[s[k]]) && ! is_zero(f[s[k1]])) {
                            // xk is inside edge {j0,j[k+1]} of triangle K:
                            size_t edge_iloc = edge_T_iloc (s[k], s[k1]);
                            size_t edge_idx = K.edge (edge_iloc);
	                    if (marked_edge [edge_idx] == not_marked) {
	                        marked_edge [edge_idx] = vertex_intersection_list.size();
                                vertex_intersection_list.push_back (x[k]);
                            }
	                    S[k] = marked_edge [edge_idx];
                        } else { // xk is at edge boundary: a vertex of the 2d mesh
                            size_t vertex_idx = is_zero(f[s[k]]) ? K[s[k]] : K[s[k1]];
	                    if (marked_vertex [vertex_idx] == not_marked) {
	                        marked_vertex [vertex_idx] = vertex_intersection_list.size();
                                vertex_intersection_list.push_back (Lambda.vertex(vertex_idx));
                            }
	                    S[k] = marked_vertex [vertex_idx];
                        }
                    }
		    if (!triangle_3d_only) {
                        check_macro (S[0] != S[1] && S[1] != S[2] && S[2] != S[3] && S[3] != S[0],
				"degenerate 3d intersection");
                            // S[0] == S[j] when is_zero(f[j[0]]) but f[j[0]] != 0, i.e. precision pbs
		        i_element_face = element_intersection_list.size();
                        element_intersection_list.push_back (S);
		    } else {
			// one K -> two (S1,S2) faces: table element2face may return a pair of size_t
			error_macro ("split quadri into 2 triangle: not yet supported");
#ifdef TODO
	                // split quadri into 2 triangles
                        element_type S1, S2;
                        S1.set_name ('t');
		        S1 [0] = S[0];
		        S1 [1] = S[1];
		        S1 [2] = S[2];
                        check_macro (S1[0] != S1[1] && S1[1] != S1[2] && S1[2] != S1[0],
				"degenerate 3d intersection");
		        i_element_face1 = element_intersection_list.size();
                        element_intersection_list.push_back (S1);

                        S2.set_name ('t');
		        S2 [0] = S[0];
		        S2 [1] = S[2];
		        S2 [2] = S[3];
                        check_macro (S2[0] != S2[1] && S2[1] != S2[2] && S2[2] != S2[0],
				"degenerate 3d intersection");
		        i_element_face2 = element_intersection_list.size();
                        element_intersection_list.push_back (S2);
#endif // TODO
		    }
                }
            }
            break;
          }
        default : {
            error_macro("level set intersection: element not yet implemented: " << K.name());
          }
        }
        element2face [i_element_band] = i_element_face;
        i_element_band++;
    }
    geo gamma;
    gamma.build (element_intersection_list, vertex_intersection_list);
    gamma.set_name ("zero_level_set_" + itos(Lambda.zero_level_set_number())
    	+ "_from_" + Lambda.name());
    return gamma;
}
geo
zero_level_set (const field& phi) {
    std::vector<size_t> elt2face;
    return zero_level_set (phi, elt2face);
}

bool
belongs_to_band (const geo_element& K, const ublas::vector<Float>& f) 
{
        switch (K.variant()) {
        case geo_element::t: { return belongs_to_band_t(f); }
        case geo_element::T: { return belongs_to_band_T(f); }
        default : {
            error_macro("level set intersection: element not yet implemented: " << K.name());
          }
	}
}
// ===============================================================================
// the band class: allocator
// ===============================================================================
band::band (field phi_h_lambda)
 : _phi_h_lambda (phi_h_lambda)
{
  _gamma_h = zero_level_set (phi_h_lambda, _element2face);
  space Bh (phi_h_lambda, "banded_level_set");
  check_macro (Bh.is_on_boundary_domain(), "unexpected banded level set space");
  domain band_dom = Bh.get_boundary_domain();
  _phi_h_beta = field (Bh);
#ifdef TODO
  _phi_h_beta = phi_h_lambda [band_dom]; // TODO
#else // TODO
  _phi_h_beta.from_boundary_val (phi_h_lambda, 0, band_dom);
#endif // TODO
}

} // namespace
