/*
 * Copyright (C) 2007-2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	0
#define DEBUG_CONTROL_FLOW_REGS	0
#define DEBUG_CONTROL_FLOW_IRQS	0

/* FIXME */
#define CPU_APIC_VERSION	1
#define CPU_APIC_MAX_LVT	4

#ifdef INCLUDE

#endif /* INCLUDE */
#ifdef STATE
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT

struct {
/*020*/ /* APIC ID Register */
#if 1	/* Pentium and P6 family */
	unsigned int phys_apic_id : 4;
#elif 0 /* Pentium 4 and Xeon */
	unsigned int phys_apic_id : 8;
#endif

/*080*/ /* Task Priority Register */
	unsigned int tpr : 8;

/*090*/ /* Arbitration Priority Register */
	unsigned int apr : 8;

/*0A0*/ /* Processor Priority Register */
	unsigned int ppr : 8;

/*0D0*/ /* Logical Destination Register */
	unsigned int ldr : 8;

/*0E0*/ /* Destination Format Register */
	unsigned int dfr_model : 4;

/*0F0*/ struct { /* Spurious Interrupt Vector Register */
		unsigned int spurious_vector : 8;
		unsigned int apic_enabled : 1;
		unsigned int focus_cpu : 1;
	} svr;

/*100*/ /* In Service Register */
	uint32_t isr[8];

/*180*/ /* Trigger Mode Register */
	uint32_t tmr[8];

/*200*/ /* Interrupt Request Register */
	uint32_t irr[8];

/*280*/ /* Error Status Register */
	unsigned int send_cs_error : 1;
	unsigned int receive_cs_error : 1;
	unsigned int send_accept_error : 1;
	unsigned int receive_accept_error : 1;
	unsigned int send_illegal_vector : 1;
	unsigned int receive_illegal_vector : 1;
	unsigned int illegal_register_address : 1;

/*300*/ struct { /* Interrupt Command Register 1 */
		unsigned int vector : 8;
		unsigned int delivery_mode : 3;
		unsigned int destination_mode : 1;
		unsigned int delivery_status : 1;
		unsigned int level : 1;
		unsigned int trigger : 1;
		unsigned int shorthand : 2;
	} icr1;

/*310*/ struct { /* Interrupt Command Register 2 */
		unsigned int destination : 8;
	} icr2;

/*320*/ struct { /* LVT - Timer */
		unsigned int vector : 8;
		unsigned int delivery_status : 1;
		unsigned int mask : 1;
		unsigned int timer_mode : 1;
	} lvt_timer;

/*340*/ struct { /* LVT - Performance Counter */
		unsigned int vector : 8;
		unsigned int delivery_mode : 3;
		unsigned int delivery_status : 1;
		unsigned int mask : 1;
	} lvt_pc;

/*350*/
/*360*/ struct { /* LVT - LINT0/1 */
		unsigned int vector : 8;
		unsigned int delivery_mode : 3;
		unsigned int delivery_status : 1;
		unsigned int polarity : 1;
		unsigned int remote_irr : 1;
		unsigned int trigger : 1;
		unsigned int mask : 1;
	} lvt_lint[2];

/*370*/ struct { /* LVT - Error */
		unsigned int vector : 8;
		unsigned int delivery_status : 1;
		unsigned int mask : 1;
	} lvt_error;

/*380*/ /* Timer Initial Count Register */
	uint32_t timer_icr;

/*390*/ /* Timer Current Count Register */
	uint32_t timer_ccr;

/*3E0*/ /* Timer Divide Configuration Register */
	unsigned int timer_dcr : 3;

	unsigned long long timer_event;
	int timer_running;

	int extint_pending;
	uint8_t extint_pri;

	uint64_t base;
	int apic_enable;
	int bsp;

	unsigned long tsc_to_bus;
} NAME;

#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
#endif /* STATE */
#ifdef EXPORT

/*forward*/ static int
NAME_(mx)(void *_cpssp, paddr_t paddr, unsigned int bs, udata_t *valp);
/*forward*/ static int
NAME_(mr)(void *_cpssp, paddr_t paddr, unsigned int bs, udata_t *valp);
/*forward*/ static int
NAME_(mw)(void *_cpssp, paddr_t paddr, unsigned int bs, udata_t val);
/*forward*/ static void
NAME_(map_r)(struct cpssp *cpssp, paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t *), void **csp, char **haddrp);
/*forward*/ static void
NAME_(map_w)(struct cpssp *cpssp, paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t), void **csp, char **haddrp);
/*forward*/ static void
NAME_(map_x)(struct cpssp *cpssp, paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t *), void **csp, char **haddrp);

/*forward*/ static void
NAME_(unmap)(struct cpssp *cpssp, paddr_t paddr, paddr_t len);

/*forward*/ static void
NAME_(lint0_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(lint1_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static uint8_t
NAME_(irq_ack)(struct cpssp *cpssp);

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT

/*forward*/ static int
NAME_(wrmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t val);
/*forward*/ static int
NAME_(rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp);

/*forward*/ static void
NAME_(tpr_set)(struct cpssp *cpssp, uint8_t val);
/*forward*/ static uint8_t
NAME_(tpr_get)(struct cpssp *cpssp);

/*forward*/ static void
NAME_(eoi_receive)(struct cpssp *cpssp, uint8_t vector);
/*forward*/ static void
NAME_(msg0_receive)(struct cpssp *cpssp,
		unsigned int destination_mode,
		unsigned int delivery_mode,
		unsigned int level,
		unsigned int trigger_mode,
		uint8_t vector,
		uint8_t destination);
/*forward*/ static void
NAME_(status0_receive)(struct cpssp *cpssp, unsigned int status);
/*forward*/ static void
NAME_(msg1_receive)(struct cpssp *cpssp, uint8_t processor_priority);
/*forward*/ static void
NAME_(status1_receive)(struct cpssp *cpssp, unsigned int status);
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */

/*forward*/ static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT

static int
NAME_(bit_find)(uint32_t *v, unsigned int vsize)
{
	int i;
	int j;

	for (i = (vsize / 32) - 1; ; i--) {
		uint32_t m;

		if (i < 0) {
			return -1;
		}
		m = v[i];
		if (m == 0) {
			continue;
		}
		for (j = 31; ; j--) {
			if (m & (1 << j)) {
				return i * 32 + j;
			}
		}
	}
}

static void
NAME_(bit_set)(uint32_t *v, unsigned int bitno)
{
	v[bitno / 32] |= 1 << (bitno & 0x1f);
}

static void
NAME_(bit_clr)(uint32_t *v, unsigned int bitno)
{
	v[bitno / 32] &= ~(1 << (bitno & 0x1f));
}

static int
NAME_(bit_test)(uint32_t *v, unsigned int bitno)
{
	return v[bitno / 32] >> (bitno & 0x1f) & 1;
}

static inline int
NAME_(extint_pri)(struct cpssp *cpssp)
{
	if (cpssp->NAME.extint_pending) {
		return cpssp->NAME.extint_pri;
	} else {
		return -1;
	}
}

static inline int
NAME_(smp_pri)(struct cpssp *cpssp)
{
	int smp_irr;
	int smp_isr;
	int smp_ppr;

	smp_irr = NAME_(bit_find)(cpssp->NAME.irr, 256);
	smp_isr = NAME_(bit_find)(cpssp->NAME.isr, 256);
	smp_ppr = cpssp->NAME.tpr;

	if (0 <= smp_irr
	 && smp_isr < smp_irr
	 && smp_ppr <= smp_irr) {
		return smp_irr;
	} else {
		return -1;
	}
}

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	int extint_pri;
	int smp_pri;

	extint_pri = NAME_(extint_pri)(cpssp);
	smp_pri = NAME_(smp_pri)(cpssp);

	if (0 <= extint_pri
	 || (cpssp->NAME.apic_enable
	  && cpssp->NAME.svr.apic_enabled
	  && 0 <= smp_pri)) {
		NAME_(core_irq_set)(cpssp, 1);
	} else {
		NAME_(core_irq_set)(cpssp, 0);
	}
}

static uint8_t
NAME_(_irq_ack)(struct cpssp *cpssp)
{
	int extint_pri;
	int smp_pri;
	uint8_t vec;

	/* Get external interrupt priority. */
	extint_pri = NAME_(extint_pri)(cpssp);

	/* Get SMP interrupt priority */
	smp_pri = NAME_(smp_pri)(cpssp);

	if (extint_pri < 0
	 && smp_pri < 0) {
		/*
		 * Spurious interrupt.
		 */
		vec = cpssp->NAME.svr.spurious_vector;

	} else if (extint_pri < smp_pri) {
		/*
		 * SMP with high priority.
		 */
		vec = smp_pri;

		NAME_(bit_clr)(cpssp->NAME.irr, vec);
		NAME_(bit_set)(cpssp->NAME.isr, vec);
		if (NAME_(bit_test)(cpssp->NAME.tmr, vec)) {
			NAME_(bit_clr)(cpssp->NAME.tmr, vec);
			/* Send EOI to all IOAPICs. */
			sig_icc_bus_eoi(cpssp->icc_bus, cpssp, vec);
		}

	} else { assert(smp_pri <= extint_pri);
		/*
		 * EXTINT with high priority.
		 */
		cpssp->NAME.extint_pending = 0;

		NAME_(bus_ack)(cpssp, &vec);
	}

	NAME_(irq_update)(cpssp);

	return vec;
}

static int
NAME_(affected)(
	struct cpssp *cpssp,
	unsigned int destination_mode,
	uint8_t destination
)
{
	if (destination_mode == 0) {
		/*
		 * Physical Destination
		 */
		/* High order bits are ignored. */
		destination &= 0x0f;

		if (destination == cpssp->NAME.phys_apic_id
		 || destination == 0xf) {
			return 1;
		} else {
			return 0;
		}
	} else {
		/*
		 * Logical Destination
		 */
		switch (cpssp->NAME.dfr_model) {
		case 0x0: /* Cluster Model */
			if ((destination >> 4) == (cpssp->NAME.ldr >> 4)
			 || (destination >> 4) == 0xf) {
				return (destination & cpssp->NAME.ldr & 0xf) ? 1 : 0;
			} else {
				return 0;
			}
		case 0xf: /* Flat Model */
			return (destination & cpssp->NAME.ldr) != 0;
		default:
			assert(0);
		}
	}
}

static void
NAME_(deliver_irq_local)(
	struct cpssp *cpssp,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector
)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW_IRQS) {
		fprintf(stderr, "%s: delivery_mode=%d, level=%d, trigger_mode=%d, vector=0x%02x\n",
				__FUNCTION__,
				delivery_mode,
				level, trigger_mode, vector);
	}

	switch (delivery_mode) {
	case 0: /* FIXED */
		NAME_(bit_set)(cpssp->NAME.irr, vector);
		if (trigger_mode) {
			NAME_(bit_set)(cpssp->NAME.tmr, vector);
		} else {
			NAME_(bit_clr)(cpssp->NAME.tmr, vector);
		}
		NAME_(irq_update)(cpssp);
		break;

	case 1: /* Lowest Priority */
		/* Same as FIXED for now. FIXME */
		NAME_(bit_set)(cpssp->NAME.irr, vector);
		if (trigger_mode) {
			NAME_(bit_set)(cpssp->NAME.tmr, vector);
		} else {
			NAME_(bit_clr)(cpssp->NAME.tmr, vector);
		}
		NAME_(irq_update)(cpssp);
		break;

	case 2: /* SMI */
		NAME_(core_smi_set)(cpssp, 1);
		NAME_(core_smi_set)(cpssp, 0);
		break;

	case 4: /* NMI */
		NAME_(core_nmi_set)(cpssp, 1);
		NAME_(core_nmi_set)(cpssp, 0);
		break;

	case 5: /* INIT */
		if (level == 0) {
			/* Set arbitration ID to value of APIC ID. */
			/* Nothing to do, yet... */
		} else {
			NAME_(init_by_apic)(cpssp);
		}
		break;

	case 6: /* STARTUP */
		NAME_(startup_by_apic)(cpssp, vector);
		break;

	case 7: /* EXTINT */
		cpssp->NAME.extint_pending = 1;
		cpssp->NAME.extint_pri = vector;
		NAME_(irq_update)(cpssp);
		break;

	default:
		fprintf(stderr, "delivery_mode=%d\n", delivery_mode);
		assert(0); /* Cannot happen. */
	}
}

static void
NAME_(deliver_eoi_local)(struct cpssp *cpssp, uint8_t vec)
{
	/* FIXME VOSSI */
}

static void
NAME_(deliver_eoi)(struct cpssp *cpssp, uint8_t vec)
{
	NAME_(deliver_eoi_local)(cpssp, vec);
	/* FIXME VOSSI */
}

static void
NAME_(lintX_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW_IRQS) {
		fprintf(stderr, "%s: lint%d set to %d\n", __FUNCTION__,
				nr, val);
	}

	if (cpssp->NAME.apic_enable
	 && cpssp->NAME.svr.apic_enabled) {
		/*
		 * APIC enabled.
		 */
		val ^= cpssp->NAME.lvt_lint[nr].polarity;
		val &= ! cpssp->NAME.lvt_lint[nr].mask;

		if (val) {
			NAME_(deliver_irq_local)(cpssp,
				cpssp->NAME.lvt_lint[nr].delivery_mode,
				1, /* Level */
				cpssp->NAME.lvt_lint[nr].trigger,
				cpssp->NAME.lvt_lint[nr].vector);
		}
	} else {
		/*
		 * APIC disabled.
		 */
		if (val) {
			switch (nr) {
			case 0: /* IRQ */
				NAME_(deliver_irq_local)(cpssp,
						7, 1, 0, 0x00);
				break;
			case 1: /* NMI */
				NAME_(deliver_irq_local)(cpssp,
						4, 1, 1, 0x00);
				break;
			default:
				assert(0);
			}
		}
	}
}

static void
NAME_(eoi_receive)(struct cpssp *cpssp, uint8_t vector)
{
	assert(0); /* FIXME */
}

static void
NAME_(msg0_receive)(
	struct cpssp *cpssp,
	unsigned int destination_mode,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector,
	uint8_t destination
)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW_IRQS) {
		fprintf(stderr, "%s: destination_mode=%d, delivery_mode=%d, level=%d, trigger_mode=%d, vector=0x%02x, destination=0x%02x\n",
				__FUNCTION__,
				destination_mode, delivery_mode,
				level, trigger_mode, vector, destination);
	}

	if (NAME_(affected)(cpssp, destination_mode, destination)) {
		NAME_(deliver_irq_local)(cpssp, delivery_mode, level,
				trigger_mode, vector);
	}
}

static void
NAME_(status0_receive)(struct cpssp *cpssp, unsigned int status)
{
	assert(0); /* FIXME */
}

static void
NAME_(msg1_receive)(struct cpssp *cpssp, uint8_t processor_priority)
{
	assert(0); /* FIXME */
}

static void
NAME_(status1_receive)(struct cpssp *cpssp, unsigned int status)
{
	assert(0); /* FIXME */
}

/*forward*/ static void
NAME_(timer_event)(void *_cpssp);

static void
NAME_(timer_update)(struct cpssp *cpssp)
{
	unsigned long long tsc;
	unsigned long long ticks;
	int irq;

	tsc = time_virt() - cpssp->NAME.timer_event;
	ticks = (tsc / cpssp->NAME.tsc_to_bus) >> cpssp->NAME.timer_dcr;
	cpssp->NAME.timer_event += (ticks << cpssp->NAME.timer_dcr) * cpssp->NAME.tsc_to_bus;

	irq = 0;
	while (cpssp->NAME.timer_ccr != 0
	    && 0 < ticks) {
		if (ticks < cpssp->NAME.timer_ccr) {
			cpssp->NAME.timer_ccr -= ticks;
			ticks = 0;
		} else {
			ticks -= cpssp->NAME.timer_ccr;
			cpssp->NAME.timer_ccr = 0;
			if (cpssp->NAME.lvt_timer.timer_mode) {
				/* Periodic Timer */
				cpssp->NAME.timer_ccr = cpssp->NAME.timer_icr;
			}
			irq = 1;
		}
	}
	if (irq
	 && ! cpssp->NAME.lvt_timer.mask) {
		NAME_(deliver_irq_local)(
				cpssp,
				0, /* Delivery Mode FIXED */
				1, /* Level */
				0, /* Edge Triggered */
				cpssp->NAME.lvt_timer.vector);
	}
}

static void
NAME_(timer_stop)(struct cpssp *cpssp)
{
	if (cpssp->NAME.timer_running) {
		int ret;

		ret = time_call_delete(NAME_(timer_event), cpssp);
		assert(ret == 0);

		cpssp->NAME.timer_running = 0;
	}
}

static void
NAME_(timer_start)(struct cpssp *cpssp)
{
	if (cpssp->NAME.timer_ccr != 0) {
		unsigned long long ticks;
		unsigned long long tsc;

		ticks = cpssp->NAME.timer_ccr << cpssp->NAME.timer_dcr;
		tsc = cpssp->NAME.timer_event + ticks * cpssp->NAME.tsc_to_bus;
		time_call_at(tsc, NAME_(timer_event), cpssp);

		cpssp->NAME.timer_running = 1;
	}
}

static void
NAME_(timer_event)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* Timer already stopped by time code. */
	cpssp->NAME.timer_running = 0;

	NAME_(timer_update)(cpssp);

	NAME_(timer_start)(cpssp);
}

static void
NAME_(read)(struct cpssp *cpssp, paddr_t reg, udata_t *valp)
{
	udata_t val;

	switch (reg) {
	case 0x020: /* ID */
		val = cpssp->NAME.phys_apic_id << 24;
		break;

	case 0x030: /* LVT */
		val = CPU_APIC_MAX_LVT << 16;
		val |= 1 << 4; /* Internal APIC */
		val |= CPU_APIC_VERSION << 0;
		break;

	case 0x080: /* TASKPRI */
		val = cpssp->NAME.tpr;
		break;

	case 0x090: /* ARBPRI */
		val = 0; /* FIXME VOSSI */
		break;

	case 0x0a0: /* PROCPRI */
		val = 0; /* FIXME VOSSI */
		break;

	case 0x0b0: /* EOI */
		/* Write-only. */
		val = 0;
		break;

	case 0x0d0: /* LDR */
		val = cpssp->NAME.ldr << 24;
		break;

	case 0x0e0: /* DFR */
		val = cpssp->NAME.dfr_model << 28;
		val |= 0x0fffffff;
		break;

	case 0x0f0: /* SPIV */
		val = cpssp->NAME.svr.focus_cpu << 9;
		val |= cpssp->NAME.svr.apic_enabled << 8;
		val |= cpssp->NAME.svr.spurious_vector << 0;
		break;

	case 0x100: /* ISR */
	case 0x110:
	case 0x120:
	case 0x130:
	case 0x140:
	case 0x150:
	case 0x160:
	case 0x170:
		val = cpssp->NAME.isr[(reg >> 4) & 7];
		break;

	case 0x180: /* TMR */
	case 0x190:
	case 0x1a0:
	case 0x1b0:
	case 0x1c0:
	case 0x1d0:
	case 0x1e0:
	case 0x1f0:
		val = cpssp->NAME.tmr[(reg >> 4) & 7];
		break;

	case 0x200: /* IRR */
	case 0x210:
	case 0x220:
	case 0x230:
	case 0x240:
	case 0x250:
	case 0x260:
	case 0x270:
		val = cpssp->NAME.irr[(reg >> 4) & 7];
		break;

	case 0x280: /* ESR */
		val = 0;
		val |= cpssp->NAME.illegal_register_address << 7;
		val |= cpssp->NAME.receive_illegal_vector << 6;
		val |= cpssp->NAME.send_illegal_vector << 5;
		/* Bit 4: reserved */
		val |= cpssp->NAME.receive_accept_error << 3;
		val |= cpssp->NAME.send_accept_error << 2;
		val |= cpssp->NAME.receive_cs_error << 1;
		val |= cpssp->NAME.send_cs_error << 0;
		break;

	case 0x300: /* ICR */
		val = 0;
		val |= cpssp->NAME.icr1.shorthand << 18;
		val |= cpssp->NAME.icr1.trigger << 15;
		val |= cpssp->NAME.icr1.level << 14;
		val |= cpssp->NAME.icr1.delivery_status << 12;
		val |= cpssp->NAME.icr1.destination_mode << 11;
		val |= cpssp->NAME.icr1.delivery_mode << 8;
		val |= cpssp->NAME.icr1.vector << 0;
		break;

	case 0x310: /* ICR2 */
		val = 0;
		val |= cpssp->NAME.icr2.destination << 24;
		break;

	case 0x320: /* LVTT */
		val = 0;
		val |= cpssp->NAME.lvt_timer.timer_mode << 17;
		val |= cpssp->NAME.lvt_timer.mask << 16;
		val |= cpssp->NAME.lvt_timer.delivery_status << 12;
		val |= cpssp->NAME.lvt_timer.vector << 0;
		break;

#if 4 <= CPU_APIC_MAX_LVT
	case 0x340: /* LVTPC */
		val = 0;
		val |= cpssp->NAME.lvt_pc.mask << 16;
		val |= cpssp->NAME.lvt_pc.delivery_status << 12;
		val |= cpssp->NAME.lvt_pc.delivery_mode << 8;
		val |= cpssp->NAME.lvt_pc.vector << 0;
		break;
#endif
	case 0x350: /* LVT0 */
		val = 0;
		val |= cpssp->NAME.lvt_lint[0].mask << 16;
		val |= cpssp->NAME.lvt_lint[0].trigger << 15;
		val |= cpssp->NAME.lvt_lint[0].remote_irr << 14;
		val |= cpssp->NAME.lvt_lint[0].polarity << 13;
		val |= cpssp->NAME.lvt_lint[0].delivery_status << 12;
		val |= cpssp->NAME.lvt_lint[0].delivery_mode << 8;
		val |= cpssp->NAME.lvt_lint[0].vector << 0;
		break;

	case 0x360: /* LVT1 */
		val = 0;
		val |= cpssp->NAME.lvt_lint[1].mask << 16;
		val |= cpssp->NAME.lvt_lint[1].trigger << 15;
		val |= cpssp->NAME.lvt_lint[1].remote_irr << 14;
		val |= cpssp->NAME.lvt_lint[1].polarity << 13;
		val |= cpssp->NAME.lvt_lint[1].delivery_status << 12;
		val |= cpssp->NAME.lvt_lint[1].delivery_mode << 8;
		val |= cpssp->NAME.lvt_lint[1].vector << 0;
		break;

#if 3 <= CPU_APIC_MAX_LVT
	case 0x370: /* LVTERR */
		val = 0;
		val |= cpssp->NAME.lvt_error.mask << 16;
		val |= cpssp->NAME.lvt_error.delivery_status << 12;
		val |= cpssp->NAME.lvt_error.vector << 0;
		break;
#endif

	case 0x380: /* TMICT */
		val = cpssp->NAME.timer_icr;
		break;

	case 0x390: /* TMCCT */
		NAME_(timer_stop)(cpssp);
		NAME_(timer_update)(cpssp);
		val = cpssp->NAME.timer_ccr;
		NAME_(timer_start)(cpssp);
		break;

	case 0x3e0: /* TDCR */
		val = 0;
		val |= (((cpssp->NAME.timer_dcr - 1) >> 2) & 1) << 3;
		val |= (((cpssp->NAME.timer_dcr - 1) >> 0) & 3) << 0;
		break;

	default:
		fprintf(stderr, "%s: reg=0x%03x\n", __FUNCTION__, reg);
		assert(0); /* FIXME VOSSI */
	}

	if (2 <= loglevel + DEBUG_CONTROL_FLOW_REGS) {
		fprintf(stderr, "APIC: Reading 0x%08lx from register 0x%03x\n",
				(unsigned long) val, (unsigned int) reg);
	}

	*valp = val;
}

static void
NAME_(write)(struct cpssp *cpssp, paddr_t reg, udata_t val)
{
	int vec;
	unsigned int destination_mode;
	uint8_t destination;

	if (2 <= loglevel + DEBUG_CONTROL_FLOW_REGS) {
		fprintf(stderr, "APIC: Writing 0x%08lx to register 0x%03x\n",
				(unsigned long) val, (unsigned int) reg);
	}

	switch (reg) {
	case 0x020: /* ID */
#if 1 /* Pentium and P6 family */
		cpssp->NAME.phys_apic_id = (val >> 24) & 0xf;
#elif 0 /* Pentium 4 and Xeon */
		cpssp->NAME.phys_apic_id = (val >> 24) & 0xff;
#endif
		break;

	case 0x030: /* LVR */
		/* Read-only. */
		break;

	case 0x080: /* TASKPRI */
		cpssp->NAME.tpr = (val >> 0) & 0xff;
		break;

	case 0x090: /* ARBPRI */
		/* Read-only. */
		break;

	case 0x0a0: /* PROCPRI */
		/* Read-only. */
		break;

	case 0x0b0: /* EOI */
		vec = NAME_(bit_find)(cpssp->NAME.isr, 256);
		if (0 <= vec) {
			NAME_(bit_clr)(cpssp->NAME.isr, vec);
			if (NAME_(bit_test)(cpssp->NAME.tmr, vec)) {
				NAME_(deliver_eoi)(cpssp, vec);
			}
		}

		NAME_(irq_update)(cpssp);
		break;

	case 0x0d0: /* LDR */
		cpssp->NAME.ldr = (val >> 24) & 0xff;
		break;

	case 0x0e0: /* DFR */
		cpssp->NAME.dfr_model = (val >> 28) & 0xf;
		break;

	case 0x0f0: /* SPIV */
		val |= 0xf; /* Bit 3-0: 1111; read-only */
		cpssp->NAME.svr.focus_cpu = (val >> 9) & 1;
		cpssp->NAME.svr.apic_enabled = (val >> 8) & 1;
		cpssp->NAME.svr.spurious_vector = (val >> 0) & 0xff;
		break;

	case 0x100: /* ISR */
	case 0x110:
	case 0x120:
	case 0x130:
	case 0x140:
	case 0x150:
	case 0x160:
	case 0x170:
		/* Read-only. */
		break;

	case 0x180: /* TMR */
	case 0x190:
	case 0x1a0:
	case 0x1b0:
	case 0x1c0:
	case 0x1d0:
	case 0x1e0:
	case 0x1f0:
		/* Read-only. */
		break;

	case 0x200: /* IRR */
	case 0x210:
	case 0x220:
	case 0x230:
	case 0x240:
	case 0x250:
	case 0x260:
	case 0x270:
		/* Read-only. */
		break;

	case 0x280: /* ESR */
		/* Read-only. */
		break;

	case 0x300: /* ICR */
		cpssp->NAME.icr1.shorthand = (val >> 18) & 3;
		cpssp->NAME.icr1.trigger = (val >> 15) & 1;
		cpssp->NAME.icr1.level = (val >> 14) & 1;
		/* Writing to ICR1 always triggers an interrupt */
		cpssp->NAME.icr1.destination_mode = (val >> 11) & 1;
		cpssp->NAME.icr1.delivery_mode = (val >> 8) & 7;
		cpssp->NAME.icr1.vector = (val >> 0) & 0xff;

		switch (cpssp->NAME.icr1.shorthand) {
		case 0: /* No Shorthand */
			destination_mode = cpssp->NAME.icr1.destination_mode;
			destination = cpssp->NAME.icr2.destination;
			break;
		case 1: /* Self */
		case 2: /* All Including Self */
		case 3: /* All Excluding Self */
			destination_mode = 0;
			destination = 0xff;
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		cpssp->NAME.icr1.delivery_status = 1;

		if (cpssp->NAME.icr1.shorthand != 3) {
			/* Send IRQ to own APIC. */
			NAME_(msg0_receive)(
					cpssp,
					destination_mode,
					cpssp->NAME.icr1.delivery_mode,
					cpssp->NAME.icr1.level,
					cpssp->NAME.icr1.trigger,
					cpssp->NAME.icr1.vector,
					destination);
		}
		if (cpssp->NAME.icr1.shorthand != 1) {
			/* Send IRQ to other APICs. */
			sig_icc_bus_msg0(cpssp->icc_bus, cpssp,
					destination_mode,
					cpssp->NAME.icr1.delivery_mode,
					cpssp->NAME.icr1.level,
					cpssp->NAME.icr1.trigger,
					cpssp->NAME.icr1.vector,
					destination);
		}

		cpssp->NAME.icr1.delivery_status = 0;
		break;

	case 0x310: /* ICR2 */
		cpssp->NAME.icr2.destination = (val >> 24) & 0xff;
		break;

	case 0x320: /* LVTT */
		cpssp->NAME.lvt_timer.timer_mode = (val >> 17) & 1;
		cpssp->NAME.lvt_timer.mask = (val >> 16) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME.lvt_timer.vector = (val >> 0) & 0xff;
		break;

#if 4 <= CPU_APIC_MAX_LVT
	case 0x340: /* LVTPC */
		cpssp->NAME.lvt_pc.mask = (val >> 16) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME.lvt_pc.delivery_mode = (val >> 8) & 7;
		cpssp->NAME.lvt_pc.vector = (val >> 0) & 0xff;
		break;
#endif

	case 0x350: /* LVT0 */
		cpssp->NAME.lvt_lint[0].mask = (val >> 16) & 1;
		cpssp->NAME.lvt_lint[0].trigger = (val >> 15) & 1;
		cpssp->NAME.lvt_lint[0].remote_irr = (val >> 14) & 1;
		cpssp->NAME.lvt_lint[0].polarity = (val >> 13) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME.lvt_lint[0].delivery_mode = (val >> 8) & 7;
		cpssp->NAME.lvt_lint[0].vector = (val >> 0) & 0xff;
		break;

	case 0x360: /* LVT1 */
		cpssp->NAME.lvt_lint[1].mask = (val >> 16) & 1;
		cpssp->NAME.lvt_lint[1].trigger = (val >> 15) & 1;
		cpssp->NAME.lvt_lint[1].remote_irr = (val >> 14) & 1;
		cpssp->NAME.lvt_lint[1].polarity = (val >> 13) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME.lvt_lint[1].delivery_mode = (val >> 8) & 7;
		cpssp->NAME.lvt_lint[1].vector = (val >> 0) & 0xff;
		break;

#if 3 <= CPU_APIC_MAX_LVT
	case 0x370: /* LVTERR */
		cpssp->NAME.lvt_error.mask = (val >> 16) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME.lvt_error.vector = (val >> 0) & 0xff;
		break;
#endif

	case 0x380: /* TMICT */
		/* Writing to TMICT starts the timer */
		NAME_(timer_stop)(cpssp);
		cpssp->NAME.timer_event = time_virt();
		cpssp->NAME.timer_icr = val;
		cpssp->NAME.timer_ccr = val;
		NAME_(timer_start)(cpssp);
		break;

	case 0x390: /* TMCCT */
		/* Read-only. */
		break;

	case 0x3e0: /* TDCR */
		/*
		 * 0000 -> 1
		 * 0001 -> 2
		 * 0010 -> 3
		 * 0011 -> 4
		 * 1000 -> 5
		 * 1001 -> 6
		 * 1010 -> 7
		 * 1011 -> 0
		 */
		val |= (val >> 1) & 4;
		val += 1;
		val &= 7;
		cpssp->NAME.timer_dcr = val;
		break;

	default:
		fprintf(stderr, "%s: reg=0x%03x\n", __FUNCTION__, reg);
		assert(0); /* FIXME VOSSI */
	}
}

static int
NAME_(wrmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t val)
{
	switch (ecx) {
	case 0x1b: /* MSR_IA32_APICBASE */
		if (2 <= loglevel + DEBUG_CONTROL_FLOW_REGS) {
			fprintf(stderr, "APIC: Writing 0x%08lx to msr\n",
					(unsigned long) val);
		}

		if (cpssp->NAME.apic_enable) {
			/* Invalidate old mapping. */
			NAME_(cache2_unmap)(cpssp, cpssp->NAME.base, 0x1000);
		}

		cpssp->NAME.base = val & ~0xfff;
		cpssp->NAME.apic_enable = (val >> 11) & 1;

		if (cpssp->NAME.apic_enable) {
			/* Invalidate old mapping. */
			NAME_(cache2_unmap)(cpssp, cpssp->NAME.base, 0x1000);
		}
		return 1;

	default:
		return 0;
	}
}

static int
NAME_(rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp)
{
	switch (ecx) {
	case 0x1b: /* MSR_IA32_APICBASE */
		*valp = cpssp->NAME.base
			| (cpssp->NAME.apic_enable << 11)
			| (cpssp->NAME.bsp << 8);

		if (2 <= loglevel + DEBUG_CONTROL_FLOW_REGS) {
			fprintf(stderr, "APIC: Reading 0x%08lx from msr\n",
					(unsigned long) *valp);
		}
		return 1;

	default:
		return 0;
	}
}

static void
NAME_(tpr_set)(struct cpssp *cpssp, uint8_t val)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW_REGS) {
		fprintf(stderr, "APIC: Writing 0x%02x to tpr\n",
				(unsigned int) val);
	}

	cpssp->NAME.tpr = (val & 0x0f) << 4;
	NAME_(irq_update)(cpssp);
}

static uint8_t
NAME_(tpr_get)(struct cpssp *cpssp)
{
	uint8_t val;

	val = cpssp->NAME.tpr >> 4;

	if (2 <= loglevel + DEBUG_CONTROL_FLOW_REGS) {
		fprintf(stderr, "APIC: Reading 0x%02x from tpr\n",
				(unsigned int) val);
	}

	return val;
}

#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */

static int
NAME_(mx)(void *_cpssp, paddr_t paddr, unsigned int bs, udata_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(bus_mx)(cpssp, paddr, bs, valp);
}

static int
NAME_(mr)(void *_cpssp, paddr_t paddr, unsigned int bs, udata_t *valp)
{
	struct cpssp *cpssp = _cpssp;

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	if ((paddr & ~0xff0) == cpssp->NAME.base
	 && bs == 0b1111) {
		NAME_(read)(cpssp, paddr & 0xff0, valp);
		return 0;
	} else
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
	{
		return NAME_(bus_mr)(cpssp, paddr, bs, valp);
	}
}

static int
NAME_(mw)(void *_cpssp, paddr_t paddr, unsigned int bs, udata_t val)
{
	struct cpssp *cpssp = _cpssp;

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	if ((paddr & ~0xff0) == cpssp->NAME.base
	 && bs == 0b1111) {
		NAME_(write)(cpssp, paddr & 0xff0, val);
		return 0;
	} else
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
	{
		return NAME_(bus_mw)(cpssp, paddr, bs, val);
	}
}

static void
NAME_(map_r)(
	struct cpssp *cpssp,
	paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t *),
	void **csp,
	char **haddrp
)
{
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	if ((paddr & ~0xff0) == cpssp->NAME.base) {
		*cfp = NAME_(mr);
		*csp = cpssp;
		*haddrp = NULL;
	} else
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
	{
		NAME_(bus_map_r)(cpssp, paddr, cfp, csp, haddrp);
	}
}

static void
NAME_(map_w)(
	struct cpssp *cpssp,
	paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t),
	void **csp,
	char **haddrp
)
{
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	if ((paddr & ~0xff0) == cpssp->NAME.base) {
		*cfp = NAME_(mw);
		*csp = cpssp;
		*haddrp = NULL;
	} else
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
	{
		NAME_(bus_map_w)(cpssp, paddr, cfp, csp, haddrp);
	}
}

static void
NAME_(map_x)(
	struct cpssp *cpssp,
	paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t *),
	void **csp,
	char **haddrp
)
{
	NAME_(bus_map_x)(cpssp, paddr, cfp, csp, haddrp);
}

static void
NAME_(unmap)(struct cpssp *cpssp, paddr_t paddr, paddr_t len)
{
	NAME_(cache2_unmap)(cpssp, paddr, len);
}

static void
NAME_(lint0_set)(struct cpssp *cpssp, unsigned int val)
{
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	NAME_(lintX_set)(cpssp, 0, val);
#else
	NAME_(core_irq_set)(cpssp, val);
#endif
}

static void
NAME_(lint1_set)(struct cpssp *cpssp, unsigned int val)
{
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	NAME_(lintX_set)(cpssp, 1, val);
#else
	NAME_(core_nmi_set)(cpssp, val);
#endif
}

static uint8_t
NAME_(irq_ack)(struct cpssp *cpssp)
{
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	return NAME_(_irq_ack)(cpssp);
#else
	uint8_t vec;

	NAME_(bus_ack)(cpssp, &vec);

	return vec;
#endif
}

static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int val)
{
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	cpssp->NAME.phys_apic_id = 0; /* FIXME */
	cpssp->NAME.dfr_model = 0xF;
	cpssp->NAME.svr.spurious_vector = 0xff;
	cpssp->NAME.svr.apic_enabled = 0;
	cpssp->NAME.svr.focus_cpu = 1;
	cpssp->NAME.lvt_timer.timer_mode = 0;
	cpssp->NAME.lvt_timer.mask = 1;
	cpssp->NAME.lvt_timer.vector = 0x00;
	cpssp->NAME.lvt_pc.mask = 1;
	cpssp->NAME.lvt_pc.delivery_mode = 0;
	cpssp->NAME.lvt_pc.vector = 0x00;
	cpssp->NAME.lvt_lint[0].mask = 1;
	cpssp->NAME.lvt_lint[0].trigger = 0;
	cpssp->NAME.lvt_lint[0].polarity = 0;
	cpssp->NAME.lvt_lint[0].delivery_mode = 0;
	cpssp->NAME.lvt_lint[0].vector = 0x00;
	cpssp->NAME.lvt_lint[1].mask = 1;
	cpssp->NAME.lvt_lint[1].trigger = 0;
	cpssp->NAME.lvt_lint[1].polarity = 0;
	cpssp->NAME.lvt_lint[1].delivery_mode = 0;
	cpssp->NAME.lvt_lint[1].vector = 0x00;
	cpssp->NAME.lvt_error.mask = 1;
	cpssp->NAME.lvt_error.vector = 0x00;
	cpssp->NAME.base = 0x00000000fee00000ULL;
	cpssp->NAME.apic_enable = 1;

	cpssp->NAME.timer_ccr = 0;
	cpssp->NAME.timer_event = 0;
	cpssp->NAME.timer_running = 0;
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
}

static void
NAME_(create)(struct cpssp *cpssp)
{
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	cpssp->NAME.bsp = 1; /* FIXME */
	cpssp->NAME.tsc_to_bus = TIME_HZ / SIG_HOST_BUS_HZ;
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef CPU_APIC_VERSION
#undef CPU_APIC_MAX_LVT

#undef DEBUG_CONTROL_FLOW
#undef DEBUG_CONTROL_FLOW_REGS
#undef DEBUG_CONTROL_FLOW_IRQS
