/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	0

/*
 * Instruction Set Summary
 *
 * 0000 0000 0000 0000	nop (85)

 * 0000 0001 dddd rrrr	movw (80)

 * 0000 0010 dddd rrrr	muls (82)
 * 0000 0011 0ddd 1rrr	fmul (59)
 * 0000 0011 0ddd 0rrr	mulsu (83)
 * 0000 0011 1ddd 0rrr	fmuls (60)
 * 0000 0011 1ddd 1rrr	fmulsu (61)

 * 0000 01rd dddd rrrr	cpc (50)
 * 0000 10rd dddd rrrr	sbc (97)
 * 0000 11rd dddd rrrr	add (6) (lsl)
 * 0001 00rd dddd rrrr	cpse (52)
 * 0001 01rd dddd rrrr	cp (49)
 * 0001 10rd dddd rrrr	sub (123)
 * 0001 11rd dddd rrrr	adc (5) (rol)
 * 0010 00rd dddd rrrr	and (8) (tst)
 * 0010 01rd dddd rrrr	eor (58) (clr)
 * 0010 10rd dddd rrrr	or (86)
 * 0010 11rd dddd rrrr	mov (79)

 * 1001 0110 KKdd KKKK	adiw (7)
 * 1001 0111 KKdd KKKK	sbiw (102)

 * 0011 KKKK dddd KKKK	cpi (51)
 * 0100 KKKK dddd KKKK	sbci (98)
 * 0101 KKKK dddd KKKK	subi (134)
 * 0110 KKKK dddd KKKK	ori (87) (sbr)
 * 0111 KKKK dddd KKKK	andi (9) (cbr)
 * 1110 KKKK dddd KKKK	ldi Rd,K (73) (ser)

 * 1001 000d dddd 0000 kkkk kkkk kkkk kkkk	lds Rd,k (74)
 * 1001 001d dddd 0000 kkkk kkkk kkkk kkkk	sts k,Rd (121)

 * 1001 010k kkkk 110k kkkk kkkk kkkk kkkk	jmp k (66)
 * 1001 010k kkkk 111k kkkk kkkk kkkk kkkk	call k (36)

 * 1001 000d dddd 0001	ld Rd,Z+ (72)
 * 1001 000d dddd 0010	ld Rd,-Z (72)
 * 1001 000d dddd 0100	lpm Rd,Z (76)
 * 1001 000d dddd 0101	lpm Rd,Z+ (76)
 * 1001 000d dddd 0110	elpm Rd,RAMPZ:Z (57)
 * 1001 000d dddd 0111	elpm Rd,RAMPZ:Z+ (57)
 * 1001 000d dddd 1001	ld Rd,Y+ (71)
 * 1001 000d dddd 1010	ld Rd,-Y (71)
 * 1001 000d dddd 1100	ld X (70)
 * 1001 000d dddd 1101	ld X+ (70)
 * 1001 000d dddd 1110	ld -X (70)

 * 1001 001r rrrr 0001	st Z+ (120)
 * 1001 001r rrrr 0010	st -Z (120)
 * 1001 001r rrrr 1001	st Y+ (119)
 * 1001 001r rrrr 1010	st -Y (119)
 * 1001 001r rrrr 1100	st X (118)
 * 1001 001r rrrr 1101	st X+ (118)
 * 1001 001r rrrr 1110	st -X (118)

 * 1001 001r rrrr 0100	xch (128)
 * 1001 001r rrrr 0101	las (68)
 * 1001 001r rrrr 0110	lac (67)
 * 1001 001r rrrr 0111	lat (69)

 * 1001 000d dddd 1111	pop (89)
 * 1001 001d dddd 1111	push (90)

 * 1001 010d dddd 0000	com (48)
 * 1001 010d dddd 0001	neg (84)
 * 1001 010d dddd 0010	swap (125)
 * 1001 010d dddd 0011	inc (65)
 * 1001 010d dddd 0101	asr (10
 * 1001 010d dddd 0110	lsr (78)
 * 1001 010d dddd 0111	ror (96)
 * 1001 010d dddd 1010	dec (53)

 * 1001 0101 1010 1000  wdr (127)

 * 1001 0100 0bbb 1000	bset (sec, seh, sei, sen, ses, set, sev, sez)
 * 1001 0100 1bbb 1000	bclr (11) (clc, clh, cli, cln, cls, clt, clv, clz)

 * 1001 1000 AAAA Abbb	cbi (37)
 * 1001 1001 AAAA Abbb	sbic (100)
 * 1001 1010 AAAA Abbb	sbi (99)
 * 1001 1011 AAAA Abbb	sbis (101)

 * 1001 11rd dddd rrrr	mul (81)

 * 1011 0AAd dddd AAAA	in (64)
 * 1011 1AAr rrrr AAAA	out (88)

 * 10q0 qq0d dddd 0qqq	ldd Rd,Z+q (72)
 * 10q0 qq0d dddd 1qqq	ldd Rd,Y+q (71)
 * 10q0 qq1r rrrr 0qqq	std Z+q (120)
 * 10q0 qq1r rrrr 1qqq	std Y+q (119)

 * 1100 kkkk kkkk kkkk	rjmp (94)
 * 1101 kkkk kkkk kkkk	rcall (91)

 * 1001 0100 0000 1001	ijmp (63)
 * 1001 0100 0001 1001	eijmp (56)
 * 1001 0100 KKKK 1011	des (54)

 * 1001 0101 0000 1001	icall (62)
 * 1001 0101 0000 1000	ret (92)
 * 1001 0101 0001 1000	reti (93)
 * 1001 0101 0001 1001	eicall (55)
 * 1001 0101 1000 1000	sleep (115)
 * 1001 0101 1001 1000	break (17)
 * 1001 0101 1010 1000	wdr (127)
 * 1001 0101 1100 1000	lpm R0,Z (76)
 * 1001 0101 1101 1000	elpm R0,RAMPZ:Z (57)
 * 1001 0101 1110 1000	spm Z+ (116)
 * 1001 0101 1110 1000	spm (117)
 * 1001 0101 1111 1000	spm Z+ (117)

 * 1010 0kkk dddd kkkk	lds Rd,k (75)
 * 1010 1kkk dddd kkkk	sts k,Rd (122)

 * 1111 100d dddd 0bbb	bld (12)
 * 1111 101d dddd 0bbb	bst (35)
 * 1111 110r rrrr 0bbb	sbrc (104)
 * 1111 111r rrrr 0bbb	sbrs (105)

 * 1111 00kk kkkk kbbb	brbs (14) (brcs, breq, brhs, ...)
 * 1111 01kk kkkk kbbb	brbc (13) (brcc, brge, brhc, ...)
 */

#ifdef INCLUDE

#include "jit-comp.h"
#include "jit-cache.h"

#endif /* INCLUDE */
#ifdef STATE

/* Page 11 */
#define NAME_REG_X	0x1a
#define NAME_REG_Y	0x1c
#define NAME_REG_Z	0x1e

struct {
	uint8_t skip;
	uint8_t hlt;
	uint16_t pc;
	uint16_t sp;

	uint8_t reg[32];

	/* SREG */
	uint8_t idelay;
	uint8_t i;
	uint8_t t;
	uint8_t h;
	uint8_t s;
	uint8_t v;
	uint8_t n;
	uint8_t z;
	uint8_t c;

	uint64_t irr;

	struct jit_comp jit_comp;
	struct jit_cache jit_cache;
	int cnt;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

#define SCONST(c) \
	jit_sconst(func, c)
#define UCONST(c) \
	jit_uconst(func, c)
#define SATTR(v) \
	jit_star(func, -1, sizeof(((struct cpssp *) 0)->v), jit_add(func, jcpssp, jit_uconst(func, offsetof(struct cpssp, v))))
#define UATTR(v) \
	jit_star(func, 1, sizeof(((struct cpssp *) 0)->v), jit_add(func, jcpssp, jit_uconst(func, offsetof(struct cpssp, v))))
#define UATTRARR(v, i) \
	jit_star(func, 1, sizeof(((struct cpssp *) 0)->v[0]), \
			jit_add(func, \
				jit_add(func, jcpssp, jit_uconst(func, offsetof(struct cpssp, v))), \
				jit_mul(func, jit_uconst(func, sizeof(((struct cpssp *) 0)->v[0])), i) \
			) \
		)
#define SATTRARR(v, i) \
	jit_star(func, -1, sizeof(((struct cpssp *) 0)->v[0]), \
			jit_add(func, \
				jit_add(func, jcpssp, jit_uconst(func, offsetof(struct cpssp, v))), \
				jit_mul(func, jit_uconst(func, sizeof(((struct cpssp *) 0)->v[0])), i) \
			) \
		)


static void
NAME_(dump)(struct cpssp *cpssp)
{
	unsigned int i;

	fprintf(stderr, "pc=%04x sp=%03x skip=%d hlt=%d\n",
			cpssp->NAME.pc * 2, cpssp->NAME.sp,
			cpssp->NAME.skip, cpssp->NAME.hlt);

	for (i = 0; i < 32; i++) {
		if (i % 16 != 0) {
			fprintf(stderr, " ");
		}
		fprintf(stderr, "%02x", cpssp->NAME.reg[i]);
		if (i % 16 == 15) {
			fprintf(stderr, "\n");
		}
	}

	fprintf(stderr, "i=%d", cpssp->NAME.i);
	fprintf(stderr, " ");
	fprintf(stderr, "t=%d", cpssp->NAME.t);
	fprintf(stderr, " ");
	fprintf(stderr, "h=%d", cpssp->NAME.h);
	fprintf(stderr, " ");
	fprintf(stderr, "s=%d", cpssp->NAME.s);
	fprintf(stderr, " ");
	fprintf(stderr, "v=%d", cpssp->NAME.v);
	fprintf(stderr, " ");
	fprintf(stderr, "n=%d", cpssp->NAME.n);
	fprintf(stderr, " ");
	fprintf(stderr, "z=%d", cpssp->NAME.z);
	fprintf(stderr, " ");
	fprintf(stderr, "c=%d", cpssp->NAME.c);
	fprintf(stderr, "\n");
}

static void
NAME_(reg8_set_res)(struct cpssp *cpssp, uint16_t r, uint16_t res)
{
	cpssp->NAME.reg[r] = res;
}

static void
NAME_(compile_reg8_set_res)(
	struct jit_func *func,
	struct jit_expr *jcpssp,
	struct jit_expr *r,
	struct jit_expr *jres
)
{
	// assert(/* 0 <= r && */ r < 32);

	/* cpssp->NAME.reg[r] = cpssp->NAME.res; */
	jit_assign(func, UATTRARR(NAME.reg, r), jres);
}

static uint16_t
NAME_(reg8_get_op1)(struct cpssp *cpssp, uint16_t r)
{
	return cpssp->NAME.reg[r];
}

static void
NAME_(compile_reg8_get_op1)(
	struct jit_func *func,
	struct jit_expr *ret,
	struct jit_expr *jcpssp,
	struct jit_expr *r
)
{
	// assert(/* 0 <= r && */ r < 32);

	/* return cpssp->NAME.reg[r]; */
	jit_assign(func, ret, UATTRARR(NAME.reg, r));
}

static void
NAME_(compile_reg8_get_op2)(
	struct jit_func *func,
	struct jit_expr *jop2,
	struct jit_expr *jcpssp,
	struct jit_expr *r
)
{
	// assert(/* 0 <= r && */ r < 32);

	/* op2 = cpssp->NAME.reg[r]; */
	jit_assign(func, jop2, UATTRARR(NAME.reg, r));
}

static void
NAME_(compile_reg16_set_addr)(
	struct jit_func *func,
	struct jit_expr *jcpssp,
	struct jit_expr *r,
	struct jit_expr *jaddr
)
{
	// assert(/* 0 <= r && */ r < 31);

	/* cpssp->NAME.reg[r + 1] = (uint8_t) (addr >> 8); */
	jit_assign(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 1))),
			jit_lsr(func, jaddr, UCONST(8)));
	/* cpssp->NAME.reg[r + 0] = (uint8_t) (addr >> 0); */
	jit_assign(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 0))),
			jit_lsr(func, jaddr, UCONST(0)));
}

static void
NAME_(compile_reg16_set_res)(
	struct jit_func *func,
	struct jit_expr *jcpssp,
	struct jit_expr *r,
	struct jit_expr *jres
)
{
	// assert(/* 0 <= r && */ r < 31);

	/* cpssp->NAME.reg[r + 1] = (uint8_t) (cpssp->NAME.res >> 8); */
	jit_assign(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 1))),
			jit_lsr(func, jres, UCONST(8)));
	/* cpssp->NAME.reg[r + 0] = (uint8_t) (cpssp->NAME.res >> 0); */
	jit_assign(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 0))),
			jit_lsr(func, jres, UCONST(0)));
}

static void
NAME_(compile_reg16_get_addr)(
	struct jit_func *func,
	struct jit_expr *jaddr,
	struct jit_expr *jcpssp,
	struct jit_expr *r
)
{
	// assert(/* 0 <= r && */ r < 31);

	/* addr = (NAME.reg[r + 1] << 8) | (NAME.reg[r + 0] << 0); */
	jit_assign(func, jaddr,
		jit_or(func,
			jit_lsl(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 1))), UCONST(8)),
			jit_lsl(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 0))), UCONST(0))
		)
	);
}

static void
NAME_(compile_reg16_get_op1)(
	struct jit_func *func,
	struct jit_expr *ret,
	struct jit_expr *jcpssp,
	struct jit_expr *r
)
{
	// assert(/* 0 <= r && */ r < 31);

	/* return (NAME.reg[r + 1] << 8) | (NAME.reg[r + 0] << 0); */
	jit_assign(func, ret,
		jit_or(func,
			jit_lsl(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 1))), UCONST(8)),
			jit_lsl(func, UATTRARR(NAME.reg, jit_add(func, r, jit_uconst(func, 0))), UCONST(0))
		)
	);
}

static void
NAME_(sreg_set_res)(struct cpssp *cpssp, uint16_t res)
{
	cpssp->NAME.i = (res >> 7) & 1;
	cpssp->NAME.t = (res >> 6) & 1;
	cpssp->NAME.h = (res >> 5) & 1;
	cpssp->NAME.s = (res >> 4) & 1;
	cpssp->NAME.v = (res >> 3) & 1;
	cpssp->NAME.n = (res >> 2) & 1;
	cpssp->NAME.z = (res >> 1) & 1;
	cpssp->NAME.c = (res >> 0) & 1;
}

static uint8_t
NAME_(sreg_get_op1)(struct cpssp *cpssp)
{
	return (cpssp->NAME.i << 7)
		| (cpssp->NAME.t << 6)
		| (cpssp->NAME.h << 5)
		| (cpssp->NAME.s << 4)
		| (cpssp->NAME.v << 3)
		| (cpssp->NAME.n << 2)
		| (cpssp->NAME.z << 1)
		| (cpssp->NAME.c << 0);
}

static void
NAME_(sph_set_res)(struct cpssp *cpssp, uint16_t res)
{
	cpssp->NAME.sp = (cpssp->NAME.sp & ~0xff00) | (res << 8);
}

static uint8_t
NAME_(sph_get_op1)(struct cpssp *cpssp)
{
	return (cpssp->NAME.sp >> 8) & 0xff;
}

static void
NAME_(spl_set_res)(struct cpssp *cpssp, uint16_t res)
{
	cpssp->NAME.sp = (cpssp->NAME.sp & ~0x00ff) | (res << 0);
}

static uint8_t
NAME_(spl_get_op1)(struct cpssp *cpssp)
{
	return (cpssp->NAME.sp >> 0) & 0xff;
}

static uint16_t
NAME_(lpm_op1)(struct cpssp *cpssp, uint16_t addr)
{
	uint16_t tmp;

	NAME_(fetch)(cpssp, addr >> 1, &tmp);

	return tmp >> ((addr & 1) * 8);
}

static void
NAME_(compile_lpm_op1)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jaddr
)
{
	jit_call(func, jret, NAME_(lpm_op1), jcpssp, jaddr, NULL);
}

/*
 * See table at page 327.
 */
static void
NAME_(out_res)(struct cpssp *cpssp, uint16_t addr, uint16_t res)
{
	switch (addr) {
	case 0x5f: /* SREG */
		NAME_(sreg_set_res)(cpssp, res);
		break;
	case 0x5e: /* SPH */
		NAME_(sph_set_res)(cpssp, res);
		break;
	case 0x5d: /* SPL */
		NAME_(spl_set_res)(cpssp, res);
		break;
	default:
		NAME_(io_st)(cpssp, addr, res);
		break;
	}
}

static void
NAME_(compile_out_res)(
	struct jit_func *func,
	struct jit_expr *jcpssp,
	struct jit_expr *jaddr,
	struct jit_expr *jres
)
{
	jit_call(func, NULL, NAME_(out_res), jcpssp, jaddr, jres, NULL);
}

/*
 * See table at page 327.
 */
static uint16_t
NAME_(in_op1)(struct cpssp *cpssp, uint16_t addr)
{
	uint8_t op1;

	switch (addr) {
	case 0x5f:
		op1 = NAME_(sreg_get_op1)(cpssp);
		break;
	case 0x5e:
		op1 = NAME_(sph_get_op1)(cpssp);
		break;
	case 0x5d:
		op1 = NAME_(spl_get_op1)(cpssp);
		break;
	default:
		NAME_(io_ld)(cpssp, addr, &op1);
		break;
	}

	return (uint16_t) op1;
}

static void
NAME_(compile_in_op1)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jaddr
)
{
	jit_call(func, jret, NAME_(in_op1), jcpssp, jaddr, NULL);
}

static void
NAME_(st_res)(struct cpssp *cpssp, uint16_t addr, uint16_t res)
{
	if (addr < 0x20) {
		NAME_(reg8_set_res)(cpssp, addr, res);
	} else if (addr < 0x60) {
		NAME_(out_res)(cpssp, addr, res);
	} else {
		NAME_(ram_st)(cpssp, addr, res);
	}
}

static void
NAME_(compile_st_res)(
	struct jit_func *func,
	struct jit_expr *jcpssp,
	struct jit_expr *jaddr,
	struct jit_expr *jres
)
{
	jit_call(func, NULL, NAME_(st_res), jcpssp, jaddr, jres, NULL);
}

static uint16_t
NAME_(ld_op1)(struct cpssp *cpssp, uint16_t addr)
{
	uint16_t op1;

	if (addr < 0x20) {
		op1 = NAME_(reg8_get_op1)(cpssp, addr);
	} else if (addr < 0x60) {
		op1 = NAME_(in_op1)(cpssp, addr);
	} else {
		NAME_(ram_ld)(cpssp, addr, &op1);
	}

	return op1;
}

static void
NAME_(compile_ld_op1)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jaddr
)
{
	jit_call(func, jret, NAME_(ld_op1), jcpssp, jaddr, NULL);
}

static void
NAME_(push_res)(struct cpssp *cpssp, uint16_t res)
{
	uint16_t addr;

	addr = cpssp->NAME.sp;
	cpssp->NAME.sp = (cpssp->NAME.sp - 1) & 0xfff;

	NAME_(st_res)(cpssp, addr, res);
}

static void
NAME_(compile_push_res)(
	struct jit_func *func,
	struct jit_expr *jcpssp,
	struct jit_expr *jres
)
{
	jit_call(func, NULL, NAME_(push_res), jcpssp, jres, NULL);
}

static uint16_t
NAME_(pop_op1)(struct cpssp *cpssp)
{
	uint16_t addr;
	uint16_t op1;

	cpssp->NAME.sp = (cpssp->NAME.sp + 1) & 0x0fff;
	addr = cpssp->NAME.sp;

	op1 = NAME_(ld_op1)(cpssp, addr);

	return op1;
}

static void
NAME_(compile_pop_op1)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp
)
{
	jit_call(func, jret, NAME_(pop_op1), jcpssp, NULL);
}

static void
NAME_(push_pc)(struct cpssp *cpssp)
{
	uint16_t res;

	res = (cpssp->NAME.pc >> 8) & 0xff;
	NAME_(push_res)(cpssp, res);

	res = (cpssp->NAME.pc >> 0) & 0xff;
	NAME_(push_res)(cpssp, res);
}

static void
NAME_(compile_push_pc)(struct jit_func *func, struct jit_expr *jcpssp)
{
	jit_call(func, NULL, NAME_(push_pc), jcpssp, NULL);
}

static void
NAME_(pop_pc)(struct cpssp *cpssp)
{
	uint16_t op1;

	op1 = NAME_(pop_op1)(cpssp);
	cpssp->NAME.pc &= ~0x00ff;
	cpssp->NAME.pc |= op1 << 0;

	op1 = NAME_(pop_op1)(cpssp);
	cpssp->NAME.pc &= ~0xff00;
	cpssp->NAME.pc |= op1 << 8;
}

static void
NAME_(compile_pop_pc)(struct jit_func *func, struct jit_expr *jcpssp)
{
	jit_call(func, NULL, NAME_(pop_pc), jcpssp, NULL);
}

static void
NAME_(compile_com)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = 0xff - cpssp->NAME.op1; */
	jit_assign(func, jret, jit_sub(func, UCONST(0xff), jop1));

	/* cpssp->NAME.c = 1; */
	jit_assign(func, UATTR(NAME.c), UCONST(1));
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->NAME.res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsr(func, jret, UCONST(7)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.v = 0; */
	jit_assign(func, UATTR(NAME.v), UCONST(0));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_neg)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = 0 - op1; */
	jit_assign(func, jret, jit_neg(func, jop1));

	/* cpssp->NAME.c = cpssp->NAME.res != 0x00; */
	jit_assign(func, UATTR(NAME.c),
			jit_neq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->NAME.res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsr(func, jret, UCONST(7)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.v = cpssp->NAME.res == 0x80; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x80)));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h = ... FIXME */
	/* FIXME */
}

static void
NAME_(compile_swap)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = (op1 << 4) | (op1 >> 4); */
	jit_assign(func, jret,
		jit_or(func,
			jit_lsl(func, jop1, UCONST(4)),
			jit_lsr(func, jop1, UCONST(4))
		)
	);

	/* cpssp->NAME.c unchanged */
	/* cpssp->NAME.z unchanged */
	/* cpssp->NAME.n unchanged */
	/* cpssp->NAME.v unchanged */
	/* cpssp->NAME.s unchanged */
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_inc)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = op1 + 1; */
	jit_assign(func, jret, jit_add(func, jop1, UCONST(1)));

	/* cpssp->NAME.c unchanged */
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->NAME.res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsr(func, jret, UCONST(7)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.v = cpssp->NAME.res == 0x80; */
	jit_assign(func, UATTR(NAME.v),
			jit_eq(func, jret, UCONST(0x80)));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_asr)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = (int8_t) op1 >> 1; */
	jit_assign(func, jret,
		jit_asr(func,
			jit_conv(func, -1, 1, jop1),
			UCONST(1)
		)
	);

	/* cpssp->NAME.c = op1 & 1; */
	jit_assign(func, UATTR(NAME.c),
			jit_and(func, jop1, UCONST(1)));
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsr(func, jret, UCONST(7)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.v = cpssp->NAME.n ^ cpssp->NAME.c; */
	jit_assign(func, UATTR(NAME.v),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.c)));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_lsr)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = op1 >> 1; */
	jit_assign(func, jret, jit_lsr(func, jop1, UCONST(1)));

	/* cpssp->NAME.c = op1 & 1; */
	jit_assign(func, UATTR(NAME.c), jit_and(func, jop1, UCONST(1)));
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = 0; */
	jit_assign(func, UATTR(NAME.n), UCONST(0));
	/* cpssp->NAME.v = cpssp->NAME.n ^ cpssp->NAME.c; */
	jit_assign(func, UATTR(NAME.v),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.c)));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_ror)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = (cpssp->NAME.c << 7) | (op1 >> 1); */
	jit_assign(func, jret,
		jit_or(func,
			jit_lsl(func, UATTR(NAME.c), UCONST(7)),
			jit_lsr(func, jop1, UCONST(1))
		)
	);

	/* cpssp->NAME.c = op1 & 1; */
	jit_assign(func, UATTR(NAME.c),
			jit_and(func, jop1, UCONST(1)));
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = 0; */
	jit_assign(func, UATTR(NAME.n), UCONST(0));
	/* cpssp->NAME.v = cpssp->NAME.n ^ cpssp->NAME.c; */
	jit_assign(func, UATTR(NAME.v),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.c)));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_dec)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1
)
{
	/* cpssp->NAME.res = op1 - 1; */
	jit_assign(func, jret, jit_sub(func, jop1, UCONST(1)));

	/* cpssp->NAME.c unchanged */
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->NAME.res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsr(func, jret, UCONST(7)),
			UCONST(1)
		)
	);

	/* cpssp->NAME.v = cpssp->NAME.res == 0x7f; */
	jit_assign(func, UATTR(NAME.v),
			jit_eq(func, jret, UCONST(0x7f)));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static uint16_t
NAME_(add)(struct cpssp *cpssp, uint16_t op1, uint16_t op2)
{
	uint16_t unsigned_res;
	int16_t signed_res;
	uint16_t res;

	unsigned_res = (uint16_t) (uint8_t) op1 + (uint16_t) (uint8_t) op2;
	signed_res = (int16_t) (int8_t) op1 + (int16_t) (int8_t) op2;

	res = op1 + op2;

	cpssp->NAME.c = unsigned_res != (uint16_t) (uint8_t) res;
	cpssp->NAME.z = res == 0x00;
	cpssp->NAME.n = (res >> 7) & 1;
	cpssp->NAME.v = signed_res != (int16_t) (int8_t) res;
	cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v;
	cpssp->NAME.h = ((res ^ op1 ^ op2) >> 4) & 1;

	return res;
}

static void
NAME_(compile_add)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	jit_call(func, jret, NAME_(add), jcpssp, jop1, jop2, NULL);
}

static uint16_t
NAME_(addw)(struct cpssp *cpssp, uint16_t op1, uint16_t op2)
{
	uint32_t unsigned_res;
	int32_t signed_res;
	uint16_t res;

	unsigned_res = (uint32_t) (uint16_t) op1 + (uint32_t) (uint16_t) op2;
	signed_res = (int32_t) (int16_t) op1 + (int32_t) (int16_t) op2;

	res = op1 + op2;

	cpssp->NAME.c = unsigned_res != (uint32_t) (uint16_t) res;
	cpssp->NAME.z = res == 0x0000;
	cpssp->NAME.n = (res >> 15) & 1;
	cpssp->NAME.v = signed_res != (int32_t) (int16_t) res;
	cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v;
	/* cpssp->NAME.h unchanged */

	return res;
}

static void
NAME_(compile_addw)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	jit_call(func, jret, NAME_(addw), jcpssp, jop1, jop2, NULL);
}

static uint16_t
NAME_(adc)(struct cpssp *cpssp, uint16_t op1, uint16_t op2)
{
	uint16_t unsigned_res;
	int16_t signed_res;
	uint16_t res;

	unsigned_res = (uint16_t) (uint8_t) op1
			+ (uint16_t) (uint8_t) op2
			+ (uint16_t) cpssp->NAME.c;
	signed_res = (int16_t) (int8_t) op1
			+ (int16_t) (int8_t) op2
			+ (int16_t) cpssp->NAME.c;

	res = op1 + op2 + cpssp->NAME.c;

	cpssp->NAME.c = unsigned_res != (uint16_t) (uint8_t) res;
	cpssp->NAME.z = res == 0x00;
	cpssp->NAME.n = (res >> 7) & 1;
	cpssp->NAME.v = signed_res != (int16_t) (int8_t) res;
	cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v;
	cpssp->NAME.h = ((res ^ op1 ^ op2) >> 4) & 1; /* Correct? FIXME */

	return res;
}

static void
NAME_(compile_adc)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	jit_call(func, jret, NAME_(adc), jcpssp, jop1, jop2, NULL);
}

static void
NAME_(compile_and)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	/* cpssp->NAME.res = op1 & op2; */
	jit_assign(func, jret, jit_and(func, jop1, jop2));

	/* cpssp->NAME.c unchanged */
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->NAME.res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsr(func, jret, UCONST(7)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.v = 0; */
	jit_assign(func, UATTR(NAME.v), UCONST(0));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_eor)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	/* cpssp->NAME.res = op1 ^ op2; */
	jit_assign(func, jret, jit_xor(func, jop1, jop2));

	/* cpssp->NAME.c unchanged */
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->NAME.res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsl(func, jret, UCONST(7)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.v = 0; */
	jit_assign(func, UATTR(NAME.v), UCONST(0));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_or)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	/* cpssp->NAME.res = op1 | op2; */
	jit_assign(func, jret, jit_or(func, jop1, jop2));

	/* cpssp->NAME.c unchanged */
	/* cpssp->NAME.z = cpssp->NAME.res == 0x00; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x00)));
	/* cpssp->NAME.n = (cpssp->NAME.res >> 7) & 1; */
	jit_assign(func, UATTR(NAME.n),
		jit_and(func,
			jit_lsr(func, jret, UCONST(7)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.v = 0; */
	jit_assign(func, UATTR(NAME.v), UCONST(0));
	/* cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v; */
	jit_assign(func, UATTR(NAME.s),
			jit_xor(func, UATTR(NAME.n), UATTR(NAME.v)));
	/* cpssp->NAME.h unchanged */
}

static void
NAME_(compile_mul)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	/* cpssp->NAME.res = (uint16_t) op1 * (uint16_t) op2; */
	jit_assign(func, jret, jit_mul(func, jop1, jop2));

	/* cpssp->NAME.c = (cpssp->NAME.res >> 15) & 1; */
	jit_assign(func, UATTR(NAME.c),
		jit_and(func,
			jit_lsr(func, jret, UCONST(15)),
			UCONST(1)
		)
	);
	/* cpssp->NAME.z = cpssp->NAME.res == 0x0000; */
	jit_assign(func, UATTR(NAME.z),
			jit_eq(func, jret, UCONST(0x0000)));
	/* cpssp->NAME.n unchanged */
	/* cpssp->NAME.v unchanged */
	/* cpssp->NAME.s unchanged */
	/* cpssp->NAME.h unchanged */
}

static uint16_t
NAME_(sub)(struct cpssp *cpssp, uint16_t op1, uint16_t op2)
{
	uint16_t unsigned_res;
	int16_t signed_res;
	uint16_t res;

	unsigned_res = (uint16_t) (uint8_t) op1
			- (uint16_t) (uint8_t) op2;
	signed_res = (int16_t) (int8_t) op1
			- (int16_t) (int8_t) op2;

	res = op1 - op2;

	cpssp->NAME.c = unsigned_res != (uint16_t) (uint8_t) res;
	cpssp->NAME.z = res == 0x00;
	cpssp->NAME.n = (res >> 7) & 1;
	cpssp->NAME.v = signed_res != (int16_t) (int8_t) res;
	cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v;
	cpssp->NAME.h = ((res ^ op1 ^ op2) >> 4) & 1;

	return res;
}

static void
NAME_(compile_sub)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	jit_call(func, jret, NAME_(sub), jcpssp, jop1, jop2, NULL);
}

static uint16_t
NAME_(subw)(struct cpssp *cpssp, uint16_t op1, uint16_t op2)
{
	uint32_t unsigned_res;
	int32_t signed_res;
	uint16_t res;

	unsigned_res = (uint32_t) (uint16_t) op1 - (uint32_t) (uint16_t) op2;
	signed_res = (int32_t) (int16_t) op1 - (int32_t) (int16_t) op2;

	res = op1 - op2;

	cpssp->NAME.c = unsigned_res != (uint32_t) (uint16_t) res;
	cpssp->NAME.z = res == 0x0000;
	cpssp->NAME.n = (res >> 15) & 1;
	cpssp->NAME.v = signed_res != (int32_t) (int16_t) res;
	cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v;
	/* cpssp->NAME.h unchanged */

	return res;
}

static void
NAME_(compile_subw)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	jit_call(func, jret, NAME_(subw), jcpssp, jop1, jop2, NULL);
}

static uint16_t
NAME_(sbc)(struct cpssp *cpssp, uint16_t op1, uint16_t op2)
{
	uint16_t unsigned_res;
	int16_t signed_res;
	uint16_t res;

	unsigned_res = (uint16_t) (uint8_t) op1
			- (uint16_t) (uint8_t) op2
			- (uint16_t) cpssp->NAME.c;
	signed_res = (int16_t) (int8_t) op1
			- (int16_t) (int8_t) op2
			- (int16_t) cpssp->NAME.c;

	res = op1 - op2 - cpssp->NAME.c;

	cpssp->NAME.c = unsigned_res != (uint16_t) (uint8_t) res;
	cpssp->NAME.z &= res == 0x00;
	cpssp->NAME.n = (res >> 7) & 1;
	cpssp->NAME.v = signed_res != (int16_t) (int8_t) res;
	cpssp->NAME.s = cpssp->NAME.n ^ cpssp->NAME.v;
	cpssp->NAME.h = ((res ^ op1 ^ op2) >> 4) & 1;

	return res;
}

static void
NAME_(compile_sbc)(
	struct jit_func *func,
	struct jit_expr *jret,
	struct jit_expr *jcpssp,
	struct jit_expr *jop1,
	struct jit_expr *jop2
)
{
	jit_call(func, jret, NAME_(sbc), jcpssp, jop1, jop2, NULL);
}

static void *
NAME_(compile_interrupt)(unsigned int irq)
{
	struct jit_func *func;
	struct jit_expr *jcpssp;

	func = jit_func_alloc();
	jcpssp = jit_param_alloc(func, 0, 0);

	/* cpssp->NAME.hlt = 0; */
	jit_assign(func, UATTR(NAME.hlt), UCONST(0));

	/* cpssp->NAME.i = 0; */
	jit_assign(func, UATTR(NAME.i), UCONST(0));

	NAME_(compile_push_pc)(func, jcpssp);

	/* cpssp->NAME.pc = irq << 1; */
	jit_assign(func, UATTR(NAME.pc), UCONST(irq << 1));

	jit_assign(func, SATTR(NAME.cnt),
			jit_sub(func, SATTR(NAME.cnt), SCONST(1)));

	return jit_compile(func);
}

static void *
NAME_(compile)(struct cpssp *cpssp)
{
	struct jit_func *func;
	struct jit_expr *jcpssp;
	struct jit_expr *jaddr;
	struct jit_expr *jop1;
	struct jit_expr *jop2;
	struct jit_expr *jres;
	struct jit_label *label_else;
	struct jit_label *label_endif;
	struct jit_label *label;
	int end;
	int needskip;
	unsigned int cnt;
	uint16_t pc;
	uint16_t inst;
	uint16_t inst2;
	uint8_t b;
	uint8_t d;
	uint8_t q;
	uint8_t r;
	uint8_t A;
	uint8_t K;
	uint32_t k;

	func = jit_func_alloc();
	jcpssp = jit_param_alloc(func, 0, 0);
	jaddr = jit_local_alloc(func, 1, 2);
	jop1 = jit_local_alloc(func, 1, 2);
	jop2 = jit_local_alloc(func, 1, 2);
	jres = jit_local_alloc(func, 1, 2);

	pc = cpssp->NAME.pc;
	cnt = 0;
	end = 0;
	needskip = 1;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "\n");
		fprintf(stderr, "%s at 0x%04x i=%d t=%d h=%d s=%d v=%d n=%d z=%d c=%d\n",
				cpssp->NAME.skip ? "Skipping" : "Executing",
				pc << 1,
				cpssp->NAME.i, cpssp->NAME.t,
				cpssp->NAME.h, cpssp->NAME.s,
				cpssp->NAME.v, cpssp->NAME.n,
				cpssp->NAME.z, cpssp->NAME.c);
		for (d = 0; d < 32; d++) {
			if (d == 0 || d == 16) {
				fprintf(stderr, "\t");
			} else if (d == 8 || d == 24) {
				fprintf(stderr, "  ");
			} else {
				fprintf(stderr, " ");
			}
			fprintf(stderr, "%02x", cpssp->NAME.reg[d]);
			if (d == 15 || d == 31) {
				fprintf(stderr, "\n");
			}
		}
	}
again:	;
#if 0
	if (pc << 1 == 0xa88) {
		jit_brk(func);
	}
#endif
	NAME_(fetch)(cpssp, pc, &inst);
	pc++;
	if ((inst & 0xfe0e) == 0x940c		/* jmp (66) */
	 || (inst & 0xfe0e) == 0x940e		/* call (36) */
	 || (inst & 0xfe0f) == 0x9000		/* lds (74) */
	 || (inst & 0xfe0f) == 0x9200) {	/* sts (121) */
		NAME_(fetch)(cpssp, pc, &inst2);
		pc++;
	} else {
		inst2 = 0x0000; /* Not used. */
	}

	/* cpssp->NAME.pc = pc; */
	jit_assign(func, UATTR(NAME.pc), UCONST(pc));

	/* cpssp->NAME.idelay = 0; */
	jit_assign(func, UATTR(NAME.idelay), UCONST(0));

	label_else = jit_label_alloc(func);
	label_endif = jit_label_alloc(func);

	if (needskip) {
		/* if (cpssp->NAME.skip) { */
		jit_ifgoto(func, jit_not(func, UATTR(NAME.skip)), label_else);

		/* cpssp->NAME.skip = 0; */
		jit_assign(func, UATTR(NAME.skip), UCONST(0));

		/* } else { */
		jit_goto(func, label_endif);
		jit_label(func, label_else);
	}
	needskip = 0;

	if ((inst & 0xff8f) == 0x9408) {		/* bset (34) */
							/* sec (106) */
							/* seh (107) */
							/* sei (108) */
							/* sen (109) */
							/* ses (111) */
							/* set (112) */
							/* sev (113) */
							/* sez (114) */
		b = (inst >> 4) & 0x7;

		switch (b) {
		case 0:
			jit_assign(func, UATTR(NAME.c), UCONST(1));
			break;
		case 1:
			jit_assign(func, UATTR(NAME.z), UCONST(1));
			break;
		case 2:
			jit_assign(func, UATTR(NAME.n), UCONST(1));
			break;
		case 3:
			jit_assign(func, UATTR(NAME.v), UCONST(1));
			break;
		case 4:
			jit_assign(func, UATTR(NAME.s), UCONST(1));
			break;
		case 5:
			jit_assign(func, UATTR(NAME.h), UCONST(1));
			break;
		case 6:
			jit_assign(func, UATTR(NAME.t), UCONST(1));
			break;
		case 7:
			jit_assign(func, UATTR(NAME.i), UCONST(1));
			jit_assign(func, UATTR(NAME.idelay), UCONST(1));
			break;
		default:
			assert(0); /* Cannot happen. */
		}

	} else if ((inst & 0xff8f) == 0x9488) {		/* bclr (11) */
		b = (inst >> 4) & 0x7;

		switch (b) {
		case 0:
			jit_assign(func, UATTR(NAME.c), UCONST(0));
			break;
		case 1:
			jit_assign(func, UATTR(NAME.z), UCONST(0));
			break;
		case 2:
			jit_assign(func, UATTR(NAME.n), UCONST(0));
			break;
		case 3:
			jit_assign(func, UATTR(NAME.v), UCONST(0));
			break;
		case 4:
			jit_assign(func, UATTR(NAME.s), UCONST(0));
			break;
		case 5:
			jit_assign(func, UATTR(NAME.h), UCONST(0));
			break;
		case 6:
			jit_assign(func, UATTR(NAME.t), UCONST(0));
			break;
		case 7:
			jit_assign(func, UATTR(NAME.i), UCONST(0));
			break;
		default:
			assert(0); /* Cannot happen. */
		}

	} else if ((inst & 0xffff) == 0x9508		/* ret (92) */
		|| (inst & 0xffff) == 0x9518) {		/* reti (93) */
		NAME_(compile_pop_pc)(func, jcpssp);

		if ((inst >> 4) & 1) {
			jit_assign(func, UATTR(NAME.i), UCONST(1));
		}
		end = 1;

	} else if ((inst & 0xffff) == 0x9409) {		/* ijmp (63) */
		NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
		jit_assign(func, UATTR(NAME.pc), jaddr);
		end = 1;

	} else if ((inst & 0xffff) == 0x9419) {		/* eijmp (56) */
		assert(0); /* FIXME */

	} else if ((inst & 0xffff) == 0x9509) {		/* icall (62) */
		NAME_(compile_push_pc)(func, jcpssp);

		NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
		jit_assign(func, UATTR(NAME.pc), jaddr);
		end = 1;

	} else if ((inst & 0xffff) == 0x9519) {		/* eicall (55) */
		assert(0); /* FIXME */

	} else if ((inst & 0xffff) == 0x9598) {		/* break (17) */
		assert(0); /* FIXME */

	} else if ((inst & 0xffff) == 0x9588) {		/* sleep (115) */
		jit_assign(func, UATTR(NAME.hlt), UCONST(1));
		end = 1;

	} else if ((inst & 0xfe0e) == 0x940c		/* jmp (66) */
		|| (inst & 0xfe0e) == 0x940e) {		/* call (36) */
		k = ((inst >> 4) & 0x1f) << 17;
		k |= ((inst >> 0) & 0x1) << 16;
		k |= inst2;

		switch (inst & 0xfe0e) {
		case 0x940c:
			/* jmp */
			break;
		case 0x940e:
			/* call */
			NAME_(compile_push_pc)(func, jcpssp);
			break;
		default:
			assert(0);
		}

		jit_assign(func, UATTR(NAME.pc), UCONST(k));
		end = 1;

	} else if ((inst & 0xf000) == 0xc000		/* rjmp (94) */
		|| (inst & 0xf000) == 0xd000) {		/* rcall (91) */
		k = inst & 0xfff;
		k = (int32_t) k << (32-12);
		k = (int32_t) k >> (32-12);

		switch (inst & 0xf000) {
		case 0xc000:
			/* rjmp */
			break;
		case 0xd000:
			/* rcall */
			NAME_(compile_push_pc)(func, jcpssp);
			break;
		default:
			assert(0);
		}

		jit_assign(func, UATTR(NAME.pc), UCONST(pc + k));
		end = 1;

	} else if ((inst & 0xff00) == 0x0100) {		/* movw (80) */
		d = ((inst >> 4) & 0xf) << 1;
		r = (inst & 0xf) << 1;

		NAME_(compile_reg16_get_op1)(func, jop1, jcpssp, UCONST(r));
		jit_assign(func, jres, jop1);
		NAME_(compile_reg16_set_res)(func, jcpssp, UCONST(d), jres);

	} else if ((inst & 0xff00) == 0x9600		/* adiw (7) */
		|| (inst & 0xff00) == 0x9700) {		/* sbiw (102) */
		d = (inst >> 4) & 0x3;
		d = 24 + d * 2;
		K = ((inst >> 6) & 0x3) << 4;
		K |= inst & 0xf;

		NAME_(compile_reg16_get_op1)(func, jop1, jcpssp, UCONST(d));
		jit_assign(func, jop2, UCONST(K));
		switch (inst & 0xff00) {
		case 0x9600:
			NAME_(compile_addw)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x9700:
			NAME_(compile_subw)(func, jres, jcpssp, jop1, jop2);
			break;
		default:
			assert(0);
		}
		NAME_(compile_reg16_set_res)(func, jcpssp, UCONST(d), jres);

	} else if ((inst & 0xff00) == 0x9800		/* cbi (37) */
		|| (inst & 0xff00) == 0x9a00) {		/* sbi (99) */
		A = (inst >> 3) & 0x1f;
		b = inst & 0x7;

		jit_assign(func, jaddr, UCONST(A + 0x20));
		NAME_(compile_in_op1)(func, jop1, jcpssp, jaddr);
		switch (inst & 0xff00) {
		case 0x9800:
			/* cbi */
			jit_assign(func, jres,
					jit_and(func, jop1, UCONST(~(1 << b) & 0xff)));
			break;
		case 0x9a00:
			/* sbi */
			jit_assign(func, jres,
					jit_or(func, jop1, UCONST(1 << b)));
			break;
		default:
			assert(0);
		}
		NAME_(compile_out_res)(func, jcpssp, jaddr, jres);

	} else if ((inst & 0xff00) == 0x9900		/* sbic (100) */
		|| (inst & 0xff00) == 0x9b00) {		/* sbis (101) */
		A = (inst >> 3) & 0x1f;
		b = (inst >> 0) & 0x7;

		/* cpssp->NAME.addr = A; */
		jit_assign(func, jaddr, UCONST(A + 0x20));
		/* NAME_(in_op1)(cpssp); */
		NAME_(compile_in_op1)(func, jop1, jcpssp, jaddr);
		/* cpssp->NAME.res = (op1 >> b) & 1; */
		jit_assign(func, jres,
			jit_and(func,
				jit_lsr(func, jop1, UCONST(b)),
				UCONST(1)
			)
		);
		/* cpssp->NAME.skip = cpssp->NAME.res ^ ((inst >> 9) & 1) ^ 1; */
		jit_assign(func, UATTR(NAME.skip),
				jit_xor(func, jres, UCONST(((inst >> 9) & 1) ^ 1)));
		needskip = 1;

	} else if ((inst & 0xfe08) == 0xfc00		/* sbrc (104) */
		|| (inst & 0xfe08) == 0xfe00) {		/* sbrs (105) */
		r = (inst >> 4) & 0x1f;
		b = (inst >> 0) & 0x7;

		/* NAME_(reg8_get_op1)(cpssp, r); */
		NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(r));
		/* cpssp->NAME.res = (op1 >> b) & 1; */
		jit_assign(func, jres,
			jit_and(func,
				jit_lsr(func, jop1, UCONST(b)),
				UCONST(1)
			)
		);
		/* cpssp->NAME.skip = cpssp->NAME.res ^ ((inst >> 9) & 1) ^ 1; */
		jit_assign(func, UATTR(NAME.skip),
				jit_xor(func, jres, UCONST(((inst >> 9) & 1) ^ 1)));
		needskip = 1;

	} else if ((inst & 0xfe08) == 0xf800		/* bld (12) */
		|| (inst & 0xfe08) == 0xfa00) {		/* bst (35) */
		r = (inst >> 4) & 0x1f;
		b = (inst >> 0) & 0x7;

		/* NAME_(reg8_get_op1)(cpssp, r); */
		NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(r));
		switch (inst & 0xfe08) {
		case 0xf800:
			/* bld */
			/* cpssp->NAME.res = (op1 & ~(1 << b)) | (cpssp->NAME.t << b); */
			jit_assign(func, jres,
				jit_or(func,
					jit_and(func, jop1, UCONST(~(1 << b))),
					jit_lsl(func, UATTR(NAME.t), UCONST(b))
				)
			);
			/* NAME_(reg8_set_res)(cpssp, r); */
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(r), jres);
			break;
		case 0xfa00:
			/* bst */
			/* cpssp->NAME.t = (op1 >> b) & 1; */
			jit_assign(func, UATTR(NAME.t),
				jit_and(func,
					jit_lsr(func, jop1, UCONST(b)),
					UCONST(1)
				)
			);
			break;
		default:
			assert(0); /* Cannot happen. */
		}

	} else if ((inst & 0xfc00) == 0x0000	/* nop */
		|| (inst & 0xfc00) == 0x0400	/* cpc (50) */
		|| (inst & 0xfc00) == 0x0800	/* sbc (97) */
		|| (inst & 0xfc00) == 0x0c00	/* add (6) */
		|| (inst & 0xfc00) == 0x1000	/* cpse (52) */
		|| (inst & 0xfc00) == 0x1400	/* cp (49) */
		|| (inst & 0xfc00) == 0x1800	/* sub (123) */
		|| (inst & 0xfc00) == 0x1c00	/* adc (5) (rol) */
		|| (inst & 0xfc00) == 0x2000	/* and (8) */
		|| (inst & 0xfc00) == 0x2400	/* eor (58) */
		|| (inst & 0xfc00) == 0x2800	/* or (86) */
		|| (inst & 0xfc00) == 0x2c00) {	/* mov (79) */
		d = (inst >> 4) & 0x1f;
		r = ((inst >> 9) & 0x1) << 4;
		r |= (inst >> 0) & 0xf;

		NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
		NAME_(compile_reg8_get_op2)(func, jop2, jcpssp, UCONST(r));

		switch (inst & 0xfc00) {
		case 0x0000: /* nop */
			jit_assign(func, jres, UCONST(0));
			break;
		case 0x0400: /* cpc */
			NAME_(compile_sbc)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x0800: /* sbc */
			NAME_(compile_sbc)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x0c00: /* add */
			NAME_(compile_add)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x1000: /* cpse */
			jit_assign(func, UATTR(NAME.skip),
					jit_eq(func, jop1, jop2));
			needskip = 1;
			break;
		case 0x1400: /* cp */
			NAME_(compile_sub)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x1800: /* sub */
			NAME_(compile_sub)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x1c00: /* adc */
			NAME_(compile_adc)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x2000: /* and */
			NAME_(compile_and)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x2400: /* eor */
			NAME_(compile_eor)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x2800: /* or */
			NAME_(compile_or)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x2c00: /* mov */
			jit_assign(func, jres, jop2);
			break;
		default:
			assert(0);
		}

		if ((inst & 0xfc00) != 0x0000
		 && (inst & 0xfc00) != 0x0400
		 && (inst & 0xfc00) != 0x1000
		 && (inst & 0xfc00) != 0x1400) {
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
		}

	} else if ((inst & 0xfc00) == 0x9c00) {		/* mul (81) */
		d = (inst >> 4) & 0x1f;
		r = ((inst >> 9) & 0x1) << 4;
		r |= (inst >> 0) & 0xf;

		NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
		NAME_(compile_reg8_get_op2)(func, jop2, jcpssp, UCONST(r));

		NAME_(compile_mul)(func, jres, jcpssp, jop1, jop2);

		NAME_(compile_reg16_set_res)(func, jcpssp, UCONST(0), jres);

	} else if ((inst & 0xfc00) == 0xf000		/* brbs (14) */
		|| (inst & 0xfc00) == 0xf400) {		/* brbc (13) */
		k = (inst >> 3) & 0x7f;
		k = (int16_t) k << (16 - 7);
		k = (int16_t) k >> (16 - 7);
		b = (inst >> 0) & 0x7;

		switch (b) {
		case 0:
			jit_assign(func, jres, UATTR(NAME.c));
			break;
		case 1:
			jit_assign(func, jres, UATTR(NAME.z));
			break;
		case 2:
			jit_assign(func, jres, UATTR(NAME.n));
			break;
		case 3:
			jit_assign(func, jres, UATTR(NAME.v));
			break;
		case 4:
			jit_assign(func, jres, UATTR(NAME.s));
			break;
		case 5:
			jit_assign(func, jres, UATTR(NAME.h));
			break;
		case 6:
			jit_assign(func, jres, UATTR(NAME.t));
			break;
		case 7:
			jit_assign(func, jres, UATTR(NAME.i));
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		jit_assign(func, jres,
				jit_xor(func, jres, UCONST((inst >> 10) & 1)));

		label = jit_label_alloc(func);
		jit_ifgoto(func, jit_not(func, jres), label);

		jit_assign(func, UATTR(NAME.pc),
				jit_add(func, UATTR(NAME.pc), UCONST(k)));

		jit_label(func, label);
		end = 1;

	} else if ((inst & 0xff0f) == 0x940b) {		/* des (54) */
		assert(0); /* FIXME */

	} else if ((inst & 0xfe0f) == 0x9000		/* lds (74) */
		|| (inst & 0xfe0f) == 0x9200) {		/* sts (121) */
		d = (inst >> 4) & 0x1f;
		k = inst2;

		jit_assign(func, jaddr, UCONST(k));
		if ((inst & 0xfe0f) == 0x9000) {
			/* lds */
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
		} else {
			/* sts */
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
		}

	} else if ((inst & 0xd000) == 0x8000) {
		/* 0x8000: ldd Z+q (72) */
		/* 0x8008: ldd Y+q (71) */
		/* 0x8200: std Z+q (120) */
		/* 0x8208: std Y+q (119) */
		d = (inst >> 4) & 0x1f;
		q = ((inst >> 13) & 0x1) << 5;
		q |= ((inst >> 10) & 0x3) << 3;
		q |= (inst >> 0) & 0x7;

		assert(/* 0 <= d && */ d < 32);
		assert(/* 0 <= q && */ q < 64);

		if ((inst >> 3) & 1) {
			/* Y */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Y));
		} else {
			/* Z */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
		}
		jit_assign(func, jaddr,
				jit_add(func, jaddr, UCONST(q)));

		if ((inst >> 9) & 1) {
			/* std */
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
		} else {
			/* ldd */
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
		}

	} else if ((inst & 0xfe0f) == 0x9400		/* com (48) */
		|| (inst & 0xfe0f) == 0x9401		/* neg (84) */
		|| (inst & 0xfe0f) == 0x9402		/* swap (125) */
		|| (inst & 0xfe0f) == 0x9403		/* inc (65) */
		|| (inst & 0xfe0f) == 0x9405		/* asr (10) */
		|| (inst & 0xfe0f) == 0x9406		/* lsr (78) */
		|| (inst & 0xfe0f) == 0x9407		/* ror (96) */
		|| (inst & 0xfe0f) == 0x940a) {		/* dec (53) */
		d = (inst >> 4) & 0x1f;

		NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));

		switch (inst & 0xfe0f) {
		case 0x9400:
			NAME_(compile_com)(func, jres, jcpssp, jop1);
			break;
		case 0x9401:
			NAME_(compile_neg)(func, jres, jcpssp, jop1);
			break;
		case 0x9402:
			NAME_(compile_swap)(func, jres, jcpssp, jop1);
			break;
		case 0x9403:
			NAME_(compile_inc)(func, jres, jcpssp, jop1);
			break;
		case 0x9405:
			NAME_(compile_asr)(func, jres, jcpssp, jop1);
			break;
		case 0x9406:
			NAME_(compile_lsr)(func, jres, jcpssp, jop1);
			break;
		case 0x9407:
			NAME_(compile_ror)(func, jres, jcpssp, jop1);
			break;
		case 0x940a:
			NAME_(compile_dec)(func, jres, jcpssp, jop1);
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);

	} else if ((inst & 0xffff) == 0x95a8) {		/* wdr (127) */
		/* Ignore... */

	} else if ((inst & 0xfe0f) == 0x900f		/* pop (89) */
		|| (inst & 0xfe0f) == 0x920f) {		/* push (90) */
		d = (inst >> 4) & 0x1f;

		if ((inst >> 9) & 1) {
			/* push */
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_push_res)(func, jcpssp, jres);
		} else {
			/* pop */
			NAME_(compile_pop_op1)(func, jop1, jcpssp);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
		}

	} else if ((inst & 0xfe0f) == 0x9001		/* ld Z+ (72) */
		|| (inst & 0xfe0f) == 0x9002		/* ld -Z (72) */
		|| (inst & 0xfe0f) == 0x9004		/* lpm Z (76) */
		|| (inst & 0xfe0f) == 0x9005		/* lpm Z+ (76) */
		|| (inst & 0xfe0f) == 0x9009		/* ld Y+ (71) */
		|| (inst & 0xfe0f) == 0x900a		/* ld -Y (71) */
		|| (inst & 0xfe0f) == 0x900c		/* ld X (70) */
		|| (inst & 0xfe0f) == 0x900d		/* ld X+ (70) */
		|| (inst & 0xfe0f) == 0x9201		/* st Z+ (120) */
		|| (inst & 0xfe0f) == 0x9202		/* st -Z (120) */
		|| (inst & 0xfe0f) == 0x9209		/* st Y+ (119) */
		|| (inst & 0xfe0f) == 0x920a		/* st -Y (119) */
		|| (inst & 0xfe0f) == 0x920c		/* st X (118) */
		|| (inst & 0xfe0f) == 0x920d		/* st X+ (118) */
		|| (inst & 0xfe0f) == 0x920e) {		/* st -X (118) */
		d = (inst >> 4) & 0x1f;

		switch (inst & 0xfe0f) {
		case 0x9001:
			/* ld Z+ */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			jit_assign(func, jaddr,
					jit_add(func, jaddr, UCONST(1)));
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Z), jaddr);
			break;
		case 0x9002:
			/* ld -Z */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
			jit_assign(func, jaddr,
					jit_sub(func, jaddr, UCONST(1)));
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Z), jaddr);
			break;
		case 0x9004:
			/* lpm Z */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
			NAME_(compile_lpm_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			break;
		case 0x9005:
			/* lpm Z+ */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
			NAME_(compile_lpm_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			jit_assign(func, jaddr,
					jit_add(func, jaddr, UCONST(1)));
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Z), jaddr);
			break;
		case 0x9009:
			/* ld Y+ */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Y));
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			jit_assign(func, jaddr,
					jit_add(func, jaddr, UCONST(1)));
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Y), jaddr);
			break;
		case 0x900a:
			/* ld -Y */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Y));
			jit_assign(func, jaddr,
					jit_sub(func, jaddr, UCONST(1)));
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Y), jaddr);
			break;
		case 0x900c:
			/* ld X */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_X));
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			break;
		case 0x900d:
			/* ld X+ */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_X));
			NAME_(compile_ld_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
			jit_assign(func, jaddr,
					jit_add(func, jaddr, UCONST(1)));
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_X), jaddr);
			break;
		case 0x9201:
			/* st Z+ */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
			jit_assign(func, jaddr,
					jit_add(func, jaddr, UCONST(1)));
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Z), jaddr);
			break;
		case 0x9202:
			/* st -Z */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Z));
			jit_assign(func, jaddr,
					jit_sub(func, jaddr, UCONST(1)));
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Z), jaddr);
			break;
		case 0x9209:
			/* st Y+ */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Y));
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
			jit_assign(func, jaddr,
					jit_add(func, jaddr, UCONST(1)));
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Y), jaddr);
			break;
		case 0x920a:
			/* st -Y */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_Y));
			jit_assign(func, jaddr,
					jit_sub(func, jaddr, UCONST(1)));
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_Y), jaddr);
			break;
		case 0x920c:
			/* st X */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_X));
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
			break;
		case 0x920d:
			/* st X+ */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_X));
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
			jit_assign(func, jaddr,
					jit_add(func, jaddr, UCONST(1)));
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_X), jaddr);
			break;
		case 0x920e:
			/* st -X */
			NAME_(compile_reg16_get_addr)(func, jaddr, jcpssp, UCONST(NAME_REG_X));
			jit_assign(func, jaddr,
					jit_sub(func, jaddr, UCONST(1)));
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
			jit_assign(func, jres, jop1);
			NAME_(compile_st_res)(func, jcpssp, jaddr, jres);
			NAME_(compile_reg16_set_addr)(func, jcpssp, UCONST(NAME_REG_X), jaddr);
			break;
		default:
			assert(0);
		}

	} else if ((inst & 0xf800) == 0xb000		/* in (64) */
		|| (inst & 0xf800) == 0xb800) {		/* out (88) */
		r = (inst >> 4) & 0x1f;
		A = ((inst >> 9) & 0x3) << 4;
		A |= inst & 0xf;

		jit_assign(func, jaddr, UCONST(A + 0x20));
		if ((inst >> 11) & 1) {
			/* out */
			NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(r));
			jit_assign(func, jres, jop1);
			NAME_(compile_out_res)(func, jcpssp, jaddr, jres);
		} else {
			/* in */
			NAME_(compile_in_op1)(func, jop1, jcpssp, jaddr);
			jit_assign(func, jres, jop1);
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(r), jres);
		}

	} else if ((inst & 0xf000) == 0x3000		/* cpi (51) */
		|| (inst & 0xf000) == 0x4000		/* sbci (98) */
		|| (inst & 0xf000) == 0x5000		/* subi (124) */
		|| (inst & 0xf000) == 0x6000		/* ori (87) */
		|| (inst & 0xf000) == 0x7000		/* andi (9) */
		|| (inst & 0xf000) == 0xe000) {		/* ldi (73) */
		d = (inst >> 4) & 0x0f;
		d |= 0x10;
		K = ((inst >> 8) & 0xf) << 4;
		K |= (inst >> 0) & 0xf;

		NAME_(compile_reg8_get_op1)(func, jop1, jcpssp, UCONST(d));
		jit_assign(func, jop2, UCONST(K));

		switch (inst & 0xf000) {
		case 0x3000:
			/* cpi */
			NAME_(compile_sub)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x4000:
			/* sbci */
			NAME_(compile_sbc)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x5000:
			/* subi */
			NAME_(compile_sub)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x6000:
			/* ori */
			NAME_(compile_or)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0x7000:
			/* andi */
			NAME_(compile_and)(func, jres, jcpssp, jop1, jop2);
			break;
		case 0xe000:
			/* ldi */
			jit_assign(func, jres, jop2);
			break;
		default:
			assert(0);
		}

		if ((inst & 0xf000) != 0x3000) {
			NAME_(compile_reg8_set_res)(func, jcpssp, UCONST(d), jres);
		}

	} else {
		/* Unknown instruction. */
		fprintf(stderr, "Unknown instruction %04x\n", inst);
		assert(0); /* FIXME */
	}

	/* } */
	jit_label(func, label_endif);

	cnt++;

	if (! end) {
		goto again;
	}

	jit_assign(func, SATTR(NAME.cnt),
			jit_sub(func, SATTR(NAME.cnt), SCONST(cnt)));

	return jit_compile(func);
}

static void
NAME_(step)(struct cpssp *cpssp)
{
	void *func;

	cpssp->NAME.cnt++;

	if (unlikely(cpssp->NAME.irr
	  && cpssp->NAME.i
	  && ! cpssp->NAME.idelay
	  && ! cpssp->NAME.skip)) {
		/* Interrupt pending. */
		unsigned int irq;

		for (irq = 1; ; irq++) {
			assert(irq < 8 * sizeof(cpssp->NAME.irr));
			if ((cpssp->NAME.irr >> irq) & 1) {
				break;
			}
		}
		NAME_(irq_ack)(cpssp, irq);

		if (DEBUG_CONTROL_FLOW) {
			fprintf(stderr, "Irq %u at 0x%04x\n", irq, cpssp->NAME.pc << 1);
		}

		func = jit_cache_lookup(&cpssp->NAME.jit_cache, 1, irq);
		if (unlikely(! func)) {
			func = NAME_(compile_interrupt)(irq);
			jit_cache_add(&cpssp->NAME.jit_cache, 1, irq, func);
		}
		jit_exec(func, cpssp);

	} else if (unlikely(cpssp->NAME.hlt)) {
		/* Waiting... */

	} else if (unlikely(0 < cpssp->NAME.cnt)) {
		/* Normal instruction execution. */
		if (DEBUG_CONTROL_FLOW) {
			NAME_(dump)(cpssp);
		}

		func = jit_cache_lookup(&cpssp->NAME.jit_cache, 0, cpssp->NAME.pc);
		if (unlikely(! func)) {
			func = NAME_(compile)(cpssp);
			jit_cache_add(&cpssp->NAME.jit_cache, 0, cpssp->NAME.pc, func);
		}
		jit_exec(func, cpssp);
	}
}

static void
NAME_(irqN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: n=%d, val=%u\n", __FUNCTION__, n, val);
	}

	assert(val < 8 * sizeof(cpssp->NAME.irr));

	cpssp->NAME.irr &= ~((uint64_t) 1 << n);
	cpssp->NAME.irr |= (uint64_t) val << n;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int i;

	jit_comp_reset(&cpssp->NAME.jit_comp);
	jit_cache_reset(&cpssp->NAME.jit_cache);

	cpssp->NAME.skip = 0;
	cpssp->NAME.hlt = 0;
	cpssp->NAME.pc = 0x0000;
	cpssp->NAME.sp = 0x000;
	for (i = 0; i < 32; i++) {
		cpssp->NAME.reg[i] = 0x00;
	}
	cpssp->NAME.irr = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	jit_comp_create(&cpssp->NAME.jit_comp);
	jit_cache_create(&cpssp->NAME.jit_cache);
	cpssp->NAME.cnt = 0;
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
	jit_comp_destroy(&cpssp->NAME.jit_comp);
	jit_cache_destroy(&cpssp->NAME.jit_cache);
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
