// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "storage/browser/file_system/local_file_stream_writer.h"

#include <stdint.h>

#include <memory>
#include <string>

#include "base/bind.h"
#include "base/callback_helpers.h"
#include "base/files/file_util.h"
#include "base/files/scoped_temp_dir.h"
#include "base/memory/ptr_util.h"
#include "base/run_loop.h"
#include "net/base/io_buffer.h"
#include "net/base/net_errors.h"
#include "net/base/test_completion_callback.h"
#include "storage/browser/file_system/file_stream_test_utils.h"
#include "storage/browser/file_system/file_stream_writer_test.h"

namespace storage {

class LocalFileStreamWriterTest : public FileStreamWriterTest {
 public:
  LocalFileStreamWriterTest() : file_thread_("TestFileThread") {}

  void SetUp() override {
    ASSERT_TRUE(file_thread_.Start());
    ASSERT_TRUE(file_system_directory_.CreateUniqueTempDir());
  }

  void TearDown() override {
    // Give another chance for deleted streams to perform Close.
    base::RunLoop().RunUntilIdle();
    file_thread_.Stop();
    base::RunLoop().RunUntilIdle();
    ASSERT_TRUE(file_system_directory_.Delete());
  }

  base::FilePath Path(const std::string& name) {
    return file_system_directory_.GetPath().AppendASCII(name);
  }

 protected:
  bool CreateFileWithContent(const std::string& name,
                             const std::string& data) override {
    return base::WriteFile(Path(name), data);
  }

  std::unique_ptr<FileStreamWriter> CreateWriter(const std::string& name,
                                                 int64_t offset) override {
    return base::WrapUnique(
        new LocalFileStreamWriter(file_task_runner(), Path(name), offset,
                                  FileStreamWriter::OPEN_EXISTING_FILE));
  }

  bool FilePathExists(const std::string& name) override {
    return base::PathExists(Path(name));
  }

  std::string GetFileContent(const std::string& name) override {
    std::string content;
    base::ReadFileToString(Path(name), &content);
    return content;
  }

  base::SingleThreadTaskRunner* file_task_runner() const {
    return file_thread_.task_runner().get();
  }

 private:
  base::ScopedTempDir file_system_directory_;
  base::Thread file_thread_;
};

INSTANTIATE_TYPED_TEST_SUITE_P(Local,
                               FileStreamWriterTypedTest,
                               LocalFileStreamWriterTest);

}  // namespace storage
