// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef EXTENSIONS_BROWSER_VERIFIED_CONTENTS_H_
#define EXTENSIONS_BROWSER_VERIFIED_CONTENTS_H_

#include <stdint.h>

#include <map>
#include <memory>
#include <string>
#include <vector>

#include "base/containers/span.h"
#include "base/files/file_path.h"
#include "base/macros.h"
#include "base/version.h"
#include "extensions/browser/content_verifier/content_verifier_utils.h"

namespace extensions {

using CanonicalRelativePath = content_verifier_utils::CanonicalRelativePath;

// This class encapsulates the data in a "verified_contents.json" file
// generated by the webstore for a .crx file. That data includes a set of
// signed expected hashes of file content which can be used to check for
// corruption of extension files on local disk.
class VerifiedContents {
 public:
  ~VerifiedContents();

  // Returns verified contents after successfully parsing verified_contents.json
  // file at |path| and validating the enclosed signature. Returns nullptr on
  // failure.
  // Note: |public_key| must remain valid for the lifetime of the returned
  // object.
  static std::unique_ptr<VerifiedContents> CreateFromFile(
      base::span<const uint8_t> public_key,
      const base::FilePath& path);

  // Returns verified contents after successfully parsing |contents| and
  // validating the enclosed signature. Returns nullptr on failure. Note:
  // |public_key| must remain valid for the lifetime of the returned object.
  static std::unique_ptr<VerifiedContents> Create(
      base::span<const uint8_t> public_key,
      base::StringPiece contents);

  int block_size() const { return block_size_; }
  const std::string& extension_id() const { return extension_id_; }
  const base::Version& version() const { return version_; }

  bool HasTreeHashRoot(const base::FilePath& relative_path) const;

  bool TreeHashRootEquals(const base::FilePath& relative_path,
                          const std::string& expected) const;

  bool TreeHashRootEqualsForCanonicalPath(
      const CanonicalRelativePath& canonical_relative_path,
      const std::string& expected) const;

  // If InitFrom has not been called yet, or was used in "ignore invalid
  // signature" mode, this can return false.
  bool valid_signature() { return valid_signature_; }

 private:
  // Note: the public_key must remain valid for the lifetime of this object.
  explicit VerifiedContents(base::span<const uint8_t> public_key);

  // Returns the base64url-decoded "payload" field from the |contents|, if
  // the signature was valid.
  bool GetPayload(base::StringPiece contents, std::string* payload);

  // The |protected_value| and |payload| arguments should be base64url encoded
  // strings, and |signature_bytes| should be a byte array. See comments in the
  // .cc file on GetPayload for where these come from in the overall input
  // file.
  bool VerifySignature(const std::string& protected_value,
                       const std::string& payload,
                       const std::string& signature_bytes);

  // The public key we should use for signature verification.
  base::span<const uint8_t> public_key_;

  // Indicates whether the signature was successfully validated or not.
  bool valid_signature_;

  // The block size used for computing the treehash root hashes.
  int block_size_;

  // Information about which extension these signed hashes are for.
  std::string extension_id_;
  base::Version version_;

  // The expected treehash root hashes for each file.
  // For case-sensitive systems (linux/chromeos) the key is exact cased, but for
  // case-insensitive systems (win/macos) the key is lower cased to support
  // case-insensitive lookups.
  //
  // We use a multi-map here so that we can do fast lookups of paths from
  // requests on case-insensitive systems (windows, mac) where the request path
  // might not have the exact right capitalization. Note that this doesn't
  // affect case-sensitive systems (linux, chromeos) as we use the exact cased
  // keys.
  // TODO(crbug.com/29941) - we should give developers client-side warnings in
  // each of those cases, and have the webstore reject the cases they can
  // statically detect.
  typedef std::multimap<CanonicalRelativePath, std::string> RootHashes;
  RootHashes root_hashes_;

  DISALLOW_COPY_AND_ASSIGN(VerifiedContents);
};

}  // namespace extensions

#endif  // EXTENSIONS_BROWSER_VERIFIED_CONTENTS_H_
