// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_RENDERER_HOST_CODE_CACHE_HOST_IMPL_H_
#define CONTENT_BROWSER_RENDERER_HOST_CODE_CACHE_HOST_IMPL_H_

#include <string>

#include "base/memory/scoped_refptr.h"
#include "base/memory/weak_ptr.h"
#include "base/strings/string16.h"
#include "build/build_config.h"
#include "components/services/storage/public/mojom/cache_storage_control.mojom.h"
#include "content/common/content_export.h"
#include "mojo/public/cpp/base/big_buffer.h"
#include "mojo/public/cpp/bindings/pending_receiver.h"
#include "mojo/public/cpp/bindings/receiver.h"
#include "third_party/blink/public/mojom/cache_storage/cache_storage.mojom.h"
#include "third_party/blink/public/mojom/loader/code_cache.mojom.h"

class GURL;

namespace url {
class Origin;
}

namespace content {

class GeneratedCodeCache;
class GeneratedCodeCacheContext;
class RenderProcessHostImpl;

// The implementation of a CodeCacheHost, which stores and retrieves resource
// metadata, either bytecode or native code, generated by a renderer process.
// Instances of this class must be created and used on the UI thread.
class CONTENT_EXPORT CodeCacheHostImpl : public blink::mojom::CodeCacheHost {
 public:
  CodeCacheHostImpl(
      int render_process_id,
      RenderProcessHostImpl* render_process_host_impl,
      scoped_refptr<GeneratedCodeCacheContext> generated_code_cache_context,
      mojo::PendingReceiver<blink::mojom::CodeCacheHost> receiver);
  ~CodeCacheHostImpl() override;

  mojo::Receiver<blink::mojom::CodeCacheHost>& receiver() { return receiver_; }

  void SetCacheStorageControlForTesting(
      storage::mojom::CacheStorageControl* cache_storage_control);

 private:
  // blink::mojom::CodeCacheHost implementation.
  void DidGenerateCacheableMetadata(blink::mojom::CodeCacheType cache_type,
                                    const GURL& url,
                                    base::Time expected_response_time,
                                    mojo_base::BigBuffer data) override;
  void FetchCachedCode(blink::mojom::CodeCacheType cache_type,
                       const GURL& url,
                       FetchCachedCodeCallback) override;
  void ClearCodeCacheEntry(blink::mojom::CodeCacheType cache_type,
                           const GURL& url) override;
  void DidGenerateCacheableMetadataInCacheStorage(
      const GURL& url,
      base::Time expected_response_time,
      mojo_base::BigBuffer data,
      const url::Origin& cache_storage_origin,
      const std::string& cache_storage_cache_name) override;

  // Helpers.
  GeneratedCodeCache* GetCodeCache(blink::mojom::CodeCacheType cache_type);
  void OnReceiveCachedCode(FetchCachedCodeCallback callback,
                           const base::Time& response_time,
                           mojo_base::BigBuffer data);

  // Our render process host ID, used to bind to the correct render process.
  const int render_process_id_;

  // CodeCacheHostImpl is owned by this RenderProcessHostImpl.
  // This pointer may be null in tests.
  RenderProcessHostImpl* const render_process_host_impl_;

  // Used to override the CacheStorageControl from the RHPI as needed.
  storage::mojom::CacheStorageControl* cache_storage_control_for_testing_ =
      nullptr;

  scoped_refptr<GeneratedCodeCacheContext> generated_code_cache_context_;

  mojo::Receiver<blink::mojom::CodeCacheHost> receiver_;

  base::WeakPtrFactory<CodeCacheHostImpl> weak_ptr_factory_{this};

  DISALLOW_COPY_AND_ASSIGN(CodeCacheHostImpl);
};

}  // namespace content

#endif  // CONTENT_BROWSER_RENDERER_HOST_CODE_CACHE_HOST_IMPL_H_
