#include "adaptor.h"

#include <goodix/goodix.h>
#include <qdbusconnection.h>
#include <qdbuscontext.h>
#include <qmetaobject.h>
#include <qobjectdefs.h>

#include <QDir>
#include <QJsonDocument>
#include <QJsonObject>
#include <filesystem>
#include <memory>
#include <optional>

#include "goodixadaptor.h"

#define TEMPLATE "/var/lib/goodix/"

QDebug operator<<(QDebug d, const goodix::FpRequest &request)
{
    d << "{\n"
      << "    \"requestType\": " << request.requestType << "\n"
      << "    union: {\n"
      << "        \"RequestEnrollResult\": " << request.RequestEnrollResult
      << "\n"
      << "        \"RequestIdentifyResult\": " << request.RequestIdentifyResult
      << "\n"
      << "    },\n"
      << "    union: {\n"
      << "        \"enrollingProgress\": " << request.enrollingProgress << "\n"
      << "        \"matchIndex\": " << request.matchIndex << "\n"
      << "    },\n"
      << "    \"matchedAccountinfo\": "
      << QByteArray::fromRawData((const char *) request.matchedAccountinfo,
                                 sizeof(request.matchedAccountinfo))
      << "\n}";

    return d;
}

static goodix::FpRequest copyFpRequest(goodix::FpRequest *request)
{
    goodix::FpRequest result;

    result.requestType           = request->requestType;
    result.RequestEnrollResult   = request->RequestEnrollResult;
    result.RequestIdentifyResult = request->RequestIdentifyResult;
    result.enrollingProgress     = request->enrollingProgress;
    result.matchIndex            = request->matchIndex;
    strncpy((char *) result.matchedAccountinfo,
            (char *) request->matchedAccountinfo,
            sizeof(request->matchedAccountinfo));

    return result;
}

namespace Enroll {
enum Status {
    COMPILED     = 0,
    FAILED       = 1,
    PASS         = 2,
    RETRY        = 3,
    DISCONNECTED = 4,
};
enum Failed {
    UNKNOW       = 0,
    INTERRUPTION = 1,
    REPLICATED   = 2,
    DATA_FULL    = 3,
};
enum Retry {
    TOUCH_SHORT         = 1,
    GRAPHICS_INVALID    = 2,
    HIGH_REPLICATED     = 3,
    FINGER_EXIST        = 4,
    SWIPE_TOO_SHORT     = 5,
    FINGER_NOT_CENTERED = 6,
    REMOVE_AND_RETRY    = 7,
};
};  // namespace Enroll
namespace Verify {
enum Status {
    MATCH        = 0,
    NOT_MATCH    = 1,
    ERROR        = 2,
    RETRY        = 3,
    DISCONNECTED = 4,
};
enum Failed {
    UNKNOWN_ERROR = 1,
    UNAVAILABLE   = 2,
};
enum Retry {
    SWIPE_TOO_SHORT     = 1,
    FINGER_NOT_CENTERED = 2,
    REMOVE_AND_RETRY    = 3,
    TOUCH_SHORT         = 4,
    QUALITY_BAD         = 5,
};
}  // namespace Verify

struct EnrollStatus {
    uint8_t subcode;
    uint8_t progress;
};

struct VerifyStatus {
    uint8_t subcode;
};

QJsonObject EnrollStatusJson(const EnrollStatus &status)
{
    QJsonObject obj;
    obj["subcode"]  = status.subcode;
    obj["progress"] = status.progress;
    return obj;
}

QJsonObject VerifyStatusJson(const VerifyStatus &status)
{
    QJsonObject obj;
    obj["subcode"] = status.subcode;
    return obj;
}

using FpRequestPtr = std::shared_ptr<goodix::FpRequest>;

enum class GoodixType {
    Verify,
    Enroll,
};

Q_DECLARE_METATYPE(GoodixType)

class GoodixInstance : public QObject {
    Q_OBJECT
    GoodixInstance()
    {
        qRegisterMetaType<GoodixType>("GoodixType");
        qRegisterMetaType<FpRequestPtr>("FpRequestPtr");
    }
    ~GoodixInstance() {}
    std::optional<QString> id;
    std::optional<bool>    claimed;
    FpRequestPtr           verifyRequest = nullptr;
    FpRequestPtr           enrollRequest = nullptr;
    AdaptorPrivate        *m_private     = nullptr;

public:
    static GoodixInstance &Instance()
    {
        static GoodixInstance Instance;
        return Instance;
    }

    void setFpRequest(FpRequestPtr request)
    {
        enrollRequest = request;
        emit requestChanged(GoodixType::Enroll, request);
    }

    void setVerifyRequest(FpRequestPtr request)
    {
        verifyRequest = request;
        emit requestChanged(GoodixType::Verify, request);
    }

    std::optional<QString> Id() const
    {
        return id;
    }

    bool lock(const QString &id)
    {
        if (this->id.has_value() && this->id == id) {
            return true;
        }

        if (!this->claimed.has_value()) {
            this->id      = id;
            this->claimed = true;

            return true;
        }

        return false;
    }

    bool unlock(const QString &id, bool really = false)
    {
        if (!this->claimed.has_value() || !this->id.has_value() ||
            this->id != id) {
            return false;
        }

        if (really) {
            this->id.reset();
            this->claimed.reset();
        }

        return true;
    }

    std::optional<bool> Claimed() const
    {
        return claimed;
    }

    FpRequestPtr EnrollRequest() const
    {
        return enrollRequest;
    }

    FpRequestPtr VerifyRequest() const
    {
        return verifyRequest;
    }

Q_SIGNALS:
    void requestChanged(GoodixType type, FpRequestPtr request);

public:
    QString                   enrollFinger;
    QString                   verifyFinger;
    const goodix::FpInitValue fpInitValue{ TEMPLATE };
};

static void FpAsyncEnrollFinger(goodix::FpRequest *request)
{
    FpRequestPtr ptr =
        std::make_shared<goodix::FpRequest>(copyFpRequest(request));
    GoodixInstance::Instance().setFpRequest(ptr);
}

static void FpAsyncIdentifyFinger(goodix::FpRequest *request)
{
    FpRequestPtr ptr =
        std::make_shared<goodix::FpRequest>(copyFpRequest(request));
    GoodixInstance::Instance().setVerifyRequest(ptr);
}

class AdaptorPrivate {
    Q_DECLARE_PUBLIC(Adaptor)
    Adaptor *const                q_ptr;
    QScopedPointer<GoodixAdaptor> adaptor;

public:
    AdaptorPrivate(Adaptor *parent)
        : q_ptr(parent), adaptor(new GoodixAdaptor(q_ptr))
    {
        init();

        QObject::connect(q_ptr, &Adaptor::VerifyStatus, adaptor.data(),
                         &GoodixAdaptor::VerifyStatus);
        QObject::connect(q_ptr, &Adaptor::EnrollStatus, adaptor.data(),
                         &GoodixAdaptor::EnrollStatus);
        q_ptr->connect(
            &GoodixInstance::Instance(), &GoodixInstance::requestChanged, q_ptr,
            [=](GoodixType type, const FpRequestPtr &request) {
                qInfo() << "----- begin -----";
                qDebug() << *request;
                auto id = GoodixInstance::Instance().Id();
                switch (type) {
                    case GoodixType::Verify: {
                        qInfo() << "[Type] Verify";
                        QJsonObject    subcode;
                        Verify::Status code;
                        switch (request->RequestIdentifyResult) {
                            case goodix::FpIdentifyResult::FP_IDENTIFY_MATCH:
			    case goodix::FpIdentifyResult::FP_IDENTIFY_MATCH_TEMPLATE_UPDATED:
                                code = Verify::Status::MATCH;
                                break;
                            case goodix::FpIdentifyResult::FP_IDENTIFY_NO_MATCH:
                                code    = Verify::Status::NOT_MATCH;
                                subcode = VerifyStatusJson(
                                    { .subcode =
                                          Verify::Retry::FINGER_NOT_CENTERED });
                                break;
                            default:
                                code    = Verify::Status::RETRY;
                                subcode = VerifyStatusJson(
                                    { .subcode = Verify::Retry::TOUCH_SHORT });
                                verifyAgain();
                        }
                        emit q_ptr->VerifyStatus(
                            id.value(), code, QJsonDocument(subcode).toJson());
                    } break;
                    case GoodixType::Enroll: {
                        qInfo() << "[Type] Enroll";
                        QJsonObject subcode;
                        int         code = -1;
                        switch (request->RequestEnrollResult) {
                            case goodix::FpEnrollResult::FP_ENROLL_FAIL:
                                code    = Enroll::Status::FAILED;
                                subcode = EnrollStatusJson(
                                    { .subcode = Enroll::Failed::UNKNOW });
                                break;
                            case goodix::FpEnrollResult::FP_ENROLL_PASS:
                                code    = Enroll::Status::PASS;
                                subcode = EnrollStatusJson(
                                    { .progress = (uint8_t) request
                                                      ->enrollingProgress });
                                enrollAgain();
                                break;
                            case goodix::FpEnrollResult::FP_ENROLL_POOR_QUALITY:
                                code    = Enroll::Status::RETRY;
                                subcode = EnrollStatusJson(
                                    { .subcode =
                                          Enroll::Retry::GRAPHICS_INVALID });
                                enrollAgain();
                                break;
                            case goodix::FpEnrollResult::FP_ENROLL_COMPLETE:
                                qInfo() << "[Enroll]: complete.";
                                code = Enroll::Status::COMPILED;
                                store();
                                break;
                            default:
                                //TODO: 假设重新识别的错误都是接触过短
                                code    = Enroll::Status::RETRY;
                                subcode = EnrollStatusJson(
                                    { .subcode =
                                          Enroll::Retry::SWIPE_TOO_SHORT });
                                enrollAgain();
                        }
                        emit q_ptr->EnrollStatus(
                            id.value(), code, QJsonDocument(subcode).toJson());
                    } break;
                }
            });
    }

    ~AdaptorPrivate() {}

    void handleDBus()
    {
        const QMetaObject   *obj{ adaptor->metaObject() };
        const QMetaClassInfo info{ obj->classInfo(
            obj->indexOfClassInfo("D-Bus Interface")) };
        QDBusConnection::systemBus().registerService(info.value());
        QDBusConnection::systemBus().registerObject("/finger", q_ptr);
    }

    bool lock()
    {
        if (!GoodixInstance::Instance().Claimed().has_value()) {
            return false;
        }

        return GoodixInstance::Instance().Claimed().value();
    }

    void init() {
        qInfo() << "[FpInit] "
        << goodix::FpInit(GoodixInstance::Instance().fpInitValue);
    }

    void exit() {
        qInfo() << "[FpExit]";
        goodix::FpExit();
    }

    void store()
    {
        auto storeDir = QString("%1/%2").arg(TEMPLATE).arg(
            GoodixInstance::Instance().Id().value());

        if (!std::filesystem::is_directory(storeDir.toStdString()) &&
            !std::filesystem::create_directory(storeDir.toStdString())) {
            qWarning() << "[FpStoreTemplate] create " + storeDir + " failed!";
            return;
        }

        qInfo() << "[FpStoreTemplate] "
                << goodix::FpStoreTemplate(
                       (char *) QString("%1/%2/%3")
                           .arg(TEMPLATE)
                           .arg(GoodixInstance::Instance().Id().value())
                           .arg(GoodixInstance::Instance().enrollFinger)
                           .toUtf8()
                           .data(),
                       (uint8_t *) GoodixInstance::Instance()
                           .Id()
                           ->toUtf8()
                           .data());
        goodix::FpRefreshTemplates();
    }

    void Claim(const QString &id, bool claimed)
    {
        if (claimed) {
            if (!GoodixInstance::Instance().lock(id)) {
                q_ptr->sendErrorReply(QDBusError::AccessDenied,
                                      "device is claimed.");
                return;
            }
            emit q_ptr->stateChanged();
            for (int i = 0; i <= 3; ++i) {
                int startSession = goodix::FpStartSession();
                qInfo() << "[FpStartSession] " << startSession;

                if (startSession != 0 && goodix::FpS3StartSession() != 0) {
                    qInfo() << "start session failed, restarting... "
                            << goodix::FpStopSession();
                    exit();
                    init();
                    continue;
                }

                unsigned char version[64];
                qInfo() << "[FpVersion] " << goodix::FpGetVersion(version);
                qInfo() << "[FpVersion] "
                        << QString::fromUtf8((char *) version);
                break;
            }
        }
        else {
            if (!GoodixInstance::Instance().unlock(id)) {
                q_ptr->sendErrorReply(QDBusError::AccessDenied,
                                      "device is claimed.");
                return;
            }
            if (GoodixInstance::Instance().unlock(id, true)) {
                qInfo() << "[FpStopSession]" << goodix::FpStopSession();
                emit q_ptr->stateChanged();
                return;
            }
        }
    }

    void DeleteAllFingers(const QString &id)
    {
        // TODO: check lock state

        auto split = [](const std::string &string,
                        const char         split) -> std::vector<std::string> {
            std::vector<std::string> result;
            std::string              tmp;

            for (const auto c : string) {
                if (c == split) {
                    result.push_back(tmp);
                    tmp = {};
                    continue;
                }
                tmp.push_back(c);
            }

            return result;
        };

        for (const auto &entry :
             std::filesystem::directory_iterator(TEMPLATE)) {
            auto filename = entry.path().string();
            auto result   = split(filename, '.');
            if (!result.empty() && result.front() != id.toStdString()) {
                if (!std::filesystem::remove(entry)) {
                    q_ptr->sendErrorReply(QDBusError::AccessDenied,
                                          "Delete all fingers failed.");
                }
            }
        }
        goodix::FpRefreshTemplates();
    }

    void DeleteFinger(const QString &id, const QString &finger)
    {
        // TODO: check lock state
        const QString &path{
            QString("%1/%2/%3").arg(TEMPLATE).arg(id).arg(finger)
        };

        if (!std::filesystem::remove(path.toStdString())) {
            q_ptr->sendErrorReply(QDBusError::AccessDenied,
                                  "Delete the finger failed.");
            return;
        }
        goodix::FpRefreshTemplates();
    }

    void Enroll(const QString &id, const QString &finger)
    {
        if (!GoodixInstance::Instance().lock(id)) {
            q_ptr->sendErrorReply(QDBusError::AccessDenied,
                                  "claimed id is different.");
            return;
        }

        //TODO: check finger

        GoodixInstance::Instance().enrollFinger = finger;
        enrollAgain();
    }

    void enrollAgain()
    {
        if (GoodixInstance::Instance().enrollFinger.isEmpty()) {
            return;
        }
        qInfo() << "[FpAsyncEnrollFinger] "
                << goodix::FpAsyncEnrollFinger(FpAsyncEnrollFinger);
    }

    QStringList ListFingers(const QString &id)
    {
        QDir dir(QString("%1/%2").arg(TEMPLATE).arg(id));
        return dir.entryList(QDir::NoDotAndDotDot | QDir::Files);
    }

    void RenameFinger(const QString &id,
                      const QString &finger,
                      const QString &newName)
    {
        // TODO: check lock state
        const QString &old(
            QString("%1/%2/%3").arg(TEMPLATE).arg(id).arg(finger));
        if (!QFile::exists(old)) {
            q_ptr->sendErrorReply(QDBusError::AccessDenied,
                                  "Finger is not exists.");
            return;
        }
        const QString &newFile(
            QString("%1/%2/%3").arg(TEMPLATE).arg(id).arg(newName));
        if (!QFile::rename(old, newFile)) {
            q_ptr->sendErrorReply(QDBusError::AccessDenied,
                                  "Rename finger failed.");
        }
        goodix::FpRefreshTemplates();
    }

    void StopEnroll()
    {
        goodix::FpScanCancel();
    }

    void StopVerify()
    {
        goodix::FpScanCancel();
    }

    void Verify(const QString &finger)
    {
        GoodixInstance::Instance().verifyFinger = finger;
        verifyAgain();
    }

    void verifyAgain()
    {
        goodix::FpAsyncIdentifyFingerWithoutAccountinfo(FpAsyncIdentifyFinger);
    }

    int capability() const
    {
        return 0;
    }

    QString name() const
    {
        return "Goodix";
    }

    /*
    * enum {
    *     DeviceStateNormal  = 1 << 0,  // 设备正常可用 D
    *    DeviceStateClaimed = 1 << 1,  // 设备被独占
    * };
    */
    int state() const
    {
        enum Result {
            DeviceStateNormal  = 1 << 0,
            DeviceStateClaimed = 1 << 1,
        };

        if (GoodixInstance::Instance().Id().has_value() &&
            GoodixInstance::Instance().Claimed().has_value()) {
            return GoodixInstance::Instance().Claimed()
                       ? Result::DeviceStateClaimed
                       : Result::DeviceStateNormal;
        }

        return Result::DeviceStateNormal;
    }

    /*
     * enum {
     *     DeviceTypeUnknown = 0,
     *     DeviceTypeScanning,
     *     DeviceTypeTouch,
     * };
     */
    int type() const
    {
        return 2;
    }
};

Adaptor::Adaptor(QObject *parent)
    : QObject(parent), d_ptr(new AdaptorPrivate(this))
{
    d_ptr->handleDBus();
}

Adaptor::~Adaptor() {}

void Adaptor::Claim(const QString &id, bool claimed)
{
    Q_D(Adaptor);
    return d->Claim(id, claimed);
}

void Adaptor::DeleteAllFingers(const QString &id)
{
    Q_D(Adaptor);
    return d->DeleteAllFingers(id);
}

void Adaptor::DeleteFinger(const QString &id, const QString &finger)
{
    Q_D(Adaptor);
    return d->DeleteFinger(id, finger);
}

void Adaptor::Enroll(const QString &id, const QString &finger)
{
    Q_D(Adaptor);
    return d->Enroll(id, finger);
}

QStringList Adaptor::ListFingers(const QString &id)
{
    Q_D(Adaptor);
    return d->ListFingers(id);
}

void Adaptor::RenameFinger(const QString &id,
                           const QString &finger,
                           const QString &newName)
{
    Q_D(Adaptor);
    return d->RenameFinger(id, finger, newName);
}

void Adaptor::StopEnroll()
{
    Q_D(Adaptor);
    return d->StopEnroll();
}

void Adaptor::StopVerify()
{
    Q_D(Adaptor);
    return d->StopVerify();
}

void Adaptor::Verify(const QString &finger)
{
    Q_D(Adaptor);
    return d->Verify(finger);
}
int Adaptor::capability() const
{
    Q_D(const Adaptor);
    return d->capability();
}

QString Adaptor::name() const
{
    Q_D(const Adaptor);
    return d->name();
}

int Adaptor::state() const
{
    Q_D(const Adaptor);
    return d->state();
}

int Adaptor::type() const
{
    Q_D(const Adaptor);
    return d->type();
}

#include "adaptor.moc"
