/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <stdio.h>
#include <assert.h>
#include <stdbool.h>
#include <math.h>
#include <limits.h>

#include "glue.h"

#include "red_pitaya.h"

#define COMP_(x) red_pitaya_ ## x

#define INCLUDE
#include "arch_adc.c"
#include "arch_channel.c"
#include "arch_channel_c.c"
#include "arch_control_handler.c"
#include "arch_dac.c"
#include "arch_digital_channel.c"
#include "arch_event_handler.c"
#include "arch_graph_dumper.c"
#include "arch_potentiometer.c"
#include "arch_scpi_server.c"
#include "arch_trigger_system.c"
#undef INCLUDE

#define SCREEN_WIDTH	400
#define SCREEN_HEIGHT	800
#define SUB_SCREEN_HEIGHT 40
#define SUB_SCREEN_LEFT 40
#define GRID_STEP	20
#define CHANNEL_STEP	40
#define DIGITAL_CHANNEL_GRID_STEP	10
#define EVENT_BUFFER_SIZE	1000000
#define RLE_BUFFER_SIZE		5000000
#define NUMBER_ANALOG_CHANNELS 5
#define NUMBER_DIGITAL_CHANNELS 5

struct RLE_buffer_item {
	int16_t value;
	int8_t sample_count;
	unsigned long long time_stamp;
};

struct event_measure_point{
	unsigned long long time_stamp;
	bool hi; /* true means high, false means low */
};

struct cpssp {
	struct sig_opt_rgb *port_monitor;
	struct sig_integer *port_sample_rate_sim_gui;
	struct sig_integer *port_vert_offset_sim_gui;
	struct sig_integer *port_sec_div_sim_gui;
	struct sig_integer *port_volt_div_sim_gui;
	struct sig_integer *port_amp_div_sim_gui;
	struct sig_integer *port_trigger_channel_sim_gui;
	struct sig_integer *port_trigger_edge_sim_gui;
	struct sig_integer *port_trigger_level_sim_gui;
	unsigned long long start_time;
	unsigned long long end_time;
	bool running;
	unsigned long long sample_rate;
	int pixel_step;
	double pixel_step_d;
	int selected_channel;
	unsigned int call_after_ticks;
	unsigned long long number_samples_total;
	double sweep_time;
	unsigned long long draw_tick_counter;
	bool initial_start;
	double screen_start_time;
	unsigned int ms_div;
	int trigger_channel;
	int trigger_level;
	bool trigger_edge;
	bool trigger_armed;
	double ticks_per_sample;
	double sec_div;
	unsigned long long trigger_time_stamp;

	unsigned long long last_time_virt[NUMBER_ANALOG_CHANNELS];
	unsigned long long buffer_item_skipped_samples[NUMBER_ANALOG_CHANNELS];
	double scale_factor[NUMBER_ANALOG_CHANNELS];
	unsigned int first_values[NUMBER_ANALOG_CHANNELS];
	int vertical_offset[NUMBER_ANALOG_CHANNELS + NUMBER_DIGITAL_CHANNELS];
	int channel_values[NUMBER_ANALOG_CHANNELS];
	struct RLE_buffer_item RLE_buffer[NUMBER_ANALOG_CHANNELS][RLE_BUFFER_SIZE];
	int RLE_buffer_pos[NUMBER_ANALOG_CHANNELS];
	int RLE_buffer_read_pos[NUMBER_ANALOG_CHANNELS];
	unsigned int start_buffer_pos[NUMBER_ANALOG_CHANNELS];
	long long start_samples_skipped[NUMBER_ANALOG_CHANNELS];
	unsigned int last_on_screen_y[NUMBER_ANALOG_CHANNELS];
	int center_buffer_pos[NUMBER_ANALOG_CHANNELS];
	long long center_sample_pos[NUMBER_ANALOG_CHANNELS];
	double pixel_step_overflow[NUMBER_ANALOG_CHANNELS];
	int first_buffer_pos[NUMBER_ANALOG_CHANNELS];
	long long first_sample_pos[NUMBER_ANALOG_CHANNELS];
	int last_buffer_pos[NUMBER_ANALOG_CHANNELS];
	long long last_sample_pos[NUMBER_ANALOG_CHANNELS];
	int trigger_buffer_pos[NUMBER_ANALOG_CHANNELS];
	long long trigger_sample_pos[NUMBER_ANALOG_CHANNELS];
	int total_number_buffer_items[NUMBER_ANALOG_CHANNELS];
	long long center_time_stamp;
	long long center_pos_ticks;
	long long first_pos_ticks;
	long long last_pos_ticks;
	int first_event_buffer_pos[NUMBER_DIGITAL_CHANNELS];
	int last_event_buffer_pos[NUMBER_DIGITAL_CHANNELS];
	bool digital_channel_hi[NUMBER_DIGITAL_CHANNELS];
	unsigned int event_pointer[NUMBER_DIGITAL_CHANNELS];
	struct event_measure_point event_buffer[NUMBER_DIGITAL_CHANNELS][EVENT_BUFFER_SIZE];
	int registered_events[NUMBER_DIGITAL_CHANNELS];
	int total_number_events[NUMBER_DIGITAL_CHANNELS];
	unsigned long long timer_tick_count;

#define STATE

#define NAME		adc
#define NAME_(x)	adc_ ## x
#include "arch_adc.c"
#undef NAME_
#undef NAME

#define NAME		channel_1
#define NAME_(x)	channel_1_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_2
#define NAME_(x)	channel_2_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_3
#define NAME_(x)	channel_3_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_4
#define NAME_(x)	channel_4_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_5
#define NAME_(x)	channel_5_ ## x
#include "arch_channel_c.c"
#undef NAME_
#undef NAME

#if 0
#define NAME		channel_6
#define NAME_(x)	channel_6_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_7
#define NAME_(x)	channel_7_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_8
#define NAME_(x)	channel_8_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_9
#define NAME_(x)	channel_9_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_10
#define NAME_(x)	channel_10_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME
#endif

#define NAME		control_handler
#define NAME_(x)	control_handler_ ## x
#include "arch_control_handler.c"
#undef NAME_
#undef NAME

#define NAME		dac
#define NAME_(x)	dac_ ## x
#include "arch_dac.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_1
#define NAME_(x)	digital_channel_1_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_2
#define NAME_(x)	digital_channel_2_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_3
#define NAME_(x)	digital_channel_3_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_4
#define NAME_(x)	digital_channel_4_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_5
#define NAME_(x)	digital_channel_5_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#if 0
#define NAME		digital_channel_6
#define NAME_(x)	digital_channel_6_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_7
#define NAME_(x)	digital_channel_7_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_8
#define NAME_(x)	digital_channel_8_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_9
#define NAME_(x)	digital_channel_9_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_10
#define NAME_(x)	digital_channel_10_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME
#endif

#define NAME		event_handler
#define NAME_(x)	event_handler_ ## x
#include "arch_event_handler.c"
#undef NAME_
#undef NAME

#define NAME		graph_dumper
#define NAME_(x)	graph_dumper_ ## x
#include "arch_graph_dumper.c"
#undef NAME_
#undef NAME

#define NAME		scpi_server
#define NAME_(x)	scpi_server_ ## x
#include "arch_scpi_server.c"
#undef NAME_
#undef NAME

#define NAME		trigger_system
#define NAME_(x)	trigger_system_ ## x
#include "arch_trigger_system.c"
#undef NAME_
#undef NAME

#undef STATE
};

#define EXPORT

#define NAME		adc
#define NAME_(x)	adc_ ## x
#include "arch_adc.c"
#undef NAME_
#undef NAME

#define NAME		channel_1
#define NAME_(x)	channel_1_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_2
#define NAME_(x)	channel_2_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_3
#define NAME_(x)	channel_3_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_4
#define NAME_(x)	channel_4_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_5
#define NAME_(x)	channel_5_ ## x
#include "arch_channel_c.c"
#undef NAME_
#undef NAME

#define NAME		control_handler
#define NAME_(x)	control_handler_ ## x
#include "arch_control_handler.c"
#undef NAME_
#undef NAME

#define NAME		dac
#define NAME_(x)	dac_ ## x
#include "arch_dac.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_1
#define NAME_(x)	digital_channel_1_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_2
#define NAME_(x)	digital_channel_2_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_3
#define NAME_(x)	digital_channel_3_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_4
#define NAME_(x)	digital_channel_4_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_5
#define NAME_(x)	digital_channel_5_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#if 0
#define NAME		digital_channel_6
#define NAME_(x)	digital_channel_6_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_7
#define NAME_(x)	digital_channel_7_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_8
#define NAME_(x)	digital_channel_8_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_9
#define NAME_(x)	digital_channel_9_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_10
#define NAME_(x)	digital_channel_10_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME
#endif

#define NAME		event_handler
#define NAME_(x)	event_handler_ ## x
#include "arch_event_handler.c"
#undef NAME_
#undef NAME

#define NAME		graph_dumper
#define NAME_(x)	graph_dumper_ ## x
#include "arch_graph_dumper.c"
#undef NAME_
#undef NAME

#define NAME		scpi_server
#define NAME_(x)	scpi_server_ ## x
#include "arch_scpi_server.c"
#undef NAME_
#undef NAME

#define NAME		trigger_system
#define NAME_(x)	trigger_system_ ## x
#include "arch_trigger_system.c"
#undef NAME_
#undef NAME

#undef EXPORT

static bool
subtract_samples(
	struct cpssp *cpssp,
	int channel,
	long long samples_to_subtract,
	int start_buffer_pos,
	int *dest_buffer_pos,
	long long start_sample_pos,
	long long *dest_sample_pos,
	long long *remaining_samples
)
{
	return dac_subtract_samples(cpssp,
			channel,
			samples_to_subtract,
			start_buffer_pos,
			dest_buffer_pos,
			start_sample_pos,
			dest_sample_pos,
			remaining_samples);
}

static void
volt_div_set_sim(void *_cpssp, int val, int channel)
{
	control_handler_volt_div_set_sim(_cpssp, val, channel);
}

static void
amp_div_set_sim(void *_cpssp, int val, int channel)
{
	control_handler_amp_div_set_sim(_cpssp, val, channel);
}

static void
sample_rate_set_sim(void *_cpssp, int val)
{
	control_handler_sample_rate_set_sim(_cpssp, val);
}

static void
sec_div_set_sim(void *_cpssp, int val)
{
	control_handler_sec_div_set_sim(_cpssp, val);
}

static void
offset_set_sim(void *_cpssp, int channel, double value)
{
	control_handler_offset_set_sim(_cpssp, channel, value);
}

static void
register_event(struct cpssp *cpssp, int channel, bool rising_edge)
{
	event_handler_register_event(cpssp, channel, rising_edge);
}

static void
draw_pixels(struct cpssp *cpssp)
{
	dac_draw_pixels(cpssp);
}

static void
wipe_screen(struct cpssp *cpssp)
{
	dac_wipe_screen(cpssp);
}

static void
update_pixel_buffer(struct cpssp *cpssp)
{
	dac_update_pixel_buffer(cpssp);
}

static void
export_data(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;
	int mV;

	mV = SIG_mV(val);
	if (0 < mV) {
		graph_dumper_dump_data(cpssp);
	}
}

static void
register_trigger(void *_cpssp, int channel)
{
	trigger_system_register_trigger(_cpssp, channel);
}

static void
trigger_channel_set_sim(void *_cpssp, int channel)
{
	control_handler_trigger_channel_set_sim(_cpssp, channel);
}

static void
trigger_edge_set_sim(void *_cpssp, int edge)
{
	control_handler_trigger_edge_set_sim(_cpssp, edge);
}

static void
trigger_level_set_sim(void *_cpssp, int level)
{
	control_handler_trigger_level_set_sim(_cpssp, level);
}

static char
get_channel_type(void *_cpssp, int channel)
{
	struct cpssp *cpssp = _cpssp;

	if (channel == 0) {
		return channel_1_get_channel_type(cpssp);
	} else if (channel == 1) {
		return channel_2_get_channel_type(cpssp);
	} else if (channel == 2) {
		return channel_3_get_channel_type(cpssp);
	} else if (channel == 3) {
		return channel_4_get_channel_type(cpssp);
	} else if (channel == 4) {
		return channel_5_get_channel_type(cpssp);
	}
	return 'f';
}

#define BEHAVIOR

#define NAME		adc
#define NAME_(x)	adc_ ## x
#include "arch_adc.c"
#undef NAME_
#undef NAME

#define NAME		channel_1
#define NAME_(x)	channel_1_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_2
#define NAME_(x)	channel_2_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_3
#define NAME_(x)	channel_3_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_4
#define NAME_(x)	channel_4_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_5
#define NAME_(x)	channel_5_ ## x
#include "arch_channel_c.c"
#undef NAME_
#undef NAME
#if 0

#define NAME		channel_6
#define NAME_(x)	channel_6_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_7
#define NAME_(x)	channel_7_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_8
#define NAME_(x)	channel_8_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_9
#define NAME_(x)	channel_9_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME

#define NAME		channel_10
#define NAME_(x)	channel_10_ ## x
#include "arch_channel.c"
#undef NAME_
#undef NAME
#endif

#define NAME		control_handler
#define NAME_(x)	control_handler_ ## x
#include "arch_control_handler.c"
#undef NAME_
#undef NAME

#define NAME		dac
#define NAME_(x)	dac_ ## x
#include "arch_dac.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_1
#define NAME_(x)	digital_channel_1_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_2
#define NAME_(x)	digital_channel_2_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_3
#define NAME_(x)	digital_channel_3_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_4
#define NAME_(x)	digital_channel_4_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_5
#define NAME_(x)	digital_channel_5_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#if 0
#define NAME		digital_channel_6
#define NAME_(x)	digital_channel_6_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_7
#define NAME_(x)	digital_channel_7_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_8
#define NAME_(x)	digital_channel_8_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_9
#define NAME_(x)	digital_channel_9_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME

#define NAME		digital_channel_10
#define NAME_(x)	digital_channel_10_ ## x
#include "arch_digital_channel.c"
#undef NAME_
#undef NAME
#endif

#define NAME		event_handler
#define NAME_(x)	event_handler_ ## x
#include "arch_event_handler.c"
#undef NAME_
#undef NAME

#define NAME		graph_dumper
#define NAME_(x)	graph_dumper_ ## x
#include "arch_graph_dumper.c"
#undef NAME_
#undef NAME

#define NAME		scpi_server
#define NAME_(x)	scpi_server_ ## x
#include "arch_scpi_server.c"
#undef NAME_
#undef NAME

#define NAME		trigger_system
#define NAME_(x)	trigger_system_ ## x
#include "arch_trigger_system.c"
#undef NAME_
#undef NAME

#undef BEHAVIOR

void *
COMP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_voltage_1,
	struct sig_std_logic *port_voltage_2,
	struct sig_std_logic *port_voltage_3,
	struct sig_std_logic *port_voltage_4,
	struct sig_std_logic *port_voltage_5,
	struct sig_std_logic *port_voltage_6,
	struct sig_std_logic *port_voltage_7,
	struct sig_std_logic *port_voltage_8,
	struct sig_std_logic *port_voltage_9,
	struct sig_std_logic *port_voltage_10,
	struct sig_std_logic *port_digi_1,
	struct sig_std_logic *port_digi_2,
	struct sig_std_logic *port_digi_3,
	struct sig_std_logic *port_digi_4,
	struct sig_std_logic *port_digi_5,
	struct sig_std_logic *port_digi_6,
	struct sig_std_logic *port_digi_7,
	struct sig_std_logic *port_digi_8,
	struct sig_std_logic *port_digi_9,
	struct sig_std_logic *port_digi_10,
	struct sig_integer *port_sample_rate_gui_sim,
	struct sig_integer *port_sample_rate_sim_gui,
	struct sig_integer *port_sec_div_gui_sim,
	struct sig_integer *port_sec_div_sim_gui,
	struct sig_integer *port_volt_div_gui_sim,
	struct sig_integer *port_volt_div_sim_gui,
	struct sig_integer *port_channel_select,
	struct sig_integer *port_trigger_sim_gui,
	struct sig_integer *port_trigger_gui_sim,
	struct sig_integer *port_vert_offset_sim_gui,
	struct sig_integer *port_vert_offset_gui_sim,
	struct sig_opt_rgb *port_monitor,
	struct sig_std_logic *port_vcc,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_start,
	struct sig_std_logic *port_stop,
	struct sig_std_logic *port_scroll_left,
	struct sig_std_logic *port_scroll_right,
	struct sig_std_logic *port_export,
	struct sig_integer *port_trigger_channel_gui_sim,
	struct sig_integer *port_trigger_channel_sim_gui,
	struct sig_integer *port_trigger_edge_gui_sim,
	struct sig_integer *port_trigger_edge_sim_gui,
	struct sig_integer *port_trigger_level_gui_sim,
	struct sig_integer *port_trigger_level_sim_gui,
	struct sig_integer *port_amp_div_gui_sim,
	struct sig_integer *port_amp_div_sim_gui
)
{
	static const struct sig_std_logic_funcs channel_1_voltage_func = {
		.std_logic_set = channel_1_voltage_changed,
	};
	static const struct sig_std_logic_funcs channel_2_voltage_func = {
		.std_logic_set = channel_2_voltage_changed,
	};
	static const struct sig_std_logic_funcs channel_3_voltage_func = {
		.std_logic_set = channel_3_voltage_changed,
	};
	static const struct sig_std_logic_funcs channel_4_voltage_func = {
		.std_logic_set = channel_4_voltage_changed,
	};
	static const struct sig_std_logic_funcs channel_5_voltage_func = {
		.std_logic_set = channel_5_current_changed,
	};
#if 0
	static const struct sig_std_logic_funcs channel_6_voltage_func = {
		.std_logic_set = channel_6_current_changed,
	};
	static const struct sig_std_logic_funcs channel_7_voltage_func = {
		.std_logic_set = channel_7_voltage_changed,
	};
	static const struct sig_std_logic_funcs channel_8_voltage_func = {
		.std_logic_set = channel_8_voltage_changed,
	};
	static const struct sig_std_logic_funcs channel_9_voltage_func = {
		.std_logic_set = channel_9_voltage_changed,
	};
	static const struct sig_std_logic_funcs channel_10_voltage_func = {
		.std_logic_set = channel_10_voltage_changed,
	};
#endif
	static const struct sig_std_logic_funcs digital_channel_1_func = {
		.std_logic_set = digital_channel_1_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_2_func = {
		.std_logic_set = digital_channel_2_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_3_func = {
		.std_logic_set = digital_channel_3_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_4_func = {
		.std_logic_set = digital_channel_4_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_5_func = {
		.std_logic_set = digital_channel_5_event_triggered,
	};
#if 0
	static const struct sig_std_logic_funcs digital_channel_6_func = {
		.std_logic_set = digital_channel_6_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_7_func = {
		.std_logic_set = digital_channel_7_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_8_func = {
		.std_logic_set = digital_channel_8_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_9_func = {
		.std_logic_set = digital_channel_9_event_triggered,
	};
	static const struct sig_std_logic_funcs digital_channel_10_func = {
		.std_logic_set = digital_channel_10_event_triggered,
	};
#endif
	static const struct sig_integer_funcs sample_rate_gui_sim_func = {
		.set = control_handler_sample_rate_set_gui,
	};
	static const struct sig_integer_funcs sec_div_func = {
		.set =  control_handler_sec_div_set_gui,
	};
	static const struct sig_integer_funcs volt_div_func = {
		.set =  control_handler_volt_div_set_gui,
	};
	static const struct sig_integer_funcs amp_div_func = {
		.set =  control_handler_amp_div_set_gui,
	};
	static const struct sig_integer_funcs channel_select_func = {
		.set =  control_handler_channel_set_gui,
	};
	static const struct sig_integer_funcs vert_offset_gui_sim_func = {
		.set =  control_handler_offset_set_gui,
	};
	static const struct sig_std_logic_funcs start_func = {
		.std_logic_set = adc_start_acquisition,
	};
	static const struct sig_std_logic_funcs stop_func = {
		.std_logic_set = adc_stop_acquisition,
	};
	static const struct sig_std_logic_funcs scroll_left_func = {
		.std_logic_set = dac_scroll_left,
	};
	static const struct sig_std_logic_funcs scroll_right_func = {
		.std_logic_set = dac_scroll_right,
	};
	static const struct sig_std_logic_funcs export_func = {
		.std_logic_set = export_data,
	};
	static const struct sig_integer_funcs trigger_channel_gui_sim_func = {
		.set = control_handler_trigger_channel_set_gui,
	};
	static const struct sig_integer_funcs trigger_edge_gui_sim_func = {
		.set = control_handler_trigger_edge_set_gui,
	};
	static const struct sig_integer_funcs trigger_level_gui_sim_func = {
		.set = control_handler_trigger_level_set_gui,
	};
	struct cpssp *cpssp;
	struct RLE_buffer_item item;
	struct event_measure_point event_item;
	int i, j;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);
	cpssp->running = true;

	/*TODO refactor resetting channels*/
	for (i = 0; i < NUMBER_ANALOG_CHANNELS; i++) {
		cpssp->RLE_buffer_pos[i] = 0;
		cpssp->RLE_buffer_read_pos[i] = 0;
		cpssp->buffer_item_skipped_samples[i] = 0;
		cpssp->last_on_screen_y[i] = -1 ;
		cpssp->pixel_step_overflow[i] = 0.0;
		cpssp->first_buffer_pos[i] = -1;
		cpssp->first_sample_pos[i] = -1;
		for (j = 0; j < RLE_BUFFER_SIZE; j++) {
			item = cpssp->RLE_buffer[i][j];
			item.sample_count = -1;
			item.time_stamp = 0;
			cpssp->RLE_buffer[i][j] = item;
		}
	}
	for (i = 0; i < NUMBER_DIGITAL_CHANNELS; i++) {
		cpssp->digital_channel_hi[i] = false;
		cpssp->registered_events[i] = 0;
		cpssp->event_pointer[i] = 0;
		cpssp->first_event_buffer_pos[i] = 0;

		for (j = 0; j < EVENT_BUFFER_SIZE; j++) {
			event_item = cpssp->event_buffer[i][j];
			event_item.time_stamp = 0;
			cpssp->event_buffer[i][j] = event_item;
		}
	}
	/* create channels */
	channel_1_create(cpssp, 0);
	channel_2_create(cpssp, 1);
	channel_3_create(cpssp, 2);
	channel_4_create(cpssp, 3);
	channel_5_create(cpssp, 4);
#if 0
	channel_6_create(cpssp, 5);
	channel_7_create(cpssp, 6);
	channel_8_create(cpssp, 7);
	channel_9_create(cpssp, 8);
	channel_10_create(cpssp, 9);
#endif
	digital_channel_1_create(cpssp, 0);
	digital_channel_2_create(cpssp, 1);
	digital_channel_3_create(cpssp, 2);
	digital_channel_4_create(cpssp, 3);
	digital_channel_5_create(cpssp, 4);
#if 0
	digital_channel_6_create(cpssp, 5);
	digital_channel_7_create(cpssp, 6);
	digital_channel_8_create(cpssp, 7);
	digital_channel_9_create(cpssp, 8);
	digital_channel_10_create(cpssp, 9);
#endif
	cpssp->selected_channel = 0;
	cpssp->number_samples_total = 0;
	cpssp->screen_start_time = 0.0;
	cpssp->trigger_armed = true;
	/* connect analog and digital signals */
	sig_std_logic_connect_in(port_voltage_1, cpssp, &channel_1_voltage_func);
	sig_std_logic_connect_in(port_voltage_2, cpssp, &channel_2_voltage_func);
	sig_std_logic_connect_in(port_voltage_3, cpssp, &channel_3_voltage_func);
	sig_std_logic_connect_in(port_voltage_4, cpssp, &channel_4_voltage_func);
	sig_std_logic_connect_in(port_voltage_5, cpssp, &channel_5_voltage_func);
#if 0
	sig_std_logic_connect_in(port_voltage_6, cpssp, &channel_6_voltage_func);
	sig_std_logic_connect_in(port_voltage_7, cpssp, &channel_7_voltage_func);
	sig_std_logic_connect_in(port_voltage_8, cpssp, &channel_8_voltage_func);
	sig_std_logic_connect_in(port_voltage_9, cpssp, &channel_9_voltage_func);
	sig_std_logic_connect_in(port_voltage_10, cpssp, &channel_10_voltage_func);
#endif
	sig_std_logic_connect_in(port_digi_1, cpssp, &digital_channel_1_func);
	sig_std_logic_connect_in(port_digi_2, cpssp, &digital_channel_2_func);
	sig_std_logic_connect_in(port_digi_3, cpssp, &digital_channel_3_func);
	sig_std_logic_connect_in(port_digi_4, cpssp, &digital_channel_4_func);
	sig_std_logic_connect_in(port_digi_5, cpssp, &digital_channel_5_func);
#if 0
	sig_std_logic_connect_in(port_digi_6, cpssp, &digital_channel_6_func);
	sig_std_logic_connect_in(port_digi_7, cpssp, &digital_channel_7_func);
	sig_std_logic_connect_in(port_digi_8, cpssp, &digital_channel_8_func);
	sig_std_logic_connect_in(port_digi_9, cpssp, &digital_channel_9_func);
	sig_std_logic_connect_in(port_digi_10, cpssp, &digital_channel_10_func);
#endif
	sig_integer_connect_in(port_sec_div_gui_sim, cpssp, &sec_div_func);
	sig_integer_connect_in(port_volt_div_gui_sim, cpssp, &volt_div_func);
	sig_integer_connect_in(port_amp_div_gui_sim, cpssp, &amp_div_func);
	sig_integer_connect_in(port_vert_offset_gui_sim, cpssp, &vert_offset_gui_sim_func);
	sig_integer_connect_in(port_channel_select, cpssp, &channel_select_func);
	sig_integer_connect_in(port_sample_rate_gui_sim, cpssp, &sample_rate_gui_sim_func);
	sig_std_logic_connect_in(port_start, cpssp, &start_func);
	sig_std_logic_connect_in(port_stop, cpssp, &stop_func);
	sig_std_logic_connect_in(port_scroll_left, cpssp, &scroll_left_func);
	sig_std_logic_connect_in(port_scroll_right, cpssp, &scroll_right_func);
	sig_std_logic_connect_in(port_export, cpssp, &export_func);
	sig_integer_connect_in(port_trigger_channel_gui_sim, cpssp,
			&trigger_channel_gui_sim_func);
	sig_integer_connect_in(port_trigger_edge_gui_sim, cpssp,
			&trigger_edge_gui_sim_func);
	sig_integer_connect_in(port_trigger_level_gui_sim, cpssp,
			&trigger_level_gui_sim_func);

	sig_integer_connect_out(port_sample_rate_sim_gui, cpssp, 0);
	sig_integer_connect_out(port_vert_offset_sim_gui, cpssp, 0);
	sig_integer_connect_out(port_sec_div_sim_gui, cpssp, 0);
	sig_integer_connect_out(port_volt_div_sim_gui, cpssp, 0);
	sig_integer_connect_out(port_amp_div_sim_gui, cpssp, 0);
	sig_integer_connect_out(port_trigger_channel_sim_gui, cpssp, 0);
	sig_integer_connect_out(port_trigger_edge_sim_gui, cpssp, 0);
	sig_integer_connect_out(port_trigger_level_sim_gui, cpssp, 0);

	cpssp->port_monitor = port_monitor;
	cpssp->port_sample_rate_sim_gui = port_sample_rate_sim_gui;
	cpssp->port_vert_offset_sim_gui = port_vert_offset_sim_gui;
	cpssp->port_sec_div_sim_gui = port_sec_div_sim_gui;
	cpssp->port_volt_div_sim_gui = port_volt_div_sim_gui;
	cpssp->port_amp_div_sim_gui = port_amp_div_sim_gui;
	cpssp->port_trigger_channel_sim_gui = port_trigger_channel_sim_gui;
	cpssp->port_trigger_edge_sim_gui = port_trigger_edge_sim_gui;
	cpssp->port_trigger_level_sim_gui = port_trigger_level_sim_gui;

	cpssp->running = false;
	cpssp->initial_start = true;

	/* default values */
	for (i = 0; i < NUMBER_ANALOG_CHANNELS; i++) {
		cpssp->scale_factor[i] = 2500 / GRID_STEP;
	}
	cpssp->trigger_channel = 0;
	trigger_level_set_sim(cpssp, 2500);
	cpssp->trigger_edge = true;
	control_handler_sample_rate_set_sim(cpssp, 2000000);
	if (scpi_server_open_file(cpssp, "scpi_script.txt")) {
		scpi_server_read_commands_from_file(cpssp);
		scpi_server_close_file(cpssp);
	}
	cpssp->timer_tick_count = 0;

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
