package storage

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"os"
	"testing"
	"time"

	"github.com/google/go-cloud/blob"

	"github.com/myml/cloud-blob/ucloud"
	"github.com/myml/cloud-blob/upyun"
)

// 测试所选云服务,支持upyun和ucloud和sync(deepin-sync)
const Cloud = "sync"

var store *Storage

func TestNewStorage(t *testing.T) {
	token := os.Getenv("DEEPIN_SYNC_TOKEN")
	t.Log(Backend(token))
	var err error
	store, err = NewStorage(token)
	if err != nil {
		panic(err)
	}
	switch Cloud {
	case "upyun":
		var (
			Bucket   = os.Getenv("UPYUN_BUCKET")
			Operator = os.Getenv("UPYUN_OPERATOR")
			Password = os.Getenv("UPYUN_PASSWORD")
		)
		auth := upyunBlob.MakeAuth(Operator, Password)
		store.Bucket = blob.NewBucket(upyunBlob.OpenUpyunBucket(Bucket, auth))
	case "ucloud":
		var (
			Bucket     = os.Getenv("UCLOUD_BUCKET")
			PublicKey  = os.Getenv("UCLOUD_PUBLIC")
			PrivateKey = os.Getenv("UCLOUD_PRIVATE")
		)
		auth := ucloudBlob.MakeAuth(PublicKey, PrivateKey)
		store.Bucket = blob.NewBucket(ucloudBlob.OpenUcloudBucket(Bucket, auth))
	}
}

func TestSet(t *testing.T) {
	_, err := store.Set(context.TODO(), "test", []byte("hello,world"), nil)
	if err != nil {
		t.Fatal(err)
	}
}

func TestSetVersion(t *testing.T) {
	_, err := store.Set(context.TODO(), "test", []byte("this is version test"), &SetOptions{NewVersion: true})
	if err != nil {
		t.Fatal(err)
	}
}

func TestGet(t *testing.T) {
	testContent := []byte("this is test")
	_, err := store.Set(context.TODO(), "test", testContent, nil)
	if err != nil {
		t.Fatal(err)
	}
	resp, err := store.Get(context.TODO(), "test", nil)
	if err != nil {
		t.Fatal(err)
		return
	}
	if !bytes.Equal(testContent, resp.Content) {
		t.Fatal("no equal")
	}
}

func TestGetLaster(t *testing.T) {
	notLasterContent := []byte("this is not the laster version")
	lasterContent := []byte("this is laster version")
	_, err := store.Set(context.TODO(), "index", notLasterContent, &SetOptions{NewVersion: true})
	if err != nil {
		t.Fatal(err)
	}
	time.Sleep(time.Second)
	_, err = store.Set(context.TODO(), "index", lasterContent, &SetOptions{NewVersion: true})
	if err != nil {
		t.Fatal(err)
	}
	resp, err := store.Get(context.TODO(), "index", &GetOptions{Version: VersionLatest})
	if err != nil {
		t.Fatal(err)
		return
	}
	if !bytes.Equal(lasterContent, resp.Content) {
		t.Fatal("no equal")
	}
}

func TestNotExist(t *testing.T) {
	file := fmt.Sprintf("NotFountTest-%d", time.Now().Unix())
	_, err := store.Get(context.TODO(), file, nil)
	if err != ErrNotFound {
		t.Error(err)
	}

	_, err = store.Get(context.TODO(), file, &GetOptions{Version: VersionLatest})
	if err != ErrNotFound {
		t.Error(err)
	}
}

func TestGetVersion(t *testing.T) {
	versionContent := []byte("this is one version")
	version, err := store.Set(context.TODO(), "test", versionContent, &SetOptions{NewVersion: true})
	if err != nil {
		t.Fatal(err)
		return
	}
	resp, err := store.Get(context.TODO(), "test", &GetOptions{Version: version})
	if err != nil {
		t.Fatal(err)
		return
	}
	if !bytes.Equal(versionContent, resp.Content) {
		t.Fatal("no equal")
	}
}

func TestList(t *testing.T) {
	list, err := store.List(context.TODO(), &ListOptions{Prefix: "test"})
	if err != nil {
		t.Fatal(err)
		return
	}
	for {
		obj, err := list.Next(context.TODO())
		if err != nil && err != io.EOF {
			t.Fatal(err)
			return
		}
		if obj == nil {
			break
		}
		t.Log(obj)
	}
}

func TestRemoveVersion(t *testing.T) {
	version, err := store.Set(context.TODO(), "test", []byte("remove version test"), &SetOptions{NewVersion: true})
	if err != nil {
		t.Fatal(err)
		return
	}
	time.Sleep(time.Second * 1)
	err = store.Remove(context.TODO(), "test", &DeleteOptions{Version: version})
	if err != nil {
		t.Fatal(err)
	}
}

func TestRemove(t *testing.T) {
	time.Sleep(time.Second * 1)
	err := store.Remove(context.TODO(), "test", nil)
	if err != nil {
		t.Fatal(err)
	}
}
