// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/sharing/click_to_call/click_to_call_ui_controller.h"

#include <memory>

#include "base/guid.h"
#include "base/memory/ptr_util.h"
#include "base/strings/strcat.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/sharing/fake_device_info.h"
#include "chrome/browser/sharing/features.h"
#include "chrome/browser/sharing/mock_sharing_service.h"
#include "chrome/browser/sharing/sharing_constants.h"
#include "chrome/browser/sharing/sharing_service_factory.h"
#include "chrome/test/base/testing_profile.h"
#include "components/gcm_driver/fake_gcm_driver.h"
#include "components/gcm_driver/instance_id/instance_id_driver.h"
#include "components/sync/protocol/sync.pb.h"
#include "components/sync_device_info/fake_device_info_tracker.h"
#include "components/sync_preferences/testing_pref_service_syncable.h"
#include "content/public/test/browser_task_environment.h"
#include "content/public/test/web_contents_tester.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "url/gurl.h"

using ::testing::Eq;
using ::testing::Property;

namespace {

const char kPhoneNumber[] = "073%2087%202525%2078";
const char kExpectedPhoneNumber[] = "073 87 2525 78";
const char kReceiverGuid[] = "test_receiver_guid";
const char kReceiverName[] = "test_receiver_name";

class ClickToCallUiControllerTest : public testing::Test {
 public:
  ClickToCallUiControllerTest() = default;

  void SetUp() override {
    web_contents_ =
        content::WebContentsTester::CreateTestWebContents(&profile_, nullptr);
    SharingServiceFactory::GetInstance()->SetTestingFactory(
        &profile_, base::BindRepeating([](content::BrowserContext* context)
                                           -> std::unique_ptr<KeyedService> {
          return std::make_unique<testing::NiceMock<MockSharingService>>();
        }));
    ClickToCallUiController::ShowDialog(
        web_contents_.get(), /*initiating_origin=*/base::nullopt,
        GURL(base::StrCat({"tel:", kPhoneNumber})), false);
    controller_ = ClickToCallUiController::GetOrCreateFromWebContents(
        web_contents_.get());
  }

 protected:
  testing::NiceMock<MockSharingService>* service() {
    return static_cast<testing::NiceMock<MockSharingService>*>(
        SharingServiceFactory::GetForBrowserContext(&profile_));
  }

  content::BrowserTaskEnvironment task_environment_;
  TestingProfile profile_;
  std::unique_ptr<content::WebContents> web_contents_;
  ClickToCallUiController* controller_ = nullptr;
};
}  // namespace

MATCHER_P(ProtoEquals, message, "") {
  std::string expected_serialized, actual_serialized;
  message.SerializeToString(&expected_serialized);
  arg.SerializeToString(&actual_serialized);
  return expected_serialized == actual_serialized;
}

// Check the call to sharing service when a device is chosen.
TEST_F(ClickToCallUiControllerTest, OnDeviceChosen) {
  std::unique_ptr<syncer::DeviceInfo> device_info =
      CreateFakeDeviceInfo(kReceiverGuid, kReceiverName);

  chrome_browser_sharing::SharingMessage sharing_message;
  sharing_message.mutable_click_to_call_message()->set_phone_number(
      kExpectedPhoneNumber);
  EXPECT_CALL(
      *service(),
      SendMessageToDevice(
          Property(&syncer::DeviceInfo::guid, kReceiverGuid),
          Eq(base::TimeDelta::FromSeconds(kSharingMessageTTLSeconds.Get())),
          ProtoEquals(sharing_message), testing::_));
  controller_->OnDeviceChosen(*device_info.get());
}

// Check the call to sharing service to get all synced devices.
TEST_F(ClickToCallUiControllerTest, GetSyncedDevices) {
  EXPECT_CALL(*service(),
              GetDeviceCandidates(
                  Eq(sync_pb::SharingSpecificFields::CLICK_TO_CALL_V2)));
  controller_->GetDevices();
}
