// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/cert_provisioning/cert_provisioning_worker.h"

#include "base/callback.h"
#include "base/test/gmock_callback_support.h"
#include "base/time/time.h"
#include "chrome/browser/chromeos/attestation/mock_tpm_challenge_key_subtle.h"
#include "chrome/browser/chromeos/attestation/tpm_challenge_key_subtle.h"
#include "chrome/browser/chromeos/cert_provisioning/cert_provisioning_common.h"
#include "chrome/browser/chromeos/platform_keys/mock_platform_keys_service.h"
#include "chrome/browser/chromeos/platform_keys/platform_keys_service_factory.h"
#include "chrome/test/base/testing_browser_process.h"
#include "chrome/test/base/testing_profile_manager.h"
#include "components/policy/core/common/cloud/mock_cloud_policy_client.h"
#include "components/prefs/testing_pref_service.h"
#include "content/public/test/browser_task_environment.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace em = enterprise_management;

using base::TimeDelta;
using base::test::RunOnceCallback;
using testing::_;
using testing::Mock;

namespace chromeos {
namespace cert_provisioning {
namespace {

// Generated by chrome/test/data/policy/test_certs/create_test_certs.sh
const char kFakeCertificate[] = R"(-----BEGIN CERTIFICATE-----
MIIDJzCCAg+gAwIBAgIBATANBgkqhkiG9w0BAQsFADAXMRUwEwYDVQQDDAxyb290
X2NhX2NlcnQwHhcNMjAwMjI1MTUyNTU2WhcNMzAwMjIyMTUyNTU2WjAUMRIwEAYD
VQQDDAkxMjcuMC4wLjEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDW
druvpaJovmyWzIcjtsSk/lp319+zNPSYGLzJzTeEmnFoDf/b89ft6xR1NIahmvVd
UHGOMlzgDKnNkqWw+pgpn6U8dk+leWnwlUefzDz7OY8qXfX29Vh0m/kATQc64lnp
rX19fEi2DOgH6heCQDSaHI/KAnAXccwl8kdGuTEnvdzbdHqQq8pPGpEqzC/NOjk7
kDNkUt0J74ZVMm4+jhVOgZ35mFLtC+xjfycBgbnt8yfPOzmOMwXTjYDPNaIy32AZ
t66oIToteoW5Ilg+j5Mto3unBDHrw8rml3+W/nwHuOPEIgBqLQFfWtXpuX8CbcS6
SFNK4hxCJOvlzUbgTpsrAgMBAAGjgYAwfjAMBgNVHRMBAf8EAjAAMB0GA1UdDgQW
BBRDEl1/2pL5LtKnpIly+XCj3N6MwDAfBgNVHSMEGDAWgBQrwVEnUQZlX850A2N+
URfS8BxoyzAdBgNVHSUEFjAUBggrBgEFBQcDAQYIKwYBBQUHAwIwDwYDVR0RBAgw
BocEfwAAATANBgkqhkiG9w0BAQsFAAOCAQEAXZd+Ul7GUFZPLSiTZ618hUI2UdO0
7rtPwBw3TephWuyEeHht+WhzA3sRL3nprEiJqIg5w/Tlfz4dsObpSU3vKmDhLzAx
HJrN5vKdbEj9wyuhYSRJwvJka1ZOgPzhQcDQOp1SqonNxLx/sSMDR2UIDMBGzrkQ
sDkn58N5eWm+hZADOAKROHR47j85VcsmYGK7z2x479YzsyWyOm0dbACXv7/HvFkz
56KvgxRaPZQzQUg5yuXa21IjQz07wyWSYnHpm2duAbYFl6CTR9Rlj5vpRkKsQP1W
mMhGDBfgEskdbM+0agsZrJupoQMBUbD5gflcJlW3kwlboi3dTtiGixfYWw==
-----END CERTIFICATE-----)";

const char kCertProfileId[] = "cert_profile_1";
// Prefix + certificate profile name.
const char kKeyName[] = "cert-provis-cert_profile_1";
const char kCertScopeStrUser[] = "google/chromeos/user";
const char kCertScopeStrDevice[] = "google/chromeos/device";
const char kPublicKey[] = "fake_public_key_1";
const char kInvalidationTopic[] = "fake_invalidation_topic_1";
const char kDataToSign[] = "fake_data_to_sign_1";
const em::HashingAlgorithm kProtoHashAlgo = em::HashingAlgorithm::SHA256;
const platform_keys::HashAlgorithm kPkHashAlgo =
    platform_keys::HashAlgorithm::HASH_ALGORITHM_SHA256;
const char kChallenge[] = "fake_va_challenge_1";
const char kChallengeResponse[] = "fake_va_challenge_response_1";
const char kSignature[] = "fake_signature_1";

// Using macros to reduce boilerplate code, but keep real line numbers in
// error messages in case of expectation failure. They use some of protected
// fields of CertProvisioningWorkerTest class and may be considered as extra
// methods of it.
#define EXPECT_PREPARE_KEY_OK(PREPARE_KEY_FUNC)                        \
  {                                                                    \
    auto public_key_result =                                           \
        attestation::TpmChallengeKeyResult::MakePublicKey(kPublicKey); \
    EXPECT_CALL(*tpm_challenge_key_impl_, PREPARE_KEY_FUNC)            \
        .Times(1)                                                      \
        .WillOnce(RunOnceCallback<4>(public_key_result));              \
  }

#define EXPECT_SIGN_CHALLENGE_OK(SIGN_CHALLENGE_FUNC)              \
  {                                                                \
    auto sign_challenge_result =                                   \
        attestation::TpmChallengeKeyResult::MakeChallengeResponse( \
            kChallengeResponse);                                   \
    EXPECT_CALL(*tpm_challenge_key_impl_, SIGN_CHALLENGE_FUNC)     \
        .Times(1)                                                  \
        .WillOnce(RunOnceCallback<2>(sign_challenge_result));      \
  }

#define EXPECT_REGISTER_KEY_OK(REGISTER_KEY_FUNC)            \
  {                                                          \
    auto register_key_result =                               \
        attestation::TpmChallengeKeyResult::MakeSuccess();   \
    EXPECT_CALL(*tpm_challenge_key_impl_, REGISTER_KEY_FUNC) \
        .Times(1)                                            \
        .WillOnce(RunOnceCallback<0>(register_key_result));  \
  }

#define EXPECT_START_CSR_OK(START_CSR_FUNC)                           \
  {                                                                   \
    EXPECT_CALL(cloud_policy_client_, START_CSR_FUNC)                 \
        .Times(1)                                                     \
        .WillOnce(RunOnceCallback<3>(                                 \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS,        \
            /*response_error=*/base::nullopt,                         \
            /*try_again_later_ms=*/base::nullopt, kInvalidationTopic, \
            kChallenge, kProtoHashAlgo, kDataToSign));                \
  }

#define EXPECT_START_CSR_OK_WITHOUT_VA(START_CSR_FUNC)                \
  {                                                                   \
    EXPECT_CALL(cloud_policy_client_, START_CSR_FUNC)                 \
        .Times(1)                                                     \
        .WillOnce(RunOnceCallback<3>(                                 \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS,        \
            /*response_error=*/base::nullopt,                         \
            /*try_again_later_ms=*/base::nullopt, kInvalidationTopic, \
            /*va_challenge=*/"", kProtoHashAlgo, kDataToSign));       \
  }

#define EXPECT_START_CSR_TRY_LATER(START_CSR_FUNC, DELAY_MS)              \
  {                                                                       \
    EXPECT_CALL(cloud_policy_client_, START_CSR_FUNC)                     \
        .Times(1)                                                         \
        .WillOnce(RunOnceCallback<3>(                                     \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS,            \
            /*response_error=*/base::nullopt,                             \
            /*try_again_later_ms=*/(DELAY_MS), /*invalidation_topic=*/"", \
            /*va_challenge=*/"",                                          \
            enterprise_management::HashingAlgorithm::                     \
                HASHING_ALGORITHM_UNSPECIFIED,                            \
            /*data_to_sign=*/""));                                        \
  }

#define EXPECT_START_CSR_INVALID_REQUEST(START_CSR_FUNC)                     \
  {                                                                          \
    EXPECT_CALL(cloud_policy_client_, START_CSR_FUNC)                        \
        .Times(1)                                                            \
        .WillOnce(RunOnceCallback<3>(                                        \
            policy::DeviceManagementStatus::DM_STATUS_REQUEST_INVALID,       \
            /*response_error=*/base::nullopt,                                \
            /*try_again_later_ms=*/base::nullopt, /*invalidation_topic=*/"", \
            /*va_challenge=*/"",                                             \
            enterprise_management::HashingAlgorithm::                        \
                HASHING_ALGORITHM_UNSPECIFIED,                               \
            /*data_to_sign=*/""));                                           \
  }

#define EXPECT_START_CSR_CA_ERROR(START_CSR_FUNC)                            \
  {                                                                          \
    EXPECT_CALL(cloud_policy_client_, START_CSR_FUNC)                        \
        .Times(1)                                                            \
        .WillOnce(RunOnceCallback<3>(                                        \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS,               \
            /*response_error=*/CertProvisioningResponseError::CA_ERROR,      \
            /*try_again_later_ms=*/base::nullopt, /*invalidation_topic=*/"", \
            /*va_challenge=*/"",                                             \
            enterprise_management::HashingAlgorithm::                        \
                HASHING_ALGORITHM_UNSPECIFIED,                               \
            /*data_to_sign=*/""));                                           \
  }

#define EXPECT_START_CSR_TEMPORARY_UNAVAILABLE(START_CSR_FUNC)               \
  {                                                                          \
    EXPECT_CALL(cloud_policy_client_, START_CSR_FUNC)                        \
        .Times(1)                                                            \
        .WillOnce(RunOnceCallback<3>(                                        \
            policy::DeviceManagementStatus::DM_STATUS_TEMPORARY_UNAVAILABLE, \
            /*response_error=*/base::nullopt,                                \
            /*try_again_later_ms=*/base::nullopt, /*invalidation_topic=*/"", \
            /*va_challenge=*/"",                                             \
            enterprise_management::HashingAlgorithm::                        \
                HASHING_ALGORITHM_UNSPECIFIED,                               \
            /*data_to_sign=*/""));                                           \
  }

#define EXPECT_FINISH_CSR_OK(FINISH_CSR_FUNC)                                 \
  {                                                                           \
    EXPECT_CALL(cloud_policy_client_, FINISH_CSR_FUNC)                        \
        .Times(1)                                                             \
        .WillOnce(RunOnceCallback<5>(                                         \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS, base::nullopt, \
            base::nullopt));                                                  \
  }

#define EXPECT_FINISH_CSR_TRY_LATER(FINISH_CSR_FUNC, DELAY_MS)                \
  {                                                                           \
    EXPECT_CALL(cloud_policy_client_, FINISH_CSR_FUNC)                        \
        .Times(1)                                                             \
        .WillOnce(RunOnceCallback<5>(                                         \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS, base::nullopt, \
            /*try_again_later_ms=*/(DELAY_MS)));                              \
  }

#define EXPECT_DOWNLOAD_CERT_OK(DOWNLOAD_CERT_FUNC)                           \
  {                                                                           \
    EXPECT_CALL(cloud_policy_client_, DOWNLOAD_CERT_FUNC)                     \
        .Times(1)                                                             \
        .WillOnce(RunOnceCallback<3>(                                         \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS, base::nullopt, \
            base::nullopt, kFakeCertificate));                                \
  }

#define EXPECT_DOWNLOAD_CERT_TRY_LATER(DOWNLOAD_CERT_FUNC, DELAY_MS)          \
  {                                                                           \
    EXPECT_CALL(cloud_policy_client_, DOWNLOAD_CERT_FUNC)                     \
        .Times(1)                                                             \
        .WillOnce(RunOnceCallback<3>(                                         \
            policy::DeviceManagementStatus::DM_STATUS_SUCCESS, base::nullopt, \
            /*try_again_later_ms=*/(DELAY_MS), /*certificate=*/""));          \
  }

#define EXPECT_SIGN_RSAPKC1_DIGEST_OK(SIGN_FUNC)                         \
  {                                                                      \
    EXPECT_CALL(*platform_keys_service_, SIGN_FUNC)                      \
        .Times(1)                                                        \
        .WillOnce(RunOnceCallback<4>(kSignature, /*error_message=*/"")); \
  }

#define EXPECT_IMPORT_CERTIFICATE_OK(IMPORT_FUNC)            \
  {                                                          \
    EXPECT_CALL(*platform_keys_service_, IMPORT_FUNC)        \
        .Times(1)                                            \
        .WillOnce(RunOnceCallback<2>(/*error_message=*/"")); \
  }

class CallbackObserver {
 public:
  MOCK_METHOD(void, Callback, (bool is_success));
};

class CertProvisioningWorkerTest : public ::testing::Test {
 public:
  CertProvisioningWorkerTest() { Init(); }
  CertProvisioningWorkerTest(const CertProvisioningWorkerTest&) = delete;
  CertProvisioningWorkerTest& operator=(const CertProvisioningWorkerTest&) =
      delete;
  ~CertProvisioningWorkerTest() override = default;

  void SetUp() override {
    // Replace TpmChallengeKey.
    auto mock_tpm_challenge_key_subtle_impl =
        std::make_unique<attestation::MockTpmChallengeKeySubtle>();
    tpm_challenge_key_impl_ = mock_tpm_challenge_key_subtle_impl.get();
    attestation::TpmChallengeKeySubtleFactory::SetForTesting(
        std::move(mock_tpm_challenge_key_subtle_impl));
    ASSERT_TRUE(tpm_challenge_key_impl_);

    // There should not be any calls to callback before this expect is
    // overridden.
    EXPECT_CALL(callback_observer_, Callback).Times(0);
  }

  void TearDown() override {
    EXPECT_FALSE(
        attestation::TpmChallengeKeySubtleFactory::WillReturnTestingInstance());
  }

 protected:
  void Init() {
    testing_profile_manager_ = std::make_unique<TestingProfileManager>(
        TestingBrowserProcess::GetGlobal());
    ASSERT_TRUE(testing_profile_manager_->SetUp());
    testing_profile_ =
        testing_profile_manager_->CreateTestingProfile("user@gmail.com");
    ASSERT_TRUE(testing_profile_);

    platform_keys_service_ =
        static_cast<platform_keys::MockPlatformKeysService*>(
            platform_keys::PlatformKeysServiceFactory::GetInstance()
                ->SetTestingFactoryAndUse(
                    testing_profile_,
                    base::BindRepeating(
                        &platform_keys::BuildMockPlatformKeysService)));
    ASSERT_TRUE(platform_keys_service_);
  }

  void FastForwardBy(TimeDelta delta) {
    task_environment_.FastForwardBy(delta);
  }

  CertProvisioningWorkerCallback GetCallback() {
    return base::BindOnce(&CallbackObserver::Callback,
                          base::Unretained(&callback_observer_));
  }

  content::BrowserTaskEnvironment task_environment_{
      base::test::TaskEnvironment::TimeSource::MOCK_TIME};

  CallbackObserver callback_observer_;

  std::unique_ptr<TestingProfileManager> testing_profile_manager_;
  TestingProfile* testing_profile_ = nullptr;

  TestingPrefServiceSimple testing_pref_service_;

  policy::MockCloudPolicyClient cloud_policy_client_;
  attestation::MockTpmChallengeKeySubtle* tpm_challenge_key_impl_ = nullptr;
  platform_keys::MockPlatformKeysService* platform_keys_service_ = nullptr;
};

// Checks that the worker makes all necessary requests to other modules during
// success scenario.
TEST_F(CertProvisioningWorkerTest, Success) {
  CertProfile cert_profile{kCertProfileId};
  CertProvisioningWorkerImpl worker(CertScope::kUser, testing_profile_,
                                    &testing_pref_service_, cert_profile,
                                    &cloud_policy_client_, GetCallback());

  {
    testing::InSequence seq;

    EXPECT_PREPARE_KEY_OK(StartPrepareKeyStep(
        attestation::AttestationKeyType::KEY_USER, kKeyName,
        /*profile=*/_, /*key_name_for_spkac=*/"", /*callback=*/_));

    EXPECT_START_CSR_OK(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));

    EXPECT_SIGN_CHALLENGE_OK(StartSignChallengeStep(
        kChallenge, /*include_signed_public_key=*/true, /*callback=*/_));

    EXPECT_REGISTER_KEY_OK(StartRegisterKeyStep);

    EXPECT_SIGN_RSAPKC1_DIGEST_OK(
        SignRSAPKCS1Digest(platform_keys::kTokenIdUser, kDataToSign, kPublicKey,
                           kPkHashAlgo, /*callback=*/_));

    EXPECT_FINISH_CSR_OK(ClientCertProvisioningFinishCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, kChallengeResponse,
        kSignature, /*callback=*/_));

    EXPECT_DOWNLOAD_CERT_OK(ClientCertProvisioningDownloadCert(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));

    EXPECT_IMPORT_CERTIFICATE_OK(ImportCertificate(
        platform_keys::kTokenIdUser, /*certificate=*/_, /*callback=*/_));
  }

  EXPECT_CALL(callback_observer_, Callback(true)).Times(1);
  worker.DoStep();
}

// Checks that the worker makes all necessary requests to other modules during
// success scenario when VA challenge is not received.
TEST_F(CertProvisioningWorkerTest, NoVaSuccess) {
  CertProfile cert_profile{kCertProfileId};
  CertProvisioningWorkerImpl worker(CertScope::kUser, testing_profile_,
                                    &testing_pref_service_, cert_profile,
                                    &cloud_policy_client_, GetCallback());
  EXPECT_CALL(*tpm_challenge_key_impl_, StartRegisterKeyStep).Times(0);

  {
    testing::InSequence seq;

    EXPECT_PREPARE_KEY_OK(StartPrepareKeyStep(
        attestation::AttestationKeyType::KEY_USER, kKeyName,
        /*profile=*/_, /*key_name_for_spkac=*/"", /*callback=*/_));

    EXPECT_START_CSR_OK_WITHOUT_VA(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));

    EXPECT_REGISTER_KEY_OK(StartRegisterKeyStep);

    EXPECT_SIGN_RSAPKC1_DIGEST_OK(
        SignRSAPKCS1Digest(platform_keys::kTokenIdUser, kDataToSign, kPublicKey,
                           kPkHashAlgo, /*callback=*/_));

    EXPECT_FINISH_CSR_OK(ClientCertProvisioningFinishCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey,
        /*va_challenge_response=*/"", kSignature, /*callback=*/_));

    EXPECT_DOWNLOAD_CERT_OK(ClientCertProvisioningDownloadCert(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));

    EXPECT_IMPORT_CERTIFICATE_OK(ImportCertificate(
        platform_keys::kTokenIdUser, /*certificate=*/_, /*callback=*/_));
  }

  EXPECT_CALL(callback_observer_, Callback(true)).Times(1);
  worker.DoStep();
}

// Checks that when the server returns try_again_later field, the worker will
// retry a request when it asked to continue the provisioning.
TEST_F(CertProvisioningWorkerTest, TryLaterManualRetry) {
  CertProfile cert_profile{kCertProfileId};
  CertProvisioningWorkerImpl worker(CertScope::kDevice, testing_profile_,
                                    &testing_pref_service_, cert_profile,
                                    &cloud_policy_client_, GetCallback());
  const TimeDelta delay = TimeDelta::FromSeconds(30);

  {
    testing::InSequence seq;

    EXPECT_PREPARE_KEY_OK(StartPrepareKeyStep(
        attestation::AttestationKeyType::KEY_DEVICE, /*key_name=*/"",
        /*profile=*/_, /*key_name_for_spkac=*/kKeyName, /*callback=*/_));

    EXPECT_START_CSR_TRY_LATER(
        ClientCertProvisioningStartCsr(kCertScopeStrDevice, kCertProfileId,
                                       kPublicKey, /*callback=*/_),
        delay.InMilliseconds());

    worker.DoStep();
    EXPECT_EQ(worker.GetState(),
              CertProvisioningWorkerState::kKeypairGenerated);
  }

  {
    testing::InSequence seq;

    EXPECT_START_CSR_OK(ClientCertProvisioningStartCsr(
        kCertScopeStrDevice, kCertProfileId, kPublicKey, /*callback=*/_));

    EXPECT_SIGN_CHALLENGE_OK(StartSignChallengeStep(
        kChallenge, /*include_signed_public_key=*/true, /*callback=*/_));

    EXPECT_REGISTER_KEY_OK(StartRegisterKeyStep);

    EXPECT_SIGN_RSAPKC1_DIGEST_OK(SignRSAPKCS1Digest);

    EXPECT_FINISH_CSR_TRY_LATER(
        ClientCertProvisioningFinishCsr(kCertScopeStrDevice, kCertProfileId,
                                        kPublicKey, kChallengeResponse,
                                        kSignature, /*callback=*/_),
        delay.InMilliseconds());

    worker.DoStep();
    EXPECT_EQ(worker.GetState(), CertProvisioningWorkerState::kSignCsrFinished);
  }

  {
    testing::InSequence seq;

    EXPECT_FINISH_CSR_OK(ClientCertProvisioningFinishCsr(
        kCertScopeStrDevice, kCertProfileId, kPublicKey, kChallengeResponse,
        kSignature, /*callback=*/_));

    EXPECT_DOWNLOAD_CERT_TRY_LATER(
        ClientCertProvisioningDownloadCert(kCertScopeStrDevice, kCertProfileId,
                                           kPublicKey, /*callback=*/_),
        delay.InMilliseconds());

    worker.DoStep();
    EXPECT_EQ(worker.GetState(),
              CertProvisioningWorkerState::kFinishCsrResponseReceived);
  }

  {
    testing::InSequence seq;

    EXPECT_DOWNLOAD_CERT_OK(ClientCertProvisioningDownloadCert(
        kCertScopeStrDevice, kCertProfileId, kPublicKey, /*callback=*/_));

    EXPECT_IMPORT_CERTIFICATE_OK(ImportCertificate(
        platform_keys::kTokenIdSystem, /*certificate=*/_, /*callback=*/_));

    EXPECT_CALL(callback_observer_, Callback(true)).Times(1);

    worker.DoStep();
    EXPECT_EQ(worker.GetState(), CertProvisioningWorkerState::kSucceed);
  }
}

// Checks that when the server returns try_again_later field, the worker will
// automatically retry a request after some time.
TEST_F(CertProvisioningWorkerTest, TryLaterWait) {
  CertProfile cert_profile{kCertProfileId};
  CertProvisioningWorkerImpl worker(CertScope::kUser, testing_profile_,
                                    &testing_pref_service_, cert_profile,
                                    &cloud_policy_client_, GetCallback());

  const TimeDelta start_csr_delay = TimeDelta::FromSeconds(30);
  const TimeDelta finish_csr_delay = TimeDelta::FromSeconds(30);
  const TimeDelta download_cert_server_delay = TimeDelta::FromMilliseconds(100);
  const TimeDelta download_cert_real_delay = TimeDelta::FromSeconds(10);
  const TimeDelta small_delay = TimeDelta::FromMilliseconds(500);

  {
    testing::InSequence seq;

    EXPECT_PREPARE_KEY_OK(StartPrepareKeyStep(
        attestation::AttestationKeyType::KEY_USER, kKeyName,
        /*profile=*/_, /*key_name_for_spkac=*/"", /*callback=*/_));

    EXPECT_START_CSR_TRY_LATER(
        ClientCertProvisioningStartCsr(kCertScopeStrUser, kCertProfileId,
                                       kPublicKey, /*callback=*/_),
        start_csr_delay.InMilliseconds());

    worker.DoStep();
    EXPECT_EQ(worker.GetState(),
              CertProvisioningWorkerState::kKeypairGenerated);
  }

  {
    testing::InSequence seq;

    EXPECT_START_CSR_OK(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));

    EXPECT_SIGN_CHALLENGE_OK(StartSignChallengeStep(
        kChallenge, /*include_signed_public_key=*/true, /*callback=*/_));

    EXPECT_REGISTER_KEY_OK(StartRegisterKeyStep);

    EXPECT_SIGN_RSAPKC1_DIGEST_OK(
        SignRSAPKCS1Digest(platform_keys::kTokenIdUser, kDataToSign, kPublicKey,
                           kPkHashAlgo, /*callback=*/_));

    EXPECT_FINISH_CSR_TRY_LATER(
        ClientCertProvisioningFinishCsr(kCertScopeStrUser, kCertProfileId,
                                        kPublicKey, kChallengeResponse,
                                        kSignature, /*callback=*/_),
        finish_csr_delay.InMilliseconds());

    FastForwardBy(start_csr_delay + small_delay);
    EXPECT_EQ(worker.GetState(), CertProvisioningWorkerState::kSignCsrFinished);
  }

  {
    testing::InSequence seq;

    EXPECT_FINISH_CSR_OK(ClientCertProvisioningFinishCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, kChallengeResponse,
        kSignature, /*callback=*/_));

    EXPECT_DOWNLOAD_CERT_TRY_LATER(
        ClientCertProvisioningDownloadCert(kCertScopeStrUser, kCertProfileId,
                                           kPublicKey, /*callback=*/_),
        download_cert_server_delay.InMilliseconds());

    FastForwardBy(finish_csr_delay + small_delay);
    EXPECT_EQ(worker.GetState(),
              CertProvisioningWorkerState::kFinishCsrResponseReceived);
  }

  {
    testing::InSequence seq;

    EXPECT_DOWNLOAD_CERT_OK(ClientCertProvisioningDownloadCert);

    EXPECT_IMPORT_CERTIFICATE_OK(ImportCertificate(
        platform_keys::kTokenIdUser, /*certificate=*/_, /*callback=*/_));

    FastForwardBy(small_delay);
    // Check that minimum wait time is not too small even if the server
    // has responded with a small one.
    EXPECT_EQ(worker.GetState(),
              CertProvisioningWorkerState::kFinishCsrResponseReceived);

    EXPECT_CALL(callback_observer_, Callback(true)).Times(1);
    FastForwardBy(download_cert_real_delay + small_delay);
    EXPECT_EQ(worker.GetState(), CertProvisioningWorkerState::kSucceed);
  }
}

// Checks that when the server returns error status, the worker will enter an
// error state and stop the provisioning.
TEST_F(CertProvisioningWorkerTest, StatusErrorHandling) {
  CertProfile cert_profile{kCertProfileId};
  CertProvisioningWorkerImpl worker(CertScope::kUser, testing_profile_,
                                    &testing_pref_service_, cert_profile,
                                    &cloud_policy_client_, GetCallback());

  {
    testing::InSequence seq;

    EXPECT_PREPARE_KEY_OK(StartPrepareKeyStep(
        attestation::AttestationKeyType::KEY_USER, kKeyName,
        /*profile=*/_, /*key_name_for_spkac=*/"", /*callback=*/_));

    EXPECT_START_CSR_INVALID_REQUEST(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));
  }

  EXPECT_CALL(callback_observer_, Callback(false)).Times(1);
  worker.DoStep();
}

// Checks that when the server returns response error, the worker will enter an
// error state and stop the provisioning. Also check factory.
TEST_F(CertProvisioningWorkerTest, ResponseErrorHandling) {
  CertProfile cert_profile{kCertProfileId};
  auto worker = CertProvisioningWorkerFactory::Get()->Create(
      CertScope::kUser, testing_profile_, &testing_pref_service_, cert_profile,
      &cloud_policy_client_, GetCallback());

  {
    testing::InSequence seq;

    EXPECT_PREPARE_KEY_OK(StartPrepareKeyStep(
        attestation::AttestationKeyType::KEY_USER, kKeyName,
        /*profile=*/_, /*key_name_for_spkac=*/"", /*callback=*/_));

    EXPECT_START_CSR_CA_ERROR(ClientCertProvisioningStartCsr);
  }

  EXPECT_CALL(callback_observer_, Callback(false)).Times(1);
  worker->DoStep();
}

// Checks that when the server returns TEMPORARY_UNAVAILABLE status code, the
// worker will automatically retry a request using exponential backoff strategy.
TEST_F(CertProvisioningWorkerTest, BackoffStrategy) {
  CertProfile cert_profile{kCertProfileId};
  CertProvisioningWorkerImpl worker(CertScope::kUser, testing_profile_,
                                    &testing_pref_service_, cert_profile,
                                    &cloud_policy_client_, GetCallback());

  TimeDelta next_delay = TimeDelta::FromSeconds(30);
  const TimeDelta small_delay = TimeDelta::FromMilliseconds(500);

  {
    testing::InSequence seq;

    EXPECT_PREPARE_KEY_OK(StartPrepareKeyStep(
        attestation::AttestationKeyType::KEY_USER, kKeyName,
        /*profile=*/_, /*key_name_for_spkac=*/"", /*callback=*/_));

    EXPECT_START_CSR_TEMPORARY_UNAVAILABLE(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));
    worker.DoStep();
  }

  Mock::VerifyAndClearExpectations(&cloud_policy_client_);

  {
    EXPECT_START_CSR_TEMPORARY_UNAVAILABLE(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));
    FastForwardBy(next_delay + small_delay * 10);
    next_delay *= 2;
  }

  Mock::VerifyAndClearExpectations(&cloud_policy_client_);

  {
    EXPECT_START_CSR_TEMPORARY_UNAVAILABLE(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));
    FastForwardBy(next_delay + small_delay * 10);
    next_delay *= 2;
  }

  Mock::VerifyAndClearExpectations(&cloud_policy_client_);

  {
    EXPECT_START_CSR_TEMPORARY_UNAVAILABLE(ClientCertProvisioningStartCsr(
        kCertScopeStrUser, kCertProfileId, kPublicKey, /*callback=*/_));
    FastForwardBy(next_delay + small_delay);
    next_delay *= 2;
  }
}

}  // namespace
}  // namespace cert_provisioning
}  // namespace chromeos
