/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.shiro.web.servlet

import org.junit.Test

import javax.servlet.ServletContext
import javax.servlet.http.HttpServletResponse
import javax.servlet.http.HttpSession

import static org.easymock.EasyMock.*
import static org.junit.Assert.*

/**
 * Unit tests for {@link ShiroHttpServletResponse}.
 */
class ShiroHttpServletResponseTest {

    private static String URL_SESSION_ID = "url_session_id"

    @Test
    void testEncodeURLNoSessionId() {

        def servletContext = createStrictMock(ServletContext)
        def httpServletResponse = createStrictMock(HttpServletResponse)
        def shiroHttpServletRequest = setupRequestMock()
        expect(shiroHttpServletRequest.getSession(false)).andReturn(null)
        expect(shiroHttpServletRequest.getAttribute(ShiroHttpServletRequest.SESSION_ID_URL_REWRITING_ENABLED)).andReturn(true)
        replay shiroHttpServletRequest, servletContext, httpServletResponse

        def shiroHttpServletResponse = new ShiroHttpServletResponse(httpServletResponse, servletContext, shiroHttpServletRequest)

        assertEquals "/foobar", shiroHttpServletResponse.encodeURL("/foobar")
        verify shiroHttpServletRequest, servletContext, httpServletResponse
    }

    @Test
    void testEncodeURLSessionIdInURL() {

        def servletContext = createStrictMock(ServletContext)
        def httpServletResponse = createStrictMock(HttpServletResponse)
        def session = createMock(HttpSession)
        def shiroHttpServletRequest = setupRequestMock()
        expect(session.getId()).andReturn(URL_SESSION_ID).anyTimes()
        expect(shiroHttpServletRequest.getSession(false)).andReturn(session)
        expect(shiroHttpServletRequest.getSession()).andReturn(session)
        expect(shiroHttpServletRequest.isRequestedSessionIdFromCookie()).andReturn(false)
        expect(shiroHttpServletRequest.getAttribute(ShiroHttpServletRequest.SESSION_ID_URL_REWRITING_ENABLED)).andReturn(true)
        replay shiroHttpServletRequest, servletContext, httpServletResponse, session

        def shiroHttpServletResponse = new ShiroHttpServletResponse(httpServletResponse, servletContext, shiroHttpServletRequest)

        assertEquals "/foobar;JSESSIONID=" + URL_SESSION_ID, shiroHttpServletResponse.encodeURL("/foobar")
        verify shiroHttpServletRequest, servletContext, httpServletResponse, session
    }

    @Test
    void testEncodeURLSessionIdInCookie() {

        def servletContext = createStrictMock(ServletContext)
        def httpServletResponse = createStrictMock(HttpServletResponse)
        def session = createMock(HttpSession)
        def shiroHttpServletRequest = setupRequestMock()
        expect(shiroHttpServletRequest.getAttribute(ShiroHttpServletRequest.SESSION_ID_URL_REWRITING_ENABLED)).andReturn(false)
        replay shiroHttpServletRequest, servletContext, httpServletResponse, session

        def shiroHttpServletResponse = new ShiroHttpServletResponse(httpServletResponse, servletContext, shiroHttpServletRequest)

        assertEquals "/foobar", shiroHttpServletResponse.encodeURL("/foobar")
        verify shiroHttpServletRequest, servletContext, httpServletResponse, session
    }

    @Test
    void testEncodeURLSessionIdInWhenRewriteDisabled() {

        def servletContext = createStrictMock(ServletContext)
        def httpServletResponse = createStrictMock(HttpServletResponse)
        def session = createMock(HttpSession)
        def shiroHttpServletRequest = setupRequestMock()
        expect(shiroHttpServletRequest.getAttribute(ShiroHttpServletRequest.SESSION_ID_URL_REWRITING_ENABLED)).andReturn(false)
        replay shiroHttpServletRequest, servletContext, httpServletResponse, session

        def shiroHttpServletResponse = new ShiroHttpServletResponse(httpServletResponse, servletContext, shiroHttpServletRequest)

        assertEquals "/foobar", shiroHttpServletResponse.encodeURL("/foobar")
        verify shiroHttpServletRequest, servletContext, httpServletResponse, session
    }

    /**
     * Tests if the request attribute {@link ShiroHttpServletRequest}.SESSION_ID_URL_REWRITING_ENABLED is not false
     * boolean, the default behavior is to encode the URL.
     */
    @Test
    void testEncodeURLSessionIdInWhenRewriteInvalid() {

        def servletContext = createStrictMock(ServletContext)
        def httpServletResponse = createStrictMock(HttpServletResponse)
        def session = createMock(HttpSession)
        def shiroHttpServletRequest = setupRequestMock()
        expect(session.getId()).andReturn(URL_SESSION_ID).anyTimes()
        expect(shiroHttpServletRequest.getSession(false)).andReturn(session)
        expect(shiroHttpServletRequest.getSession()).andReturn(session)
        expect(shiroHttpServletRequest.isRequestedSessionIdFromCookie()).andReturn(false)
        expect(shiroHttpServletRequest.getAttribute(ShiroHttpServletRequest.SESSION_ID_URL_REWRITING_ENABLED)).andReturn("something-else")
        replay shiroHttpServletRequest, servletContext, httpServletResponse, session

        def shiroHttpServletResponse = new ShiroHttpServletResponse(httpServletResponse, servletContext, shiroHttpServletRequest)

        assertEquals "/foobar;JSESSIONID=" + URL_SESSION_ID, shiroHttpServletResponse.encodeURL("/foobar")
        verify shiroHttpServletRequest, servletContext, httpServletResponse, session
    }

    /**
     * Tests if the request attribute {@link ShiroHttpServletRequest}.SESSION_ID_URL_REWRITING_ENABLED is null,
     * the default behavior is NOT to encode the URL.
     */
    @Test
    void testEncodeURLSessionIdInWhenRewriteInvalidAndNull() {

        def servletContext = createStrictMock(ServletContext)
        def httpServletResponse = createStrictMock(HttpServletResponse)
        def session = createMock(HttpSession)
        def shiroHttpServletRequest = setupRequestMock()
        expect(session.getId()).andReturn(URL_SESSION_ID).anyTimes()
        expect(shiroHttpServletRequest.getSession(false)).andReturn(session)
        expect(shiroHttpServletRequest.getSession()).andReturn(session)
        expect(shiroHttpServletRequest.isRequestedSessionIdFromCookie()).andReturn(false)
        expect(shiroHttpServletRequest.getAttribute(ShiroHttpServletRequest.SESSION_ID_URL_REWRITING_ENABLED)).andReturn(null)
        replay shiroHttpServletRequest, servletContext, httpServletResponse, session

        def shiroHttpServletResponse = new ShiroHttpServletResponse(httpServletResponse, servletContext, shiroHttpServletRequest)

        assertEquals "/foobar;JSESSIONID=" + URL_SESSION_ID, shiroHttpServletResponse.encodeURL("/foobar")
        verify shiroHttpServletRequest, servletContext, httpServletResponse, session
    }


    private static ShiroHttpServletRequest setupRequestMock() {
        def shiroHttpServletRequest = createMock(ShiroHttpServletRequest)

        expect(shiroHttpServletRequest.getScheme()).andReturn("http").anyTimes()
        expect(shiroHttpServletRequest.getServerName()).andReturn("localhost").anyTimes()
        expect(shiroHttpServletRequest.getServerPort()).andReturn(8080).anyTimes()
        expect(shiroHttpServletRequest.getContextPath()).andReturn("/").anyTimes()


        return shiroHttpServletRequest
    }

}
