//              -*- mode: C++ -*-
// Copyright(C) 2005,2007 Stefan Siegl <stesie@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>

#include <klocale.h>
#include <kmessagebox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <kconfigbase.h>
#include <qlistview.h>
#include <knuminput.h>
#include <ktextedit.h>
#include <kdebug.h>

#include "silcprotocol.h"

#include "silceditaccountwidget.h"
#include "silcaccount.h"

SilcEditAccountWidget::SilcEditAccountWidget(SilcProtocol *,
					     SilcAccount *account,
					     QWidget *parent)
  : SilcEditAccount(parent), KopeteEditAccountWidget(account),
    serverKeysToDelete()
{
  if(account) {
    // editing an already existing SilcAccount, therefore show it's current
    // settings in our SilcEditAccountWidget
    NickName->setText(account->myself()->nickName());
    UserName->setText(account->userName());
    RealName->setText(account->realName());
    RemoteHost->setText(account->hostName());
    ExcludeConnect->setChecked(account->excludeConnect());

    // set message sign flags ...
    chkSignChannelMessages->setChecked(account->signChannelMessages());
    chkSignChannelActions->setChecked(account->signChannelActions());
    chkSignPrivateMessages->setChecked(account->signPrivateMessages());
    chkSignPrivateActions->setChecked(account->signPrivateActions());
    chkDisplayImagesInline->setChecked(account->displayImagesInline());

    // set File Transfer settings ...
    chkUseSilcMime->setChecked(account->useSilcMime());
    chkFtAutoAccept->setChecked(account->ftAutoAccept());
    bindSelection->setCurrentItem(account->getFtBind());

    // set Default Messages settings ...
    QuitMessage->setText(account->quitMessage());

    // list available server fingerprints ...
    QStringList list =
      account->configGroup()->readListEntry("serverkey_hostlist");
    for(QStringList::Iterator it = list.begin(); it != list.end(); it ++) {
      const int i = (* it).findRev(':');
      if(i < 0) {
	std::cerr << "invalid hostlist entry " << (* it).latin1() << std::endl;
	continue;
      }

      const QString port = (* it).mid(i + 1);
      const int j = (* it).findRev('_', i);
      if(j < 0) {
	std::cerr << "invalid hostlist entry " << (* it).latin1() << std::endl;
	continue;
      }

      const QString ip = (* it).mid(j + 1, i - j - 1);
      const QString host = (* it).left(j);
      const QString hostent = QString("serverkey_%2").arg(* it);
      const QString &finger = account->configGroup()->readEntry(hostent);

      QListViewItem *item = new QListViewItem(listServerFingerprints, 
					      host, ip, port, finger);
      if(! item) {
	std::cerr << "unable to allocate QListViewItem" << std::endl;
	continue;
      }

      listServerFingerprints->insertItem(item);
    }
    // set moods
    setAttributeMood((SilcTK::SilcAttributeMood) account->getAttributeMood());
    // set contact mediums
    setAttributeContact
      ((SilcTK::SilcAttributeContact) account->getAttributeContact());

    // set other attributes
    attrTimezone->setChecked(account->getAttributeTimezone());
    chkAttr->setChecked(account->getAttributesAllowed());
    attrLanguage->setText(account->getAttributeLanguage());
    attrStatus->setText(account->getAttributeMessage());

    chkAttrGeo->setChecked(account->getGeoAllowed());
    attrGeoLong->setValue(account->getGeoLongitude());
    attrGeoLat->setValue(account->getGeoLatitude());
    attrGeoAlt->setValue(account->getGeoAltitude());

    slotGeoAllowToggle(account->getGeoAllowed());
    slotAllowToggle(account->getAttributesAllowed());

  }

  QObject::connect((QObject *) cmdRemove, SIGNAL(clicked()), 
		   this, SLOT(slotRemoveServerKey()));
  QObject::connect((QObject *) chkAttrGeo, SIGNAL(toggled(bool)), 
		   this, SLOT(slotGeoAllowToggle(bool)));
  QObject::connect((QObject *) chkAttr, SIGNAL(toggled(bool)), 
		   this, SLOT(slotAllowToggle(bool)));
}

SilcEditAccountWidget::~SilcEditAccountWidget()
{
}

bool
SilcEditAccountWidget::validateData(void)
{
  if(NickName->text().isEmpty())
    KMessageBox::sorry(this, i18n("<qt>You must enter a nickname.</qt>"));
  else if(NickName->text().contains("@") || NickName->text().contains(" ")) {
    //BUG: this is a hack, do it properly
    KMessageBox::sorry(this, 
    i18n("<qt>Please remove any @ or space from the nickname</qt>"));
  }
  else if(RemoteHost->text().isEmpty())
    KMessageBox::sorry(this, i18n("<qt>You have to specify a "
				  "remote host.</qt>"));
  else
    return true; // all the stuff is valid

  return false;
}

Kopete::Account *
SilcEditAccountWidget::apply(void)
{
  
  const QString nickName = this->NickName->text();
  const QString remoteHost = RemoteHost->text();

  SilcAccount *a;

  if(! this->account()) {
    a = new SilcAccount(SilcProtocol::protocol(),
			QString("%1@%2").arg(nickName).arg(remoteHost), 
			nickName.utf8());
    if(! a) return NULL;
    this->setAccount(a);
  }
  else {
    a = static_cast<SilcAccount *>(this->account());
    a->setNickName(nickName);
  }

  if(!validateData()) {
    return a;
  }

  a->setRealName(RealName->text());
  a->setUserName(UserName->text());
  a->setHostName(remoteHost);
  a->setExcludeConnect(ExcludeConnect->isChecked());

  a->setSignChannelMessages(chkSignChannelMessages->isChecked());
  a->setSignChannelActions(chkSignChannelActions->isChecked());
  a->setSignPrivateMessages(chkSignPrivateMessages->isChecked());
  a->setSignPrivateActions(chkSignPrivateActions->isChecked());
  a->setDisplayImagesInline(chkDisplayImagesInline->isChecked());

  a->setUseSilcMime(chkUseSilcMime->isChecked());
  a->setFtAutoAccept(chkFtAutoAccept->isChecked());
  a->setFtBind(bindSelection->currentItem());

  a->setQuitMessage(QuitMessage->text());
  a->setAttributeMood(getAttributeMood());
  a->setAttributeContact(getAttributeContact());

  // delete the keys, queued for deletion
  QStringList list = a->configGroup()->readListEntry("serverkey_hostlist");
  for(QStringList::Iterator it = serverKeysToDelete.begin(); 
      it != serverKeysToDelete.end(); it ++) {
    a->configGroup()->deleteEntry(QString("serverkey_%1").arg(* it));
    list.remove(* it);
  }
  a->configGroup()->writeEntry("serverkey_hostlist", list);

  a->setGeoInformations(chkAttrGeo->isChecked(), attrGeoLong->value(),
                        attrGeoLat->value(), attrGeoAlt->value());
  a->setAttributeLanguage(attrLanguage->text());
  a->setAttributeMessage(attrStatus->text());
  a->setAttributesAllowed(chkAttr->isChecked());
  a->setAttributeTimezone(attrTimezone->isChecked());

  a->updateAttributes();

  return a;
}



void
SilcEditAccountWidget::slotRemoveServerKey(void)
{
  QListViewItem *sel = listServerFingerprints->selectedItem();
  if(! sel) return;

  QString hostent = 
    QString("%1_%2:%3").arg(sel->text(0)).arg(sel->text(1)).arg(sel->text(2));

  serverKeysToDelete += hostent; // queue for real deletion
  delete sel; // remove item from list
}
void
SilcEditAccountWidget::slotGeoAllowToggle(bool value)
{
  attrGeoLong->setEnabled(value);
  attrGeoLat->setEnabled(value);
  attrGeoAlt->setEnabled(value);
}

void 
SilcEditAccountWidget::slotAllowToggle(bool value)
{
  attrTimezone->setEnabled(value);
  attrLanguage->setEnabled(value);
  attrMoodHappy->setEnabled(value);
  attrMoodSad->setEnabled(value);
  attrMoodAngry->setEnabled(value);
  attrMoodJealous->setEnabled(value);
  attrMoodAshamed->setEnabled(value);
  attrMoodInvincible->setEnabled(value);
  attrMoodInLove->setEnabled(value);
  attrMoodSleepy->setEnabled(value);
  attrMoodBored->setEnabled(value);
  attrMoodExcited->setEnabled(value);
  attrMoodAnxious->setEnabled(value);
  attrContactEmail->setEnabled(value);
  attrContactPhone->setEnabled(value);
  attrContactPaging->setEnabled(value);
  attrContactSMS->setEnabled(value);
  attrContactMMS->setEnabled(value);
  attrContactChat->setEnabled(value);
  attrContactVideo->setEnabled(value);
  attrStatus->setEnabled(value);  
  chkAttrGeo->setEnabled(value);
  slotGeoAllowToggle(value & chkAttrGeo->isChecked());
}

void
SilcEditAccountWidget::setAttributeMood(SilcTK::SilcAttributeMood mood)
{
  attrMoodHappy->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_HAPPY);
  attrMoodSad->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_SAD);
  attrMoodAngry->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_ANGRY);
  attrMoodJealous->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_JEALOUS);
  attrMoodAshamed->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_ASHAMED);
  attrMoodInvincible->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_INVINCIBLE);
  attrMoodInLove->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_INLOVE);
  attrMoodSleepy->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_SLEEPY);
  attrMoodBored->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_BORED);
  attrMoodExcited->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_EXCITED);
  attrMoodAnxious->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_ANXIOUS);
}

SilcTK::SilcAttributeMood 
SilcEditAccountWidget::getAttributeMood(void)
{
  SilcTK::SilcUInt32 mood = 0;
  
  if(attrMoodHappy->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_HAPPY;
  if(attrMoodSad->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_SAD;
  if(attrMoodAngry->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_ANGRY;
  if(attrMoodJealous->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_JEALOUS;
  if(attrMoodAshamed->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_ASHAMED;
  if(attrMoodInvincible->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_INVINCIBLE;
  if(attrMoodInLove->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_INLOVE;
  if(attrMoodSleepy->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_SLEEPY;
  if(attrMoodBored->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_BORED;
  if(attrMoodExcited->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_EXCITED;
  if(attrMoodAnxious->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_ANXIOUS;
  
  return (SilcTK::SilcAttributeMood) mood;  
}

void 
SilcEditAccountWidget::setAttributeContact(SilcTK::SilcAttributeContact contact)
{
  attrContactEmail->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_EMAIL);
  attrContactPhone->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_CALL);
  attrContactPaging->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_PAGE);
  attrContactSMS->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_SMS);
  attrContactMMS->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_MMS);
  attrContactChat->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_CHAT);
  attrContactVideo->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_VIDEO);
}

SilcTK::SilcAttributeContact 
SilcEditAccountWidget::getAttributeContact(void)
{
  SilcTK::SilcUInt32 contact = 0;
  if(attrContactEmail->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_EMAIL;
  if(attrContactPhone->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_CALL;
  if(attrContactPaging->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_PAGE;
  if(attrContactSMS->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_SMS;
  if(attrContactMMS->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_MMS;
  if(attrContactChat->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_CHAT;
  if(attrContactVideo->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_VIDEO;

  return (SilcTK::SilcAttributeContact) contact;
}

#include "silceditaccountwidget.moc"
