//                       -*- mode: C++ -*-
//
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
//
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>
#include <assert.h>
#include <kdebug.h>
#include <qfile.h>
#include <qimage.h>
#include <kstandarddirs.h>

#include "silcbuddyattributes.h"
#include "silcbuddycontact.h"


// sorry for this hack, unfortunately we need it for
// the macros of recent libsilc to work ...
typedef unsigned char SilcUInt8;

SilcBuddyAttributes::SilcBuddyAttributes(SilcBuddyContact *bud) 
{
  buddy = bud;
  _vcard = NULL;
  _mood = SilcTK::SILC_ATTRIBUTE_MOOD_NORMAL;
  _contactinfo = SilcTK::SILC_ATTRIBUTE_CONTACT_NONE;
  memset(&_geolocation, 0, sizeof(_geolocation));
}


SilcBuddyAttributes::~SilcBuddyAttributes() {}

void
SilcBuddyAttributes::updateAttributes(SilcTK::SilcDList attrs)
{
  assert(attrs);
    
  SilcTK::SilcAttributePayload attr;
  SilcTK::SilcAttribute attribute;
  SilcTK::SilcMime mime;
  char p[1024];
  bool preferred_language_start = FALSE;
  QString status_text;
  QString status_message;
  QString mime_type;
  char *ptr;
  SilcTK::SilcUInt32 len;


  SilcTK::silc_dlist_start(attrs);
  while ((attr = (SilcTK::SilcAttributePayload) SilcTK::silc_dlist_get(attrs))
         != SILC_LIST_END) {
    if (silc_attribute_get_flags(attr) != SILC_ATTRIBUTE_FLAG_VALID)
      continue;
    attribute = SilcTK::silc_attribute_get_attribute(attr);
    switch(attribute) {
    case SILC_ATTRIBUTE_USER_INFO:
      silc_free(_vcard);
      SilcTK::silc_attribute_get_object(attr, (void *)&_vcard, sizeof(_vcard));
      break;
    case SILC_ATTRIBUTE_STATUS_MOOD:
      SilcTK::silc_attribute_get_object(attr, (void *)&_mood, sizeof(_mood));
      break;
    case SILC_ATTRIBUTE_STATUS_FREETEXT:
      memset(&p, 0, sizeof(p));
      SilcTK::silc_attribute_get_object(attr, (void *)&p, sizeof(p) - 1);
      status_text = QString::fromUtf8(p);
      break;
    case SILC_ATTRIBUTE_STATUS_MESSAGE:
      mime = SilcTK::silc_mime_alloc();
      // untested
      if (!SilcTK::silc_attribute_get_object(attr, (void *)mime, sizeof(*mime)))
        break;
      ptr = (char *)SilcTK::silc_mime_get_field(mime, "Content-Type");
      if(!ptr) {
        status_message = QString::fromUtf8
          ((char *)SilcTK::silc_mime_get_data(mime, NULL));
        goto mime_free;
      }
      mime_type = QString(ptr);
      if ( mime_type.startsWith("text/plain") 
           || mime_type.startsWith("text/html") ) {
        if (mime_type.contains("utf-8")) 
          status_message = QString::fromUtf8((char *)SilcTK::silc_mime_get_data(mime, NULL));
        else
          status_message = QString::fromLatin1((char *)SilcTK::silc_mime_get_data(mime, NULL));
        if (mime_type.startsWith("text/plain")) 
          status_message = status_message.replace("\n", "<br>");
      } 
      else if ( mime_type.startsWith("multipart/alternate") ) {
        SilcTK::SilcDList parts = SilcTK::silc_mime_get_multiparts(mime, NULL);
        SilcTK::SilcMime part;

        SilcTK::silc_dlist_start(parts);
        while ((part = (SilcTK::SilcMime)SilcTK::silc_dlist_get(parts)) != SILC_LIST_END) {
          mime_type = SilcTK::silc_mime_get_field(part, "Content-Type");

          if ( mime_type.startsWith("text/plain") 
               || mime_type.startsWith("text/html") ) {
            if (mime_type.contains("utf-8"))
              status_message = QString::fromUtf8((char *)SilcTK::silc_mime_get_data(part, NULL));
            else
              status_message = QString::fromLatin1((char *)SilcTK::silc_mime_get_data(part, NULL));
            if (mime_type.startsWith("text/plain")) 
              status_message = status_message.replace("\n", "<br>");
          }
        }
      }
      goto mime_free;
    case SILC_ATTRIBUTE_PREFERRED_LANGUAGE:
      memset(&p, 0, sizeof(p));
      SilcTK::silc_attribute_get_object(attr, (void *)&p, sizeof(p)-1);
      if(!preferred_language_start) {
        _language = QString::fromUtf8(p);
        preferred_language_start = TRUE;
      }
      else
        _language.append(", " + QString::fromUtf8(p));
      break;
    case SILC_ATTRIBUTE_TIMEZONE:
      memset(&p, 0, sizeof(p));
      SilcTK::silc_attribute_get_object(attr, (void *)&p, sizeof(p)-1);
      _timezone = QString::fromUtf8(p);
      break;
    case SILC_ATTRIBUTE_PREFERRED_CONTACT:
      SilcTK::silc_attribute_get_object(attr, (void *)&_contactinfo, sizeof(_contactinfo));
      break;
    case SILC_ATTRIBUTE_GEOLOCATION:
      SilcTK::silc_attribute_get_object(attr, (void *)&_geolocation, sizeof(_geolocation));
      break;
    case SILC_ATTRIBUTE_USER_ICON:
      mime = SilcTK::silc_mime_alloc();
      if (!SilcTK::silc_attribute_get_object(attr, (void *)mime, sizeof(*mime)))
        break;
      ptr = (char *)SilcTK::silc_mime_get_field(mime, "Content-Type");
      if (!ptr)
        goto mime_free; 
      
      mime_type = QString(ptr);
      if (mime_type.startsWith("image/")) {
        QString newLocation(locateLocal("appdata", "silcphotos/"+ buddy->fingerprint()));

        ptr = (char *)SilcTK::silc_mime_get_data(mime, &len);
        if (!ptr || !len) 
          goto mime_free;
   
        QFile file(newLocation);
        file.open(IO_WriteOnly);
        file.writeBlock(ptr, len);
        file.close();
      
        QImage image(newLocation);
        if (image.height() > 96 || image.width() > 96)
          image = image.smoothScale(96, 96, QImage::ScaleMin);
        image.save(newLocation, "PNG");

        buddy->setProperty(Kopete::Global::Properties::self()->photo(), newLocation);
      }
mime_free:
      SilcTK::silc_mime_free(mime);
      break;
    }

  }
  if (! status_message.isEmpty())
    _status_text = status_message;
  else if (! status_text.isEmpty())
    _status_text = status_text;
}

QString 
SilcBuddyAttributes::getInformations() const
{
  QString info;
  
  info += i18n("<b><u>Mood:</u></b> ") + getMood() + "<br>";
  if(!getContact().isEmpty()) 
    info += i18n("<b><u>Preferred medium:</u></b> ")
      + getContact() + "<br>";
  if(_timezone)
    info += i18n("<b><u>Timezone:</u></b> ") + _timezone + "<br>";
  if(_language)
    info += i18n("<b><u>Language:</u></b> ") + _language + "<br>";
  // FIXME: Hack
  if(memcmp(&_geolocation, "\0\0\0\0", 4) != 0)
    info += i18n("<b><u>Location on Earth:</u></b><br>")
      + "<table><tr><td></td><td>"
      + i18n("Longitude: ") + _geolocation.longitude + "<br>"
      + i18n("Latitude: ") + _geolocation.latitude + "<br>"
      + i18n("Altitude: ") + _geolocation.altitude + "</td></tr></table>";
  if(_status_text)
    info += i18n("<b><u>Status Message:</u></b><br>") + _status_text + "<br>";

  return info;
}

QString
SilcBuddyAttributes::getContact() const
{
  QStringList contacts;
  if (! _contactinfo)
    return QString::null;

  if (_contactinfo & SilcTK::SILC_ATTRIBUTE_CONTACT_EMAIL)
    contacts.append(i18n("email"));
  if (_contactinfo & SilcTK::SILC_ATTRIBUTE_CONTACT_CALL)
    contacts.append(i18n("phone call"));
  if (_contactinfo & SilcTK::SILC_ATTRIBUTE_CONTACT_PAGE)
    contacts.append(i18n("paging"));
  if (_contactinfo & SilcTK::SILC_ATTRIBUTE_CONTACT_SMS)
    contacts.append(i18n("SMS"));
  if (_contactinfo & SilcTK::SILC_ATTRIBUTE_CONTACT_MMS)
    contacts.append(i18n("MMS"));
  if (_contactinfo & SilcTK::SILC_ATTRIBUTE_CONTACT_CHAT)
    contacts.append(i18n("chatting"));
  if (_contactinfo & SilcTK::SILC_ATTRIBUTE_CONTACT_VIDEO)
    contacts.append(i18n("video conference"));

  if (contacts.size() == 0)
    return QString::null;
  return contacts.join(", ");
}

QString 
SilcBuddyAttributes::getMood() const
{
  QStringList moods;
  /* I love vim macros */
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_HAPPY)
    moods.append(i18n("happy"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_SAD)
    moods.append(i18n("sad"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_ANGRY)
    moods.append(i18n("angry"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_JEALOUS)
    moods.append(i18n("jealous"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_ASHAMED)
    moods.append(i18n("ashamed"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_INVINCIBLE)
    moods.append(i18n("invincible"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_INLOVE)
    moods.append(i18n("in love"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_SLEEPY)
    moods.append(i18n("sleepy"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_BORED)
    moods.append(i18n("bored"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_EXCITED)
    moods.append(i18n("excited"));
  if (_mood & SilcTK::SILC_ATTRIBUTE_MOOD_ANXIOUS)
    moods.append(i18n("anxious"));

  if (moods.size() == 0)
    moods.append(QString(i18n("normal")));

  return moods.join(", ");
}

