#include "global.h"
#include "images.h"

#ifndef VRENGD


static const char *img_types[] = {
  EXT_GIF,  MIME_GIF,  (char *) TYPE_GIF,
  EXT_JPG,  MIME_JPEG, (char *) TYPE_JPG,
  EXT_JPEG, MIME_JPEG, (char *) TYPE_JPG,
  EXT_PPM,  MIME_PPM,  (char *) TYPE_PPM,
  EXT_PNG,  MIME_PNG,  (char *) TYPE_PNG,
  EXT_PCX,  MIME_PCX,  (char *) TYPE_PCX,
  EXT_RGB,  MIME_RGB,  (char *) TYPE_SGI,
  EXT_RGBA, MIME_RGBA, (char *) TYPE_SGI,
  EXT_BW,   MIME_BW,   (char *) TYPE_SGI,
  EXT_PHP,  MIME_PLAIN,(char *) TYPE_PLAIN,
  NULL,     NULL,      (char *) TYPE_UNKNOWN
};

int getImageTypeByMime(char *str)
{
  const char **t = img_types;

  while (t[1] != NULL) {
    if (!strcmp(t[1], str))
      return (int) t[2];
    t += 3;
  }
  return (int) t[2];
}

int getImageTypeByExt(char *str)
{
  char *p;
  const char **t = img_types;

  p = str + strlen(str)-1;
  while (p>str && *p != '.') p--;
  if (*p == '.') {
    p++;
    while (t[0] != NULL) {
      if (!strcmp(t[0], p))
        return (int) t[2];
      t += 3;
    }
  }
  return (int) t[2];
}

Image *
newImage(u_int16 width, u_int16 height, u_int8 format, u_int8 type)
{
  Image *image = NULL;

  if ((image = (Image *) calloc(1, sizeof(Image))) == NULL) {
    warning("can't alloc image");
    return NULL;
  }
  image->xsize = width;
  image->ysize = height;
  image->format = format;
  image->type = type;
  if ((image->pixmap = (u_int8 *) calloc(width*height, format)) == NULL) {
    warning("can't calloc pixmap");
    return NULL;
  }
  return image;
}

void freeImage(Image *image)
{
  if (image == NULL)
    return;
  free(image->pixmap); image->pixmap = NULL;
  free(image); image = NULL;
}
    

/* Resampling */

/*
 * bilinear interpolation with xf,yf normalized to 2^16
 */
static inline
int interpolate(int v00, int v01, int v10, int v11, int xf, int yf)
{
  int v0, v1, v;

  v0 = v00+(((v01-v00)*xf) >> INTERP_NORM_BITS);
  v1 = v10+(((v11-v10)*xf) >> INTERP_NORM_BITS);
  v = v0 + (((v1-v0) * yf) >> INTERP_NORM_BITS);
  return v;
}

/* 
 * called by updateTextures
 * TODO: more accurate resampling 
 */
Image *
resizeImage(u_int16 width_new, u_int16 height_new, u_int8 format, u_int8 type, Image *image_old)
{
  Image *image_new = NULL;
  u_int16 width_old, height_old, x, y;
  u_int8 *pixmap_new, *pixmap_old;
  u_int16 y1, x1inc, y1inc;

  if (type > 1)
    type = IMAGE_FIX;
  if ((image_new = newImage(width_new, height_new, format, type)) == NULL)
    return NULL;
  pixmap_new = (u_int8 *) image_new->pixmap;

  width_old = image_old->xsize;
  height_old = image_old->ysize;
  pixmap_old = (u_int8 *)image_old->pixmap;
  
  x1inc = ((width_old - 1) * INTERP_NORM1) / (width_new - 1);
  y1inc = ((height_old - 1) * INTERP_NORM1) / (height_new - 1);

  for (y1=0, y=0; y < height_new; y++) {
    u_int16 x1;

    for (x1=0, x=0; x < width_new; x++) {
      u_int16 xi, yi, xf, yf, j;

      xi = x1 >> (INTERP_NORM1_BITS - INTERP_NORM_BITS);
      yi = y1 >> (INTERP_NORM1_BITS - INTERP_NORM_BITS);
      xf = x1 & ((1 << (INTERP_NORM1_BITS - INTERP_NORM_BITS))-1);
      yf = y1 & ((1 << (INTERP_NORM1_BITS - INTERP_NORM_BITS))-1);
      if (xi < (width_old-1) && yi < (height_old-1)) {
        for (j=0; j < format; j++) {
          pixmap_new[j] = interpolate(pixmap_old[(yi*width_old+xi)*format+j],
                                      pixmap_old[(yi*width_old+xi+1)*format+j],
                                      pixmap_old[((yi+1)*width_old+xi)*format+j],
                                      pixmap_old[((yi+1)*width_old+xi+1)*format+j],
                                      xf, yf);
        }
      }
      else {
        for (j=0; j < format; j++)
          pixmap_new[j] = pixmap_old[(yi*width_old+xi)*format+j];
      }
      pixmap_new += format;
      x1 += x1inc;
    }
    y1 += y1inc;
  }

  return image_new;
}

#endif /* !VRENGD */
