#ifndef VRENGD

#include <GL/gl.h>

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "col.h"	/* COL_ONCE */
#include "user.h"	/* USER_TYPE */
#include "ball.h"	/* BALL_TYPE */
#include "water.h"

#include "zv.h"		/* parseGeometry */
#include "texture.h"	/* VRENG_TEXTURE_TYPE */
#include "helpers.h"	/* playSound */


#ifndef __sgi
#define sinf(x) ((float)sin((x)))
#else
#define sinf(x) ((float)Sin((x)))
#endif


const WClass Water::wclass(WATER_TYPE, "Water", Water::creator);


/* create from a fileline */
void Water::creator(char *l)
{
  new Water(l);
}

Water::Water(char *l)
{
  l = parseName(l, this);
  l = parsePosition(l, this);
  amplitude = WATER_AMPLITUDE; /* 0.03 */
  if (isdigit((int) *l))
    amplitude = atof(l); l = strtok(NULL, SEP);
  freq = WATER_FREQ; /* 5.0 */
  if (isdigit((int) *l))
    freq = atof(l); l = strtok(NULL, SEP);
  phase = WATER_PHASE; /* 0.0001 */
  if (isdigit((int) *l))
    phase = atof(l); l = strtok(NULL, SEP);
  l = parseURL(l, this);
  soh = parseGeometry(l);

  nature.collision = COL_ONCE;
  nature.movable = VR_NO_ELEM_MOVE;
  nature.renderable = VR_SPECIAL_RENDER;
  initializeObject(this, WATER_TYPE, VR_MOBILE);

  posx = pos.x;
  posy = pos.y;
  posz = pos.z + 0.15;
  rotx = RADIAN2DEGREE(pos.az);
  roty = 90.0;
  rotz = RADIAN2DEGREE(pos.ax);
  scalex = pos.bbsize.v[0];
  scaley = pos.bbsize.v[1];
  scalez = pos.bbsize.v[2];
  move.perm_sec = 1;

  static TextureCacheEntry *tc = NULL;

  if (tc == NULL) {
    boolean loaded = FALSE;
    for (int i=0; !loaded && i < 5; i++) {
      if ((tc = getTextureEntryByUrl(name.url)) == NULL) {
        trace(DBG_WMGT, "unable to find texture %s", name.url);
	sleep(1);
	continue;
      }
      if (tc->image == NULL) {
        trace(DBG_WMGT, "texture image null, url=%s", name.url);
	sleep(1);
	continue;
      }
      loaded = TRUE;
    }
    texture = tc->num;
    trace(DBG_WMGT, "water texture %s, texture=%d width=%d height=%d format=%d",
          name.url, tc->num,
          tc->image->xsize, tc->image->ysize, tc->image->format);
  }
}

static float off;

void Water::render()
{
  static float tcolor[] = {0.4, 0.7, 1.0, 1.0};	/* triangles color */
  static float wcolor[] = {0.8, 0.8, 1.0, 1.0};	/* water color */

  glPushMatrix();
  glTranslatef(posx, posy, posz);
  glRotatef(rotx, 1, 0, 0);
  glRotatef(roty, 0, 1, 0);
  glRotatef(rotz, 0, 0, 1);
  glScalef(scaley, 1, scalex);

  glMatrixMode(GL_TEXTURE);	
  glPushMatrix();
  glTranslatef(ttrans[0], ttrans[1], 0);
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, texture);
#if !defined(WITH_TINYGL)	/* glTexEnvfv missing */
  glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, wcolor);
  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_BLEND);
#endif
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, tcolor);

  float d = 1./WATER_MESH;

  for (int i = 0; i < WATER_MESH; i++) {
    glBegin(GL_TRIANGLE_STRIP);
    for (int j = 0; j < WATER_MESH; j++) {
      float s = (float)j*d;
      float t = (float)i*d;
      float x = -1. + 2.*s;
      float z = -1. + 2.*t;
      float y = amplitude * sinf(freq*M_2PI*t+off);
      glTexCoord2f(s, t); glVertex3f(x, y, z);

      s += d; t += d;
      x = -1. + 2.*s;
      z = -1. + 2.*t;
      y = amplitude * sinf(freq*M_2PI*t+off);
      glTexCoord2f(s, t); glVertex3f(x, y, z);

      off += phase;
      if (off > 20.)
        off = 0.;
    }
    glEnd();
  }

#if !defined(WITH_TINYGL)	/* glTexEnvf missing */
  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
#else
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
#endif
  glDisable(GL_TEXTURE_2D);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

/* system of equations handling permanent motion */
void Water::changePermanent(float lasting)
{
  /* water animation */
  ttrans[0] += .005;
  if (ttrans[0] >= 1.) ttrans[0] = 0.;
  ttrans[1] -= .0025;
  if (ttrans[1] <= 0.) ttrans[1] = 1.;
}

void Water::whenIntersect(WObject *pcur, WObject *pold)
{
  switch (pcur->noh.type) {
  case BALL_TYPE:
    pcur->pos.z += BALL_DELTAZ;
    updateObjectIn3D(pcur);
    updateBB(pcur);
    nature.collision = COL_EVER;
    break;
  case USER_TYPE:
    updateObjectIn3D(pcur);
    updateBB(pcur);
    playSound(BUBBLESSND);
    nature.collision = COL_ONCE;
    break;
  }
}

void waterInitFuncList(void) { }

#endif /* !VRENGD */
