#include "global.h"
#include "vncclient.h"


VNCRGB::VNCRGB() {
  Red = 0;
  Green = 0;
  Blue = 0;
}

VNCRGB::VNCRGB(const CARD32 &pixel) {
  Red = (CARD8) (pixel);
  Green = (CARD8) (pixel >> 8);
  Blue = (CARD8) (pixel >> 16);
}

VNCRGB::VNCRGB(CARD8 red, CARD8 green, CARD8 blue) {
  Red = red;
  Green = green;
  Blue = blue;
}

/* 
   CONSTRUCTORS
*/

VNCClient::VNCClient(char *server, int port, char *passFile) : RFB_proto(server,port,passFile)
{ 
  ViewOnly = false;
  FrameBuffer = NULL;
  framebufferWidth = 0;
  framebufferHeight = 0;
  serverCutText = NULL;
  newServerCutText = false;
}

VNCClientTextured::VNCClientTextured(char *server, int port, char *passFile) : VNCClient(server,port,passFile)
{ 
  realScreenWidth = 0;
  realScreenHeight = 0;
}

bool VNCClient::VNCInit()
{
  if (RFB_proto.ConnectToRFBServer() && RFB_proto.InitialiseRFBConnection()) {

    RFB_proto.SetDefaultVisual();

    if (!RFB_proto.SetFormatAndEncodings()) {
      trace(DBG_FORCE, "VNCInit: unable to Set Default PixelFormat");
      VNCClose();
      return false;
    }

    framebufferWidth = RFB_proto.si.framebufferWidth;
    framebufferHeight = RFB_proto.si.framebufferHeight;
    FrameBuffer = (VNCRGB *) malloc( framebufferWidth * framebufferHeight * sizeof(VNCRGB) );

    if (!FrameBuffer) {
      trace(DBG_FORCE, "VNCInit: unable to allocate memory for FrameBuffer");
      VNCClose();
      return false;
    }
    return true;
  }
  else {
    trace(DBG_FORCE, "VNCInit: connection or initialization impossible");
    VNCClose();
  }
  return false;
}     

bool VNCClient::VNCInit(int flag)
{
  if (RFB_proto.ConnectToRFBServer() && RFB_proto.InitialiseRFBConnection()) {

    switch (flag) {
    case BPP8:
      //here we do 8pp
      RFB_proto.SetVisual8();
      break;
    case BPP16:
      //here we do 16pp
      RFB_proto.SetVisual16();
      break;
    default:
      //ok we keep 32bpp
      RFB_proto.SetDefaultVisual();
      break;
    }

    if (!RFB_proto.SetFormatAndEncodings()) {
      trace(DBG_FORCE, "VNCInit: unable to Set Default PixelFormat");
      VNCClose();
      return false;
    }

    framebufferWidth = RFB_proto.si.framebufferWidth;
    framebufferHeight = RFB_proto.si.framebufferHeight;
    FrameBuffer = (VNCRGB *) malloc( framebufferWidth * framebufferHeight * sizeof(VNCRGB) );

    if (!FrameBuffer) {
      trace(DBG_FORCE, "VNCInit: unable to allocate memory for FrameBuffer");
      VNCClose();
      return false;
    }
    return true;
  }
  else {
    trace(DBG_FORCE, "VNCInit: connection or initialization impossible");
    VNCClose();
  }
  return false;
}     

bool VNCClientTextured::VNCInit()
{
  if (RFB_proto.ConnectToRFBServer() && RFB_proto.InitialiseRFBConnection()) {
    
    RFB_proto.SetDefaultVisual();

    if (!RFB_proto.SetFormatAndEncodings()) {
      trace(DBG_FORCE, "VNCInit: unable to Set Default PixelFormat");
      VNCClose();
      return false;
    }

    realScreenWidth = RFB_proto.si.framebufferWidth;
    realScreenHeight = RFB_proto.si.framebufferHeight;

    CARD16 PowerOf2 = 1;
    for (; PowerOf2 < realScreenWidth; PowerOf2 *= 2) {}
    framebufferWidth = PowerOf2;
    PowerOf2 = 1;
    for (; PowerOf2 < realScreenHeight; PowerOf2 *= 2) {}
    framebufferHeight = PowerOf2;

    FrameBuffer = (VNCRGB *) malloc(framebufferWidth * framebufferHeight * sizeof(VNCRGB));

    trace(DBG_VNC, "Yo: %d %d", framebufferWidth, framebufferHeight);

    if (!FrameBuffer) {
      trace(DBG_FORCE, "VNCInit: unable to allocate memory for FrameBuffer");
      VNCClose();
      return false;
    }
    return true;
  }
  else {
    trace(DBG_FORCE, "VNCInit: connection or initialization impossible");
    VNCClose();
  }
  return false;
}

bool VNCClientTextured::VNCInit(int flag)
{
  if (RFB_proto.ConnectToRFBServer() && RFB_proto.InitialiseRFBConnection()) {

    switch (flag) {
    case BPP8:
      //here we do 8pp
      RFB_proto.SetVisual8();
      break;
    case BPP16:
      //here we do 16pp
      RFB_proto.SetVisual16();
      break;
    default:
      //ok we keep 32bpp
      RFB_proto.SetDefaultVisual();
      break;
    }

    if (!RFB_proto.SetFormatAndEncodings()) {
      trace(DBG_FORCE, "VNCInit: unable to Set Default PixelFormat");
      VNCClose();
      return false;
    }

    realScreenWidth = RFB_proto.si.framebufferWidth;
    realScreenHeight = RFB_proto.si.framebufferHeight;

    CARD16 PowerOf2 = 1;
    for (; PowerOf2 < realScreenWidth; PowerOf2 *= 2) {}
    framebufferWidth = PowerOf2;
    PowerOf2 = 1;
    for (; PowerOf2 < realScreenHeight; PowerOf2 *= 2) {}
    framebufferHeight = PowerOf2;

    FrameBuffer = (VNCRGB *) malloc( framebufferWidth * framebufferHeight * sizeof(VNCRGB) );

    if (!FrameBuffer) {
      trace(DBG_FORCE, "VNCInit: unable to allocate memory for FrameBuffer");
      VNCClose();
      return false;
    }
    return true;
  }
  else {
    trace(DBG_FORCE, "VNCInit: connection or initialization impossible");
    VNCClose();
  }
  return false;
}

bool VNCClient::VNCClose()
{
  if (FrameBuffer)
    free(FrameBuffer);
  close(getSock());
  return true;
}

/*
 * SendRFBEvent is an action which sends an RFB event.  It can be used in two
 * ways.  Without any parameters it simply sends an RFB event corresponding to
 * the X event which caused it to be called.  With parameters, it generates a
 * "fake" RFB event based on those parameters.  The first parameter is the
 * event type, either "ptr", "keydown", "keyup" or "key" (down&up).  For a
 * "key" event the second parameter is simply a keysym string as understood by
 * XStringToKeysym().  For a "ptr" event, the following three parameters are
 * just X, Y and the button mask (0 for all up, 1 for button1 down, 2 for
 * button2 down, 3 for both, etc).
 */
void VNCClient::SendRFBEvent(char **params, Cardinal *num_params)
{
  KeySym ks;
  int buttonMask, x, y;

  if (ViewOnly || !FrameBuffer) return;

  if (strncasecmp(params[0],"key",3) == 0) {
    if (*num_params != 2) {
      trace(DBG_FORCE, "Invalid params: SendRFBEvent(key|keydown|keyup,<keysym>)");
      return;
    }
    ks = XStringToKeysym(params[1]);
    if (ks == NoSymbol) {
      trace(DBG_FORCE, "Invalid keysym '%s' passed to SendRFBEvent", params[1]);
      return;
    }
    if (strcasecmp(params[0],"keydown") == 0) {
      RFB_proto.SendKeyEvent(ks, 1);
    } else if (strcasecmp(params[0],"keyup") == 0) {
      RFB_proto.SendKeyEvent(ks, 0);
    } else if (strcasecmp(params[0],"key") == 0) {
      RFB_proto.SendKeyEvent(ks, 1);
      RFB_proto.SendKeyEvent(ks, 0);
    } else {
      trace(DBG_FORCE, "Invalid event '%s' passed to SendRFBEvent", params[0]);
      return;
    }
  }
  else if (strcasecmp(params[0],"ptr") == 0) {
    if (*num_params == 4) {
      x = atoi(params[1]);
      y = atoi(params[2]);
      buttonMask = atoi(params[3]);
      RFB_proto.SendPointerEvent(x, y, buttonMask);
    }
    else {
      trace(DBG_FORCE, "Invalid params: SendRFBEvent(ptr,<x>,<y>,<buttonMask>)");
      return;
    }
  }
  else
    trace(DBG_FORCE, "Invalid event '%s' passed to SendRFBEvent", params[0]);
}

CARD8 VNCClient::RescalePixValue( CARD32 Pix, CARD8 Shift, CARD16 Max)
{
  return (CARD8) (((Pix >> Shift)&Max) * (256 / (Max+1))) ;
}

// can be used with a CARD8, CARD16 or CARD32 Pixel
VNCRGB VNCClient::CardToVNCRGB(CARD32 Pixel)
{
  rfbPixelFormat PF = RFB_proto.myFormat; 

  return VNCRGB(RescalePixValue(Pixel, PF.redShift, PF.redMax), RescalePixValue(Pixel, PF.greenShift, PF.greenMax), RescalePixValue(Pixel, PF.blueShift, PF.blueMax));
}

// All the methods we need to handle a given rect message
// And update the FrameBuffer

bool VNCClient::HandleRAW8(int rx, int ry, int rw, int rh)
{
  CARD8 *src = (CARD8 *) buffer;
  VNCRGB *dest = (FrameBuffer + ry * framebufferWidth + rx);

  for (int h = 0; h < rh; h++) {
    for (int w = 0; w < rw; w++)  
      *dest++ = CardToVNCRGB((CARD32) *src++);
    dest += framebufferWidth-rw;
  }
  return true;
}

bool VNCClient::HandleRAW16(int rx, int ry, int rw, int rh)
{
  CARD16 *src = (CARD16 *) buffer;
  VNCRGB *dest = (FrameBuffer + ry * framebufferWidth + rx);

  for (int h = 0; h < rh; h++) {
    for (int w = 0; w < rw; w++)  
      *dest++ = CardToVNCRGB( (CARD32) Swap16IfLE(*src++));
    dest += framebufferWidth-rw;
  }
  return true;
}

bool VNCClient::HandleRAW32(int rx, int ry, int rw, int rh)
{
  CARD32 *src = (CARD32 *) buffer;
  VNCRGB *dest = (FrameBuffer + ry * framebufferWidth + rx);
  
  trace(DBG_VNC, "VNCClient::HandleRAW32: rx=%d ry=%d rw=%d rh=%d",rx,ry,rw,rh);

  for (int h = 0; h < rh; h++) {
    for (int w = 0; w < rw; w++)  
      *dest++ = CardToVNCRGB(Swap32IfLE(*src++));
    dest += framebufferWidth-rw;
  }
  return true;
}

bool VNCClient::HandleCR(int srcx, int srcy, int rx, int ry, int rw, int rh)
{
  VNCRGB *src;
  VNCRGB *dest;

  VNCRGB *buf = new VNCRGB[rh*rw];

  src = FrameBuffer + (srcy * framebufferWidth + srcx);
  dest = buf;

  for (int h = 0; h < rh; h++) {
    memcpy(dest,src,rw*sizeof(VNCRGB));
    src += framebufferWidth;
    dest += rw;
  }

  src = buf;
  dest = FrameBuffer + (ry * framebufferWidth + rx);

  for (int h = 0; h < rh; h++) {
    memcpy(dest,src,rw*sizeof(VNCRGB));
    src +=rw;
    dest += framebufferWidth;
  }
  delete[] buf;
  return true;
}

void VNCClient::FillRect(int rx, int ry, int rw, int rh, VNCRGB pixel)
{
  VNCRGB * dest = FrameBuffer + (ry * framebufferWidth + rx);
  
  for (int h = 0; h < rh; h++) {
    for (int w = 0; w < rw; w++)
      *dest++ = pixel;
    dest += framebufferWidth - rw;
  }
}

bool VNCClient::HandleRRE8(int rx, int ry, int rw, int rh)
{
  rfbRREHeader hdr;
  CARD8 pix;
  rfbRectangle subrect;
  VNCRGB pixel;
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&hdr, sz_rfbRREHeader))
    return false;
  
  hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
    return false;
  
  pixel = CardToVNCRGB((CARD32) pix);
  
  FillRect(rx, ry, rw, rh, pixel);
  
  for (unsigned int i = 0; i < hdr.nSubrects; i++) {
    if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
      return false;
    
    if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&subrect, sz_rfbRectangle))
      return false;
    
    subrect.x = Swap16IfLE(subrect.x);
    subrect.y = Swap16IfLE(subrect.y);
    subrect.w = Swap16IfLE(subrect.w);
    subrect.h = Swap16IfLE(subrect.h);
    
    pixel = CardToVNCRGB((CARD32) pix);
    
    FillRect(subrect.x, subrect.y, subrect.w, subrect.h, pixel);
  }
  return true;
}

bool VNCClient::HandleRRE16(int rx, int ry, int rw, int rh)
{
  rfbRREHeader hdr;
  CARD16 pix;
  rfbRectangle subrect;
  VNCRGB pixel;
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&hdr, sz_rfbRREHeader))
    return false;
  
  hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
    return false;
  
  pixel = CardToVNCRGB((CARD32) Swap16IfLE(pix));
  
  FillRect(rx, ry, rw, rh, pixel);
  
  for (unsigned int i = 0; i < hdr.nSubrects; i++) {
    if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
      return false;
    
    if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&subrect, sz_rfbRectangle))
      return false;
    
    subrect.x = Swap16IfLE(subrect.x);
    subrect.y = Swap16IfLE(subrect.y);
    subrect.w = Swap16IfLE(subrect.w);
    subrect.h = Swap16IfLE(subrect.h);
    
    pixel = CardToVNCRGB((CARD32) Swap16IfLE(pix));
    
    FillRect(subrect.x, subrect.y, subrect.w, subrect.h, pixel);
  }
  return true;
}

bool VNCClient::HandleRRE32(int rx, int ry, int rw, int rh)
{
  rfbRREHeader hdr;
  CARD32 pix;
  rfbRectangle subrect;
  VNCRGB pixel;
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&hdr, sz_rfbRREHeader))
    return false;
  
  hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
    return false;
  
  pixel = CardToVNCRGB(Swap32IfLE(pix));
  
  FillRect(rx, ry, rw, rh, pixel);
  
  for (unsigned int i = 0; i < hdr.nSubrects; i++) {
    if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
      return false;
    
    if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&subrect, sz_rfbRectangle))
      return false;
    
    subrect.x = Swap16IfLE(subrect.x);
    subrect.y = Swap16IfLE(subrect.y);
    subrect.w = Swap16IfLE(subrect.w);
    subrect.h = Swap16IfLE(subrect.h);
    
    pixel = CardToVNCRGB(Swap32IfLE(pix));
    
    FillRect(subrect.x, subrect.y, subrect.w, subrect.h, pixel);
  }
  return true;
}

bool VNCClient::HandleCoRRE8(int rx, int ry, int rw, int rh)
{
  rfbRREHeader hdr;
  CARD8 pix;
  CARD8 *ptr;
  int x, y, ww, hh;
  VNCRGB pixel;
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&hdr, sz_rfbRREHeader))
    return false;
  
  hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
    return false;

  pixel = CardToVNCRGB((CARD32) pix);
  FillRect(rx, ry, rw, rh, pixel);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, hdr.nSubrects * 5))
    return false;
  
  ptr = (CARD8 *)buffer;
  
  for (unsigned int i = 0; i < hdr.nSubrects; i++) {
    pix = *ptr++;
    x = *ptr++;
    y = *ptr++;
    ww = *ptr++;
    hh = *ptr++;
    
    pixel = CardToVNCRGB((CARD32) pix);
    FillRect(rx+x, ry+y, ww, hh, pixel);
  }
  return true;
}

bool VNCClient::HandleCoRRE16(int rx, int ry, int rw, int rh)
{
  rfbRREHeader hdr;
  CARD16 pix;
  CARD8 *ptr;
  int x, y, ww, hh;
  VNCRGB pixel;
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&hdr, sz_rfbRREHeader))
    return false;
  
  hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
    return false;

  pixel = CardToVNCRGB((CARD32) Swap16IfLE(pix));
  FillRect(rx, ry, rw, rh, pixel);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, hdr.nSubrects * 6))
    return false;
  
  ptr = (CARD8 *)buffer;
  
  for (unsigned int i = 0; i < hdr.nSubrects; i++) {
    pix = *(CARD16 *)ptr;
    ptr += 2;
    x = *ptr++;
    y = *ptr++;
    ww = *ptr++;
    hh = *ptr++;
    
    pixel = CardToVNCRGB((CARD32) Swap16IfLE(pix));
    FillRect(rx+x, ry+y, ww, hh, pixel);
  }
  return true;
}

bool VNCClient::HandleCoRRE32(int rx, int ry, int rw, int rh)
{
  rfbRREHeader hdr;
  CARD32 pix;
  CARD8 *ptr;
  int x, y, ww, hh;
  VNCRGB pixel;
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&hdr, sz_rfbRREHeader))
    return false;
  
  hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&pix, sizeof(pix)))
    return false;

  pixel = CardToVNCRGB(Swap32IfLE(pix));
  FillRect(rx, ry, rw, rh, pixel);
  
  if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, hdr.nSubrects * 8))
    return false;
  
  ptr = (CARD8 *)buffer;
  
  for (unsigned int i = 0; i < hdr.nSubrects; i++) {
    pix = *(CARD32 *)ptr;
    ptr += 4;
    x = *ptr++;
    y = *ptr++;
    ww = *ptr++;
    hh = *ptr++;
    
    pixel = CardToVNCRGB(Swap32IfLE(pix));
    FillRect(rx+x, ry+y, ww, hh, pixel);
  }
  return true;
}

#define GET_PIXEL8(pix, ptr) ((pix) = *(ptr)++)
#define GET_PIXEL16(pix, ptr) (((CARD8*)&(pix))[0] = *(ptr)++, \
			       ((CARD8*)&(pix))[1] = *(ptr)++)
#define GET_PIXEL32(pix, ptr) (((CARD8*)&(pix))[0] = *(ptr)++, \
			       ((CARD8*)&(pix))[1] = *(ptr)++, \
			       ((CARD8*)&(pix))[2] = *(ptr)++, \
			       ((CARD8*)&(pix))[3] = *(ptr)++)

bool VNCClient::HandleHextile8(int rx, int ry, int rw, int rh)
{
  CARD8 bg, fg;
  int i;
  CARD8 *ptr;
  int x, y, w, h;
  int sx, sy, sw, sh;
  CARD8 subencoding;
  CARD8 nSubrects;
  VNCRGB pixel;

  for (y = ry; y < ry+rh; y += 16) {
    for (x = rx; x < rx+rw; x += 16) {
      w = h = 16;
      if (rx+rw - x < 16)
	w = rx+rw - x;
      if (ry+rh - y < 16)
	h = ry+rh - y;

      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&subencoding, 1))
	return false;
      
      if (subencoding & rfbHextileRaw) {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, w * h ))
	  return false;
	
	(void) HandleRAW8(x, y, w, h);
	continue;
      }

      if (subencoding & rfbHextileBackgroundSpecified)
	if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&bg, sizeof(bg)))
	  return false;
      
      pixel = CardToVNCRGB((CARD32) bg);
      FillRect(x, y, w, h, pixel);
      
      if (subencoding & rfbHextileForegroundSpecified)
	if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&fg, sizeof(fg)))
	  return false;
      
      if (!(subencoding & rfbHextileAnySubrects))
	continue;

      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&nSubrects, 1))
	return false;

      ptr = (CARD8 *)buffer;

      if (subencoding & rfbHextileSubrectsColoured) {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, nSubrects * 3))
	  return false;

	for (i = 0; i < nSubrects; i++) {
	  fg = *ptr++;
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
	  pixel = CardToVNCRGB((CARD32) fg);
	  FillRect(x+sx, y+sy, sw, sh, pixel);
	}
      }
      else {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, nSubrects * 2))
	  return false;

	for (i = 0; i < nSubrects; i++) {
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
	  pixel = CardToVNCRGB((CARD32) fg);
	  FillRect(x+sx, y+sy, sw, sh, pixel);
	}
      }
    }
  }
  return true;
}

bool VNCClient::HandleHextile16(int rx, int ry, int rw, int rh)
{
  CARD16 bg, fg;
  int i;
  CARD8 *ptr;
  int x, y, w, h;
  int sx, sy, sw, sh;
  CARD8 subencoding;
  CARD8 nSubrects;
  VNCRGB pixel;

  for (y = ry; y < ry+rh; y += 16) {
    for (x = rx; x < rx+rw; x += 16) {
      w = h = 16;
      if (rx+rw - x < 16)
	w = rx+rw - x;
      if (ry+rh - y < 16)
	h = ry+rh - y;

      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&subencoding, 1))
	return false;
      if (subencoding & rfbHextileRaw) {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, w * h * 2))
	  return false;
	
	(void) HandleRAW16(x, y, w, h);
	continue;
      }

      if (subencoding & rfbHextileBackgroundSpecified)
	if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&bg, sizeof(bg)))
	  return false;
      
      pixel = CardToVNCRGB((CARD32) Swap16IfLE(bg));
      FillRect(x, y, w, h, pixel);
      
      if (subencoding & rfbHextileForegroundSpecified)
	if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&fg, sizeof(fg)))
	  return false;
      if (!(subencoding & rfbHextileAnySubrects))
	continue;

      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&nSubrects, 1))
	return false;

      ptr = (CARD8 *)buffer;

      if (subencoding & rfbHextileSubrectsColoured) {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, nSubrects * 4))
	  return false;

	for (i = 0; i < nSubrects; i++) {
	  GET_PIXEL16(fg, ptr);
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
	  pixel = CardToVNCRGB((CARD32) Swap16IfLE(fg));
	  FillRect(x+sx, y+sy, sw, sh, pixel);
	}
      }
      else {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, nSubrects * 2))
	  return false;

	for (i = 0; i < nSubrects; i++) {
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
	  pixel = CardToVNCRGB((CARD32) Swap16IfLE(fg));
	  FillRect(x+sx, y+sy, sw, sh, pixel);
	}
      }
    }
  }
  return true;
}

bool VNCClient::HandleHextile32(int rx, int ry, int rw, int rh)
{
  CARD32 bg, fg;
  int i;
  CARD8 *ptr;
  int x, y, w, h;
  int sx, sy, sw, sh;
  CARD8 subencoding;
  CARD8 nSubrects;
  VNCRGB pixel;

  trace(DBG_VNC, "VNCClient::HandleHextile32: rx=%d ry=%d rw=%d rh=%d",rx,ry,rw,rh);

  for (y = ry; y < ry+rh; y += 16) {
    for (x = rx; x < rx+rw; x += 16) {
      w = h = 16;
      if (rx+rw - x < 16)
	w = rx+rw - x;
      if (ry+rh - y < 16)
	h = ry+rh - y;

      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&subencoding, 1))
	return false;
      if (subencoding & rfbHextileRaw) {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, w * h * 4))
	  return false;
	
	(void) HandleRAW32(x, y, w, h);
	continue;
      }

      if (subencoding & rfbHextileBackgroundSpecified)
	if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&bg, sizeof(bg)))
	  return false;
      
      pixel = VNCRGB(Swap32IfLE(bg));
      FillRect(x, y, w, h, pixel);
      
      if (subencoding & rfbHextileForegroundSpecified)
	if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&fg, sizeof(fg)))
	  return false;
      if (!(subencoding & rfbHextileAnySubrects))
	continue;

      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&nSubrects, 1))
	return false;

      ptr = (CARD8 *)buffer;

      if (subencoding & rfbHextileSubrectsColoured) {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, nSubrects * 6))
	  return false;

	for (i = 0; i < nSubrects; i++) {
	  GET_PIXEL32(fg, ptr);
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
	  pixel = VNCRGB(Swap32IfLE(fg));
	  FillRect(x+sx, y+sy, sw, sh, pixel);
	}
      }
      else {
	if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer, nSubrects * 2))
	  return false;

	for (i = 0; i < nSubrects; i++) {
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
	  pixel = VNCRGB(Swap32IfLE(fg));
	  FillRect(x+sx, y+sy, sw, sh, pixel);
	}
      }
    }
  }
  return true;
}

/*
 * HandleRFBServerMessage.
 */
bool VNCClient::HandleRFBServerMessage()
{
  rfbServerToClientMsg msg;

  if (!FrameBuffer)
    return false;
  if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&msg, 1))
    return false;

  switch (msg.type) {

  case rfbSetColourMapEntries:
  {
    int i;
    CARD16 rgb[3];

    trace(DBG_FORCE, "HandleRFBServerMessage: rfbSetColourMapEntries not supported yet");

    if (!RFB_proto.VNC_sock.ReadFromRFBServer(((char *)&msg) + 1,
			   sz_rfbSetColourMapEntriesMsg - 1))
      return false;

    msg.scme.firstColour = Swap16IfLE(msg.scme.firstColour);
    msg.scme.nColours = Swap16IfLE(msg.scme.nColours);

    for (i = 0; i < msg.scme.nColours; i++) {
      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)rgb, 6))
	return false;
      /*
      // Here we should store the colormap
      xc.pixel = msg.scme.firstColour + i;
      xc.red = Swap16IfLE(rgb[0]);
      xc.green = Swap16IfLE(rgb[1]);
      xc.blue = Swap16IfLE(rgb[2]);
      xc.flags = DoRed|DoGreen|DoBlue;
      XStoreColor(dpy, cmap, &xc);
      */
    }
    break;
  }

  case rfbFramebufferUpdate:
  {
    rfbFramebufferUpdateRectHeader rect;
    int linesToRead;
    int bytesPerLine;
    int i;

    if (!RFB_proto.VNC_sock.ReadFromRFBServer(((char *)&msg.fu) + 1, sz_rfbFramebufferUpdateMsg - 1))
      return false;

    msg.fu.nRects = Swap16IfLE(msg.fu.nRects);

    for (i = 0; i < msg.fu.nRects; i++) {
      if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&rect, sz_rfbFramebufferUpdateRectHeader))
	return false;

      rect.r.x = Swap16IfLE(rect.r.x);
      rect.r.y = Swap16IfLE(rect.r.y);
      rect.r.w = Swap16IfLE(rect.r.w);
      rect.r.h = Swap16IfLE(rect.r.h);

      rect.encoding = Swap32IfLE(rect.encoding);

      if ((rect.r.x + rect.r.w > RFB_proto.si.framebufferWidth) ||
	  (rect.r.y + rect.r.h > RFB_proto.si.framebufferHeight)) {
	  trace(DBG_FORCE, "Rect too large: %dx%d at (%d, %d)",
		           rect.r.w, rect.r.h, rect.r.x, rect.r.y);
	  return false;
      }
      if ((rect.r.h * rect.r.w) == 0) {
	trace(DBG_FORCE, "Zero RFB_proto.size rect - ignoring");
	continue;
      }

      switch (rect.encoding) {

      case rfbEncodingRaw:
	bytesPerLine = rect.r.w * RFB_proto.myFormat.bitsPerPixel / 8;
	linesToRead = BUFFER_SIZE / bytesPerLine;

	while (rect.r.h > 0) {
	  if (linesToRead > rect.r.h)
	    linesToRead = rect.r.h;
	  if (!RFB_proto.VNC_sock.ReadFromRFBServer(buffer,bytesPerLine * linesToRead))
	    return false;

	  switch (RFB_proto.myFormat.bitsPerPixel) {
	  case 8:
	    if (!HandleRAW8(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	      return false;
	    break;
	  case 16:
	    if (!HandleRAW16(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	      return false;
	    break;
	  case 32:
	    if (!HandleRAW32(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	      return false;
	    break;
	  }
	  rect.r.h -= linesToRead;
	  rect.r.y += linesToRead;
	}
	break;

      case rfbEncodingCopyRect:
      {
	rfbCopyRect cr;
	if (!RFB_proto.VNC_sock.ReadFromRFBServer((char *)&cr, sz_rfbCopyRect))
	  return false;

	cr.srcX = Swap16IfLE(cr.srcX);
	cr.srcY = Swap16IfLE(cr.srcY);
	
	if (!HandleCR(cr.srcX, cr.srcY, rect.r.x, rect.r.y, rect.r.w, rect.r.h))
	  return false;
	break;
      }

      case rfbEncodingRRE:
	switch (RFB_proto.myFormat.bitsPerPixel) {
	case 8:
	  if (!HandleRRE8(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	case 16:
	  if (!HandleRRE16(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	case 32:
	  if (!HandleRRE32(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	}
	break;

      case rfbEncodingCoRRE:
	switch (RFB_proto.myFormat.bitsPerPixel) {
	case 8:
	  if (!HandleCoRRE8(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	case 16:
	  if (!HandleCoRRE16(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	case 32:
	  if (!HandleCoRRE32(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	}
	break;

      case rfbEncodingHextile:
	switch (RFB_proto.myFormat.bitsPerPixel) {
	case 8:
	  if (!HandleHextile8(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	case 16:
	  if (!HandleHextile16(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	case 32:
	  if (!HandleHextile32(rect.r.x,rect.r.y,rect.r.w,rect.r.h))
	    return false;
	  break;
	}
	break;

      default:
	trace(DBG_FORCE, "Unknown rect encoding %d", (int)rect.encoding);
	return false;
      }
    }
    if (!SendIncrementalFramebufferUpdateRequest())
      return false;
    break;
  }

  case rfbBell:
    //XBell(dpy,100);
    break;

  case rfbServerCutText:
    if (!RFB_proto.VNC_sock.ReadFromRFBServer(((char *)&msg) + 1,sz_rfbServerCutTextMsg - 1))
      return false;
    
    msg.sct.length = Swap32IfLE(msg.sct.length);

    if (serverCutText)
      free(serverCutText);
    serverCutText = (char *) malloc(msg.sct.length+1);

    if (!RFB_proto.VNC_sock.ReadFromRFBServer(serverCutText, msg.sct.length))
      return false;

    serverCutText[msg.sct.length] = 0;
    newServerCutText = true;
    break;

  default:
    trace(DBG_FORCE, "Unknown message type %d from VNC server", msg.type);
    return false;
  }
  trace(DBG_VNC, "Server Message Handled successfully!");
  return true;
}

int VNCClient::getSock()
{
  return RFB_proto.getSock();
}

bool VNCClient::SendIncrementalFramebufferUpdateRequest()
{
  return RFB_proto.SendIncrementalFramebufferUpdateRequest();
}

bool VNCClient::SendFramebufferUpdateRequest(int x, int y, int w, int h, bool incremental)
{
  return RFB_proto.SendFramebufferUpdateRequest(x, y, w, h, incremental);
}

