/*
   File     : http.c
   Author   : Lionel ULMER
   Creation : 19/02/97
   Updates  : 20/12/97 - Philippe Dax

   Functions "utils" network: call nameserver, connection to
   a HTTP server, URL parsing.
*/

#ifndef VRENGD

#include "global.h"
#include "texture.h"	/* TextureCacheEntry */


static u_int8 proxy=0, noproxy=0;
static u_int16 portproxy;
static char *domnoproxy, *hostproxy;


/* check if http proxy */
static
void httpProxy(void)
{
  char envproxy[90];
  char *p;
  static boolean done = FALSE;

  if (done)
    return;
  if (httpproxystr)
    p = httpproxystr;
  else
    p = getenv(HTTP_PROXY);	/* syntax: http://hostname:port/ */
  if (p && *p) {
    hostproxy = (char *) calloc(1, strlen(p));
    if (p[strlen(p) - 1] == '/')
      p[strlen(p) - 1] = '\0';
    strcpy(envproxy, p);
    p = strrchr(envproxy, ':');
    *p = '\0';
    portproxy = atoi(++p);
    if ((p = strrchr(envproxy, '/')) == NULL)
      sprintf(hostproxy, "http://%s", envproxy);
    else
      strcpy(hostproxy, ++p);
    proxy = 1;
    trace(DBG_HTTP, "proxy=%s:%d", hostproxy, portproxy);
  }
  if (noproxystr)
    p = noproxystr;
  else
    p = getenv(NO_PROXY);
  if (p && *p) {
    domnoproxy = (char *) calloc(1, strlen(p));
    strcpy(domnoproxy, p);
    noproxy = 1;
  }
  initResolve();
  done = TRUE;
}

static
int openFile(char *path)
{
  int fd = -1;

#if defined(WIN32) && !defined(CYGWIN32)
  fd = CreateFile(path, GENERIC_READ, 0, NULL, OPEN_EXISTING,
                        FILE_ATTRIBUTE_NORMAL, NULL);
#else
  fd = open(path, O_RDONLY);
#endif
  return fd;
}

/* fill buffer answer */
static
int httpAnswer(int s, char *answer, int max)
{
#if defined(WIN32) && !defined(CYGWIN32)
  return recv(s, answer, max, 0);
#else
  return read(s, answer, max);
#endif
}

/* send request to server */
static
int httpSend(int s, char *req, int lreq)
{
  int sent, r;

  for (sent = 0; sent < lreq; sent += r) {
#if defined(WIN32) && !defined(CYGWIN32)
    if ((r = write(s, req + sent, lreq - sent, 0)) == -1) {
#else
    if ((r = write(s, req + sent, lreq - sent)) == -1) {
#endif
      perror("httpSend: write");
      return BADSEND;
    }
  }
  return 0;
}

/* connect to server defined by sa */
static
int httpConnect(struct sockaddr_in *sa)
{
  int s;
  
  if ((s = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
    perror("httpConnect: socket");
    return BADSOCKET;
  }
  if (connect(s, (struct sockaddr*) sa, sizeof(struct sockaddr_in)) < 0) {
    perror("httpConnect: connect");
    close(s);
    return BADCONNECT;
  }
  return s;
}

/* URL parsing */
static
int httpParseURL(char *url, char *host, char *scheme, char *path)
{
  int type;
  char *p;
  static char prev_host[64];
  
  /* parse scheme */
  if (url == 0 || strlen(url) == 0) {
    *scheme = '\0';
    trace(DBG_HTTP, "BAD URL: %s", url);
    return URLBAD;
  }
  if (!strncmp(url, "http://", 7)) {    
    type = URLHTTP;
    strcpy(scheme, "http");
    url += 7;
  } else if (!strncmp(url, "telnet://", 9)) {
    type = URLTELNET;
    strcpy(scheme, "telnet");
    url += 9;
  } else if (!strncmp(url, "file:", 5)) {
    type = URLFILE;
    strcpy(scheme, "file");
    url += 5;
  } else if (!strncmp(url, "sip:", 4)) {
    type = URLSIP;
    strcpy(scheme, "sip");
    url += 4;
  } else if (!strncmp(url, "rtsp://", 7)) {
    type = URLRTSP;
    strcpy(scheme, "rtsp");
    url += 7;
  } else if (!strncmp(url, "ftp://", 6)) {
    type = URLFTP;
    strcpy(scheme, "ftp");
    url += 6;
  } else {
    trace(DBG_HTTP, "relative URL: %s", url);
#if 0
    *scheme = '\0';
    return URLBAD;
#endif
    strcpy(scheme, "http");
    strcpy(host, prev_host);
    if (*url !='/')
      sprintf(path, "/%s", url);
    else
      strcpy(path, url);
    return URLHTTP;
  }

  /* then parse host's name */
  if (type != URLFILE) {
    p = prev_host;
    while ((*url != ':') && (*url != '/') && (*url != '\0')) {
      *host++ = *url;
      *p++ = *url;	/* keep current host */
      url++;
    }
    *host = '\0';
    *p = '\0';
    /* trace(DBG_FORCE, "prev_host = %s", prev_host); */
  }

  /* then parse pathname */
  if (*url == '\0')
    strcpy(path, "/");
  else /* file */
    strcpy(path, url);
  return type;
}

/* http dowloading by threads */
void *
launchThread(void *info)
{
  int fd, type, ret, eoheader, headline;
  struct sockaddr_in sa;
  struct hostent *hp;
  ThreadLaunch *tl;
  HttpCon *handle;
  char *ptr;
  char host[80], scheme[8], path[MAXBUF], req[MAXBUF], httpmess[MAXBUF];

  httpProxy();

  tl = (ThreadLaunch *) info;
  if ((handle = (HttpCon *) calloc(1, sizeof(HttpCon))) == NULL)
    fatal("launchThread: can't alloc handle");

  beginThread(tl);
  trace(DBG_HTTP, "Opening url %s", tl->url);
  memset(host, 0, sizeof(host));
  memset(scheme, 0, sizeof(scheme));
  memset(path, 0, sizeof(path));
  type = httpParseURL(tl->url, host, scheme, path);
  trace(DBG_HTTP, "type=%d, scheme=%s host=%s path=%s", type,scheme,host,path);

  switch (type) {

  case URLFILE:
    if ((fd = openFile(path)) < 0)
      tl->callback(tl->handle, NULL);
    else {
      handle->fd = fd;
      handle->bptr = -1;
      tl->callback(tl->handle, (void *) handle);
    }
    goto fin;

  case URLHTTP:
    trace(DBG_HTTP, "HTTP: %s://%s/%s", scheme, host, path);
again:
    if (proxy && (!noproxy || strstr(host, domnoproxy) == 0)) {
      if ((hp = my_gethostbyname(hostproxy)) == NULL) {
        trace(DBG_HTTP, "my_gethostbyname hostproxy=%s", hostproxy);
	proxy = 0;
        noproxy = 0;
        goto again;
      }
      memset(&sa, 0, sizeof(sa));
      sa.sin_family = AF_INET;
      sa.sin_port = htons(portproxy);
      memcpy(&sa.sin_addr, hp->h_addr_list[0], hp->h_length);
      my_free_netdb(hp);
      ret = 0;
    }
    else {
      if ((ret = resolve(host, scheme, &sa)) != 0)
        trace(DBG_FORCE, "can't resolve %s err=%d", host, ret);
    }
    if (ret < 0) {	
      tl->callback(tl->handle, NULL);
      goto fin;
    }
    if ((fd = httpConnect(&sa)) < 0) {
      trace(DBG_FORCE, "can't connect %s fd=%d", host, fd);
      tl->callback(tl->handle, NULL);
      goto fin;
    }

    /* send GET request with adding useful infos */
    if (proxy && (!noproxy || strstr(host, domnoproxy) == 0))
      sprintf(req, "GET %s?version=%s&target=%s-%s%s HTTP/1.0\r\n\r\n",
        tl->url, VERSION, machinename, systemname, releasename);
    else
      sprintf(req, "GET %s?version=%s&target=%s-%s%s HTTP/1.0\r\n\r\n",
        path, VERSION, machinename, systemname, releasename);
    if (httpSend(fd, req, strlen(req)) < 0) {
      trace(DBG_FORCE, "can't httpSend req=%s fd=%d", req, fd);
      tl->callback(tl->handle, NULL);
      close(fd);
      goto fin;
    }

    /* parse header HTTP/1.0 */
    eoheader = 0;
    headline = 1; /* position first line */
    ptr = handle->buf;

    do {
/*******************************
      signal(SIGALRM, sigalrm);
      alarm(TIMEOUT_HTTP);
      if (setjmp(jmphttp)) {
        trace(DBG_HTTP, "Timeout on server");
        tl->callback(tl->handle, NULL);
        goto fin;
      }
*******************************/
      if ((handle->bl = httpAnswer(fd, ptr, MAXNCBUF)) <= 0) {
        eoheader = 1;	/* end of http header reached */
        break;
      }
      for (handle->bptr = 0; 1; ) {	
        int i = 0;
        char line[MAXBUF];

	while ((handle->bptr < handle->bl) && (ptr[handle->bptr] != '\n'))
	  line[i++] = ptr[handle->bptr++];
	if (handle->bptr < handle->bl) {
	  if (line[0] == '\r') { /* test end of header */
	    handle->bptr++;
	    eoheader = 1;
	    break;
	  }
	  line[i-1] = '\0'; /* null before last (\r) */
	  handle->bptr++;
	  trace(DBG_HTTP, "->%s", line);
	  if (headline) {
	    /* first line => get error code */
            int httperr, major, minor;

	    sscanf(line, "HTTP/%d.%d %d", &major, &minor, &httperr);
	    strcpy(httpmess, line+12);
	    trace(DBG_HTTP, "HTTP-Code_err (%d.%d): %d - %s %s",
		   major, minor, httperr, httpmess, tl->url);
	    if (httperr != HTTP_OK) {	      
	      tl->callback(tl->handle, NULL);
	      close(fd);
	      goto fin;
	    }
	    headline = 0; /* headline done */
	  }
          if (!strncmp(line, "Content-Type: ", 14)) {
            char *p;
            TextureCacheEntry *tc;

            if ((p = strchr(line, '/')) != NULL) {
              p++;
              trace(DBG_HTTP, "mime=%s %s", p, tl->url);
              /* only for textures */
              if (tl->handle) {
                tc = (TextureCacheEntry *) tl->handle;
                strcpy(tc->mime, p);
              }
            }
	  }
          i = 0;
	}
        else {
	  /* get datas by httpRead() by readCfgDatas() in wmgt */
	  break;
	}
      }
    } while (!eoheader);    

    handle->fd = fd;
    
    /* Call the callback */
    tl->callback(tl->handle, (void *) handle);
    break;

  case URLTELNET:
    if ((fd = httpConnect(&sa)) < 0) {
      tl->callback(tl->handle, NULL);
      goto fin;
    }
    sprintf(req, "\r\n\r\n");
    if (httpSend(fd, req, strlen(req)) < 0) {
      tl->callback(tl->handle, NULL);
      close(fd);
      goto fin;
    }
    break;

  case URLSIP:
  case URLRTSP:
  case URLFTP:
    warning("FTP/SIP/RTSP protocols not yet implemented");
    tl->callback(tl->handle, NULL);
    goto fin; 

  default:
    if (type)
      notice("unknown scheme = %d", type);
    tl->callback(tl->handle, NULL);
    goto fin;
  }
  close(fd);

  /* End */
fin:
  endThread(tl);
  if (tl) free(tl);
  if (handle) free(handle);
  return NULL;
}

int httpOpen(const char *url, void (*callback)(void *,void *), void *handle, int block)
{
  ThreadLaunch *tl;
  
  trace(DBG_HTTP, "httpOpen: %s", url);
  if ((tl = (ThreadLaunch *) calloc(1, sizeof(ThreadLaunch))) == NULL)
    fatal("httpOpen: can't alloc tl");

  tl->handle = handle;
  tl->callback = callback;
  tl->block = block;
  tl->wait = NULL;
  strcpy(tl->url, url);
  if (block == THREAD_NO_NONBLOCK) {
    return fifoThread(tl);
  }
  else {
    launchThread((void *) tl);
    return 0;
  }
}

void httpClose(void *_handle)
{
  HttpCon *handle = (HttpCon *) _handle;
  
  if (handle == NULL)
    return;
  close(handle->fd);
  free(handle);	/* Warning at bugs... */
  handle = NULL;
}

int httpRead(void *_handle, char *buf, int maxl)
{
  HttpCon *handle = (HttpCon *) _handle;

  if (handle == NULL) {
    trace(DBG_FORCE, "httpRead: can't download, handle NULL");
    return BADHANDLE;
  }

  int l = 0, len;

  /* Modifie par Fabrice, l= longueur lue, maxl= longueur restante a recevoir */
  if (handle->bptr < 0)
    len = 0;
  else
    len = handle->bl - handle->bptr;
  if (len > 0) {
    if (len > maxl)
      len = maxl;
    memcpy(buf, handle->buf + handle->bptr, len);
    l += len;
    maxl -= len;
    handle->bptr += len;
  }
  while (maxl > 0) {
#if defined(WIN32) && !defined(CYGWIN32)
    ReadFile(handle->fd, buf + l, maxl, &len, NULL);
#else
    len = read(handle->fd, buf + l, maxl);
#endif
    if (len < 0) {
      trace(DBG_FORCE, "BADREAD len=%d fd=%d l=%d maxl=%d bptr=%p bl=%p", len, handle->fd, l, maxl, handle->bptr, handle->bl);
      perror("httpRead");
      return BADREAD;
    }
    if (len == 0)
      break;
    l += len;
    maxl -= len;
  }
  return l;
}

/* converts host/port into sockaddr */
int resolve(char *hoststr, char *portstr, struct sockaddr_in *sa)
{
  struct hostent *hp;
  struct servent *sp;
  struct hostent host_hack;  
  u_int16 port;
  char *addr_hack;
  char addr[4];
  
  if (!strncmp(hoststr, "localhost", 9))
    strcpy(hoststr, "127.0.0.1");
  if (isdigit((int) *hoststr)) { /* numeric IPv4 address */
    int a0,a1,a2,a3;
    sscanf(hoststr, "%d.%d.%d.%d", &a0,&a1,&a2,&a3);
    addr[0]=a0;
    addr[1]=a1;
    addr[2]=a2;
    addr[3]=a3;

#if defined(WITH_PTHREAD) && defined(HAVE_GETHOSTBYNAME_R)
    hp = my_gethostbyaddr_r(addr);
#else
    hp = my_gethostbyaddr(addr);
#endif
    if (hp == NULL) {
      /* Bletcherous hack in case you do not have a nameserver */
      hp = &host_hack;
      hp->h_addrtype = AF_INET;
      addr_hack = addr;
      hp->h_addr_list = &addr_hack;
      hp->h_length = 4;
    }
  }
  else { /* hostname */
#if defined(WITH_PTHREAD) && defined(HAVE_GETHOSTBYNAME_R)
    hp = my_gethostbyname_r(hoststr);
#else
    hp = my_gethostbyname(hoststr);
#endif
    if (hp == NULL)
      return BADNAME;
  }

  if (isdigit((int) *portstr))
    port = htons(atoi(portstr));
  else {
#if defined(WITH_PTHREAD) && defined(HAVE_GETHOSTBYNAME_R)
    /*sp = my_getservbyname_r(portstr);*/
#else
    /*sp = my_getservbyname(portstr);*/
#endif
    sp = NULL;
    if (sp == NULL) {
      if (!strcmp(portstr, "http"))
        port = htons(HTTPPORT);
      else
        return BADSERV;
    }
    else {
      port = sp->s_port;
      my_free_netdb(sp);
    }
  }
  sa->sin_family = hp->h_addrtype;
  memcpy(&sa->sin_addr, hp->h_addr_list[0], hp->h_length);
  sa->sin_port = port;
  my_free_netdb(hp);

  return 0;
}

static u_int8 htbuf[BUFSIZ];
static int32 htbuf_pos = 0;
static int32 htbuf_len = 0;

void httpClearBuf(void)
{
  htbuf_pos = htbuf_len = 0;
}

int httpGetChar(void *handle)
{
  if (htbuf_pos >= htbuf_len) {	// eob
    htbuf_pos = 0;
    if ((htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf))) == 0)
      return -1;	// eof
    if (htbuf_len < 0)
      return -2;	// err
  }
  return htbuf[htbuf_pos++];
}

u_int8 httpGetByte(void *handle)
{
  if (htbuf_pos >= htbuf_len) {
    htbuf_pos = 0;
    htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf));
  }
  return htbuf[htbuf_pos++];
}

u_int8 httpGetC(void *th, ReadImageFunc read_func)
{
  if (htbuf_pos >= htbuf_len) {
    htbuf_pos = 0;
    htbuf_len = read_func(th, (char *) htbuf, sizeof(htbuf));
  }
  return htbuf[htbuf_pos++];
}

u_int32 httpGetInt(void *handle)
{
  u_int32 val;

  if (htbuf_pos >= htbuf_len) {
    htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val = htbuf[htbuf_pos++];
  if (htbuf_pos >= htbuf_len) {
    htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val |= (htbuf[htbuf_pos++] << 8);
  if (htbuf_pos >= htbuf_len) {
    htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val |= (htbuf[htbuf_pos++] << 16);
  if (htbuf_pos >= htbuf_len) {
    htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val |= (htbuf[htbuf_pos++] << 24);
  return val;
}

float httpGetFloat(void *handle)
{
  u_int32 nb = httpGetInt(handle);
  return *((float *) &nb);
}

int httpGetLine(void *handle, char *line)
{
  int i = 0;
  
  while (1) {
    int c = httpGetChar(handle);

    if (c == '\n')
      break;
    if (c < 0) {
      line[i] = '\0';
      return -1;
    }
    line[i++] = c;
  }
  line[i++] = '\0';
  return 0;
}

int httpFread(char *ptr, int size, int nitems, void *handle)
{
  int toread, len = nitems * size;

  while (len > 0) {
    if (htbuf_pos >= htbuf_len) {
      if ((htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf))) < 0)
	return (nitems - (len / size));
      htbuf_pos = 0;
    }
    toread = (len < (htbuf_len - htbuf_pos)) ? len : (htbuf_len - htbuf_pos);
    memcpy(ptr, htbuf + htbuf_pos, toread);
    htbuf_pos += toread;
    ptr += toread;
    len -= toread;
  }
  return nitems;
}

u_int32 httpGetBuf(void *handle, char *buffer, int maxlen)
{
  int32 in_pos = htbuf_len - htbuf_pos;

  if (in_pos >= maxlen) {
    memcpy(buffer, htbuf, maxlen);
    htbuf_pos += maxlen;
    return maxlen;
  }
  else {
    memcpy(buffer, htbuf, in_pos);
    htbuf_pos = htbuf_len;
    return in_pos + httpRead(handle, (char *) buffer+in_pos, maxlen-in_pos);
  }
}

u_int32 httpSkip(void *handle, int32 skiplen)
{
  int32 in_pos = htbuf_len - htbuf_pos;

  if (in_pos >= skiplen) {
    htbuf_pos += skiplen;
    return 0;
  }
  else {
    skiplen -= in_pos;
    while (skiplen > 0) {
      if ((htbuf_len = httpRead(handle, (char *) htbuf, sizeof(htbuf))) == 0)
        break;
      if (skiplen >= htbuf_len) {
        skiplen -= htbuf_len;
        htbuf_pos = htbuf_len;
      }
      else {
        htbuf_pos = skiplen;
        skiplen = 0;
      }
    }
    return skiplen;
  }
}

#endif /* !VRENGD */
