#ifndef VHDLTYPE_HH
#define VHDLTYPE_HH
//---------------------------------------------------------------------------
// Copyright (c) 1995-1996 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.


//---------------------------------------------------------------------------

#include <iostream>
#include "tyvis/TypeKind.hh"
#include "tyvis/Types.hh"
#include "tyvis/ArrayInfo.hh"
#include "tyvis/ObjectBase.hh"

using std::ostream;
using std::cout;

class SignalNetinfo;
class VHDLData;
class VHDLVTime;
template <class Type> class Variable;
class AccessVariable;

class TypeInfo;
class ScalarType;
class EnumerationType;
class PhysicalType;
class VHDLKernel;
class SignalBase;

/** The class VHDLType.

    This class is the base class for all the types that get used in a VHDL
    simulation.  The primary functionality of this class is to act as a
    interface class for all the types so that interaction between various
    modules is easier.  
*/
class VHDLType : public LValue {
public:
  /// The (default) destructor
  virtual ~VHDLType() {}

  /** This method is used to print a VHDLType object.  This method should not
      be used to dump ArrayType objects to achieve texio support.  This method
      is just a utility method.  More for debugging and maybe for tracing
      signals/variables.  In any case, the use of textio methods is encouraged.
      
      @param os The output stream to which the data must be dumped (default = cout).
  */
  virtual void print(ostream &os = cout) const = 0;

  /** Assignment operator for VHDL types.  NOTE that this is more of an internal
      method.  It should NOT be used to assign values for variables and signals.
      The assignVal method should be used for that purpose.  This method is a
      pure virtual method.  Every type derived from this class must overload
      this method and perform all the necessary operations.

      @param source The source VHDLType object.
      @return The ArrayType object with new value (*this)
  */
  virtual VHDLType &operator=(const VHDLType &source) = 0;

  /** This method is used to assign a driving value for the type object.  This
      method is overloaded by composite types (Array/Record) to perform the
      assignment correctly (as per the functionality specification of the
      types).

      @param The source VHDLType object
      @return The VHDLType object with new value (*this)
  */
  virtual VHDLType &assignVal(const VHDLType &source);

  /** Method to access a given element at the given position in a array.  Hence,
      it is valid only for ArrayTypes.  If this method is invoked on other
      types, it will report error and abort.  This is more of an internal access
      method for convinience.  This method should be moved to protected or
      private section of the class later on, if its visibility is not necessary.
      In general direct use of this method is discouraged.

      @param pos The offset/index within the array.
      @return The VHDLType constained at the given "pos".
  */
  virtual VHDLType &operator[](const int pos) const;

  /** Method to access a given element at the given position in a array.  This
      method is valid only for ArrayType objects and will report error and abort
      in all other cases.  This method is "safe" and must be used as the
      standard API call.

      @param pos The position/offset/index within the array
      @return The VHDLType constained at the given "pos"
  */
  virtual VHDLType &operator[](const ScalarType &pos) const;

  /** Method to access a given element at the given position in a array.  This
      method is valid only for ArrayType objects and will report error and abort
      in all other cases.  This method is "safe" and can be used as the standard
      API call.  This method is more of a convinience method (it calls other
      methods with suitable type casts to get its work done).

      @param pos The position/offset/index within the array
      @return The VHDLType constained at the given "pos"
  */
  virtual VHDLType &operator[](const VHDLType &pos) const;
  
  /** A convinience method to determine the object type of the elements
      constained by this array type.  Note: The method "get_kind()" MUST NOT be
      overloaded by a derived type.  This is to ensure that the type is
      recognized correctly.  It must be overloaded ONLY by the kernel types
      viz. "ScalarType", "ArrayType", "RecordType".

      @return The type of the object (signal/variable etc.).
  */
  virtual Type get_kind() const;

  /** Convenience method to determine if this VHDLType object is a scalar type.

      @return The method returns "true" if the object is a scalar type else it returns "false".
  */
  virtual bool is_scalar_type() const { return false; }

  /** Obtain the number of elements contained by a given dimension.  This method
      is valid only for ArrayTypes and RecordTypes.  If invoked on other types,
      this method will report error and abort.
      
      @param dimension The dimension of interest (1 or 2).
      @return The number of elements in that given dimension.
  */
  virtual int get_number_of_elements(int dimension = 0) const;
  
  /** A utility method that does not do bounds translation.  This is a purely
      internal method.  This should NOT be utilized by application programs.
      This method is valid only for ArrayTypes and in all other cases, the method
      will report an error and abort when invoked.
      
      @param index The offset within the array
      @return The element at the given offset.
  */
  virtual VHDLType &get_element(const int ) const;
  

  /** This method is used to obtain the object associated with this type.  This
      method is valid only for scalar types.  In all other cases, the method
      will report an error and abort.

      @return The pointer to the object contained by this type.
  */
  virtual ObjectBase* getObject() const;

  /** This method is used to set the pointer to the object associated with this
      type.  This method is valid only for scalar types.  In all other cases,
      the method will report an error and abort.
  */
  virtual void setObject(ObjectBase*);
  
  /** The bounds for a given dimension of the object.  This method may be
      invoked onlyu on Array Types. In the case or ArrayTypes the dimension may
      be 0 or 1.

      @param dimension The dimension of interest.
      @return The bounds for the given dimension.  */
  virtual const ArrayInfo &get_bounds(int dimension) const;

 /** A convinience method to determine the objecttype of the elements constained
      by this type.  This method must be overloaded (with a valid method) by all
      dervied classes.

      @return The type of the objects contained (signal/variable etc.).
 */
  virtual ObjectBase::ObjectType getKind() const = 0;

  /** Returns a clone of the object.  All the fields, dimensions, and elements
      are duplicated.  The return value is a fresh/new duplicate and the caller
      must ensure that the return value is deleted (otherwise, memory will leak)

      @return A pointer to the clone of the object.
  */  
  virtual VHDLType* clone() const = 0;

  /** Obtain the leftValue() of a scalar Type.  This method is valid only for
      scalar types.  It is a internal coninience method and must not be used by
      the user.

      @return UniversalValue of Type'Left of this object
  */
  virtual const VHDLData &leftValue();

    /** The string representation for the elements for the given ArrayType.
      This method is valid only for one-dimensional ArrayTypes whose elements
      are of EnumerationType.  The caller must remember to delete the return
      value.  Otherwise memory will leak.

      @return The "c" string representation for the elements of the ArrayType object */
  virtual char* getString() const;

  /** Obtain the left bounds for this object.  It must be noted that the bounds
      for an object and bounds for a type are essentially different.  However,
      they could be the same.  This method is valid for most types.  In the case
      of scalar types, the dimension (parameter) is ignored.  In the case or
      ArrayTypes the dimension may be 1 or 2.  In the case of RecordTypes, the
      dimension may be 1 to noOfFields.

      @param dimension The dimension of interest.
      @return The left bound for the requested dimension.
  */
  virtual int left(int dimension) const;

  /** Obtain the left bounds for this object.  It must be noted that the bounds
      for an object and bounds for a type are essentially different.  However,
      they could be the same.  This method is valid for most types.  In the case
      of scalar types, the dimension (parameter) is ignored.  In the case or
      ArrayTypes the dimension may be 1 or 2.  In the case of RecordTypes, the
      dimension may be 1 to noOfFields.  This method is the standard API used by
      Savant.

      @param dimension The dimension of interest.
      @return The left bound for the requested dimension.
  */
  VHDLType &left(const ScalarType &dimension) const;

  /** Obtain the right bounds for this object.  It must be noted that the bounds
      for an object and bounds for a type are essentially different.  However,
      they could be the same.  This method is valid for most types.  In the case
      of scalar types, the dimension (parameter) is ignored.  In the case or
      ArrayTypes the dimension may be 1 or 2.  In the case of RecordTypes, the
      dimension may be 1 to noOfFields.

      @param dimension The dimension of interest.
      @return The right bound for the requested dimension.
  */  
  virtual int right(int) const;


  /** Obtain the right bounds for this object.  It must be noted that the bounds
      for an object and bounds for a type are essentially different.  However,
      they could be the same.  This method is valid for most types.  In the case
      of scalar types, the dimension (parameter) is ignored.  In the case or
      ArrayTypes the dimension may be 1 or 2.  In the case of RecordTypes, the
      dimension may be 1 to noOfFields.  This method is the standard API used by
      Savant.

      @param dimension The dimension of interest.
      @return The left bound for the requested dimension.
  */
  VHDLType &right(const ScalarType &dimension) const;

  /** Obtain the direction for a given dimension of the Type.  In the case of
      scalar types, the dimension (parameter) is ignored.  In the case or
      ArrayTypes the dimension may be 1 or 2.  In the case of RecordTypes, the
      dimension may be 1 to noOfFields.

      @param dimension The dimension of interest (1/2).
      @return The direction for the specified dimension (to/downto).
  */
  virtual ArrayInfo::ArrayDirn_t dirn(int) const;

  /** Obtain the number of elements in a given dimension of a ArrayType.  This
      method is valid only for ArrayTypes.
      
      @param dimension The dimension of interest (1/2).
      @return The direction for the specified dimension (to/downto).
  */
  virtual int length(int) const;

  virtual int savantwrite( AccessVariable  &) const;
  virtual int savantwrite( AccessType  &) const;
  virtual int savantread( AccessVariable  &);
  virtual int savantread( AccessType  &);

  /** The resolution method for a composite resolved signal.  The TYPE's resolve
      is called only for composite resolved signals.  This resolve goes down to
      first sub-element of the VHDLType and calls the sub-elements resolve, but
      which actually does the resolution for the whole composite type.

      @param process A pointer to the process containing this object.
      @return The resolved value for this type.
  */
  virtual VHDLType *resolve( VHDLKernel *process );

  virtual void setResolutionFunctionId(int resolutionFnId);
  virtual void setTypeConversionFunctionId(int typeConversionFnId);
  virtual void updateEffVal(const VHDLType*);
  virtual void setParentCompositeType(VHDLType*);
  virtual void setCompositeResolvedSignal(bool);
  virtual void setElaborationInfo(const VHDLType &);
  virtual void setAttrib(AttribType, VHDLType&);
  virtual void initializeImplicitSignal(AttribType);
  virtual bool is_driver_already_set() const;
  virtual bool is_resolved_signal() const;
  virtual bool _is_composite_resolved_type() const;
  
  virtual bool _is_signal() const { return false; }
  virtual SignalBase* locateSig(int sigId);

  virtual SignalBase *findSigInBlock( int sigId, VHDLKernel* srcId );
  virtual SignalBase *findSigInBlock( int sigId, int processId );

  virtual void set_sourcebase_delete_flag(bool)const;
  virtual VHDLType *getParentCompositeType();

  virtual void resolveAndUpdate(VHDLKernel*);
  virtual void setBusKind();
  virtual void dump_connectivity_info(ofstream&){}
  virtual TypeInfo &getTypeInfo() const;

  virtual void Add( const VHDLType  &);
  virtual void Add( const SignalNetinfo * );
  virtual void Add( SignalNetinfo &sig );
  virtual void Add( VHDLKernel *processPtr );
  virtual void Add( VHDLKernel *processPtr, int sigid );

  virtual void assignSignal( VHDLKernel *destProcess, 
			     VHDLKernel *srcId, 
			     const VHDLType &src,
			     const PhysicalType &delay, 
			     const PhysicalType &rejTime, 
			     const ArrayInfo &dinfo,
			     const ArrayInfo &sinfo );


  /** Whenever a component is declared we create a derivation of
      _savant_entity_elab and we have the signals in the port. When it is
      actually bound to the real entity architecture, the signals inside
      the _savant_component are having different Signal ID's which are
      initialized in the SignalNetinfo node.  This was a hurdle to the
      TypeConversion. It was found that there is no harm in signals inside
      the _savant_entity_elab_instance having the same Signal ID's as the
      Signal ID's of the entity architecture to which it is bound to during
      instantiation or component configuration. This method in fact gets
      the signal ID's from the entity architecture and assigns to the
      corresponding signals in the component class. */
  virtual void copyId( const VHDLType &src );

  bool _is_alias() const { return is_alias; }

  /**
     Currently types are used interchangeably with their objects.  This
     method supports the RValue interface by proxying to the object.
  */
  const VHDLData &readVal() const { return getObject()->readVal(); }

  /**
     Currently types are used interchangeably with their objects.  This
     method supports the RValue interface by proxying to the object.
  */
  void updateVal( const VHDLData &newValue ) { getObject()->updateVal( newValue ); }
  /**
     Currently types are used interchangeably with their objects.  This
     method supports the RValue interface by proxying to the object.
  */
  int getIntValue() const { return getObject()->getIntValue(); };
  LONG getInt64Value() const { return getObject()->getInt64Value(); };
  double getDoubleValue() const { return getObject()->getDoubleValue(); };

  const EnumerationType &vhdlEqual( const RValue & ) const;
  const EnumerationType &vhdlNotEqual( const RValue & ) const;
  const EnumerationType &vhdlGreaterThan( const RValue & ) const;
  const EnumerationType &vhdlGreaterThanEqual( const RValue & ) const;
  const EnumerationType &vhdlLessThan( const RValue & ) const;
  const EnumerationType &vhdlLessThanEqual( const RValue & ) const;

  //@} // End of Public class Methods of VHDLType
  
protected:
  /**@name Public Class Methods of VHDLType. */
  //@{

  /** This is the default constructor.  However, this constructor is rather evil
      because it let's the user build object's that may not have sufficient
      information (such as TypeInfo data) for their working.  Right now, it is
      not very clear if this constructor is essential.  It should be removed if
      it is not necessary.

      @param alias A flag (true/flase) to indicate if this object is an alias.
  */
  VHDLType(bool alias = false) : is_alias(alias) {}


  /**@name Protected Class Methods of VHDLType. */
  //@{

  /// Flag to indicate alias objects
  bool is_alias;

  //@} // End of Protected Class Methods of VHDLType.
};

extern "C" bool eatwhite( AccessVariable &line );

/**
   This is the newer style of publishing.
*/
extern void assignVariable( LValue &dest, const RValue &newData );

/**
   This function is to ease to the new style of publishing.  Once we have
   ArrayData I think we won't need this any more.
*/
extern void assignVariable( VHDLType &destObject, const RValue &srcObject, 
			    const ArrayInfo &dinfo, const ArrayInfo &sinfo);

/**
   This function is used by the old-style publishing.
*/
extern void assignVariable( VHDLType &destObject, const VHDLType &srcObject, 
			    const ArrayInfo &dinfo, const ArrayInfo &sinfo);

extern ostream &operator<<(ostream &, const VHDLType &);

#endif
