#include "ScalarType.hh"
#include "AccessVariable.hh"
#include "SignalBase.hh"
#include "std_standardPkg.hh"
#include "SignalNetinfo.hh"
#include "VHDLProcess.hh"

using std::ends;

ScalarType::ScalarType(ObjectBase::ObjectType, bool alias) : VHDLType(alias) { 
  object = NULL;
}

ScalarType::ScalarType(const ScalarType& s) : VHDLType() {
  object   = s.object->clone();	// Returns an identical object.
  is_alias = s.is_alias;
}

ScalarType::~ScalarType() {
  if(is_alias == false && object != NULL) {
    delete object;
    object = NULL;
  }
}

char *
ScalarType::getString() const {
  char* ptr = new char[2];
  ptr[0] = (char)object->getIntValue();
  ptr[1] = '\0';
  return ptr;
}

int 
ScalarType::left() const {
  return 1;
}

ArrayInfo::ArrayDirn_t
ScalarType::dirn() const {
  return ArrayInfo::to;
}

int 
ScalarType::right() const {
  return 1;
}

void
ScalarType::increment() {
  ((UniversalInteger &) readVal()).increment();
}

bool
ScalarType::overflow() const
{
  return ((UniversalInteger &) readVal()).overflow();
}

void
ScalarType::decrement() {
  ((UniversalInteger &) readVal()).decrement();
}

void 
ScalarType::print(ostream& os) const {
  object->print(os);
}

VHDLType & 
ScalarType::operator =(const VHDLType& val) {
  object->operator=(*val.getObject());
  //I have commented this out because the destinations alias should
  //remain the same and not be copied from the source 
  //is_alias = val.is_alias;
  if (&getTypeInfo() != &TypeInfo::getNullTypeInfo())  {
    this->getTypeInfo().operator=(val.getTypeInfo());
  }
  return *this;
}

bool 
ScalarType::operator==( const RValue &val ) const {
  return val.operator==(*object);
}
bool 
ScalarType::operator!=( const RValue &val ) const {
  return object->readVal() != val;
}
bool 
ScalarType::operator>( const RValue &val ) const {
  return object->readVal() > val;
}
bool 
ScalarType::operator>=( const RValue &val ) const {
  return object->readVal() >= val; 
}
bool 
ScalarType::operator<( const RValue &val ) const {
  return object->readVal() < val; 
}
bool 
ScalarType::operator<=( const RValue &val ) const {
  return object->readVal() <= val; 
}

VHDLType& 
ScalarType::assignVal(const VHDLType& val) {
  if(val._is_alias() == true) {
    if ( object != NULL ){
      delete object;
    }
    object = val.getObject();
    is_alias = true;
  } else {
    *this = val;
  }
  return (*this);
}

ScalarType& 
ScalarType::operator =(const ScalarType& val) {
  ASSERT( object != 0 );
  object->operator=(*val.getObject());
  return (*this);
}

ObjectBase::ObjectType
ScalarType::getKind() const {
  return object->getKind();
}

VHDLType*
ScalarType::clone() const {
  VHDLType *retval = new ScalarType(*this);
  return retval;
}

void
ScalarType::setResolutionFunctionId(int resolutionFnId) {
  ASSERT(this->getKind() == ObjectBase::SIGNAL_NETINFO);
  this->getObject()->setResolutionFunctionId(resolutionFnId);
}

void
ScalarType::setTypeConversionFunctionId(int typeConversionFnId) {
  ASSERT(this->getKind() == ObjectBase::SIGNAL_NETINFO);
  this->getObject()->setTypeConversionFunctionId(typeConversionFnId);
}

void
ScalarType::setElaborationInfo(const VHDLType &obj_info) {
  ASSERT(_is_signal() == true);
  object->setElaborationInfo( *const_cast<ObjectBase *>(dynamic_cast<const ScalarType &>(obj_info).object) );
}

void
ScalarType::setAttrib(AttribType typ, VHDLType& attr) {
  ASSERT(_is_signal() == true);
  object->setAttrib(typ, attr);
}

void
ScalarType::updateEffVal(const VHDLType* newEffVal) {
  ASSERT(this->getKind() == ObjectBase::SIGNAL);
  ASSERT(newEffVal->is_scalar_type() == true);
  this->getObject()->updateVal(((ScalarType*)newEffVal)->readVal());
}

void
ScalarType::setParentCompositeType(VHDLType* ptr) {
  this->getObject()->setParentCompositeType(ptr);
}

void
ScalarType::setCompositeResolvedSignal(bool val) {
  this->getObject()->setCompositeResolvedSignal(val);
}

void
ScalarType::dump_connectivity_info(ofstream& fileStream) {
  switch(get_kind()) {
  case INTEGER_TYPE:
  case ENUMERATION_TYPE:
  case REAL_TYPE:
  case PHYSICAL_TYPE:
    ASSERT(getObject()->getKind() == ObjectBase::SIGNAL_NETINFO);
    ((SignalNetinfo*)getObject())->dump_connectivity_info(fileStream);
    ((SignalNetinfo*)getObject())->dump_driver_data(fileStream);
    break;

  default:
    break;
  }
}

VHDLType*
ScalarType::resolve( VHDLKernel *processPtr ){
  VHDLType *retval = getObject()->resolve(processPtr);
  // we check whether this scalar type is part of a composite resolved
  // type. If it is so, then we don't care. Otherwise, we need to update
  // the value of this scalar type with the value what we get from the
  // resolve

  // checking whether this scalar type is part of a composite resolved
  // type is done using the VHDLType *parentCompositeType value. If it
  // is not NULL, then this scalar type is part of a composite resolved
  // type.

  if ( getObject()->getParentCompositeType() == NULL){
    assignVariable( *this->getObject(), *retval->getObject() );
  }

  return retval;
}

int
ScalarType::savantwrite( AccessVariable &line ) const {
  ostringstream tempStr;
  tempStr << line.getVal();

  readVal().savantwrite(tempStr);
  
  string tempString = tempStr.str();
  line.setVal( tempString );
  
  return NORMAL_RETURN;
}

int
ScalarType::savantwrite(AccessType &line) const {
  this->readVal().savantwrite(line);

  return NORMAL_RETURN;
}

int
ScalarType::savantread(AccessVariable & line){
  string tempString = line.getVal();
  VHDLData *tempData = readVal().clone();

  tempData->savantread( const_cast<char *>(tempString.c_str()) );
  getObject()->updateVal(*tempData);

  delete tempData;

  line.setVal( tempString );
  
  return NORMAL_RETURN;
}

int
ScalarType::savantread(AccessType &line) {
  VHDLData   *tempData;
  
  tempData = this->readVal().clone();
  tempData->savantread(line);
  this->getObject()->updateVal(*tempData);

  delete tempData;

  return NORMAL_RETURN;
}

const VHDLData&
ScalarType::leftValue() {
  cerr << "ERROR!!! ScalarType::leftValue() called " << endl;
  abort();
  return *(new UniversalInteger(-1));
}

void
ScalarType::initializeImplicitSignal(AttribType attribType) {
  ObjectBase* object = this->getObject();
  SignalBase* signal = NULL;
  UniversalInteger tmp_integer(0);
  switch(attribType) {
  case LAST_EVENT:
  case LAST_ACTIVE:
    object->updateVal(UniversalLongLongInteger(-1));
    break;
  case EVENT:
  case ACTIVE:
    object->updateVal(UNIVERSAL_FALSE);
    break;
  case QUIET:
  case STABLE:
    signal = (SignalBase*)object;
    signal->updateDrvVal(&UNIVERSAL_TRUE, NULL, NULL);
    signal->updateEffVal(&UNIVERSAL_TRUE, NULL, NULL);
    break;
  case TRANSACTION:
    signal = (SignalBase*)object;
    signal->updateDrvVal(&tmp_integer, NULL, NULL);
    signal->updateEffVal(&tmp_integer, NULL, NULL);
    break;
  case LAST_VALUE:
    object->updateVal(leftValue());
    break;
  default:
    cerr << "Yet to take care of initializing Implicit attribute: " << attribType << endl;
    break;
  };
}

UniversalInteger
getIndex(const char* ptr, char** imageMap, int number_of_elements) {
  for(int i=0; i< number_of_elements; i++) {
    if ((strlen(imageMap[i]) == 3) && 
	(imageMap[i][0] == '\'') && (imageMap[i][2] == '\'')) {
      if(*ptr == imageMap[i][1]) {
	return UniversalInteger(i+1);
      }
    }
    else {
      if(strcmp(ptr, imageMap[i]) == 0) {
	return UniversalInteger(i+1);
      }
    }
  }
  
  return UniversalInteger(-1);
}
  
SignalBase *
ScalarType::locateSig(int sigId) {
  ASSERT(_is_signal() == true);
  SignalBase *ptr = (SignalBase*)getObject();
  ASSERT ( ptr != NULL );
  if( ptr->getSigId() == sigId ){
    return ptr;
  }
  return NULL;
}

void
ScalarType::setBusKind() {
  object->setBusKind();
}

SignalBase*
ScalarType::findSigInBlock(int sigId, VHDLProcess* srcId){
  SignalBase *driver_ptr = NULL;
  Block *driverList = NULL;
  int index;

  
  ASSERT(_is_signal() == true);
  ASSERT(getObject() != NULL);
  driverList = ((SignalBase *)getObject())->getAdditionalDriverList();


  driver_ptr =  ((SignalBase *)getObject())->findSigInBlock(sigId, srcId);
  if ( driver_ptr != NULL ){
    return driver_ptr;
  }
  
  if ( driverList != NULL ){
    for ( index = 0 ; index < driverList->getNumberOfElements() ; index++){
      driver_ptr = ((VHDLType *)driverList->getElement(index))->findSigInBlock(sigId, srcId);
      if ( driver_ptr != NULL ){
	return driver_ptr;
      }
    }
  }
  
  return NULL;
}

void 
ScalarType::Add( const VHDLType &rhs ){
  SignalNetinfo *addTo = dynamic_cast<SignalNetinfo*>( getObject() );
  SignalNetinfo *toAdd = dynamic_cast<SignalNetinfo*>( rhs.getObject() );
  ASSERT( addTo != 0 );
  ASSERT( toAdd != 0 );
  addTo->Add(toAdd);
}

void 
ScalarType::Add( const SignalNetinfo *ptr ){
  SignalNetinfo *addTo = dynamic_cast<SignalNetinfo*>( getObject() );
  ASSERT( addTo != 0 );
  addTo->Add(ptr);
}

void 
ScalarType::Add( VHDLProcess *processPtr, int sigId ){
  SignalNetinfo *addTo = dynamic_cast<SignalNetinfo*>( getObject() );
  ASSERT( addTo != 0 );
  addTo->Add( processPtr, sigId );
}

void 
ScalarType::copyId( const VHDLType &src ){
  SignalNetinfo *destSiginfo = dynamic_cast<SignalNetinfo *>(getObject());
  SignalNetinfo *srcSiginfo = dynamic_cast<SignalNetinfo *>(src.getObject());
  destSiginfo->id = srcSiginfo->id;
}

ostream&
operator<<(ostream &os, const ScalarType &st) {
  os << st.getObject()->readVal();

  return os;
}
