#ifndef RECORDTYPE_HH
#define RECORDTYPE_HH

//---------------------------------------------------------------------------
// Copyright (c) 1995-1996 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.
//---------------------------------------------------------------------------

#include "tyvis/VHDLType.hh"
#include "tyvis/TypeInfo.hh"
#include "tyvis/ElementAssociation.hh"

class ObjectBase;
class VHDLKernel;
class EnumerationType;
class SignalBase;

/** The class RecordType.

    This class represents all the record types and record subtype that get used
    in a VHDL simulation.  THe primary functionality of this class is to act as
    a place holder for the elements contained by an record type.  The basic
    structure for the record type is provided by the RecordTypeInfo class that
    contains all the necessary type information for each field of the RecordType
    object (signal/variable).
*/
    
class RecordType : public VHDLType {
public:
  /** This is the basic constructor for the record type.  This constructor should
      be used by all applications to build default RecordType objects.

      @param objType  The type of the object (variable/signal etc.).
      @param typeInfo The RecordTypeInfo structure associated with this object.
      @param resolveFnId Resolution function id, if any (-1 for invalid).
  */
  RecordType(ObjectBase::ObjectType objType, const RecordTypeInfo &typeInfo, ResolutionFnId_t resolveFnId);
  
  /** This constructor is used to create a new record type object whose fields
      are initialized using a given source record type object.
      
      @param objType  The type of the object (variable/signal etc.).
      @param typeInfo The RecordTypeInfo structure associated with this object.
      @param resolveFnId Resolution function id, if any (-1 for invalid).
      @param source The source RecrodType object.
  */
  RecordType(ObjectBase::ObjectType objType, const RecordTypeInfo &typeInfo,
	     ResolutionFnId_t resolveFnId, const RecordType &source);
  
  /** This constructor is used to obtain a duplicate of a given record type with
      different object types (.ie. variable/signal etc.).  This constructor is
      handly to build record types that have the structure but different
      objects.

      @param objType The type object the object contained by the record.
      @param source  The source RecordType object whose structure will be used.
  */
  RecordType(ObjectBase::ObjectType objType, const RecordType &source);

  /** This constructor can be used to build an alias for a given type of
      RecordType object.  The constructor ustilizes a given RecordType object to
      build another RecordType object (an alias).  Each field of the RecordType
      is an alias for the corresponding field in the source RecordType object.

      @param alias   A boolean flag to indicate if the new record type is an alias.
      @param objType The type of the objects to be constructed.
      @param source  The source RecordType object (which we are aliasing).
  */
  RecordType(bool alias, ObjectBase::ObjectType objType, const RecordType &source);

  /** A rather convinient copy constructor.

      @param source The source record type object to be used.
  */
  RecordType(const RecordType& source);
  
  /** Constructor for aggregate initialization of constrained record types.
      This constructor is typically used to initialize a record type with a set
      of constants.  The number of AssociationElements (pointers) provided in
      the constructor must match the number of fields in the RecordType (as
      defined in the typeInfo structure) and their types must match.

      NOTE: The constructor deletes the AssociationElement pointers.  Hence, the
      caller must "new" the pointers during the constructor call.

      @param objType  The type of the objects (signal/variable etc.) to be created.
      @param typeInfo     The RecordTypeInfo to be associated with this object.
      @param noOfElements The number of element associations in the constructor.
  */
  RecordType(ObjectBase::ObjectType objType, const RecordTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, int noOfElements, ...);

  /** Constructor for creating a temporary record aggregate This constructor is
      typically used to crate a record type with a set of fields from another
      record type particularly in variable/signal assignment statements where in
      the target is a record aggregate.  The number of AssociationElements
      (pointers) provided in the constructor must match the number of fields in
      the RecordType (as defined in the typeInfo structure) and their types must
      match.

      NOTE: The constructor deletes the AssociationElement pointers.  Hence, the
      caller must "new" the pointers during the constructor call.

      @param objType  The type of the objects (signal/variable etc.) to be created.
      @param typeInfo     The RecordTypeInfo to be associated with this object.
      @param noOfElements The number of element associations in the constructor.
  */
  RecordType(bool alias, ObjectBase::ObjectType objType, const RecordTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, int noOfElements, ...);

  /** A default constructor -- preferably not to be used */
  RecordType(ObjectBase::ObjectType);

  /** The destructor for RecordType.  */
  virtual ~RecordType();

  /** This method is used to identify the type.

      Note: The method "get_kind()" MUST NOT be overloaded by a derived type.
      This is to ensure that the type is recognized correctly.
  */
  Type get_kind() const { return RECORD_TYPE; }

  /** A convinience method to determine the object type (variable/signal etc.)
      of the elements constained by this record type.

      @return The type of the objects contained in the record (signal/variable etc.).  */
  virtual ObjectBase::ObjectType getKind() const;

 /** Returns a clone of the object.  All the fields/elements are duplicated.
      The return value is a fresh/new duplicate and the caller must ensure that
      the return value is deleted (otherwise, memory will leak)

      @return A pointer to the clone of the object.
 */
  VHDLType* clone() const;

  /** Obtain the number of fields in this record type.  This method overrides
      the virtual method defined in the base class (VHDLType) and provides a
      convinient method for handling RecordType and RecordType objects.

      @param dimension  The parameter is ignored (used ofr interface compaitablity).
      @return The number of fields in this RecordType object.
  */
  int get_number_of_elements( int = 0 ) const { return recordInfo.get_number_of_fields(); }

  /** Obtain the number of fields in this record type.
      
      @return The number of fields in this RecordType object.
  */
  int get_number_of_fields() const { return recordInfo.get_number_of_fields(); }
  

  /** A utility method to access each field of the given RecordType object.  The
      fieldNumber must be in the range: 0 < fieldNumber < numberOfFields.

      @param fieldNumber The fieldNumber (0 < fieldNumber < numberOfFields)
      @return The object corresponding to the given fieldNumber.
  */
  virtual VHDLType& get_field(int fieldNumber) const;

  /** Assignment operator for record types.  NOTE that this is more of an
      internal method.  It should NOT be used to assign values for variables and
      signals.  The assignVal method should be used for that purpose.

      @param source The source RecordType object.
      @return The RecordType object with new value (*this)
  */  
  VHDLType& operator=(const VHDLType& source);
 
  /** Assignment operator for record types.

      @param source The source RecordType object.
      @return The RecordType object with new value (*this)
  */ 
  RecordType& operator=(const RecordType& source);

  /** This method is used to compare if the given RecordType object is operator== to
      another.  Two RecordType objects are operator== if and only if: (i) They have
      the same number of fields; (ii) Each corresponding field in the two
      objects are operator==.

      @param val The RecordType object that is being compared.
      @return true/false indicating if the object is operator== or not.
  */
  bool operator==( const RValue & ) const;
  bool operator!=( const RValue & ) const;
  bool operator<( const RValue & ) const;
  bool operator<=( const RValue & ) const;
  bool operator>( const RValue & ) const;
  bool operator>=( const RValue & ) const;

  /** This method is used to assign the driving value for the record type object.

      @param The source RecordType object
      @return The RecordType object with new value (*this)
  */  
  VHDLType& assignVal(const VHDLType& val);

 /** This method is used to set the resolutionFnId for all elements that 
      contain this record type object.

      @param resolutionFnId The resolution function id.
  */
  virtual void setResolutionFunctionId(ResolutionFnId_t resolutionFnId);

  /** This method can be used to access the resolution function id for this
      record type object.

      @param The resolution function id (-1 if none present)
  */
  ResolutionFnId_t getResolutionFunctionId() const { return resolutionFunctionId; }

  /** This method is used to set the Typeconversion function id.

      @param typeConversionFnId The type conversion function id.
  */
  virtual void setTypeConversionFunctionId(TypeConversionFnId_t typeConversionFnId);

  /** This method is used to set the parent pointer to each field of the record
      type object.

      @param parent The pointer to the parent for each element.
  */  
  virtual void setParentCompositeType(VHDLType* parent);

  /** This method is used to indicate that the record type object is a composite
      resolved signal.  This method updates the corresponding fields of all the
      fields contained by this record type.

      @param flag Boolean value indicating whether the signal is composite-resolved or not.
  */  
  virtual void setCompositeResolvedSignal(bool flag);
  
 /** This method is invoked to update the elaboration information for each
      signal object contained by this record type.  The source must be a
      equivalent record type object whose objects are of type SignalNetinfo.
      This method is a part of the elaboration routine and is usually invoked
      from the VHDLKernelState::initiState().  The elaboration information
      contained in the source is filled in during elaboration.

      @param source The signal net for this record.
  */  
  void setElaborationInfo(const VHDLType&);

 /** This method is used to initialize/set the attribute object corresponding
      to the specific attribute types defined for this signal.  The source must
      be a equivalent RecordType.

      @param aType The attribute type being initialized.
      @param source The RecordType object containing the attributes.
  */  
  void setAttrib(AttribType, VHDLType&);

  /** The resolution interface for a RecordType.  The TYPE's resolve is called
       only for composite resolved signals. This resolve goes down to first
       sub-element of the VHDLType and calls the sub-elements resolve, but which
       actually does the resolution for the whole composite type.

       The TYPE's resolve is called only for composite resolved signals This
       resolve goes down to first sub-element of the VHDLType and calls the
       sub-elements resolve, but which actually does the resolution for the
       whole composite type.  The above given reason was the original reason for
       writing resolve in the RecordTypes.  But when TypeConversion came into
       picture, we needed resolve to be written for RecordTypes even if the
       RecordType was not a composite resolved signal.  In case of not a
       composite resolved type, resolve method just goes down to the sub
       elements and resolves the individual elements and then updates once the
       resolved value is obtained.

       @param process A pointer to the process containing this object.
       @return The resolved value for this object.
  */
  virtual VHDLType* resolve(VHDLKernel* process);

 /** This method is used to update the effective value of the signal objects
      contained by this record type.  The source for this method is ususally
      obtained by a call to resolve.

      @param source The source record that contains the effective value.
  */  
  virtual void updateEffVal(const VHDLType* source);


  /** Utility method to determine if the given record type object is a compositve
      resolved signal or not.

      @return Boolean value (true/false) indicating if signal is a composite resolved signal or not.
  */
  virtual bool _is_composite_resolved_type() const;
  
  /** A utility method to verify if the record type object is a signal or not.

      @return Returns true/false to indicate wether the object is a signal or not.
  */  
  virtual bool _is_signal() const { return get_field(1)._is_signal(); }

  /** This method is used to locate the driver object corresponding to the
      signalId passed in as the parameter.  This method searches all the
      elements (recursively) to locate the driver corresponding to each
      signalId.
      
      @param sigId The id of the signal whose dirver needs to be located.
      @return A pointer to the dirver (NULL if not found).
  */  
  virtual SignalBase* locateSig(int sigId);

  /** This method is used to locate the driver for a signal within a given
       process.

      @param sigId The id of the signal whose driver is required.
      @param srcId The object/process corresponding to the driver.
      @return A pointer to the driver (NULL if not found).
  */  
  virtual SignalBase* findSigInBlock(int sigId, VHDLKernel *srcId);

  /** Method to write the elements contained by the record type into the
      specified line.  This is a internal method and its direct use is
      discouraged.  This method is "deprecated" (will be removed in later release) 

      @param line The line that is going to contain the information.
      @return The return type (NORMAL_RETURN etc.)
  */  
  int savantwrite( AccessVariable &line ) const;

  /** Method to write the elements contained by the record type object into the
      specified accesstype variable.  The access type variable must be of "line"
      type.

      @param line A access variable of "line" type.
      @return The return type (NORTMAL_RETURN etc.)
  */  
  int savantwrite(AccessType &line) const;

 /** Method to read the data for the elements in the record type from the
      specified input line.  The elements in the line are altered (or consumed)
      during read.  This method is an internal method.  Its direct use is
      discouraged.  This method is "deprecated" (it will be removed in future
      release).

      @param line The line variable containing the data.
      @return The exit/return status (NORMAL_RETURN etc.)
  */  
  int savantread( AccessVariable &line );

  /** Method to read the elements contained by the record type object from the
      specified accesstype variable.  The access type variable must be of "line"
      type.
      
      @param line A access variable of "line" type.
      @return The return type (NORTMAL_RETURN etc.)
  */  
  int savantread(AccessType& line);

  /** Utility method to access the record type info. structure for a given
      record type object.

      @return The RecordTypeInfo structure for this object.
  */
  const RecordTypeInfo& getRecordTypeInfo() const;

  /** This method is used to print a RecordType object.  This method should not
      be used to dump RecordType objects to achieve texio support.  This method
      is just a utility method.  More for debugging and maybe for tracing
      signals.  In any case, the use of textio methods is encouraged.

      @param os The output stream to which the data must be dumped.
  */
  virtual void print(ostream& os) const;

  void Add( const VHDLType &rhs );
  void Add( const SignalNetinfo *rhs );
  void Add( VHDLKernel *processPtr );
  void Add( VHDLKernel *processPtr, int sigId );

  void copyId( const VHDLType &src );
  void resolveAndUpdate( VHDLKernel * );
  bool is_driver_already_set() const;
  bool is_resolved_signal() const;
  void set_sourcebase_delete_flag( bool flag ) const;

  virtual void assignSignal( VHDLKernel *destProcess, 
			     VHDLKernel *srcId, 
			     const VHDLType& src,
			     const PhysicalType& delay, 
			     const PhysicalType& rejTime, 
			     const ArrayInfo& dinfo,
			     const ArrayInfo& sinfo );


protected:
  /** This flag is for checking whether this composite type is a composite
      resolved type. This flag is set in the setCompositeResolvedSignal
      method.  It can be set even in setParentCompositeType method also. */
  bool isCompositeResolvedType;

  /** The Id for the resolution function associated with this type */
  ResolutionFnId_t resolutionFunctionId;

  /** The type info for this class */
  RecordTypeInfo recordInfo;

  /** The fields for this record type */
  VHDLType** fields;
  
private:
  VHDLType** buildRecord(ObjectBase::ObjectType objType);
};

void assignVariable(RecordType& dest, const RecordType& src, 
		    const ArrayInfo& dinfo, const ArrayInfo& sinfo);

extern ostream& operator<<(ostream &, const RecordType &);

#endif
