#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <GL/glu.h>
#include <cmath>

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#include <tulip/TlpTools.h>
#include "tulip/GlGraph.h"
#include "tulip/Glyph.h"

using namespace std;

const Color colorSelect2= Color(255, 102, 255, 255);
const Color colorMeta= Color(79, 36, 80, 255);
//====================================================
void SetMat(const Color &C) {
  float color[4];
  color[0]=((float)C.getR())/255.0;
  color[1]=((float)C.getG())/255.0;
  color[2]=((float)C.getB())/255.0;
  color[3]=1.0;
  glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
}
//=======================================================
//Calcul 3D,
//=======================================================
//Calcul de la matrice de transformation pour 
//le positionnement des flches sur les artes
void GlGraph::makeArrowMatrix(GLfloat *matrix, const Coord A, const Coord B, GLfloat sx, GLfloat sy, GLfloat sz) {
  GLfloat  xa,ya,za,xb,yb,zb;
  A.get(xa,ya,za);
  B.get(xb,yb,zb);
  //Coordonnes du centre
  GLfloat xm,ym,zm;
  //Vecteur AB
  GLfloat xab,yab,zab;
  //Vecteur V
  GLfloat xv,yv,zv;
  //Vecteur W
  GLfloat xw,yw,zw;
  //Calcul de M
  xm=xb;
  ym=yb;
  zm=zb;
  //Calcul de AB
  xab=xb-xa; if (fabs(xab)<0.001) xab=0;
  yab=yb-ya; if (fabs(yab)<0.001) yab=0;
  zab=zb-za; if (fabs(zab)<0.001) zab=0;
  //Normalisation de AB
  GLdouble dab,dw;
  dab=sqrt(xab*xab + yab*yab + zab*zab );

  if (fabs(dab)>0.0001) {
    xab=sx*xab/dab; 
    yab=sx*yab/dab; 
    zab=sx*zab/dab;
  }
  //Calcul de V
  if (fabs(xab)>0.001) {
    if (fabs(yab)>fabs(zab)) { 
      xv=yab; 
      yv=-xab; 
      zv=0;
    }
    else { 
      xv=zab; 
      yv=0; 
      zv=-xab;
    }
  }
  else 
    if (fabs(yab)>0.001) {
      xv=0; 
      yv=zab; 
      zv=-yab;
    }
    else {
      xv=-zab; 
      yv=0; 
      zv=0;
    }
  //Calcul de W
  xw= yv*zab - zv*yab;
  yw= zv*xab - xv*zab;
  zw= xv*yab - yv*xab;
  dw=sqrt( xw*xw + yw*yw + zw*zw);
  if (fabs(dw)>0.0001) {
    xw=sz*xw/dw;
    yw=sz*yw/dw;
    zw=sz*zw/dw;
  }
  matrix[0]=xw;   matrix[1]=yw;    matrix[2]=zw;    matrix[3]=0;
  matrix[4]=xv;   matrix[5]=yv;    matrix[6]=zv;    matrix[7]=0;
  matrix[8]=xab;  matrix[9]=yab;   matrix[10]=zab;  matrix[11]=0;
  matrix[12]=xm;  matrix[13]=ym;   matrix[14]=zm;   matrix[15]=1;
}
//========================================================
//Common display list
//========================================================
void GlGraph::cube(GLenum type) {
  //  GLenum type = GL_LINE_LOOP;
  /* front face */
  glBegin(type);
  glNormal3f(0.0f, 0.0f, 1.0f);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(-0.5f, -0.5f, 0.5f); 
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f(0.5f, -0.5f, 0.5f);
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(0.5f, 0.5f, 0.5f);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(-0.5f, 0.5f, 0.5f);
  glEnd();
  /* back face */
  glBegin(type);
  glNormal3f(0.0f, 0.0f, -1.0f);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f(-0.5f, -0.5f, -0.5f); 
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(-0.5f, 0.5f, -0.5f);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(0.5f, 0.5f, -0.5f);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(0.5f, -0.5f, -0.5f);
  glEnd();
  /* right face */
  glBegin(type);
  glNormal3f(1.0f, 0.0f, 0.0f);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f(0.5f, -0.5f, -0.5f); 
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(0.5f, 0.5f, -0.5f);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(0.5f, 0.5f, 0.5f);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(0.5f, -0.5f, 0.5f);
  glEnd();
  /* left face */
  glBegin(type);
  glNormal3f(-1.0f, 0.0f, 0.0f);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f(-0.5f, -0.5f, 0.5f); 
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(-0.5f, 0.5f, 0.5f);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(-0.5f, 0.5f, -0.5f);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(-0.5f, -0.5f, -0.5f);
  glEnd();
  /* top face */
  glBegin(type);
  glNormal3f(0.0f, 1.0f, 0.0f);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f(0.5f, 0.5f, 0.5f); 
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(0.5f, 0.5f, -0.5f);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(-0.5f, 0.5f, -0.5f);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(-0.5f, 0.5f, 0.5f);
  glEnd();
  /* bottom face */
  glBegin(type);
  glNormal3f(0.0f, -1.0f, 0.0f);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f(0.5f, -0.5f, -0.5f); 
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(0.5f, -0.5f, 0.5f);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(-0.5f, -0.5f, 0.5f);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(-0.5f, -0.5f, -0.5f);
  glEnd();
}
//====================================================
void GlGraph::solidCone() {
  GLUquadricObj *quadratic;
  quadratic = gluNewQuadric();
  gluQuadricNormals(quadratic, GLU_SMOOTH);
  gluQuadricTexture(quadratic, GL_TRUE);  
  glTranslatef(0.0f, 0.0f, -1.0f);
  gluQuadricOrientation(quadratic, GLU_INSIDE);
  gluDisk(quadratic, 0.0f, 0.5f, 8, 1);
  gluQuadricOrientation(quadratic, GLU_OUTSIDE);
  gluCylinder(quadratic, 0.5f, 0.0f, 1.0f, 8, 1);  
  gluDeleteQuadric(quadratic);
}
//====================================================
void GlGraph::transparentSphere() {
  glDepthMask(GL_FALSE);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA,GL_SRC_COLOR);
  SetMat(colorSelect2);
  GLUquadricObj *quadratic;
  quadratic = gluNewQuadric();
  gluQuadricDrawStyle(quadratic, GLU_FILL);
  gluQuadricNormals(quadratic, GLU_SMOOTH);
  gluQuadricTexture(quadratic, GL_FALSE); 
  gluSphere(quadratic, 1.0f, 15, 15);
  gluDeleteQuadric(quadratic);
  glDisable(GL_BLEND);
  glDepthMask(GL_TRUE);
}
//====================================================
void delList(GLuint dl) {
  if (glIsList(dl))  
    glDeleteLists(dl,1);
  if ( GLenum error=glGetError() != GL_NO_ERROR)
    std::cerr << "Open GL Error : " << error << " in " << __PRETTY_FUNCTION__ << std::endl;
}
//====================================================
void GlGraph::deleteDisplayLists() {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " (" << (int)this << ")" << endl;
#endif
  delList(metaGraphDL);
  delList(arrowDL);
  delList(selectionDL);
}
//==================================================
///Build the open gl display list
void GlGraph::buildDisplayLists() {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " (" << (int)this << ")" << endl;
#endif
  glNewList(arrowDL=glGenLists(1)    , GL_COMPILE ); {
    solidCone();
  } glEndList();  
  glNewList(selectionDL=glGenLists(1), GL_COMPILE); {
    //    transparentSphere();
    glPushAttrib(GL_ALL_ATTRIB_BITS);
    glDisable(GL_LIGHTING);
    //    glDepthMask(GL_FALSE);
    //    glEnable(GL_BLEND);
    //    glDisable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);
    glLineWidth(3);
    //    SetMat(colorSelect2);
    glColor4ub(colorSelect2[0], colorSelect2[1],  colorSelect2[2], colorSelect2[3]); 
    //    glBlendFunc(GL_ONE_MINUS_SRC_ALPHA, GL_SRC_COLOR);
    //    cube(GL_QUADS);
    //    glBlendFunc(GL_SRC_ALPHA, GL_SRC_COLOR);
    cube(GL_LINE_LOOP);
    //glDepthMask(GL_TRUE);
    //glEnable(GL_LIGHTING);
    glPopAttrib();
  } glEndList();
  glNewList(metaGraphDL=glGenLists(1), GL_COMPILE); {
    glPushAttrib(GL_ALL_ATTRIB_BITS);
    glDisable(GL_LIGHTING);
    glDepthFunc(GL_LEQUAL);
    glLineStipple(2, 0xAAAA);
    glEnable(GL_LINE_STIPPLE);
    glLineWidth(1);
    glColor4ub(colorMeta[0], colorMeta[1],  colorMeta[2], colorMeta[3]); 
    cube(GL_LINE_LOOP);
    glPopAttrib();
  } glEndList();
  GLenum error=glGetError();
  if ( error != GL_NO_ERROR)
    cerr << "Open GL Error : " << error << " in " << __PRETTY_FUNCTION__ << endl;
  Iterator<string> *itS = glyphFactory->availablePlugins();

  GlyphContext gc = GlyphContext(&_superGraph, this);
  glyphs.setAll(0);
  while (itS->hasNext()) {
    string glyphName=itS->next();
    //    cerr << "loading : " << glyphName << endl;
    Glyph *newGlyph = glyphFactory->getObject(glyphName, &gc);
    //    cerr << newGlyph << endl;
    //if (glyphFactory->objMap[glyphName]->getId() == 0) 
    //  glyphs.setAll(newGlyph);
    //    cerr << " id:" << glyphFactory->objMap[glyphName]->getId() << endl;
    glyphs.set(glyphFactory->objMap[glyphName]->getId(), newGlyph);
    //    cerr << glyphs.get(glyphFactory->objMap[glyphName]->getId()) << endl;
  } delete itS;
}
