/*
 * @(#)RiscOS.cpp 3.00 20 July 1999
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "tse3/platform/RiscOS.h"
#include "tse3/platform/midisswis.h"
#include "kernel.h"

using namespace TSE3;
using namespace TSE3::Plt;

/*
 * NB:
 *
 * This is all hypothetical code since I've not been able to test it on a
 * RISC OS platform. There isn't a decent C++ compiler, and the Acorn CFront
 * isn't quite good enough ;-)
 *
 * I include it here for completeness, and to exorcise the demons.
 *
 */


/******************************************************************************
 * RiscOsMidiSchedulerFactory class
 *****************************************************************************/

RiscOsMidiSchedulerFactory::RiscOsMidiSchedulerFactory()
{
}


RiscOsMidiSchedulerFactory::~RiscOsMidiSchedulerFactory()
{
}


MidiScheduler *RiscOsMidiSchedulerFactory::createScheduler()
{
    return new RiscOsMidiScheduler();
}


/******************************************************************************
 * RiscOsMidiScheduler class
 *****************************************************************************/

RiscOsMidiScheduler::RiscOsMidiScheduler(ostream &stream)
{
    std::cout << "RiscOsMidiScheduler created ("
              << implementationName() << ")\n";
    _swix(MIDI_Init, _IN(0) | _OUT(0), 0, &(maxPort));
}


RiscOsMidiScheduler::~RiscOsMidiScheduler()
{
    std::cout << "** RiscOsMidiScheduler deleted.\n";
}


const char *RiscOSMidiScheduler::implementationName()
{
    return "RiscOsMidiScheduler version 0.00 [dev].";
}


size_t RiscOSMidiScheduler::ports() const
{
    return maxPort;
}


const char *NullMidiScheduler::portName(size_t /*port*/) const
{
    return "RISC OS MIDI port";
}


const char *NullMidiScheduler::portType(size_t /*port*/) const
{
    return "RISC OS MIDI port";
}


bool NullMidiScheduler::portReadable(size_t /*port*/) const
{
    return true;
}


bool NullMidiScheduler::portWriteable(size_t /*port*/) const
{
    return true;
}


void RiscOsMidiScheduler::tx(MidiCommand mc)
{
    std::cout << "RiscOsMidiScheduler::tx - (now): "
              << mc.status << " on (" << mc.channel
              << "," << mc.port << ") with " << mc.data1
              << ", " << mc.data2 << "\n";
    int command = mc.channel
                + (mc.status << 4)
                + (mc.data1  << 8)
                + (mc.data2  << 16)
                + (mc.port   << 28)
    _swix(MIDI_TxCommand, _INR(0,1), command, 0);
}


void RiscOsMidiScheduler::start(Clock start)
{
    std::cout << "RiscOsMidiScheduler::start(" << start << ")\n";
    startClock = start;

    // start the fast clock
    if (_tempo == 0) _tempo = 120;
    _swix(MIDI_FastClock, _INR(0,1), 2500 / _tempo, 0);

    // start the interface sending timing clocks then send a midi stop
    //   so that the instrument doesn't play itself away!
    _swix(MIDI_TxStart, 0);
    _swix(MIDI_TxCommand, _INR(0,1), 0xfc, 0);

    _running = 1;
    notify(&MidiSchedulerListener::MidiScheduler_Started);
}


void RiscOsMidiScheduler::stop()
{
    std::cout << "RiscOsMidiScheduler::stopped at " << startClock << "\n";
    _swix(MIDI_TxStop, 0);
    _running = 0;
    notify(&MidiSchedulerListener::MidiScheduler_Stopped);
}


void RiscOsMidiScheduler::moveTo(Clock moveTime, Clock newTime)
{
    startClock -= moveTime - newTime;
    notify(&MidiSchedulerListener::MidiScheduler_Moved);
}


Clock RiscOsMidiScheduler::clock()
{
    int fcval;
    _swix(MIDI_FastClock, _IN(0) | _OUT(1), -1, &fcval);
    return msToClock(fcval);
}


int RiscOsMidiScheduler::msecs()
{
    int fcval;
    _swix(MIDI_FastClock, _IN(0) | _OUT(1), -1, &fcval);
    return fcval;
}


void RiscOsMidiScheduler::setTempo(int newTempo, Clock changeTime)
{
    std::cout << "treamMidiScheduler::setTempo - ignored\n";

    // assertion of valid input parameter
    if (newTempo < 1 || newTempo > 256) return;

    // N.B. startClock holds the scheduler clock at ms time 0
    // and by altering the time-line we have to work out a new value for
    // it to be consistent with the old one

    // work out a new value for start clock for the new tempo time-line
    //   we want as much accuracy as possible to be kept, hence the float cast
    startClock = changeTime - muldiv(newTempo, changeTime-startClock, _tempo);

    _tempo = newTempo;
}


bool RiscOsMidiScheduler::eventWaiting()
{
    // find input buffer size              // SHOULD THIS BE HELD IN VARIABLE?
    int buffer_size;
    _swix(MIDI_SetBufferSize, _INR(0,1) | _OUT(0), 0, 0, &buffer_size);

    for (int n = 0; n <= maxPort; ++n)
    {
        // find number of free bytes in input buffer
        int bufsize;
        _swix(MIDI_InqBufferSize, _IN(0) | _OUT(0), n<<1, &bufsize);

        // event waiting if the two aren't equal
        if (buffer_size != bufsize) return 1;
    }

    return 0; // all buffers empty
}


MidiEvent RiscOsMidiScheduler::rx()
{
    int c /*command*/, t /*time*/;
    _swix(MIDI_RxCommand, _IN(0) | _OUTR(0,1), -1, &c, &t);

    // return false if no event is pending
    if (c == 0) return MidiEvent();

    return MidiEvent(MidiCommand((c>>4)&0xf0, c&0xf, (c>>28)&0xf,
                                 (c>>8)&0x7f, (c>>16)&0x7f),
                     msToClock(t));
    remoteControl(e);
    return e;
}


void RiscOsMidiScheduler::tx(MidiEvent e)
{

    if (e.event.data.status == MidiCommand_Invalid) return;
    std::cout << "RiscOsMidiScheduler::tx - " << e.event.time << ": "
              << e.event.data.status << " on (" << e.event.data.channel
              << "," << e.event.data.port << ") with " << e.event.data.data1
              << ", " << e.event.data.data2 << "\n";
    int command = mc.channel
                + (mc.status << 4)
                + (mc.data1  << 8)
                + (mc.data2  << 16)
                + (mc.port   << 28)
    _swix(MIDI_TxCommand, _INR(0,1), command, clockToMs(time));
}


void RiscOSMidiScheduler::txSysEx(const unsigned char *data, size_t size)
{
    std::cout << "RiscOsMidiScheduler::txByte - (now) " << byte << "\n";
    _swix(MIDI_TxByte, _IN(0), MidiSystem_SysExStart);
    for (size_t n = 0; n < size; n++)
    {
        _swix(MIDI_TxByte, _IN(0), data[n]);
    }
    _swix(MIDI_TxByte, _IN(0), MidiSystem_SysExStop);
}
