/*
 * @(#)MidiMapper.h 3.00 25 May 1999
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifndef TSE3_MIDIMAPPER_H
#define TSE3_MIDIMAPPER_h

#include "tse3/listen/MidiMapper.h"

#include "tse3/Notifier.h"
#include "tse3/Filter.h"
#include "tse3/listen/MidiScheduler.h"

#include <utility>

namespace TSE3
{
    /**
     * The MidiMapper maintains a table mapping logical port/channel
     * destinations with where @ref MidiEvents should /really/ be sent.
     *
     * @short   MidiEvent channel/port destination mapper
     * @author  Pete Goodliffe
     * @version 3.00
     */
    class MidiMapper : public Filter,
                       public Notifier<MidiMapperListener>,
                       public Listener<MidiSchedulerListener>
    {
        public:

            /**
             * The default MidiMapper object performs no mapping whatsoever.
             *
             * @param scheduler The @ref MidiScheduler that will be
             *                  used. No events are sent to/read from the
             *                  scheduler, the MidiMapper merely keeps track
             *                  of the current number of ports through it.
             */
            MidiMapper(MidiScheduler *scheduler);
            virtual ~MidiMapper();

            /**
             * Returns the scheduler that this MidiMapper uses for port
             * information.
             */
            MidiScheduler *scheduler() const;

            /**
             * Returns the mapping for a particular channel and port.
             *
             * @param  fromChannel channel to read mapping for
             * @param  fromPort    port to read mapping for
             * @return A mapping pair (new channel, new port)
             * @see    setMap
             */
            const std::pair<int, int> map(int fromChannel, int fromPort) const;

            /**
             * Set the mapping for a particular channel and port
             *
             * @param fromChannel channel to set mapping for
             * @param fromPort    port to set mapping for
             * @param toChannel   channel to map to
             * @param toPort      port to map to
             * @see   map
             */
            void setMap(int fromChannel, int fromPort,
                        int toChannel, int toPort);

            /**
             * @reimplemented
             */
             const MidiEvent filter(const MidiEvent &me) const;

            /**
             * Reset the MidiMapper to a default state (i.e. no mappings).
             */
            void reset();

            /**
             * @reimplemented
             */
            virtual void MidiScheduler_Ports(MidiScheduler *s);

        private:

            MidiMapper &operator=(const MidiMapper &);
            MidiMapper(const MidiMapper &);

            class MidiMapperImpl *pimpl;
    };
}

#endif
