// disk.c - the disk emulation routines
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "emu/defs.h"
#include "emu/disk.h"

int disk_Open(void **FileRef, char *filename, int mode, int Blocks, int szBlock)
{
	DISK  *disk;
	char  ZeroBlock[szBlock];
	int   st;

	if (disk = (DISK *)calloc(1, sizeof(DISK))) {
		if ((disk->fd = open(filename, O_RDWR, 0)) < 0) {
			if ((disk->fd = open(filename, O_RDWR|O_CREAT, 0700)) < 0) {
				free(disk);
				return EMU_OPENERR;
			}
			memset(ZeroBlock, 0, szBlock);
			while (Blocks--)
				if((st = write(disk->fd, ZeroBlock, szBlock)) < 0) {
					close(disk->fd);
					unlink(filename);
					return EMU_OPENERR;
				}
		}
		disk->Type    = DISK_IMAGE;
		disk->Blocks  = Blocks;
		disk->szBlock = szBlock;
		*FileRef = (void *)disk;
		return EMU_OK;
	}
	*FileRef = NULL;
	return EMU_MEMERR;
}

int disk_Close(void **FileRef)
{
	DISK *disk = (DISK *)*FileRef;
	int st = 0;

	if (disk->Type == DISK_IMAGE) {
		st = close(disk->fd);
		free(disk);
	}
	*FileRef = NULL;
	return st ? EMU_IOERR : EMU_OK;
}

int disk_Seek(void *FileRef, int32 daddr, int pos)
{
	DISK *disk = (DISK *)FileRef;
	int st = 0;

	disk->Error = 0;
	if (disk->Type == DISK_IMAGE) {
		if ((st = lseek(disk->fd, daddr * disk->szBlock, pos)) < 0)
			disk->Error = errno;
	}
	return st;
}

int disk_Read(void *FileRef, uint8 *inBuffer)
{
	DISK *disk = (DISK *)FileRef;
	int st = 0;

	disk->Error = 0;
	if (disk->Type == DISK_IMAGE) {
		if ((st = read(disk->fd, inBuffer, disk->szBlock)) < 0)
			disk->Error = errno;
	}
	return st;
}

int disk_Write(void *FileRef, uint8 *outBuffer)
{
	DISK *disk = (DISK *)FileRef;
	int st = 0;

	disk->Error = 0;
	if (disk->Type == DISK_IMAGE) {
		if ((st = write(disk->fd, outBuffer, disk->szBlock)) < 0)
			disk->Error = errno;
	}
	return st;
}

int disk_GetError(void *FileRef) 
{
	DISK *disk = (DISK *)FileRef;
	
	return disk->Error;
}

#if 0
int disk_GetDiskAddr(DTYPE *dType, int Cylinder, int Track, int Sector)
{
	return (((Cylinder * dType->Tracks) + Track) * dType->Sectors) + Sector;
}
#endif
