/* Tower Toppler - Nebulus
 * Copyright (C) 2000-2006  Andreas Röver
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "event.h"
#include "decl.h"

#include <SDL.h>

EventHandler events; // global event handler object

void EventHandler::init() {
  SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
  SDL_EventState(SDL_MOUSEBUTTONDOWN, SDL_IGNORE);
  SDL_EventState(SDL_SYSWMEVENT, SDL_IGNORE);
  SDL_Init(SDL_INIT_JOYSTICK);

  SDL_EnableUNICODE(1);

  if(SDL_NumJoysticks()>0) {
    joy=SDL_JoystickOpen(0);
    if(joy) {
      printf("Joystick enabled\n"); 
    }
  } 

// zero out keymap to be sure
  for(int i=0;i<debug_ev;i++)  {
    keymap[i]=SDLK_UNKNOWN; 
    altkeymap[i]=SDLK_UNKNOWN;
  }

  keymap[fire_ev]=SDLK_SPACE;
  altkeymap[fire_ev]=SDLK_RETURN;
  keymap[left_ev]=SDLK_LEFT;
  altkeymap[left_ev]=SDLK_LEFT;
  keymap[right_ev]=SDLK_RIGHT;
  altkeymap[right_ev]=SDLK_RIGHT;
  keymap[up_ev]=SDLK_UP;
  altkeymap[up_ev]=SDLK_UP;
  keymap[down_ev]=SDLK_DOWN;
  altkeymap[down_ev]=SDLK_DOWN;
  keymap[pause_ev]=SDLK_p;
  altkeymap[pause_ev]=SDLK_p;
  keymap[break_ev]=SDLK_ESCAPE;
  altkeymap[break_ev]=SDLK_ESCAPE;

}

// 
// check_for()
// This checks to see if a particular event is currently occuring
//
bool EventHandler::check_for(const event_t event) {

  update();

  if(joy) { 
    int x=SDL_JoystickGetAxis(joy,0);
    int y=SDL_JoystickGetAxis(joy,1);
    int hat=SDL_JoystickGetHat(joy,0);
    switch(event) {
      case left_ev:
	if((x<0-joy_tolerance)||(hat==SDL_HAT_LEFT)) {
	  return true;
	}
	break;
      case right_ev:
	if((x>joy_tolerance)||(hat==SDL_HAT_RIGHT)) {
	  return true;
	}
	break;
      case up_ev:
	if((y<0-joy_tolerance)||(hat==SDL_HAT_UP)) {
	  return true;
	}
	break;
      case down_ev:
	if((y>joy_tolerance)||(hat==SDL_HAT_DOWN)) {
	  return true;
	}
	break;
      case fire_ev:
	if(SDL_JoystickGetButton(joy,0)) {
	  return true;
	}
	break;
      default:  // don't need this, but it eliminates warnings
	break;
    }
  }
 
  Uint8 *keystate=SDL_GetKeyState(NULL);


  if(keymap[event] && keystate[keymap[event]]) {
    return(true);
  } else if(altkeymap[event] && keystate[altkeymap[event]]) {
    return(true);
  }

  return false;
}

//
// get_input.  This is a non-blocking API that returns the next input 
// if provided.
//
Event *EventHandler::get_input() {
  SDL_Event e;
  while(SDL_PollEvent(&e)) {
    switch(e.type) {
      case SDL_KEYDOWN:
	for(int i=0;i<debug_ev;i++) {
	  if(keymap[i]==e.key.keysym.sym ) {
	    return(new KeyEvent((event_t)i,e.key.keysym));
	  }
	}
	for(int i=0;i<debug_ev;i++) {
	  if(altkeymap[i]==e.key.keysym.sym ) {
	    return(new KeyEvent((event_t)i,e.key.keysym));
	  }
	}
        return(new KeyEvent(key_ev,e.key.keysym));
	break;
      case SDL_MOUSEBUTTONDOWN:
	return(new MouseButtonEvent(e.button));
	break;
      case SDL_JOYAXISMOTION:
	if(e.jaxis.axis==0) {
	  if(e.jaxis.value < -joy_tolerance) {
	    return(new JoyEvent(left_ev));
	  } else if(e.jaxis.value > joy_tolerance) {
	    return(new JoyEvent(right_ev));
	  }
	} else if(e.jaxis.axis==1) {
	  if(e.jaxis.value < -joy_tolerance) {
	    return(new JoyEvent(up_ev));
	  } else if(e.jaxis.value > joy_tolerance) {
	    return(new JoyEvent(down_ev));
	  }
	}
	break;
      case SDL_JOYBUTTONDOWN:
	return(new JoyEvent(fire_ev));
	break;
      case SDL_JOYHATMOTION:
	if(e.jhat.value & SDL_HAT_LEFT) 
	  return(new JoyEvent(left_ev));
	if(e.jhat.value & SDL_HAT_RIGHT) 
	  return(new JoyEvent(right_ev));
	if(e.jhat.value & SDL_HAT_UP) 
	  return(new JoyEvent(up_ev));
	if(e.jhat.value & SDL_HAT_DOWN) 
	  return(new JoyEvent(down_ev));
	break;
      default:
	process_event(e); 
	break;
    }
  }
  return(0);
}




// get_nextkey.  This is a blocking API that returns the next key pressed
SDLKey EventHandler::get_nextkey(char *typed) {
SDL_Event e;
  while(SDL_WaitEvent(&e)) {
    if(e.type == SDL_KEYDOWN) {
      if(typed) {
        if ((e.key.keysym.unicode & 0xff80) == 0) {
          *typed=e.key.keysym.unicode & 0x7f;
	} else {
	  *typed=0;
	}
      }
      return(e.key.keysym.sym);
    }
    process_event(e); 
  }
  return(SDLK_BREAK); // should probably never happen, pick something to resolve warnings
}

//
// update()
// This will go through the event queue and return when all waiting events 
// are processed
//
void EventHandler::update() {
  SDL_Event e;
  while(SDL_PollEvent(&e)) {
    process_event(e);
  }
}

//
// wait_event()
// This is a blocking api that will return once an event occurs
//
void EventHandler::wait_event() {
  SDL_Event e;
  if(SDL_WaitEvent(&e)) {
    process_event(e);
  }
}

//
// process_event()  
// This function provides houskeeping for handling events, such as quit
//
void EventHandler::process_event(SDL_Event &e) {
  switch (e.type) {
    case SDL_QUIT:
      done();
      exit(0);
      break; 
    default:
      break;
  }
}

//
// wait_focus()
// This is a blocking api that will return once the app has focus.  If the
// app already has focus, it returns immediately
//
void EventHandler::wait_focus() {
// if we don't have focus, just hang around and wait for it
    while(!has_focus()) {
      wait_event();
    }
}


void EventHandler::done() {
    if(joy) {
	SDL_JoystickClose(joy);
    }
}

//
// get_keydef:  This gets the string value of what a key is defined as
//
const char *EventHandler::get_keydef(const event_t event) {
    if(keymap[event]) {
	return(SDL_GetKeyName(keymap[event]));
    } else {
	return("(none)");
    }
}

//
// get_keydef_sdl:  This gets the sdlkey value of what a key is defined as
//
SDLKey EventHandler::get_keydef_sdl(const event_t event) {
    if(keymap[event]) {
	return(keymap[event]);
    } else {
	return(SDLK_UNKNOWN);
    }
}

//
// set_keydef:  This sets the key for an event type
//
void EventHandler::set_keydef(const event_t event, SDLKey key) {
    keymap[event]=key;
}

//
// wait_nokey.  Wait until there is no key pressed or joystick fire
//
void EventHandler::wait_nokey(keyb_wait_proc bg) {
    bool exit;
    do {
	update();
	int numkeys;
	Uint8 *keystate=SDL_GetKeyState(&numkeys);
	exit=true;
	for(int i=null_ev+1;i<debug_ev;i++) {
	    if(keystate[keymap[i]]) {
		exit=false;
	    }
	}
	if(joy) { 
	    if(SDL_JoystickGetButton(joy,0)) {
		exit=false;
	    }
	}
	if (bg) (*bg)();
    } while(!exit);
}
