/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 364 $
 * $Date: 2012-01-22 20:27:14 +0800 (Sun, 22 Jan 2012) $
 */
#include "ThumbImage.hpp"

#include "../../image/ImageManipulation.hpp"

#include "wx/dcmemory.h"
#include "wx/dcbuffer.h"

#define ICON_SIZE 100
#define THUMB_SIZE 80

using namespace indii;

ThumbImage::ThumbImage(wxWindow *parent, const unsigned int cluster) :
    wxControl(parent, -1, wxDefaultPosition, wxDefaultSize, wxBORDER_NONE),
    model(NULL), cluster(cluster), isActive(false), isOver(false),
    fg(ICON_SIZE,ICON_SIZE), bg(ICON_SIZE,ICON_SIZE) {
  SetBackgroundStyle(wxBG_STYLE_CUSTOM);
  SetSize(ICON_SIZE,ICON_SIZE);
  SetMinSize(wxSize(ICON_SIZE,ICON_SIZE));

  fgPrepared = false;
}

void ThumbImage::setModel(ClusterModel* model) {
  this->model = model;
  ignore();

  if (model != NULL) {
    watch(model);
    prepareBackground();
    notifyNumClustersChange();
    notifySaturationDecayChange();
    notifySaturationSoftnessChange();
  }
}

void ThumbImage::notifyNumClustersChange() {
  if (cluster < model->getNumClusters()) {
    isActive = model->isShown(cluster);
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifySaturationDecayChange() {
  if (cluster < model->getNumClusters()) {
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifySaturationSoftnessChange() {
  if (cluster < model->getNumClusters()) {
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifyClusterChange(const unsigned int i) {
  //
}

void ThumbImage::notifySatChange(const unsigned int i) {
  /* pre-condition */
  assert (model != NULL);

  if (cluster == i && model->isShown(cluster) != isActive) {
    isActive = model->isShown(cluster);
    Refresh();
  }
}

void ThumbImage::OnPaint(wxPaintEvent& evt) {
  /* pre-condition */
  assert (model != NULL);

  if (cluster < model->getNumClusters()) {
    if (!fgPrepared) {
      prepareForeground();
    }

    int x, y;
    x = (ICON_SIZE - THUMB_SIZE) / 2;
    y = (ICON_SIZE - THUMB_SIZE) / 2;

    wxPaintDC dc(this);
    #ifndef WX_28
    dc.Clear();
    #endif
    if (isActive) {
      dc.SetBrush(wxBrush(*wxWHITE));
      dc.SetPen(wxPen(model->getColour(cluster), 3));
    } else if (isOver) {
      dc.SetBrush(wxBrush(*wxWHITE));
      dc.SetPen(wxPen(wxColor(128,128,128), 2));      
    } else {
      dc.SetBrush(wxBrush(*wxWHITE));
      dc.SetPen(wxPen(wxColor(128,128,128), 1));
    }
      
    dc.DrawRoundedRectangle(2, 2, ICON_SIZE - 4, ICON_SIZE - 4, 10);
    
    dc.DrawBitmap(bg, x, y);
    dc.DrawBitmap(fg, x, y);
  }
}

void ThumbImage::OnLeftDown(wxMouseEvent& evt) {
  isActive = !isActive;
  
  Refresh();
}

void ThumbImage::OnLeftUp(wxMouseEvent& evt) {
  /* pre-condition */
  assert (model != NULL);

  isActive = !model->isShown(cluster);
  Refresh();
  Update();
  model->show(cluster, isActive);
}

void ThumbImage::OnEnter(wxMouseEvent& evt) {
  isOver = true;  
  Refresh();
}

void ThumbImage::OnLeave(wxMouseEvent& evt) {
  isOver = false;  
  Refresh();
}

void ThumbImage::prepareBackground() {
  /* pre-condition */
  assert (model != NULL);

  int thumbWidth, thumbHeight;
  ImageResource* res = model->getImageResource();

  /* sizes, dimensions */
  res->fitOutside(THUMB_SIZE, THUMB_SIZE, &thumbWidth, &thumbHeight);

  /* crop */
  wxImage* image = res->get(thumbWidth, thumbHeight, true);
  if (thumbWidth > thumbHeight) {
    /* crop width */
    wxRect crop((thumbWidth - THUMB_SIZE) / 2, 0, THUMB_SIZE, THUMB_SIZE);
    bg = wxBitmap(image->GetSubImage(crop).ConvertToGreyscale());
  } else if (thumbHeight > thumbWidth) {
    /* crop height */
    wxRect crop(0, (thumbHeight - THUMB_SIZE) / 2, THUMB_SIZE, THUMB_SIZE);
    bg = wxBitmap(image->GetSubImage(crop).ConvertToGreyscale());
  } else {
    bg = wxBitmap(image->ConvertToGreyscale());
  }
}

void ThumbImage::prepareForeground() {
  /* pre-condition */
  assert (model != NULL);

  ImageResource* res = model->getImageResource();
  int thumbWidth, thumbHeight;

  /* sizes, dimensions */
  res->fitOutside(THUMB_SIZE, THUMB_SIZE, &thumbWidth, &thumbHeight);

  /* rectangle for crop */
  wxRect crop;
  if (thumbWidth > thumbHeight) {
    /* crop width */
    crop = wxRect((thumbWidth - THUMB_SIZE) / 2, 0, THUMB_SIZE, THUMB_SIZE);
  } else if (thumbHeight > thumbWidth) {
    /* crop height */
    crop = wxRect(0, (thumbHeight - THUMB_SIZE) / 2, THUMB_SIZE, THUMB_SIZE);
  } else {
    /* no crop */
    crop = wxRect(0, 0, THUMB_SIZE, THUMB_SIZE);
  }

  /* draw "blob" foreground */
  wxBitmap fg1(crop.width, crop.height); // ,32?
  wxBitmap fg2(crop.width, crop.height); // ,32?

  wxMemoryDC dc1(fg1);
  wxMemoryDC dc2(fg2);
  
  wxColour c1(model->getColour(cluster));
  wxColour c2;
  wxImage::RGBValue rgb(c1.Red(), c1.Green(), c1.Blue());
  wxImage::HSVValue hsv;
  
  hsv = wxImage::RGBtoHSV(rgb);
  hsv.value *= 0.9;
  rgb = wxImage::HSVtoRGB(hsv);
  c2.Set(rgb.red, rgb.green, rgb.blue);
  
  dc1.SetBackground(wxBrush(c1));
  dc2.SetBackground(wxBrush(c2));
  dc1.Clear();  
  dc2.Clear();

  wxImage img1(fg1.ConvertToImage());
  sparse_mask mask(model->calcMask(cluster, crop, thumbWidth, thumbHeight));
  hide(img1);
  show(img1, mask);
  
  dc2.DrawBitmap(img1, 0, 0, true);
  
  wxImage img2(fg2.ConvertToImage());
  a.resize(crop.height, crop.width);
  model->calcAlpha(cluster, crop, thumbWidth, thumbHeight, a);
  applyAlpha(img2, a);
  fg = wxBitmap(img2.Blur(2));
  
  fgPrepared = true;
}

BEGIN_EVENT_TABLE(ThumbImage, wxWindow)
EVT_PAINT(ThumbImage::OnPaint)
//EVT_LEFT_DOWN(ThumbImage::OnLeftDown)
EVT_LEFT_UP(ThumbImage::OnLeftUp)
EVT_ENTER_WINDOW(ThumbImage::OnEnter)
EVT_LEAVE_WINDOW(ThumbImage::OnLeave)
END_EVENT_TABLE()
