/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 363 $
 * $Date: 2012-01-22 18:45:43 +0800 (Sun, 22 Jan 2012) $
 */
#ifndef INDII_GUI_DOCUMENT_HPP
#define INDII_GUI_DOCUMENT_HPP

#include "PreviewPane.hpp"
#include "../model/Model.hpp"
#include "../model/Observer.hpp"
#include "../image/ImageResource.hpp"

#include "wx/panel.h"

namespace indii {
/**
 * Image document.
 */
class Document : public wxPanel, public Observer {
public:
  /**
   * Constructor.
   *
   * @param parent Parent.
   * @param res Image resource.
   * @param model Model.
   * @param control Controller.
   */
  Document(wxWindow* parent, ImageResource* res, Model* model,
      Controller* control = NULL);

  /**
   * Constructor.
   *
   * @param parent Parent.
   *
   * Use set() to set image resource and model before use.
   */
  Document(wxWindow* parent);

  /**
   * Is image saved?
   */
  bool saved();

  /**
   * Save image.
   */
  void save();

  /**
   * Save image, prompting for path.
   */
  void saveAs();

  /**
   * Does image have path?
   */
  bool hasPath();

  /**
   * Get image path.
   */
  const wxString& getPath() const;

  /**
   * Set image path.
   */
  void setPath(const wxString& path);

  /**
   * Notify.
   */
  virtual void notify();

  /**
   * Get preview pane.
   */
  PreviewPane* getPreviewPane();

  /**
   * Get model.
   */
  Model* getModel();

protected:
  /**
   * Set model and resource.
   */
  void set(ImageResource* res, Model* model, Controller* control = NULL);

  /**
   * Preview pane.
   */
  PreviewPane* previewPane;

  /**
   * Image resource.
   */
  ImageResource* res;

  /**
   * Model.
   */
  Model* model;

  /**
   * Controller.
   */
  Controller* control;

private:
  /**
   * Is the image, in its current state, saved?
   */
  bool isSaved;

  /**
   * Do we have a path for saving the image?
   */
  bool havePath;

  /**
   * Path to saved image.
   */
  wxString path;

  /**
   * Initialise document.
   */
  void init();

  DECLARE_EVENT_TABLE()
};
}

inline bool indii::Document::hasPath() {
  return havePath;
}

inline bool indii::Document::saved() {
  return isSaved;
}

inline void indii::Document::notify() {
  isSaved = false;
}

inline indii::PreviewPane* indii::Document::getPreviewPane() {
  return previewPane;
}

inline indii::Model* indii::Document::getModel() {
  return model;
}

#endif
