
/******************************************************************************
* MODULE     : glue.gen.cc
* DESCRIPTION: Glue for linking TeXmacs commands to guile
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <server.gen.h>
#include <connect.gen.h>
#include <file.gen.h>
#include <dir.gen.h>

#module code_glue
#import server
#import connect
#import file
#import dir
#include <string.h>
#include <guile/gh.h>
#include <libguile.h>
#define SCM_ARG8 8
#define SCM_ARG9 9

#ifdef DOTS_OK
typedef SCM (*FN)(...);
#else
typedef SCM (*FN)();
#endif

extern void initialize_glue_basic ();
extern void initialize_glue_editor ();
extern void initialize_glue_server ();
extern void thomas_test (string s);

/******************************************************************************
* Booleans
******************************************************************************/

#define SCM_ASSERT_BOOL(flag,arg,rout) \
  SCM_ASSERT (gh_boolean_p (flag), flag, arg, rout)

static SCM
bool_to_scm (bool flag) {
  return gh_bool2scm (flag);
}

static bool
scm_to_bool (SCM flag) {
  return gh_scm2bool (flag);
}

/******************************************************************************
* Integers
******************************************************************************/

#define SCM_ASSERT_INT(i,arg,rout) \
  SCM_ASSERT (SCM_INUMP (i), i, arg, rout);

/*static*/
SCM
int_to_scm (int i) {
  return gh_long2scm ((long) i);
}

static int
scm_to_int (SCM i) {
  return (int) gh_scm2long (i);
}

/******************************************************************************
* Strings
******************************************************************************/

#define SCM_ASSERT_STRING(s,arg,rout) \
  SCM_ASSERT (SCM_NIMP (s) && SCM_STRINGP (s), s, arg, rout)

static SCM
string_to_scm (string s) {
  char* _s= as_charp (s);
  SCM r= gh_str2scm (_s, strlen (_s));
  delete[] _s;
  return r;
}

string
scm_to_string (SCM s) {
  guile_str_size_t len_r;
  char* _r= gh_scm2newstr (s, &len_r);
  string r= _r;
  free (_r);
  return r;
}

static SCM
symbol_to_scm (string s) {
  char* _s= as_charp (s);
  SCM r= gh_symbol2scm (_s);
  delete[] _s;
  return r;
}

static string
scm_to_symbol (SCM s) {
  guile_str_size_t len_r;
  char* _r= gh_symbol2newstr (s, &len_r);
  string r= _r;
  free (_r);
  return r;
}

/******************************************************************************
* Trees
******************************************************************************/

static long tree_tag;

#define scm_is_tree(t) \
  (SCM_NIMP (t) && (((long) SCM_CAR (t)) == tree_tag))
#define SCM_ASSERT_TREE(t,arg,rout) \
  SCM_ASSERT (scm_is_tree (t), t, arg, rout)

static SCM
tree_to_scm (tree t) {
  SCM tree_smob;
  SCM_NEWCELL (tree_smob);
  SCM_SETCDR (tree_smob, (SCM) ((void*) (new tree (t))));
  SCM_SETCAR (tree_smob, tree_tag);
  return tree_smob;
}

static tree
scm_to_tree (SCM tree_smob) {
  return *((tree*) SCM_CDR (tree_smob));
}

static SCM
mark_tree (SCM tree_smob) {
  (void) tree_smob;
  return SCM_BOOL_F;
}

static scm_sizet
free_tree (SCM tree_smob) {
  tree *ptr = (tree *) SCM_CDR (tree_smob);
  delete ptr;
  return sizeof (tree); // should be replaced by total size of the tree
}

static int
print_tree (SCM tree_smob, SCM port, scm_print_state *pstate) {
  (void) pstate;
  tree   t= scm_to_tree (tree_smob);
  string s= "<tree " * tree_to_texmacs (t) * ">";
  scm_display (string_to_scm (s), port);
  return 1;
}

static SCM
cmp_tree (SCM t1, SCM t2) {
  return gh_bool2scm (scm_to_tree (t1) == scm_to_tree (t2));
}

tree
coerce_string_tree (string s) {
  return s;
}

string
coerce_tree_string (tree t) {
  return as_string (t);
}

/******************************************************************************
* Scheme trees
******************************************************************************/

#define SCM_ASSERT_SCHEME_TREE(p,arg,rout)

/*static*/
SCM
scheme_tree_to_scm (scheme_tree t) {
  if (is_atomic (t)) {
    string s= t->label;
    if (s == "#t") return SCM_BOOL_T;
    if (s == "#f") return SCM_BOOL_F;
    if (is_int (s)) return int_to_scm (as_int (s));
    if ((N(s)>=2) && (s[0]=='\42') && (s[N(s)-1]='\42'))
      return string_to_scm (s (1, N(s)-1));
    return symbol_to_scm (t->label);
  }
  else {
    int i;
    SCM p= gh_list (SCM_UNDEFINED);
    for (i=N(t)-1; i>=0; i--)
      p= gh_cons (scheme_tree_to_scm (t[i]), p);
    return p;
  }
}

scheme_tree
scm_to_scheme_tree (SCM p) {
  if (gh_list_p (p)) {
    tree t (TUPLE);
    while (!gh_null_p (p)) {
      t << scm_to_scheme_tree (gh_car (p));
      p= gh_cdr (p);
    }
    return t;
  }
  if (gh_symbol_p (p)) {
    string s= scm_to_symbol (p);
    if (s == "quote") return "'"; else return s;
  }
  if (gh_string_p (p)) return "\"" * scm_to_string (p) * "\"";
  if (SCM_INUMP (p)) return as_string (scm_to_int (p));
  if (gh_boolean_p (p)) return (scm_to_bool (p)? string ("#t"): string ("#f"));
  return "?";
}

/******************************************************************************
* TeXmacs trees
******************************************************************************/

#define texmacs_tree tree
#define SCM_ASSERT_TEXMACS_TREE SCM_ASSERT_TREE
#define texmacs_tree_to_scm tree_to_scm
#define scm_to_texmacs_tree scm_to_tree

/******************************************************************************
* Paths
******************************************************************************/

static bool
scm_is_path (SCM p) {
  if (gh_null_p (p)) return TRUE;
  else return SCM_INUMP (gh_car (p)) && scm_is_path (gh_cdr (p));
}

#define SCM_ASSERT_PATH(p,arg,rout) \
  SCM_ASSERT (scm_is_path (p), p, arg, rout)

static SCM
path_to_scm (path p) {
  if (nil (p)) return gh_list (SCM_UNDEFINED);
  else return gh_cons (int_to_scm (p->item), path_to_scm (p->next));
}

static path
scm_to_path (SCM p) {
  if (gh_null_p (p)) return path ();
  else return path (scm_to_int (gh_car (p)), scm_to_path (gh_cdr (p)));
}

/******************************************************************************
* Displays
******************************************************************************/

static long display_tag;

#define scm_is_display(t) \
  (SCM_NIMP (t) && (((long) SCM_CAR (t)) == display_tag))
#define SCM_ASSERT_DISPLAY(t,arg,rout) \
  SCM_ASSERT (scm_is_display (t), t, arg, rout)

static SCM
display_to_scm (display t) {
  SCM display_smob;
  SCM_NEWCELL (display_smob);
  SCM_SETCDR (display_smob, (SCM) ((void*) (new display (t))));
  SCM_SETCAR (display_smob, display_tag);
  return display_smob;
}

static display
scm_to_display (SCM display_smob) {
  return *((display*) SCM_CDR (display_smob));
}

static SCM
mark_display (SCM display_smob) {
  (void) display_smob;
  return SCM_BOOL_F;
}

static scm_sizet
free_display (SCM display_smob) {
  display *ptr = (display *) SCM_CDR (display_smob);
  delete ptr;
  return sizeof (display); // should be replaced by total size of the display
}

static int
print_display (SCM display_smob, SCM port, scm_print_state *pstate) {
  (void) display_smob; (void) pstate;
  string s= "<display>";
  scm_display (string_to_scm (s), port);
  return 1;
}

static SCM
cmp_display (SCM t1, SCM t2) {
  return gh_bool2scm (scm_to_display (t1) == scm_to_display (t2));
}

/******************************************************************************
* Widgets
******************************************************************************/

static long widget_tag;

#define scm_is_widget(t) \
  (SCM_NIMP (t) && (((long) SCM_CAR (t)) == widget_tag))
#define SCM_ASSERT_WIDGET(t,arg,rout) \
  SCM_ASSERT (scm_is_widget (t), t, arg, rout)

static SCM
widget_to_scm (widget t) {
  SCM widget_smob;
  SCM_NEWCELL (widget_smob);
  SCM_SETCDR (widget_smob, (SCM) ((void*) (new widget (t))));
  SCM_SETCAR (widget_smob, widget_tag);
  return widget_smob;
}

static widget
scm_to_widget (SCM widget_smob) {
  return *((widget*) SCM_CDR (widget_smob));
}

static SCM
mark_widget (SCM widget_smob) {
  (void) widget_smob;
  return SCM_BOOL_F;
}

static scm_sizet
free_widget (SCM widget_smob) {
  widget *ptr = (widget *) SCM_CDR (widget_smob);
  delete ptr;
  return sizeof (widget); // should be replaced by total size of the widget
}

static int
print_widget (SCM widget_smob, SCM port, scm_print_state *pstate) {
  (void) widget_smob; (void) pstate;
  string s= "<widget>";
  scm_display (string_to_scm (s), port);
  return 1;
}

static SCM
cmp_widget (SCM t1, SCM t2) {
  return gh_bool2scm (scm_to_widget (t1) == scm_to_widget (t2));
}

/******************************************************************************
* Several array types
******************************************************************************/

static bool
scm_is_array_tree (SCM p) {
  if (gh_null_p (p)) return TRUE;
  else return scm_is_tree (gh_car (p)) && scm_is_array_tree (gh_cdr (p));
}

#define SCM_ASSERT_ARRAY_TREE(p,arg,rout) \
  SCM_ASSERT (scm_is_array_tree (p), p, arg, rout)

/* static */ SCM
array_tree_to_scm (array<tree> a) {
  int i, n= N(a);
  SCM p= gh_list (SCM_UNDEFINED);
  for (i=n-1; i>=0; i--) p= gh_cons (tree_to_scm (a[i]), p);
  return p;
}

/* static */ array<tree>
scm_to_array_tree (SCM p) {
  array<tree> a;
  while (!gh_null_p (p)) {
    a << scm_to_tree (gh_car (p));
    p= gh_cdr (p);
  }
  return a;
}

static bool
scm_is_array_widget (SCM p) {
  if (gh_null_p (p)) return TRUE;
  else return scm_is_widget (gh_car (p)) && scm_is_array_widget (gh_cdr (p));
}

#define SCM_ASSERT_ARRAY_WIDGET(p,arg,rout) \
  SCM_ASSERT (scm_is_array_widget (p), p, arg, rout)

/* static */ SCM
array_widget_to_scm (array<widget> a) {
  int i, n= N(a);
  SCM p= gh_list (SCM_UNDEFINED);
  for (i=n-1; i>=0; i--) p= gh_cons (widget_to_scm (a[i]), p);
  return p;
}

/* static */ array<widget>
scm_to_array_widget (SCM p) {
  array<widget> a;
  while (!gh_null_p (p)) {
    a << scm_to_widget (gh_car (p));
    p= gh_cdr (p);
  }
  return a;
}

/******************************************************************************
* Initialization
******************************************************************************/

#ifdef SCM_NEWSMOB

void
initialize_glue () {
  tree_tag= scm_make_smob_type ("tree", 0);
  scm_set_smob_mark (tree_tag, mark_tree);
  scm_set_smob_free (tree_tag, free_tree);
  scm_set_smob_print (tree_tag, print_tree);
  scm_set_smob_equalp (tree_tag, cmp_tree);
  display_tag= scm_make_smob_type ("display", 0);
  scm_set_smob_mark (display_tag, mark_display);
  scm_set_smob_free (display_tag, free_display);
  scm_set_smob_print (display_tag, print_display);
  scm_set_smob_equalp (display_tag, cmp_display);
  widget_tag= scm_make_smob_type ("widget", 0);
  scm_set_smob_mark (widget_tag, mark_widget);
  scm_set_smob_free (widget_tag, free_widget);
  scm_set_smob_print (widget_tag, print_widget);
  scm_set_smob_equalp (widget_tag, cmp_widget);
  initialize_glue_basic ();
  initialize_glue_editor ();
  initialize_glue_server ();
}

#else

scm_smobfuns tree_smob_funcs = {
  mark_tree, free_tree, print_tree, cmp_tree
};

scm_smobfuns display_smob_funcs = {
  mark_display, free_display, print_display, cmp_display
};

scm_smobfuns widget_smob_funcs = {
  mark_widget, free_widget, print_widget, cmp_widget
};

void
initialize_glue () {
  tree_tag= scm_newsmob (&tree_smob_funcs);
  display_tag= scm_newsmob (&display_smob_funcs);
  widget_tag= scm_newsmob (&widget_smob_funcs);
  initialize_glue_basic ();
  initialize_glue_editor ();
  initialize_glue_server ();
}

#endif

#endmodule // code_glue
