
/******************************************************************************
* MODULE     : iterator.gen.cc
* DESCRIPTION: dynamic iterators
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <hashmap.gen.h>
#include <hashset.gen.h>
#include <iterator.gen.h>

#module code_iterator (T)
#import iterator (T)

int
iterator_rep<T>::remains () {
  if (busy()) return -1;
  else return 0;
}

iterator<T>::operator tree () {
  tree t (TUPLE);
  while (rep->busy ()) {
#ifdef no_tree_converter<T>
    t << tree ("?");
#else
    t << tree (rep->next());
#endif
  }
  return t;
}

ostream&
operator << (ostream& out, iterator<T> it) {
  bool flag=FALSE;
  out << "[ ";
  while (it->busy ()) {
    if (flag) out << ", ";
    else flag=TRUE;
    out << it->next ();
  }
  return out << " ]";
}

#endmodule // code_iterator (T)

#module code_hashset_iterator (T)
#import hashset (T)
#import iterator (T)

class hashset_iterator_rep<T>: public iterator_rep<T> {
  hashset<T> h;
  int i;
  list<T> l;
  void spool ();

public:
  hashset_iterator_rep<T> (hashset<T> h);
  bool busy ();
  T next ();
};

hashset_iterator_rep<T>::hashset_iterator_rep<T> (hashset<T> h2):
  h (h2), i (0), l (h2->a[0]) {}

void
hashset_iterator_rep<T>::spool () {
  if (i>=h->n) return;
  while (nil(l)) {
    if ((++i)>=h->n) return;
    l=h->a[i];
  }
}

bool
hashset_iterator_rep<T>::busy () {
  spool ();
  return i<h->n;
}

T
hashset_iterator_rep<T>::next () {
  if (!busy ())
    fatal_error ("end of iterator", "hashset_iterator_rep<T>::next");
  T x (l->item);
  l= l->next;
  return x;
}

iterator<T>
iterate (hashset<T> h) {
  return new hashset_iterator_rep<T> (h);
}

#endmodule // code_hashset_iterator (T)

#module code_hashmap_iterator (T,U)
#import hashmap (T,U)
#import iterator (T)

class hashmap_iterator_rep<T,U>: public iterator_rep<T> {
  hashmap<T,U> h;
  int i;
  list<hashentry<T,U>> l;
  void spool ();

public:
  hashmap_iterator_rep<T,U> (hashmap<T,U> h);
  bool busy ();
  T next ();
};

hashmap_iterator_rep<T,U>::hashmap_iterator_rep<T,U> (hashmap<T,U> h2):
  h (h2), i (0), l (h2->a[0]) {}

void
hashmap_iterator_rep<T,U>::spool () {
  if (i>=h->n) return;
  while (nil(l)) {
    if ((++i)>=h->n) return;
    l=h->a[i];
  }
}

bool
hashmap_iterator_rep<T,U>::busy () {
  spool ();
  return i<h->n;
}

T
hashmap_iterator_rep<T,U>::next () {
  if (!busy ())
    fatal_error ("end of iterator", "hashmap_iterator_rep<T,U>::next");
  T x (l->item.key);
  l= l->next;
  return x;
}

iterator<T>
iterate (hashmap<T,U> h) {
  return new hashmap_iterator_rep<T,U> (h);
}

#endmodule // code_hashmap_iterator (T,U)
