/* wwwfetch.c */

/*
  TODO:
   
   - the DOFORKS #ifdef'ed code is never used, so no forking is
     done for fetching URLs (can libwww still sometimes hang when
     fetching them?)  The forking code is broken anyway, and utterly
     different from how forking is done in other parts of xdvi, so it
     would need to be rewritten.
     
   - redesign entire hyperref module to have a single entry point
     for fetching remote files, where we can print out error messages,
     decide what to do next etc.
*/

#include "xdvi-config.h"
#if defined(HTEX) || defined(XHDVI)
#include "kpathsea/c-fopen.h"
#include "kpathsea/c-stat.h"

#include "wwwconf.h"
#include "WWWLib.h"
#include "WWWInit.h"
#include "WWWCache.h"

#define LINE 1024
#define FILELISTCHUNK 20

typedef struct {
    char *url;
    char *savefile;
    pid_t childnum;
} Fetch_Children;

#define MAXC 10	/* Don't send off more than 10 simultaneously */
Fetch_Children fetch_children[MAXC];
static int nchildren = MAXC + 1;

static int nURLs INIT(0);

/* FIXME: why do we need this struct, *and* Fetch_Children struct?? */

static struct FiletoURLconv {
	char *url;    /* Full address of the URL we have locally */
	char *file;   /* Local file name it is stored as */
} *filelist INIT(NULL);

/* access methods to filelist for external modules */
char *
htex_file_at_index(idx)
     int idx;
{
    return filelist[idx].file;
}

char *
htex_url_at_index(idx)
     int idx;
{
    return filelist[idx].url;
}


/* print errors from libwww calls */
static Boolean
www_error_print (uri, request, op, msgnum, dfault, input, reply)
     char *uri;
     HTRequest *request;
     HTAlertOpcode op;
     int msgnum;
     const char *dfault;
     void *input;
     HTAlertPar *reply;
{
    char *msg = HTDialog_errorMessage(request, op, msgnum, dfault, input);
    if (msg) {
        do_popup_message(MSG_ERR, NULL, "Couldn't fetch `%s':\nlibwww: %s\n", uri, msg);
        free(msg);
    }
    return True;
}

/* call after libwww calls to treat error messages */
static int
www_info_filter (request, response, param, status)
     HTRequest *request;
     HTResponse *response;
     void *param;
     int status;
{
    HTParentAnchor *anchor = HTRequest_anchor(request);
    char *uri = HTAnchor_address((HTAnchor*) anchor);

    switch (status) {
    case HT_LOADED:	/* success */
	break;
    default:		/* any error */
	www_error_print(uri, request, HT_A_MESSAGE, HT_MSG_NULL, NULL,
			HTRequest_error(request), NULL);
	break;
    }
    free(uri);
    return HT_OK;
}

/* wrapper for HTLoadToFile to work around a bug in that function */
int
libwww_wrapper_HTLoadToFile(const char *url, HTRequest *request, const char *filename)
{
    if (url && filename && request) {
	FILE * fp = NULL;
	
	/* file existance check omitted */

	/* open the file */
	if ((fp = xfopen_local(filename, "wb")) == NULL) {
	    HTRequest_addError(request, ERR_NON_FATAL, NO, HTERR_NO_FILE, 
			       (char *) filename, strlen(filename),
			       "HTLoadToFile"); 
	    return 0;
	}

	/* Set the output stream and start the request */
	HTRequest_setOutputFormat(request, WWW_SOURCE);
	HTRequest_setOutputStream(request, HTFWriter_new(request, fp, NO));
	if (HTLoadAbsolute(url, request) == NO) {
	    /* SU 2001/09/22: omitted closing this fp, since it will be closed a second
	       time in HTFWriter_free() (called by HTRequest_delete()), leading
	       to a segfault (this is the bug mentioned above).
	    */
	    /* fclose(fp); */
	    return 0;
	} else
	    return 1;
    }
    return 0;
}

/* Given absolute URL, open a temporary filename, and do the transfer */
int
www_fetch(url, savefile)
    char *url;
    char *savefile;
{
    int status;
    int pid;
    HTRequest *request;
    HTFormat content_type;

    if (debug & DBG_HYPER) {
	fprintf(stderr, "www_fetch called with: |%s|, savefile: |%s|\n", url,
		savefile);
    }

    /* Don't ask when overwriting temporary savefile. */
    HTAlert_setInteractive(NO);
    /* unfortunatly this makes it impossible to use any alerts,
       such as:
       HTAlert_add(www_error_print, HT_A_MESSAGE);
       so we need to do our own filtering.
       TODO: Would it be better to register a dummy handler for
       HT_A_CONFIRM instead of disabling interactive mode entirely??

       NOTE: the wrapper below doesn't check for existence of a tempfile,
       so we could use HTAlert_add again. Will need some regression testing though.
    */
    HTNet_addAfter(www_info_filter, NULL, NULL, HT_ALL, HT_FILTER_LATE);
    
    request = HTRequest_new();

    if (debug & DBG_HYPER) {
	fprintf(stderr, "calling HTLoadToFile\n");
    }

    status = libwww_wrapper_HTLoadToFile(url, request, savefile);
    
    /* Aarghh! Some libwww methods such as HTLoadFTP return non-null
       status even when fatal errors have occurred (e.g. `login
       failed'). So the following test can only catch some kinds of
       errors, and that's why we need www_info_filter.
    */
    if (status == 0) {
 	char *errmsg = HTDialog_errorMessage(request, 0, 0, NULL, HTRequest_error(request));
	do_popup_message(MSG_ERR, NULL,
			 "Couldn't open URL `%s':\nlibwww: %s\n",
			 dvi_name, errmsg);
	free(errmsg);
    }
    
    if (debug & DBG_HYPER) {
	fprintf(stderr, "after request: status %d\n", status);
    }
    /* Extract the content_type before deleting the request. */
    if (debug & DBG_HYPER) {
	/* content_type = request->response->content_type; */
	content_type = HTResponse_format(HTRequest_response(request));

	if (content_type == HTAtom_for("application/x-dvi"))
	    fprintf(stderr, "www_fetch(%s->%s) returned a dvi file.\n",
		    url, savefile);
	else
	    fprintf(stderr, "www_fetch(%s->%s) returned content-type: %s\n",
		    url, savefile, HTAtom_name(content_type));
    }
    HTRequest_delete(request);

#ifdef DOFORKS
    exit(1);	/* No cleanup! */
#else /* DOFORKS */
    return status;	/* return status: YES (1) or NO (0) */
#endif /* DOFORKS */
}

/* Turn a relative URL into an absolute one: */
void
make_absolute(rel, base, len)
    char *rel, *base;
    int len;
{
    char *cp, *parsed;

    if (base == NULL)
	return;
    cp = strchr(rel, '\n');
    if (cp)
	*cp = '\0';	/* Eliminate newline char */

    parsed = HTParse(rel, base, PARSE_ALL);
    strncpy(rel, parsed, len);
    free(parsed);
}

void
wait_for_urls()
{	/* Wait for all the children to finish... */
    int i;
#ifdef DOFORKS
    int ret, status, j;
#endif

    if (nchildren > MAXC) {
	/* Initialization needed: */
	for (i = 0; i < MAXC; i++) {
	    fetch_children[i].url = NULL;
	    fetch_children[i].savefile = NULL;
	}
    }
    else {
	for (i = 0; i < nchildren; i++) {
#ifdef DOFORKS
	    ret = wait(&status);	/* Wait for one to finish */
	    for (j = 0; j < nchildren; j++) {
		if (ret == fetch_children[j].childnum) {
		    if (debug & DBG_HYPER)
			fprintf(stderr,
				"wait_for_urls(): URL %s in file %s: status %d\n",
				fetch_children[j].url,
				fetch_children[j].savefile, status);
		    break;
		}
	    }
#else /* DOFORKS */
	    if (debug & DBG_HYPER)
		fprintf(stderr, "wait_for_urls(): URL %s in file %s\n",
			fetch_children[i].url, fetch_children[i].savefile);
#endif /* DOFORKS */
	}
    }
    nchildren = 0;
}

void
htex_cleanup ARGS((void))
{
    /* Delete all the temp files we created */
    for (; nURLs > 0; nURLs--) {
	if (debug & DBG_HYPER)
	    fprintf(stderr,"htex: trying to unlink %s\n", filelist[nURLs-1].file);
	if (unlink(filelist[nURLs - 1].file) < 0) {
	    fprintf(stderr, "Xdvi: Couldn't unlink ");
	    perror(filelist[nURLs - 1].file);
	}
    }
    HTCache_flushAll();
    HTProfile_delete();
}


/* Start a fetch of a relative URL */
int
fetch_relative_url(base_url, rel_url)
    char *base_url;
    _Xconst char *rel_url;
{
    int i, resp;
    char *tmpfile = NULL;
    char *cp;
    char buf[LINE];
    int fd;
    FILE *fp;

    if (debug & DBG_HYPER) {
	fprintf(stderr, "fetch_relative_url called: |%s|%s|!\n",
		base_url ? base_url : "NULL", rel_url);
    }
    /* Step 1: make the relative URL absolute: */
    strncpy(buf, rel_url, LINE);	/* Put it in buf */
    make_absolute(buf, base_url, LINE);

    /* Step 1.5: Check whether we already have it - if so return */
    
    /* NOTE SU: I don't think this is always desired; what if the user wants
       to fetch a new version? Can't we assume that he wants that
       if he clicks on the link a second time?
       (Also, if the temp file is deleted by accident, the URL will not be re-fetched).
       Or should reread-dvi-file() flush the filelist, as in tickrec (special.c)?
       Or should we ask interactively:

       Xdvi Info:
       File xyz already downloaded - reload?
       ------
       Yes No
    */
    /*
    for (i = 0; i < nURLs; i++) {
	if (!strcmp(buf, filelist[i].url))
	    return i;
    }
    */

    /* Step 2: Find a temporary file to store the output in */
    fd = xdvi_temp_fd(&tmpfile);
    if (debug & DBG_HYPER) {
	fprintf(stderr, "htex: created temporary file |%s|\n", tmpfile);
    }
    if (fd == -1) {
	do_popup_message(MSG_ERR,
			 "This is serious. Either you've run out of file \
descriptors, or your disk is full. Sorry, you'll need to resolve this first ...",
			 "Cannot create temporary file");
	return -1;
    }
    else {
	/* it would be nice if we could use the old Unix trick
	   unlink(file);
	   to ensure the file is deleted even if xdvi crashes,
	   but since all communication with libwww and child
	   processes is done via the file name, this is not
	   possible, and we have to rely on htex_cleanup()
	   instead.
	*/
	close(fd);
    }

    /* Step 3: Fork to fetch the URL */
    if (nchildren >= MAXC)
	wait_for_urls();	/* Wait for the old ones */

    cp = NULL;
    fetch_children[nchildren].url = MyStrAllocCopy(&cp, buf);
    cp = NULL;
    fetch_children[nchildren].savefile = MyStrAllocCopy(&cp, tmpfile);

    /* Step 4: Update the URL-filename list */
    if (nURLs == 0) {
	filelist = xmalloc(FILELISTCHUNK * sizeof *filelist);
	bzero(filelist, FILELISTCHUNK * sizeof *filelist);
    }
    else if (nURLs % FILELISTCHUNK == 0) {
	filelist = xrealloc(filelist,
			    (nURLs + FILELISTCHUNK) * sizeof *filelist);
	bzero(filelist + nURLs, FILELISTCHUNK * sizeof *filelist);
    }
    MyStrAllocCopy(&(filelist[nURLs].url), buf);
    MyStrAllocCopy(&(filelist[nURLs].file), tmpfile);
    nURLs++;
#ifdef DOFORKS
    fetch_children[nchildren].childnum = fork();
    if (fetch_children[nchildren].childnum == 0) {	/* Child process */
	www_fetch(buf, tmpfile);	/* Get the URL! */
	free(tmpfile);
	exit(0);	/* Make sure this process quits... */
    }
    nchildren++;
#else /* DOFORKS */
    resp = www_fetch(buf, tmpfile);	/* Get the URL! */
    free(tmpfile);
    if (resp == 0) {
	print_statusline(STATUS_LONG, "Error requesting URL: %s", buf);
	nURLs--;
	if (debug & DBG_HYPER) {
	    fprintf(stderr, "htex: unlinking |%s|\n", filelist[nURLs].file);
	}
	/* get rid of this temp file */
	if (unlink(filelist[nURLs].file) < 0) {
	    fprintf(stderr, "Xdvi: Couldn't unlink ");
	    perror(filelist[nURLs].file);
	}
	return -1;
    }
    else {
	nchildren++;
    }
#endif /* DOFORKS */
    close(fd);
    return nURLs - 1;
}

#endif /* HTEX || XHDVI */
