/*

    File: ext2.c

    Copyright (C) 1998-2006 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "ext2.h"
#include "intrf.h"
#include "fnctdsk.h"

#define EXT2_BLOCK_SIZE(s)      (EXT2_MIN_BLOCK_SIZE << (s)->s_log_block_size)
static int test_EXT2(t_param_disk *disk_car, const struct ext2_super_block *sb,t_partition *partition,const int debug, const int dump_ind);
static int set_EXT2_info(t_param_disk *disk_car, const struct ext2_super_block *sb,t_partition *partition,const int debug, const int dump_ind);

int check_EXT2(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char *buffer=(unsigned char*)MALLOC(EXT2_SUPERBLOCK_SIZE);
  if(disk_car->read(disk_car,EXT2_SUPERBLOCK_SIZE, buffer, partition->part_offset+0x400)!=0)
  {
    free(buffer);
    return 1;
  }
  if(test_EXT2(disk_car,(struct ext2_super_block*)buffer,partition,debug,0)!=0)
  {
    free(buffer);
    return 1;
  }
  set_EXT2_info(disk_car,(struct ext2_super_block*)buffer,partition,debug,0);
  free(buffer);
  return 0;
}

static int set_EXT2_info(t_param_disk *disk_car, const struct ext2_super_block *sb,t_partition *partition,const int debug, const int dump_ind)
{
  set_part_name(partition,sb->s_volume_name,16);
  /* sb->s_last_mounted seems to be unemployed in kernel 2.2.16 */
  if(EXT2_HAS_COMPAT_FEATURE(sb,EXT3_FEATURE_COMPAT_HAS_JOURNAL)!=0)
    strncpy(partition->info,"EXT3",sizeof(partition->info));
  else
    strncpy(partition->info,"EXT2",sizeof(partition->info));
  if(EXT2_HAS_RO_COMPAT_FEATURE(sb,EXT2_FEATURE_RO_COMPAT_LARGE_FILE)!=0)
    strcat(partition->info," Large file");
  if(EXT2_HAS_RO_COMPAT_FEATURE(sb,EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER)!=0)
    strcat(partition->info," Sparse superblock");
  if(EXT2_HAS_INCOMPAT_FEATURE(sb,EXT3_FEATURE_INCOMPAT_RECOVER)!=0)
    strcat(partition->info," Recover");
  if(EXT2_HAS_INCOMPAT_FEATURE(sb,EXT3_FEATURE_INCOMPAT_JOURNAL_DEV)!=0)
    strcat(partition->info," Journal dev");
  if(le16(sb->s_block_group_nr)!=0)
  {
    strcat(partition->info," Backup superblock");
    if(debug>0)
    {
      ecrit_rapport("\nblock_group_nr %u\n",le16(sb->s_block_group_nr));
    }
  }
  /* last mounted => date */
  return 0;
}

/*
Primary superblock is at 1024 (SUPERBLOCK_OFFSET)
Group 0 begin at s_first_data_block
*/
int recover_EXT2(t_param_disk *disk_car, const struct ext2_super_block *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if(test_EXT2(disk_car,sb,partition,debug,dump_ind)!=0)
    return 1;
  if(partition==NULL)
    return 0;
  set_EXT2_info(disk_car,sb,partition,debug,dump_ind);
  partition->part_type_i386=P_LINUX;
  partition->part_type_mac=PMAC_LINUX;
  partition->part_type_sun=PSUN_LINUX;
  partition->part_size=(uint64_t)le32(sb->s_blocks_count)*(EXT2_MIN_BLOCK_SIZE<<le32(sb->s_log_block_size));
  partition->blocksize=EXT2_MIN_BLOCK_SIZE<<le32(sb->s_log_block_size);
  if(debug>0)
  {
    ecrit_rapport("\n");
  }
  if(le16(sb->s_block_group_nr)>0)
  {
    unsigned long int boot_sector=(le32(sb->s_first_data_block)+le16(sb->s_block_group_nr)*le32(sb->s_blocks_per_group));
    if(partition->part_offset<(uint64_t)boot_sector*((EXT2_MIN_BLOCK_SIZE/disk_car->sector_size)<<le32(sb->s_log_block_size))*disk_car->sector_size)
    {
      ecrit_rapport("recover_EXT2: part_offset problem\n");
      return 1;
    }
    partition->boot_sector=boot_sector;
    partition->part_offset-=(uint64_t)boot_sector*((EXT2_MIN_BLOCK_SIZE/disk_car->sector_size)<<le32(sb->s_log_block_size))*disk_car->sector_size;
    ecrit_rapport("recover_EXT2: \"e2fsck -b %lu -B %lu device\" may be needed\n",partition->boot_sector,partition->blocksize);
  }
  else
  {
    partition->boot_sector=0;
  }
  if(debug>0)
  {
    ecrit_rapport("recover_EXT2: s_block_group_nr=%u/%u, s_mnt_count=%u/%u, s_blocks_per_group=%u\n",
    le16(sb->s_block_group_nr), (unsigned int)(le32(sb->s_blocks_count)/le32(sb->s_blocks_per_group)), le16(sb->s_mnt_count),le16(sb->s_max_mnt_count),(unsigned int)le32(sb->s_blocks_per_group));
    ecrit_rapport("recover_EXT2: boot_sector=%lu, s_blocksize=%lu\n",partition->boot_sector,partition->blocksize);
    ecrit_rapport("recover_EXT2: s_blocks_count %u\n",(unsigned int)le32(sb->s_blocks_count));
    ecrit_rapport("recover_EXT2: part_size %lu\n",(long unsigned)(partition->part_size/disk_car->sector_size));
  }
  return 0;
}

static int test_EXT2(t_param_disk *disk_car, const struct ext2_super_block *sb,t_partition *partition,const int debug, const int dump_ind)
{
    /* There is a little offset ... */
  if(le16(sb->s_magic)!=EXT2_SUPER_MAGIC)
    return 1;
  if(dump_ind!=0)
  {
    if(partition!=NULL && disk_car!=NULL)
      ecrit_rapport("\nEXT2/EXT3 magic value at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    /* There is a little offset ... */
    dump(stdscr,sb,DEFAULT_SECTOR_SIZE);
  }
  if (le32(sb->s_free_blocks_count) >= le32(sb->s_blocks_count)) return 2;
  if (le32(sb->s_free_inodes_count) >= le32(sb->s_inodes_count)) return 3;
  if (le16(sb->s_errors)!=0 &&
      (le16(sb->s_errors) != EXT2_ERRORS_CONTINUE) &&
      (le16(sb->s_errors) != EXT2_ERRORS_RO) &&
      (le16(sb->s_errors) != EXT2_ERRORS_PANIC))
    return 5;
  if ((le16(sb->s_state) & ~(EXT2_VALID_FS | EXT2_ERROR_FS))!=0)
    return 5;
  if (le32(sb->s_blocks_count) == 0) /* reject empty filesystem */
    return 6;
  if(le32(sb->s_log_block_size)>2)  /* block size max = 4096, can be 8192 on alpha */
    return 7;
  if(partition!=NULL)
  {
    if(partition->part_size!=0 && (partition->part_size<le32(sb->s_blocks_count)*(EXT2_MIN_BLOCK_SIZE<<le32(sb->s_log_block_size))))
      return 8;
    if(EXT2_HAS_COMPAT_FEATURE(sb,EXT3_FEATURE_COMPAT_HAS_JOURNAL)!=0)
      partition->upart_type=UP_EXT3;
    else
      partition->upart_type=UP_EXT2;
  }
  return 0;
}
