/* TESS: the (Te)st (S)ystem for (S)-Lang {{{
 *
 * Copyright (C) 2004 Massachusetts Institute of Technology 
 *
 * Michael S. Noble <mnoble@space.mit.edu>
 *
 * This software was partially developed by the MIT Center for Space
 * Research under contract SV1-61010 from the Smithsonian Institution.
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * the supporting documentation, and that the name of the Massachusetts
 * Institute of Technology not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission.  The Massachusetts Institute of Technology makes
 * no representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 * 
 * THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
 * INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE. }}} */

#include <slang.h>

#if SLANG_VERSION < 20000
#define SLang_set_error(x)      (SLang_Error = (x))
#endif

#include "config.h"

static int     tess_version = TESS_VERSION;
static char    *tess_version_string = TESS_VERSION_STRING;

static int     auto_summarize = 1;

/* Intrinsic function definitions {{{ */

static void exit_summary(void)
{
   unsigned int result = 1;
   (void) SLang_execute_function("tess_summary");
   (void) SLang_pop_int(&result);
   _exit(result);
}

static void type_error_hook(char *error_msg)
{
   fputs(error_msg,stderr);	/* support function, which lets scripts */
   fputs("\r\n", stderr);	/* catch normally uncatchable type */
   fflush (stderr);		/* mismatch errors w/in an ERROR_BLOCK */
#if SLANG_VERSION < 20000
   SLang_restart(0);
#endif
   SLang_set_error(SL_USAGE_ERROR);
}

static void catch_type_errors(int *catch)
{
   static void (*previous_error_hook)(char *);

   if (*catch) {

	if (SLang_Error_Hook != type_error_hook)
	   previous_error_hook = SLang_Error_Hook;

	SLang_Error_Hook = type_error_hook;
   }
   else if (SLang_Error_Hook == type_error_hook)
	   SLang_Error_Hook = previous_error_hook;
}

/* }}} */

/* Intrinsic function and variable tables {{{ */

#define I               SLANG_INT_TYPE
#define S               SLANG_STRING_TYPE
#define V               SLANG_VOID_TYPE

static SLang_Intrin_Var_Type Intrin_Vars[] =
{
   MAKE_VARIABLE("_tess_version", &tess_version, I, 1),
   MAKE_VARIABLE("_tess_auto_summarize", &auto_summarize, I, 0),
   MAKE_VARIABLE("_tess_version_string", &tess_version_string, S, 1),

   SLANG_END_INTRIN_VAR_TABLE
};

static SLang_Intrin_Fun_Type Tess_Intrins[] =
{

  MAKE_INTRINSIC_1("_tess_catch_type_errors", catch_type_errors, V, I),

  SLANG_END_INTRIN_FUN_TABLE
};

/* }}} */

/* Module initialization {{{ */

SLANG_MODULE(tess);
int init_tess_module_ns(char *ns_name)
{
   SLang_NameSpace_Type *ns = NULL;

   if (ns_name != NULL) {
	ns = SLns_create_namespace (ns_name);
	if (ns == NULL)
	   return -1;
   }

   if ( atexit(exit_summary) != 0)
      return -1;

   /* Note that the version variables go into the Global namespace */
   if (SLadd_intrin_var_table (Intrin_Vars, NULL) == -1)
	return -1;

   if (-1 == SLns_add_intrin_fun_table (ns, Tess_Intrins, "___TESS___"))
	return -1;

   return 0;
}
/* }}} */
