/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://terraform.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef __TTERRAIN_H__
#define __TTERRAIN_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#include <gtk/gtk.h>
#include <libgnomeprint/gnome-print.h>
#include "colormaps.h"
#include "tcanvas.h"

typedef enum TComposeOp TComposeOp;

enum TComposeOp
{
  T_COMPOSE_NONE,
  T_COMPOSE_REPLACE,
  T_COMPOSE_ADD,
  T_COMPOSE_SUBTRACT
};

typedef struct TTerrainObject TTerrainObject;

struct TTerrainObject
{
  gint      ox, oy;
  gfloat    x, y;
  gfloat    angle;
  gfloat    scale_x, scale_y, scale_z;
  gchar    *name;
};

void t_terrain_object_clear (TTerrainObject *object);

#define T_TYPE_TERRAIN            (t_terrain_get_type ())
#define T_TERRAIN(obj)            (GTK_CHECK_CAST ((obj), T_TYPE_TERRAIN, TTerrain))
#define T_TERRAIN_CLASS(klass)    (GTK_CHECK_CLASS_CAST ((klass), T_TYPE_TERRAIN, TTerrain))
#define T_IS_TERRAIN(obj)         (GTK_CHECK_TYPE ((obj), T_TYPE_TERRAIN))
#define T_IS_TERRAIN_CLASS(klass) (GTK_CHECK_CLASS_TYPE ((klass), T_TYPE_TERRAIN))


typedef struct TTerrainClass TTerrainClass;
typedef struct TTerrain      TTerrain;

struct TTerrainClass
{
  GtkObjectClass object_class;

  /* Signals */
  void           (*heightfield_modified) (GtkObject *object);
  void           (*title_modified)       (GtkObject *object);
  void           (*selection_modified)   (GtkObject *object);
  void           (*object_added)         (GtkObject *object,
                                          gint       item);
  void           (*object_modified)      (GtkObject *object,
                                          gint       item);
  void           (*object_deleted)       (GtkObject *object,
                                          gint       item);
};

struct TTerrain
{
  GtkObject      object;

  gchar         *filename;
  gboolean       modified;
  gboolean       autogenned; /* Set true if filename is auto generated. */

  gint           width, height;
  gfloat        *heightfield;
  gfloat        *selection;

  gfloat         camera_x, camera_y, camera_z;
  gfloat         lookat_x, lookat_y, lookat_z;
  gboolean       observe_sealevel;
  gfloat         elevation_offset;
  //gboolean       do_tile; /* used for render options, disabled/removed */
  gboolean       clouds;
  gboolean       fog;
  gboolean       filled_sea;
  gfloat         fog_offset;
  gfloat         fog_alt;
  gfloat         fog_distance;
  gfloat         time_of_day;
  gfloat         north_direction;
  gfloat         water_clarity;
  gfloat         camera_height_factor;
  gint           render_width;
  gint           render_scale_x;
  gint           render_scale_y;
  gint           render_scale_z;
  gfloat         y_scale_factor;
  gfloat         sealevel;
  gint           wireframe_resolution;
  gchar         *theme_file;
  gint           lighting_level;

  gint           contour_levels;

  /* objects is a GArray of TTerrainObjects */
  GArray        *objects;
};

extern guint title_modified_signal;
extern guint heightfield_modified_signal;
extern guint selection_modified_signal;
extern guint object_added_signal;
extern guint object_modified_signal;
extern guint object_deleted_signal;

GtkType    t_terrain_get_type             (void);
GtkObject *t_terrain_new                  (gint        width,
                                           gint        height);

void       t_terrain_set_filename         (TTerrain   *terrain,
                                           gchar      *filename);
gchar     *t_terrain_get_title            (TTerrain   *terrain);
void       t_terrain_set_modified         (TTerrain   *terrain,
                                           gboolean    modified);

gdouble    t_terrain_get_height           (TTerrain   *terrain,
                                           gdouble     x,
                                           gdouble     y);
void       t_terrain_heightfield_modified (TTerrain   *terrain);
void       t_terrain_selection_modified   (TTerrain   *terrain);
TTerrain  *t_terrain_clone                (TTerrain   *terrain);
TTerrain  *t_terrain_clone_resize         (TTerrain   *terrain,
                                           gint        width,
                                           gint        height,
					   gboolean    terrain_only);
void       t_terrain_set_size             (TTerrain   *terrain,
                                           gint        width,
                                           gint        height);
TTerrain  *t_terrain_clone_preview        (TTerrain   *terrain);
gint      *t_terrain_histogram            (TTerrain   *terrain, gint n_intervals, gint scale);
TTerrain  *t_terrain_clone_histogram      (TTerrain   *terrain, gfloat display_yscale);
void       t_terrain_copy_heightfield     (TTerrain   *from,
                                           TTerrain   *to);
void       t_terrain_copy_heightfield_and_selection (TTerrain   *from,
                                                     TTerrain   *to);
void       t_terrain_normalize            (TTerrain   *terrain,
                                           gboolean    never_grow);
void       t_terrain_crop                 (TTerrain   *terrain,
                                           gint        x1,
                                           gint        y1,
                                           gint        x2,
                                           gint        y2);
TTerrain  *t_terrain_crop_new             (TTerrain   *terrain,
                                           gint        x1,
                                           gint        y1,
                                           gint        x2,
                                           gint        y2);
void       t_terrain_invert               (TTerrain   *terrain);

void       t_terrain_pack_options         (TTerrain   *terrain,
                                           GtkWidget  *options);
void       t_terrain_unpack_options       (TTerrain   *terrain,
                                           GtkWidget  *options);
void       t_terrain_print_contour_map    (TTerrain   *terrain,
                                           GnomePrintContext *context);

void       t_terrain_ref                  (TTerrain   *terrain);
void       t_terrain_unref                (TTerrain   *terrain);

TTerrain  *t_terrain_import_heightfield   (gchar      *filename);
void       t_terrain_export_heightfield   (TTerrain   *terrain,
                                           gchar      *filename);
void       t_terrain_select_by_height     (TTerrain  *terrain,
                                           gfloat     floor,
                                           gfloat     ceil,
                                           TComposeOp op);
void       t_terrain_select               (TTerrain   *terrain,
                                           gfloat      x1,
                                           gfloat      y1,
                                           gfloat      x2,
                                           gfloat      y2,
                                           TSelectionType type,
                                           TComposeOp  op);
void       t_terrain_select_destroy       (TTerrain   *terrain);
gint       t_terrain_add_object           (TTerrain   *terrain,
                                           gint        ox,
                                           gint        oy,
                                           gfloat      x,
                                           gfloat      y,
                                           gfloat      angle,
                                           gfloat      scale_x,
                                           gfloat      scale_y,
                                           gfloat      scale_z,
                                           gchar      *object_name);
void       t_terrain_move_object          (TTerrain   *terrain,
                                           gint        item,
                                           gfloat      x,
                                           gfloat      y);
void       t_terrain_rotate_object        (TTerrain   *terrain,
                                           gint        item,
                                           gfloat      angle);
void       t_terrain_remove_object        (TTerrain   *terrain,
                                           gint        item);
void       t_terrain_remove_objects       (TTerrain   *terrain);
gint       t_terrain_seed_data            (TTerrain   *terrain, 
                                           gfloat     *data,
					   gint        width, 
					   gint        height);
gint       t_terrain_seed                 (TTerrain   *terrain,
                                           gint        new_width, 
                                           gint        new_height, 
			                   gint        sel_x,
			                   gint        sel_y, 
					   gint        sel_width, 
					   gint        sel_height);
TTerrain  *t_terrain_tile_new             (TTerrain   *terrain,
                                           gint        num_x, 
                                           gint        num_y); 


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __TTERRAIN_H__ */
