/*   FILE: bintree.c -- 
 * AUTHOR: W. Michael Petullo <strmap@flyn.org>
 *   DATE: 08 January 2000
 *   NOTE: From _Mastering Algorithms with C_ by Kyle Loudon.
 */

#include <bintree.h>
#include <stdlib.h>
#include <string.h>

/* ============================ bintree_init () ============================ */
void bintree_init(bintree_t * tree, void (*destroy) (void *data))
{
    tree->size = 0;
    tree->destroy = destroy;
    tree->root = NULL;
}

/* ============================ bintree_destroy () ========================= */
void bintree_destroy(bintree_t * tree)
{
    bintree_rem_left(tree, NULL);
    bintree_rem_right(tree, NULL);
    memset(tree, 0, sizeof(bintree_t));
}

/* ============================ bintree_ins_left () ======================== */
int bintree_ins_left(bintree_t * tree, bintree_node_t * node,
		     const void *data)
{
    bintree_node_t *new_node, **position;
    if (node == NULL) {
	if (bintree_size(tree) > 0)
	    return -1;
	position = &tree->root;
    } else {
	if (bintree_left(node) != NULL)
	    return -1;
	position = &node->left;
    }
    if ((new_node = (bintree_node_t *) malloc(sizeof(bintree_node_t))) ==
	NULL) return -1;
    new_node->data = (void *) data;
    new_node->left = NULL;
    new_node->right = NULL;
    *position = new_node;
    tree->size++;
    return 0;
}

/* ============================ bintree_ins_right () ======================= */
int bintree_ins_right(bintree_t * tree, bintree_node_t * node,
		      const void *data)
{
    bintree_node_t *new_node, **position;
    if (node == NULL) {
	if (bintree_size(tree) > 0)
	    return -1;
	position = &tree->root;
    } else {
	if (bintree_right(node) != NULL)
	    return -1;
	position = &node->right;
    }
    if ((new_node = (bintree_node_t *) malloc(sizeof(bintree_node_t))) ==
	NULL) return -1;
    new_node->data = (void *) data;
    new_node->left = NULL;
    new_node->right = NULL;
    *position = new_node;
    tree->size++;
    return 0;
}

/* ============================ bintree_rem_left () ======================== */
void bintree_rem_left(bintree_t * tree, bintree_node_t * node)
{
    bintree_node_t **position;
    if (bintree_size(tree) == 0)
	return;
    if (node == NULL)
	position = &tree->root;
    else
	position = &node->left;
    if (*position != NULL) {
	bintree_rem_left(tree, *position);
	bintree_rem_right(tree, *position);
	if (tree->destroy != NULL)
	    tree->destroy((*position)->data);
	free(*position);
	*position = NULL;
	tree->size--;
    }
}

/* ============================ bintree_rem_right () ======================= */
void bintree_rem_right(bintree_t * tree, bintree_node_t * node)
{
    bintree_node_t **position;
    if (bintree_size(tree) == 0)
	return;
    if (node == NULL)
	position = &tree->root;
    else
	position = &node->right;
    if (*position != NULL) {
	bintree_rem_left(tree, *position);
	bintree_rem_right(tree, *position);
	if (tree->destroy != NULL)
	    tree->destroy((*position)->data);
	free(*position);
	*position = NULL;
	tree->size--;
    }
}

/* ============================ bintree_merge () =========================== */
int bintree_merge(bintree_t * merge, bintree_t * left, bintree_t * right,
		  const void *data)
{
    bintree_init(merge, left->destroy);
    if (bintree_ins_left(merge, NULL, data) != 0) {
	bintree_destroy(merge);
	return -1;
    }
    bintree_root(merge)->left = bintree_root(left);
    bintree_root(merge)->right = bintree_root(right);
    merge->size = merge->size + bintree_size(left) + bintree_size(right);
    left->root = NULL;
    left->size = 0;
    right->root = NULL;
    right->size = 0;
    return 0;
}
