#   include	"bitmapConfig.h"

#   include	"bmintern.h"
#   include	<string.h>
#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Scaling, Dithereing, Antialiassing.					*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/*  Data structures for dithering.					*/
/*									*/
/************************************************************************/

typedef struct ColorValue
    {
    long	cvR;
    long	cvG;
    long	cvB;
    } ColorValue;

typedef void (*GetSourceRow) ( ColorValue * cv, const unsigned char * from,
					    const BitmapDescription * bdIn );

typedef int (*PutScreenRow)(	unsigned char *		to,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		cor,
				ColorAllocator *	ac );

static int dc1[512];
static int dc3[512];
static int dc5[512];
static int dc7[512];

static void scanFillDitherTables( void )
    {
    int		i;

    for ( i= -256; i < 0; i++ )
	{
	dc1[i+256]= -( ( 8- 1*i+ 8 )/ 16 );
	dc3[i+256]= -( ( 8- 3*i+ 8 )/ 16 );
	dc5[i+256]= -( ( 8- 5*i+ 8 )/ 16 );
	dc7[i+256]= -( ( 8- 7*i+ 8 )/ 16 );
	}

    for ( i= 0; i < 256; i++ )
	{
	dc1[i+256]= ( 1*i+ 8 )/ 16;
	dc3[i+256]= ( 3*i+ 8 )/ 16;
	dc5[i+256]= ( 5*i+ 8 )/ 16;
	dc7[i+256]= ( 7*i+ 8 )/ 16;
	}
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 8 image from rgb8 palette data.			*/
/*									*/
/************************************************************************/

static void scan_ipal(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;
    unsigned	mask;

    switch( bdIn->bdBitsPerPixel )
	{
	case 8:
	    for ( col= 0; col < bdIn->bdPixelsWide; from++, cv++, col++ )
		{
		cv->cvR += bdIn->bdRGB8Palette[*from].rgb8Red;
		cv->cvG += bdIn->bdRGB8Palette[*from].rgb8Green;
		cv->cvB += bdIn->bdRGB8Palette[*from].rgb8Blue;
		}
	    return;

	case 4: case 2: case 1:
	    mask= ( 1 << bdIn->bdBitsPerPixel )- 1;

	    for ( col= 0; col < bdIn->bdBytesPerRow; from++, col++ )
		{
		int	shift;

		for ( shift= 8- bdIn->bdBitsPerPixel; shift >= 0;
					cv++, shift -= bdIn->bdBitsPerPixel )
		    {
		    int		val= ( *from >> shift ) & mask;

		    cv->cvR += bdIn->bdRGB8Palette[val].rgb8Red;
		    cv->cvG += bdIn->bdRGB8Palette[val].rgb8Green;
		    cv->cvB += bdIn->bdRGB8Palette[val].rgb8Blue;
		    }
		}
	    return;

	case 16:
	    {
	    const BmUint16 *	psh= (const BmUint16 *)from;

	    for ( col= 0; col < bdIn->bdPixelsWide; psh++, cv++, col++ )
		{
		cv->cvR += bdIn->bdRGB8Palette[*psh].rgb8Red;
		cv->cvG += bdIn->bdRGB8Palette[*psh].rgb8Green;
		cv->cvB += bdIn->bdRGB8Palette[*psh].rgb8Blue;
		}

	    return;
	    }

	case 32:
	    {
	    const BmUint32 *	plo= (const BmUint32 *)from;

	    for ( col= 0; col < bdIn->bdPixelsWide; plo++, cv++, col++ )
		{
		cv->cvR += bdIn->bdRGB8Palette[*plo].rgb8Red;
		cv->cvG += bdIn->bdRGB8Palette[*plo].rgb8Green;
		cv->cvB += bdIn->bdRGB8Palette[*plo].rgb8Blue;
		}

	    return;
	    }

	default:
	    LDEB(bdIn->bdBitsPerPixel); return;
	}
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 8 image from rgb8 palette data with an alpha channel.	*/
/*									*/
/************************************************************************/

static void scan_ipala(		ColorValue *			cv,
				const unsigned char *		from,
				const BitmapDescription *	bdIn )
    {
    int		col;
    unsigned	mask;

    switch( bdIn->bdBitsPerPixel )
	{
	case 32:
	    {
	    const BmUint16 *	psh= (const BmUint16 *)from;

	    for ( col= 0; col < bdIn->bdPixelsWide; psh += 2, cv++, col++ )
		{
		cv->cvR += bdIn->bdRGB8Palette[*psh].rgb8Red;
		cv->cvG += bdIn->bdRGB8Palette[*psh].rgb8Green;
		cv->cvB += bdIn->bdRGB8Palette[*psh].rgb8Blue;
		}
	    }
	return;

	case 16:
	    for ( col= 0; col < bdIn->bdPixelsWide; from += 2, cv++, col++ )
		{
		cv->cvR += bdIn->bdRGB8Palette[*from].rgb8Red;
		cv->cvG += bdIn->bdRGB8Palette[*from].rgb8Green;
		cv->cvB += bdIn->bdRGB8Palette[*from].rgb8Blue;
		}
	    return;

	case 8:
	    mask= 0x0f;

	    for ( col= 0; col < bdIn->bdPixelsWide; from++, cv++, col++ )
		{
		int		val= ( *from >> 4 ) & mask;

		cv->cvR += bdIn->bdRGB8Palette[val].rgb8Red;
		cv->cvG += bdIn->bdRGB8Palette[val].rgb8Green;
		cv->cvB += bdIn->bdRGB8Palette[val].rgb8Blue;
		}
	    return;

	default:
	    LDEB(bdIn->bdBitsPerPixel); return;
	}
    }

static void scan_ibw1(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; from++, col += 8 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit++ )
	    {
	    unsigned int	val;

	    val= 255* ( ( (*from) >> ( 7- bit ) ) & 0x1 );

	    val= 255- val;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_iwb1(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; from++, col += 8 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit++ )
	    {
	    unsigned int	val;

	    val= 255* ( ( (*from) >> ( 7- bit ) ) & 0x1 );

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_ibw2(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; from++, col += 4 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 2 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xc0 ) + 0xc0/ 2 )/ 0xc0;

	    val= 255- val;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_iwb2(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; from++, col += 4 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 2 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xc0 ) + 0xc0/ 2 )/ 0xc0;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_ibw4(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; from++, col += 2 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 4 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xf0 ) + 0xf0/ 2 )/ 0xf0;

	    val= 255- val;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_iwb4(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; from++, col += 2 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 4 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xf0 ) + 0xf0/ 2 )/ 0xf0;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_ibw8(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; cv++, from++, col++ )
	{
	int		val= 255- *from;

	cv->cvR += val;
	cv->cvG += val;
	cv->cvB += val;
	}
    }

static void scan_iwb8(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; cv++, from++, col++ )
	{
	cv->cvR += *from;
	cv->cvG += *from;
	cv->cvB += *from;
	}
    }

/************************************************************************/
/*									*/
/*  Add data from one row of a 24 bits image to an acumulator array.	*/
/*									*/
/************************************************************************/

static void scan_rgb24(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bdIn )
    {
    int		col;

    for ( col= 0; col < bdIn->bdPixelsWide; cv++, col++ )
	{ cv->cvR += *(from++); cv->cvG += *(from++); cv->cvB += *(from++); }

    return;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 8 image from rgb data.					*/
/*									*/
/************************************************************************/

static int scanFillD8RgbRow(	unsigned char *		to,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		cor,
				ColorAllocator *	ca )
    {
    int			col;
    unsigned int	col332;
    AllocatorColor	ac;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; cor++, col++ )
	    {
	    int		count= 0;
	    int		cw;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    r += cor->cvR; g += cor->cvG; b += cor->cvB;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    col332= C332( r, g, b );

	    if  ( ca->caColors[col332].acAllocated == AC_UNALOCATED )
		{
		if  ( (*ca->caSystemAllocator)( &ac, ca, r, g, b ) )
		    { LDEB(col); return -1;	}
		}
	    else{ ac= ca->caColors[col332];	}

	    *(to++)= ac.acColorNumber;

	    r -= ac.acRed/ 256;		r += 256;
	    g -= ac.acGreen/ 256;	g += 256;
	    b -= ac.acBlue/ 256;	b += 256;

	    val += count;

	    cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
	    cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
	    cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

	    r= dc7[r]; g= dc7[g]; b= dc7[b];
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r += val->cvR; g += val->cvG; b += val->cvB;
		r += cor->cvR; g += cor->cvG; b += cor->cvB;

		if  ( weight > 1 )
		    { r /= weight; g /= weight; b /= weight; }

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		col332= C332( r, g, b );

		if  ( ca->caColors[col332].acAllocated == AC_UNALOCATED )
		    {
		    if  ( (*ca->caSystemAllocator)( &ac, ca, r, g, b ) )
			{ LDEB(col); return -1;	}
		    }
		else{ ac= ca->caColors[col332];	}

		*(to++)= ac.acColorNumber;

		r -= ac.acRed/ 256;	r += 256;
		g -= ac.acGreen/ 256;	g += 256;
		b -= ac.acBlue/ 256;	b += 256;

		cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
		cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
		cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

		r= dc7[r]; g= dc7[g]; b= dc7[b];

		e -= d2; cor++;
		}

	    e += e2;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 1 image from rgb data.					*/
/*									*/
/*  Routines rely on the fact that they are called in scan line order	*/
/*  and that surplus memory has been allocated for the last row.	*/
/*									*/
/************************************************************************/

static int scanFillD1RgbRow_Any(	unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					ColorAllocator *	ca )
    {
    int			col;
    unsigned int	col111;
    AllocatorColor	ac;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; cor++, col++ )
	    {
	    int		count= 0;
	    int		cw;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    r += cor->cvR; g += cor->cvG; b += cor->cvB;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    col111= C111( r, g, b );

	    if  ( ca->ca222Colors[col111].acAllocated == AC_UNALOCATED )
		{
		if  ( (*ca->caSystemAllocator)( &ac, ca, r, g, b ) )
		    { LDEB(col); return -1;	}
		}
	    else{ ac= ca->ca222Colors[col111];	}

	    *(to++)= ac.acColorNumber;

	    r -= ac.acRed/ 256;		r += 256;
	    g -= ac.acGreen/ 256;	g += 256;
	    b -= ac.acBlue/ 256;	b += 256;

	    val += count;

	    cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
	    cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
	    cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

	    r= dc7[r]; g= dc7[g]; b= dc7[b];
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r += val->cvR; g += val->cvG; b += val->cvB;
		r += cor->cvR; g += cor->cvG; b += cor->cvB;

		if  ( weight > 1 )
		    { r /= weight; g /= weight; b /= weight; }

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		col111= C111( r, g, b );

		if  ( ca->ca222Colors[col111].acAllocated == AC_UNALOCATED )
		    {
		    if  ( (*ca->caSystemAllocator)( &ac, ca, r, g, b ) )
			{ LDEB(col); return -1;	}
		    }
		else{ ac= ca->ca222Colors[col111];	}

		*(to++)= ac.acColorNumber;

		r -= ac.acRed/ 256;	r += 256;
		g -= ac.acGreen/ 256;	g += 256;
		b -= ac.acBlue/ 256;	b += 256;

		cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
		cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
		cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

		r= dc7[r]; g= dc7[g]; b= dc7[b];

		e -= d2; cor++;
		}

	    e += e2;
	    }
	}

    return 0;
    }

static int scanFillD1RgbRow_32SX(	unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					ColorAllocator *	ca )
    {
    unsigned char *	row= to;
    int			col;

    if  ( scanFillD1RgbRow_Any( to, frWide, toWide, weight, val, cor, ca ) )
	{ LDEB(1); return -1;	}

    for ( col= 0; col < toWide; to += 4, row += 32, col += 32 )
	{
	to[3]=	( row[ 0]  << 7 )	|
		( row[ 1]  << 6 )	|
		( row[ 2]  << 5 )	|
		( row[ 3]  << 4 )	|
		( row[ 4]  << 3 )	|
		( row[ 5]  << 2 )	|
		( row[ 6]  << 1 )	|
		( row[ 7]  << 0 )	;

	to[2]=	( row[ 8]  << 7 )	|
		( row[ 9]  << 6 )	|
		( row[10]  << 5 )	|
		( row[11]  << 4 )	|
		( row[12]  << 3 )	|
		( row[13]  << 2 )	|
		( row[14]  << 1 )	|
		( row[15]  << 0 )	;

	to[1]=	( row[16]  << 7 )	|
		( row[17]  << 6 )	|
		( row[18]  << 5 )	|
		( row[19]  << 4 )	|
		( row[20]  << 3 )	|
		( row[21]  << 2 )	|
		( row[22]  << 1 )	|
		( row[23]  << 0 )	;

	to[0]=	( row[24]  << 7 )	|
		( row[25]  << 6 )	|
		( row[26]  << 5 )	|
		( row[27]  << 4 )	|
		( row[28]  << 3 )	|
		( row[29]  << 2 )	|
		( row[30]  << 1 )	|
		( row[31]  << 0 )	;
	}

    return 0;
    }

static int scanFillD1RgbRow_32SS(	unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					ColorAllocator *	ca )
    {
    unsigned char *	row= to;
    int			col;

    if  ( scanFillD1RgbRow_Any( to, frWide, toWide, weight, val, cor, ca ) )
	{ LDEB(1); return -1;	}

    for ( col= 0; col < toWide; to += 4, row += 32, col += 32 )
	{
	to[3]=	( row[ 7]  << 7 )	|
		( row[ 6]  << 6 )	|
		( row[ 5]  << 5 )	|
		( row[ 4]  << 4 )	|
		( row[ 3]  << 3 )	|
		( row[ 2]  << 2 )	|
		( row[ 1]  << 1 )	|
		( row[ 0]  << 0 )	;

	to[2]=	( row[15]  << 7 )	|
		( row[14]  << 6 )	|
		( row[13]  << 5 )	|
		( row[12]  << 4 )	|
		( row[11]  << 3 )	|
		( row[10]  << 2 )	|
		( row[ 9]  << 1 )	|
		( row[ 8]  << 0 )	;

	to[1]=	( row[23]  << 7 )	|
		( row[22]  << 6 )	|
		( row[21]  << 5 )	|
		( row[20]  << 4 )	|
		( row[19]  << 3 )	|
		( row[18]  << 2 )	|
		( row[17]  << 1 )	|
		( row[16]  << 0 )	;

	to[0]=	( row[31]  << 7 )	|
		( row[30]  << 6 )	|
		( row[29]  << 5 )	|
		( row[28]  << 4 )	|
		( row[27]  << 3 )	|
		( row[26]  << 2 )	|
		( row[25]  << 1 )	|
		( row[24]  << 0 )	;
	}

    return 0;
    }

static int scanFillD1RgbRow_16SX(	unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					ColorAllocator *	ca )
    {
    unsigned char *	row= to;
    int			col;

    if  ( scanFillD1RgbRow_Any( to, frWide, toWide, weight, val, cor, ca ) )
	{ LDEB(1); return -1;	}

    for ( col= 0; col < toWide; to += 2, row += 16, col += 16 )
	{
	to[1]=	( row[ 0]  << 7 )	|
		( row[ 1]  << 6 )	|
		( row[ 2]  << 5 )	|
		( row[ 3]  << 4 )	|
		( row[ 4]  << 3 )	|
		( row[ 5]  << 2 )	|
		( row[ 6]  << 1 )	|
		( row[ 7]  << 0 )	;

	to[0]=	( row[ 8]  << 7 )	|
		( row[ 9]  << 6 )	|
		( row[10]  << 5 )	|
		( row[11]  << 4 )	|
		( row[12]  << 3 )	|
		( row[13]  << 2 )	|
		( row[14]  << 1 )	|
		( row[15]  << 0 )	;
	}

    return 0;
    }

static int scanFillD1RgbRow_16SS(	unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					ColorAllocator *	ca )
    {
    unsigned char *	row= to;
    int			col;

    if  ( scanFillD1RgbRow_Any( to, frWide, toWide, weight, val, cor, ca ) )
	{ LDEB(1); return -1;	}

    for ( col= 0; col < toWide; to += 2, row += 16, col += 16 )
	{
	to[1]=	( row[ 7]  << 7 )	|
		( row[ 6]  << 6 )	|
		( row[ 5]  << 5 )	|
		( row[ 4]  << 4 )	|
		( row[ 3]  << 3 )	|
		( row[ 2]  << 2 )	|
		( row[ 1]  << 1 )	|
		( row[ 0]  << 0 )	;

	to[0]=	( row[15]  << 7 )	|
		( row[14]  << 6 )	|
		( row[13]  << 5 )	|
		( row[12]  << 4 )	|
		( row[11]  << 3 )	|
		( row[10]  << 2 )	|
		( row[ 9]  << 1 )	|
		( row[ 8]  << 0 )	;
	}

    return 0;
    }

static int scanFillD1RgbRow_8X(		unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					ColorAllocator *	ca )
    {
    unsigned char *	row= to;
    int			col;

    if  ( scanFillD1RgbRow_Any( to, frWide, toWide, weight, val, cor, ca ) )
	{ LDEB(1); return -1;	}

    for ( col= 0; col < toWide; to++, row += 8, col += 8 )
	{
	to[0]=	( row[ 0]  << 7 )	|
		( row[ 1]  << 6 )	|
		( row[ 2]  << 5 )	|
		( row[ 3]  << 4 )	|
		( row[ 4]  << 3 )	|
		( row[ 5]  << 2 )	|
		( row[ 6]  << 1 )	|
		( row[ 7]  << 0 )	;
	}

    return 0;
    }

static int scanFillD1RgbRow_8S(		unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					ColorAllocator *	ca )
    {
    unsigned char *	row= to;
    int			col;

    if  ( scanFillD1RgbRow_Any( to, frWide, toWide, weight, val, cor, ca ) )
	{ LDEB(1); return -1;	}

    for ( col= 0; col < toWide; to++, row += 8, col += 8 )
	{
	to[0]=	( row[ 7]  << 7 )	|
		( row[ 6]  << 6 )	|
		( row[ 5]  << 5 )	|
		( row[ 4]  << 4 )	|
		( row[ 3]  << 3 )	|
		( row[ 2]  << 2 )	|
		( row[ 1]  << 1 )	|
		( row[ 0]  << 0 )	;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 16 image from rgb data.				*/
/*									*/
/************************************************************************/

static int scanFillD16RgbRow(	unsigned char *		cto,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		corx,
				ColorAllocator *	ca )
    {
    int			col;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    BmUint16 *		to= (BmUint16 *)cto;
    AllocatorColor	ac;
    int			col555;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; col++ )
	    {
	    int		count= 0;
	    int		cw;

	    r= g= b= 0;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    if  ( ca->caAllocationType == AC_CALCULATED )
		{ bmColorRgbDirect( &ac, ca, r, g, b );	}
	    else{
		col555= C555( r, g, b );

		if  ( ca->caColors[col555].acAllocated == AC_UNALOCATED )
		    {
		    if  ( (*ca->caSystemAllocator)( &ac, ca, r, g, b ) )
			{ LDEB(col); return -1;	}
		    }
		else{ ac= ca->caColors[col555];	}
		}

	    *(to++)= ac.acColorNumber;

	    val += count;
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r= val->cvR; g= val->cvG; b= val->cvB;

		if  ( weight > 1 )
		    { r /= weight; g /= weight; b /= weight; }

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		if  ( ca->caAllocationType == AC_CALCULATED )
		    { bmColorRgbDirect( &ac, ca, r, g, b );	}
		else{
		    col555= C555( r, g, b );

		    if  ( ca->caColors[col555].acAllocated == AC_UNALOCATED )
			{
			if  ( (*ca->caSystemAllocator)( &ac, ca, r, g, b ) )
			    { LDEB(col); return -1;	}
			}
		    else{ ac= ca->caColors[col555];	}
		    }

		*(to++)= ac.acColorNumber;

		e -= d2;
		}

	    e += e2;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 24 image from rgb data.				*/
/*									*/
/************************************************************************/

static int scanFillD24RgbRow(	unsigned char *		to,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		corx,
				ColorAllocator *	ac )
    {
    int			col;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; col++ )
	    {
	    int		count= 0;
	    int		cw;

	    r= g= b= 0;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    *(to++)= r; *(to++)= g; *(to++)= b;

	    val += count;
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r= val->cvR; g= val->cvG; b= val->cvB;

		if  ( weight > 1 )
		    { r /= weight; g /= weight; b /= weight; }

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		*(to++)= r; *(to++)= g; *(to++)= b;

		e -= d2;
		}

	    e += e2;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 32 image from rgb data.				*/
/*									*/
/************************************************************************/

static int scanFillD32RgbRow(	unsigned char *		cto,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		corx,
				ColorAllocator *	ca )
    {
    int			col;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    BmUint32 *		to= (BmUint32 *)cto;
    AllocatorColor	ac;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; col++ )
	    {
	    int		count= 0;
	    int		cw;

	    r= g= b= 0;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    bmColorRgbDirect( &ac, ca, r, g, b );
	    *(to++)= ac.acColorNumber;

	    val += count;
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r= val->cvR; g= val->cvG; b= val->cvB;

		if  ( weight > 1 )
		    { r /= weight; g /= weight; b /= weight; }

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		bmColorRgbDirect( &ac, ca, r, g, b );
		*(to++)= ac.acColorNumber;

		e -= d2;
		}

	    e += e2;
	    }
	}

    return 0;
    }

static void scanInitRow(	ColorValue *		cv,
				int			wide )
    {
    int			col;

    for ( col= 0; col < wide; cv++, col++ )
	{ cv->cvR= cv->cvG= cv->cvB= 0L;	}
    }

static int bmBuildImage(	ColorAllocator *		ca,
				unsigned char *			bufferOut,
				const unsigned char *		bufferIn,
				const BitmapDescription *	bdOut,
				const BitmapDescription *	bdIn,
				GetSourceRow			getSource,
				PutScreenRow			putRow )
    {
    int				rval= 0;

    int				toRow;
    int				frRow;

    const unsigned char *	from;
    unsigned char *		to;

    int				toWide= bdOut->bdPixelsWide;
    int				toHigh= bdOut->bdPixelsHigh;
    int				frWide= bdIn->bdPixelsWide;
    int				frHigh= bdIn->bdPixelsHigh;

    int				e;
    int				d2;
    int				e2;

    ColorValue *		thisRow= (ColorValue *)0;
    ColorValue *		nextCor= (ColorValue *)0;

    /*  1  */
    thisRow= (ColorValue *)malloc( ( frWide+ 9 )* sizeof(ColorValue) );
    if  ( ! thisRow )
	{ LXDEB(frWide,thisRow); rval= -1; goto ready;	}

    nextCor= (ColorValue *)malloc( ( toWide+ 9 )* sizeof(ColorValue) );
    if  ( ! nextCor )
	{ LXDEB(toWide,nextCor); rval= -1; goto ready;	}

    if  ( toHigh <= frHigh )
	{
	e2= 2* frHigh;
	d2= 2* toHigh;
	e= e2- toHigh;

	scanInitRow( nextCor+ 1, toWide );

	frRow= 0;
	for ( toRow= 0; toRow < toHigh; toRow++ )
	    {
	    int		count= 0;

	    scanInitRow( thisRow+ 1, frWide );
	    to= bufferOut+ toRow* bdOut->bdBytesPerRow;

	    while( e >= 0 )
		{
		from= bufferIn+ frRow* bdIn->bdBytesPerRow;

		(*getSource)( thisRow+ 1, from, bdIn );

		frRow++; count++; e -= d2;
		}

	    e += e2;

	    if  ( (*putRow)( to, frWide, toWide, count,
					    thisRow+ 1, nextCor+ 1, ca ) )
		{ LDEB(toRow); free( bufferOut ); rval= -1; goto ready;	}
	    }
	}
    else{
	e2= 2* toHigh;
	d2= 2* frHigh;
	e= e2- frHigh;

	toRow= 0;
	for ( frRow= 0; frRow < frHigh; frRow++ )
	    {
	    from= bufferIn+ frRow* bdIn->bdBytesPerRow;

	    scanInitRow( thisRow+ 1, frWide );
	    (*getSource)( thisRow+ 1, from, bdIn );

	    while( e >= 0 )
		{
		to= bufferOut+ toRow* bdOut->bdBytesPerRow;

		if  ( (*putRow)( to, frWide, toWide, 1,
				    thisRow+ 1, nextCor+ 1, ca ) )
		    { LDEB(toRow); free( bufferOut ); rval= -1; goto ready; }

		e -= d2; toRow++;
		}

	    e += e2;
	    }
	}

  ready:
    if  ( thisRow )
	{ free( thisRow );	}
    if  ( nextCor )
	{ free( nextCor );	}

    return rval;
    }

int bmFillImage(	ColorAllocator *		ca,
			int				bitmapUnit,
			int				swapBitmapBytes,
			int				swapBitmapBits,
			unsigned char *			bufferOut,
			const unsigned char *		bufferIn,
			BitmapDescription *		bdOut,
			const BitmapDescription *	bdIn )
    {
    PutScreenRow	putRow= (PutScreenRow)0;

    if  ( ca->caAllocationType == AC_CALCULATED )
	{
	switch( bdOut->bdBitsPerPixel )
	    {
	    case 32:
		putRow= scanFillD32RgbRow;
		break;
	    case 24:
		putRow= scanFillD24RgbRow;
		break;
	    case 16:
		putRow= scanFillD16RgbRow;
		break;
	    default:
		LDEB(bdOut->bdBitsPerPixel);
		return -1;
	    }
	}
    else{
	switch( bdOut->bdBitsPerPixel )
	    {
	    case 1:
		if  ( swapBitmapBytes )
		    {
		    switch( bitmapUnit )
			{
			case 16:
			    if  ( swapBitmapBits )
				{ putRow= scanFillD1RgbRow_16SX;	}
			    else{ putRow= scanFillD1RgbRow_16SS;	}
			break;

			case 32:
			    if  ( swapBitmapBits )
				{ putRow= scanFillD1RgbRow_32SX;	}
			    else{ putRow= scanFillD1RgbRow_32SS;	}
			break;

			default:
			    LDEB(bitmapUnit); return -1;
			}
		    }
		else{
		    if  ( swapBitmapBits )
			{ putRow= scanFillD1RgbRow_8S;		}
		    else{ putRow= scanFillD1RgbRow_8X;		}
		    }
		break;

	    case 8:
		putRow= scanFillD8RgbRow;
		break;

	    case 16:
		putRow= scanFillD16RgbRow;
		break;

	    default:
		LDEB(bdOut->bdBitsPerPixel);
		return -1;
	    }
	}

    /*  3  */
    if  ( ! dc1[0] )
	{ scanFillDitherTables();	}

    /*  4  */
    switch( bdIn->bdColorEncoding )
	{
	case BMcoRGB8PALETTE:
	    if  ( bdIn->bdHasAlpha )
		{
		if  ( bmBuildImage( ca, bufferOut, bufferIn,
				bdOut, bdIn, scan_ipala, putRow ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( bmBuildImage( ca, bufferOut, bufferIn,
				bdOut, bdIn, scan_ipal, putRow ) )
		    { LDEB(1); return -1;	}
		}
	    break;

	case BMcoBLACKWHITE:
	    switch( bdIn->bdBitsPerPixel )
		{
		case 1:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_ibw1, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 2:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_ibw2, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 4:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_ibw4, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 8:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_ibw8, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		default:
		    LLDEB(bdIn->bdColorEncoding,bdIn->bdBitsPerPixel);
		    return -1;
		}
	    break;

	case BMcoWHITEBLACK:
	    switch( bdIn->bdBitsPerPixel )
		{
		case 1:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_iwb1, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 2:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_iwb2, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 4:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_iwb4, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 8:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_iwb8, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		default:
		    LLDEB(bdIn->bdColorEncoding,bdIn->bdBitsPerPixel);
		}
	    break;

	case BMcoRGB:
	    switch( bdIn->bdBitsPerSample )
		{
		case 8:
		    if  ( bmBuildImage( ca, bufferOut, bufferIn,
					bdOut, bdIn, scan_rgb24, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		default:
		    LLDEB(bdIn->bdColorEncoding,bdIn->bdBitsPerSample);
		    return -1;
		}
	    break;
	default:
	    LDEB(bdIn->bdColorEncoding); return -1;
	}

    return 0;
    }

