/************************************************************************/
/*									*/
/*  Color allocation, mainly for bitmap images.				*/
/*									*/
/************************************************************************/

#   ifndef	BMCOLOR_H
#   define	BMCOLOR_H

/************************************************************************/
/*									*/
/*  An RGB8 Color.							*/
/*  The most usual way to display images on a computer screen.		*/
/*									*/
/*  An RGB16 Color.							*/
/*  Used to match X11's way of handling colors.				*/
/*									*/
/*  NOTE that the structs use native types so the 8/16 indication is	*/
/*	more a suggestion than an implementation reality.		*/
/*									*/
/************************************************************************/

typedef struct	RGB8Color
    {
    unsigned char	rgb8Red;
    unsigned char	rgb8Green;
    unsigned char	rgb8Blue;
    unsigned char	rgb8Alpha;
    } RGB8Color;

typedef struct	RGB16Color
    {
    unsigned short	rgb16Red;
    unsigned short	rgb16Green;
    unsigned short	rgb16Blue;
    unsigned char	rgb16Alpha;
    } RGB16Color;

/************************************************************************/
/*									*/
/*  For color approximation, anti aliasing and dithering		*/
/*									*/
/************************************************************************/

typedef struct AllocatorColor
    {
    RGB16Color		acColorValues;
    unsigned long	acColorNumber;
    unsigned char	acAllocated;
#			define		AC_UNALOCATED	0
#			define		AC_ALOCATED	1
#			define		AC_CALCULATED	2
#			define		AC_COPIED	3
    } AllocatorColor;

#   define acRed	acColorValues.rgb16Red
#   define acGreen	acColorValues.rgb16Green
#   define acBlue	acColorValues.rgb16Blue

struct ColorAllocator;

typedef int (*SystemAllocator)(		AllocatorColor *	ac,
					struct ColorAllocator *	ca,
					unsigned int		r,
					unsigned int		g,
					unsigned int		b );

typedef struct ColorAllocator
    {
    AllocatorColor	ca222Colors[64];
    AllocatorColor *	caColors;
    int			caColorCount;
    int			caDepth;
    unsigned int	caRedApproxShift;
    unsigned int	caGreenApproxShift;
    unsigned int	caBlueApproxShift;
    unsigned int	caRedPixelShift;
    unsigned int	caGreenPixelShift;
    unsigned int	caBluePixelShift;
    unsigned int	caRedMask;
    unsigned int	caGreenMask;
    unsigned int	caBlueMask;

    int			caAllocationType;
    void *		caSystemPrivate;
    SystemAllocator	caSystemAllocator;

    } ColorAllocator;

/************************************************************************/
/*									*/
/*  Macros for indexing colors in an array of colors, usually in an	*/
/*  ColorAllocator structure.						*/
/*									*/
/************************************************************************/

#   define	C555(r,g,b)	( ( ( (r) << 7 ) & 0x7c00 ) | \
				  ( ( (g) << 2 ) & 0x03e0 ) | \
				  ( ( (b) >> 3 ) & 0x001f ) )

#   define	C332(r,g,b)	( ( ( (r) >> 0 ) & 0xe0 ) | \
				  ( ( (g) >> 3 ) & 0x1c ) | \
				  ( ( (b) >> 6 ) & 0x03 ) )

#   define	C222(r,g,b)	( ( ( (r) >> 2 ) & 0x30 ) | \
				  ( ( (g) >> 4 ) & 0x0c ) | \
				  ( ( (b) >> 6 ) & 0x03 ) )

#   define	C111(r,g,b)	( ( ( (r) >> 5 ) & 0x04 ) | \
				  ( ( (g) >> 6 ) & 0x02 ) | \
				  ( ( (b) >> 7 ) & 0x01 ) )

#   define bmColorRgbDirect( ac, ca, r, g, b )			\
    {								\
    unsigned int	rr= (r) >> (ca)->caRedApproxShift;	\
    unsigned int	gg= (g) >> (ca)->caGreenApproxShift;	\
    unsigned int	bb= (b) >> (ca)->caBlueApproxShift;	\
    \
    (ac)->acRed=   ( 65535* rr )/ (ca)->caRedMask;		\
    (ac)->acGreen= ( 65535* gg )/ (ca)->caGreenMask;		\
    (ac)->acBlue=  ( 65535* bb )/ (ca)->caBlueMask;		\
    \
    (ac)->acColorNumber= ( rr << (ca)->caRedPixelShift ) +	\
		 ( gg << (ca)->caGreenPixelShift )	+	\
		 ( bb << (ca)->caBluePixelShift )	;	\
    }


/************************************************************************/
/*									*/
/*  Routine declarations:						*/
/*									*/
/************************************************************************/

extern void bmSetCalculatedShifts(	ColorAllocator *	ca,
					unsigned long		redMask,
					unsigned long		greenMask,
					unsigned long		blueMask );

extern void bmInitColorAllocator(	ColorAllocator *	ca );
extern void bmCleanColorAllocator(	ColorAllocator *	ca );

#   endif	/*  BMCOLOR_H  */

