#include "config.h"
#include <stdlib.h>
#include <pango/pango.h>
#include <panel-applet.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnome/libgnome.h>
#include <gst/gst.h>

#define FRAMES  3
#define TEACUPS 3

#define ROT_FRAMES  24
#define CUP_WIDTH  128
#define CUP_HEIGHT  96

#define DEFAULT_ASSAM	   260
#define DEFAULT_DARJEELING 220
#define DEFAULT_GREEN	   180
#define DEFAULT_MINT	   450
#define DEFAULT_HERBAL	   600
#define DEFAULT_FRUIT	   720

#define TEATIME_PREFS_CUP	     "cup"
#define TEATIME_PREFS_POPUP	     "popup"
#define TEATIME_PREFS_SOUND          "sound"
#define TEATIME_PREFS_PLAY_SOUND     "play_sound"
#define TEATIME_PREFS_SOUND_ENTRY    "sound_entry"
#define TEATIME_PREFS_SHOW_CLOCK     "show_clock"
#define TEATIME_PREFS_LAST_TEA       "last_tea"

static gchar *teas[] = {
  N_("Assam"),
  N_("Darjeling"),
  N_("Fruit Tea"),
  N_("Green Tea"),
  N_("Herbal Tea"),
  N_("Peppermint Tea")
};

static gchar *teas_pref[] = {
  "assam",
  "darjeling",
  "fruit",
  "green",
  "herbal",
  "mint"
};

static gint default_drawing_times[] = {
  DEFAULT_ASSAM,
  DEFAULT_DARJEELING,
  DEFAULT_GREEN,
  DEFAULT_MINT,
  DEFAULT_HERBAL,
  DEFAULT_FRUIT
};

static gchar *teacups[] = {
  "cup_classic",
  "cup_glass",
  "mug_soeren"
};

typedef struct{
  gboolean visible;
  
  /* label vars */
  GtkWidget *label_window;
  PangoLayout *layout;
  gint label_width;
  gint label_height;
  
  /*cup vars */
  GtkWidget *cup_window;
  GtkWidget *image[ROT_FRAMES];
  GdkBitmap *bitmap[ROT_FRAMES];
  guint timer_handle;
  gint frame;
}TeatimePopup;

typedef struct{
  GstElement *thread;
  GstElement *input;
  GstElement *decoder;
  GstElement *output;
}TeatimeSound;

typedef struct{
  gint drawing_time[6];
  gboolean popup;
  gboolean show_clock;
  gint last_tea;
  
  gint cup;
  GdkPixbuf *teacups[TEACUPS];

  gboolean play_sound;
  gchar *sound;
}TeatimeProperties;

typedef struct{
  PanelApplet *applet;
  GtkWidget *border;
  GtkWidget *area;
  gint size;
  gboolean orient_vert;
  GtkTooltips *tooltip;
  GdkPixbuf *cup_empty;
  GdkPixbuf *cup_empty_hl;
  GdkPixbuf *cup_full[FRAMES];
  GdkPixbuf *cup_full_hl[FRAMES];

  GdkGC *clock_gc;

  gboolean full;
  gboolean highlight;
  gint frame;
  gint time_left;
  guint timer_handle;

  gint last_tea;
  gint drawing_time;
  
  gchar *icon_path;
  GtkAdjustment *pref_time_adj[6];
  GtkWidget *pref_time_spin[6];

  GtkWidget *pb;
  GtkWidget *pref_cup[TEACUPS];
  GtkWidget *pref_cup_button[TEACUPS];
  
  TeatimeProperties properties;
  TeatimeSound sound;
  TeatimePopup popup;
}Teatime;

static void set_menu (Teatime *teatime);

static void
set_tooltip (Teatime *teatime)
{
  gchar *tip;

  if (teatime->full)
  {
    tip = g_strdup_printf (_("%s %i s left"),
                           _(teas[teatime->last_tea]),
                           teatime->time_left / 3);
  }
  else
  {
    tip = g_strdup_printf (_("%s (%is)"),
                           _(teas[teatime->last_tea]),
                           teatime->properties.drawing_time[teatime->last_tea]);
  }
  gtk_tooltips_set_tip (teatime->tooltip, teatime->area, tip, NULL);
  g_free (tip);
}

static void
redraw( Teatime *teatime)
{
  GdkPixbuf *active;
  gint size;
  if (!teatime->full)
    active = teatime->highlight ? teatime->cup_empty_hl : teatime->cup_empty;
  else
   active = teatime->highlight ? teatime->cup_full_hl[teatime->frame] : teatime->cup_full[teatime->frame];
  gdk_pixbuf_render_to_drawable_alpha (active, teatime->area->window,
                                       0, 0,
                                       0, 0,
                                       teatime->size, teatime->size,
                                       GDK_PIXBUF_ALPHA_FULL,
                                       1,
                                       GDK_RGB_DITHER_NONE,
                                       0, 0);
  if (!teatime->clock_gc)
  {
    static GdkColor clock_color = {0, 0xFFFF, 0x0000, 0x0000};
    GdkColormap * colormap = gtk_widget_get_colormap (gtk_widget_get_toplevel (teatime->area));
    gdk_colormap_alloc_color (colormap, &clock_color, TRUE, TRUE);
    teatime->clock_gc = gdk_gc_new (teatime->area->window);
    gdk_gc_set_foreground (teatime->clock_gc, &clock_color);
  }
  if (teatime->time_left && teatime->properties.show_clock)
    gdk_draw_arc (teatime->area->window,
                  teatime->clock_gc,
                  FALSE,
                  1, 1,
                  teatime->size -2, teatime->size -2,
                  90 * 64 , (gint)(1 - ((gfloat)teatime->time_left / (gfloat)teatime->drawing_time / 3.0) * 360.0 * 64.0));

  set_tooltip (teatime);
}

static void
enter_event (GtkWidget *widget, GdkEvent *event, Teatime *teatime)
{
  teatime->highlight = TRUE;
  redraw (teatime);
}

static void
leave_event (GtkWidget *widget, GdkEvent *event, Teatime *teatime)
{
  teatime->highlight = FALSE;
  redraw (teatime);
}

static void
animate (Teatime *teatime)
{
  teatime->frame++;
  if (teatime->frame == FRAMES)
    teatime->frame = 0;
  redraw (teatime);
}

static void
popup_hide (GtkWidget *widget, GdkEvent *event, TeatimePopup *popup)
{
  gtk_timeout_remove (popup->timer_handle);
  gtk_widget_hide (popup->cup_window);
  gtk_widget_hide (popup->label_window);
  popup->visible = FALSE;
}

static void
cup_popup (TeatimePopup *popup)
{
  static GdkPixbuf *pixbuf;
  static GdkPixmap *pixmap;
  GtkWidget *hbox;
  GtkWidget *eventbox;
  gint i;
    
  popup->cup_window = gtk_window_new (GTK_WINDOW_POPUP);
  gtk_widget_set_usize (popup->cup_window, CUP_WIDTH, CUP_HEIGHT);
  hbox = gtk_hbox_new (TRUE, 0);
  for (i=0; i<ROT_FRAMES; i++)
    gtk_box_pack_start (GTK_BOX (hbox), popup->image[i], FALSE, FALSE, 0);
  gtk_widget_shape_combine_mask (popup->cup_window, popup->bitmap[0], 0, 0);
  eventbox = gtk_event_box_new ();
  gtk_container_add (GTK_CONTAINER (eventbox), hbox);	
  gtk_container_add (GTK_CONTAINER (popup->cup_window), eventbox);
  g_signal_connect (G_OBJECT (eventbox), "button_release_event",
                    G_CALLBACK (popup_hide), popup);
  gtk_widget_show (eventbox);
  gtk_widget_show (hbox);
  gtk_widget_show (popup->image[0]);
}

static void
draw_layout(GtkWidget *widget, GdkEvent *event, TeatimePopup *popup)
{
  gchar *tea_is_ready = {N_("Your Tea\nis ready")};
  gchar *string;

  string = g_strdup_printf ("<span face=\"bold\" foreground=\"gold\">"
                            "<big><big>%s</big></big></span>", _(tea_is_ready));
  pango_layout_set_markup (popup->layout, string, -1);
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   0, 0,
                   popup->layout);
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   0, 1,
                   popup->layout);
  g_free (string);

  string = g_strdup_printf ("<span face=\"bold\" foreground=\"gold4\">"
                            "<big><big>%s</big></big></span>", _(tea_is_ready));
  pango_layout_set_markup (popup->layout, string, -1); 
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   1, 1,
                   popup->layout);
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   2, 1,
                   popup->layout);
  g_free (string);
}

static void
label_popup (TeatimePopup *popup, GtkWidget *widget)
{
  GtkWidget *area;
  GdkPixbuf *pixbuf;
  GdkPixmap *pixmap;
  GdkBitmap *bitmap;
  guchar *pixels;
  gchar *tea_is_ready = {N_("Your Tea\nis ready")};
  gchar *string; 
  gint x, y;
  GdkGC *fggc, *bggc;
  GdkGC *fggc_bit, *bggc_bit;
  GdkColor black = {0, 0, 0, 0};
  GdkColor white = {1, 0xffff, 0xffff, 0xffff};
 
  popup->label_window = gtk_window_new (GTK_WINDOW_POPUP);
  area = gtk_drawing_area_new ();

  gtk_container_add (GTK_CONTAINER (popup->label_window), area);
  g_signal_connect (G_OBJECT (area), "expose_event",
                    G_CALLBACK (draw_layout), popup);

  popup->layout = gtk_widget_create_pango_layout (widget, "");
  pango_layout_set_alignment (popup->layout, PANGO_ALIGN_CENTER);

  string = g_strdup_printf ("<span face=\"bold\" foreground=\"black\">"
                            "<big><big>%s</big></big></span>", _(tea_is_ready)); 
  pango_layout_set_markup (popup->layout, string, -1);
  g_free (string);
  pango_layout_get_pixel_size (popup->layout, &popup->label_width, &popup->label_height);
  if (popup->label_width%4)
    popup->label_width = popup->label_width - popup->label_width%4 + 4;

  gtk_widget_set_usize (popup->label_window, popup->label_width, popup->label_height); 
  pixmap = gdk_pixmap_new (widget->window, popup->label_width, popup->label_height, -1);
  bitmap = gdk_pixmap_new (NULL, popup->label_width, popup->label_height, 1);  
  
  fggc = gdk_gc_new (pixmap);
  bggc = gdk_gc_new (pixmap);
  fggc_bit = gdk_gc_new (bitmap);
  bggc_bit = gdk_gc_new (bitmap);

  gdk_gc_set_foreground (bggc, &black);
  gdk_gc_set_foreground (fggc, &white);
  gdk_gc_set_foreground (fggc_bit, &black);
  gdk_gc_set_foreground (bggc_bit, &white);
 
  gdk_draw_rectangle (pixmap, widget->style->white_gc, TRUE, 0, 0,
                      popup->label_width, popup->label_height);
  
  gdk_draw_layout (pixmap,
                   fggc,
                   0, 0, popup->layout);
  gdk_draw_layout (pixmap,
                   fggc,
                   1, 1, popup->layout);

  pixbuf = gdk_pixbuf_get_from_drawable (NULL,
                                         pixmap,
                                         gdk_drawable_get_colormap (pixmap),
                                         0, 0, 0, 0,
                                         popup->label_width, popup->label_height);
  pixels = gdk_pixbuf_get_pixels (pixbuf);

  for (y=0; y< popup->label_height; y++)
    for (x=0; x<popup->label_width; x++)
    {
      if (*(pixels+2) <255)
        gdk_draw_point (bitmap, bggc_bit, x, y);
      else
        gdk_draw_point (bitmap, fggc_bit, x, y);
      pixels+=3;      
    }
    
  gtk_widget_shape_combine_mask (popup->label_window, bitmap, 0, 0);
  gtk_widget_show (area);
}

static gint
popup_cup_timer (TeatimePopup *popup)
{
  gint hide;

  hide = popup->frame -1;
  
  if (hide < 0)
    hide = ROT_FRAMES - 1;
  gtk_widget_hide (popup->image[hide]);
  gtk_widget_show (popup->image[popup->frame]);
  gtk_widget_shape_combine_mask (popup->cup_window, popup->bitmap[popup->frame], 0, 0);
  popup->frame++;
  if (popup->frame == ROT_FRAMES)
    popup->frame = 0;
  return TRUE;  
}

void
stop_sound_thread (GstElement *element,
                   gpointer data)
{
  gst_element_set_state (GST_ELEMENT (data), GST_STATE_NULL);
}

static gint
timer(Teatime *teatime)
{
  gint center_x, center_y;
 
  if (--teatime->time_left)
  {
    animate (teatime);
    return TRUE;
  }
  
  teatime->full = FALSE;
  if (teatime->timer_handle)
    gtk_timeout_remove (teatime->timer_handle);
  redraw (teatime);
  if (teatime->properties.play_sound && teatime->properties.sound)
  {
    teatime->sound.thread = gst_thread_new ("pipe");
    teatime->sound.input = gst_element_factory_make ("filesrc", "input");
    g_signal_connect (G_OBJECT (teatime->sound.input), "eos",
                      G_CALLBACK (stop_sound_thread), teatime->sound.thread);
    teatime->sound.decoder = gst_element_factory_make ("spider", "decoder");
    teatime->sound.output = gst_element_factory_make ("osssink", "output");
    gst_bin_add_many (GST_BIN (teatime->sound.thread),
                      teatime->sound.input,
                      teatime->sound.decoder,
                      teatime->sound.output, NULL);
    gst_element_link_many (teatime->sound.input,
                           teatime->sound.decoder,
                           teatime->sound.output, NULL);   
    g_object_set (G_OBJECT (teatime->sound.input), "location", teatime->properties.sound, NULL);
    gst_element_set_state (teatime->sound.thread,  GST_STATE_PLAYING);
  }
  if (!teatime->properties.popup)
    return FALSE;
      
  teatime->popup.visible = TRUE;

  if (teatime->popup.cup_window)
  {
    gdk_window_raise (teatime->popup.cup_window->window);
    gdk_window_raise (teatime->popup.label_window->window);
    gtk_widget_hide (teatime->popup.image[teatime->popup.frame - 1]);
    gtk_widget_show (teatime->popup.image[0]);
  }
  else
  {
    cup_popup (&teatime->popup);
    label_popup (&teatime->popup, teatime->border);
  }
  teatime->popup.frame = 0;    
  gdk_window_get_size (gdk_get_default_root_window(), &center_x, &center_y);
  center_x >>= 1;
  center_y >>= 1;
  gtk_widget_set_uposition (teatime->popup.cup_window,
                            center_x - CUP_WIDTH / 2,
                            center_y - (CUP_HEIGHT + teatime->popup.label_height) / 2);
  gtk_widget_set_uposition (teatime->popup.label_window,
                            center_x - teatime->popup.label_width / 2,
                            center_y  - (CUP_HEIGHT + teatime->popup.label_height) / 2
                            + CUP_HEIGHT);

  gtk_widget_show (teatime->popup.cup_window);
  gtk_widget_show (teatime->popup.label_window);

  teatime->popup.timer_handle = g_timeout_add (60,
                                               (GSourceFunc)popup_cup_timer,
                                               &teatime->popup);
  return FALSE;
}

static void
lighten (GdkPixbuf *src, GdkPixbuf *dest, gint cup, gint size)
{
  gint i, j, rowstride;
  guchar *target_pixels;
  guchar *original_pixels;
  guchar *current_pixel;
  guchar light_val = 16;
  gint bits_per_sample;
  gint chanels;
 
  bits_per_sample = gdk_pixbuf_get_bits_per_sample (src);
  if ( bits_per_sample != 8)
  {
    g_warning (_("Only images with a color depth of 24 or 32 Bit are allowed for the Teatime Aplett\n"
               "The requested images %s* have a depth of %i"), teacups[cup], bits_per_sample);
    return;
  }
  chanels = gdk_pixbuf_get_n_channels (src);
  target_pixels = gdk_pixbuf_get_pixels (dest);
  original_pixels = gdk_pixbuf_get_pixels (src);
  rowstride = gdk_pixbuf_get_rowstride (src);
  for (i = 0; i < size; i++)
  {
    for (j = 0; j < size; j++)
    {
      current_pixel = original_pixels + i*rowstride + j*chanels;

      *(target_pixels + i*rowstride + j*chanels) = MIN( light_val + (*(current_pixel)), 255);
      *(target_pixels + i*rowstride + j*chanels + 1) = MIN( light_val + (*(current_pixel + 1)), 255);
      *(target_pixels + i*rowstride + j*chanels + 2) = MIN( light_val + (*(current_pixel + 2)), 255);
      if (chanels == 4)
        *(target_pixels + i*rowstride + j*chanels + 3) = *(current_pixel + 3);
    }
  }
}

static gboolean
load_applet_images (Teatime *teatime)
{
  GError *error = NULL;
  gint i;
  gchar *filename;
  const gchar *path;
  GdkPixbuf *tmp1 = NULL;
  GdkPixbuf *tmp2 = NULL;

  filename = g_strdup_printf ("teatime/%s_empty.png", teacups[teatime->properties.cup]);
  teatime->icon_path = gnome_program_locate_file( NULL,
                                                 GNOME_FILE_DOMAIN_PIXMAP,
                                                 filename,
                                                 FALSE,
                                                 NULL);
  if (teatime->cup_empty)
    tmp1 = teatime->cup_empty;
    
  tmp2 = gdk_pixbuf_new_from_file (teatime->icon_path, &error);
  if (error)
  {
    g_warning (G_STRLOC ": cannot open %s: %s", filename, error);
    g_error_free (error);
    return FALSE;
  }
  g_free (filename);
  teatime->cup_empty = gdk_pixbuf_scale_simple (tmp2,
                                                teatime->size, teatime->size,
                                                GDK_INTERP_HYPER);
  g_object_unref (G_OBJECT (tmp2));
  tmp2 = gdk_pixbuf_copy (teatime->cup_empty);
  if (tmp1)
    g_object_unref (G_OBJECT (tmp1));
  
  tmp1 = teatime->cup_empty_hl;
  lighten (teatime->cup_empty,
           tmp2,
           teatime->properties.cup,
           teatime->size);
  teatime->cup_empty_hl = tmp2;
  if (tmp1)
    g_object_unref (G_OBJECT (tmp1));

  for (i=0; i<FRAMES; i++)
  {
    tmp1 = NULL;
    tmp2 = NULL;
    filename = g_strdup_printf ("teatime/%s_full_%i.png",
                                teacups[teatime->properties.cup], i);
    path = gnome_program_locate_file (NULL,
                                      GNOME_FILE_DOMAIN_PIXMAP,
                                      filename,
                                      FALSE,
                                      NULL);
    if (teatime->cup_full[i])
      tmp1 = teatime->cup_full[i];
    tmp2 = gdk_pixbuf_new_from_file (path,
                                     &error);
    if (error)
    {
      g_warning (G_STRLOC ": cannot open %s: %s", filename, error);
      g_error_free (error);
      return FALSE;
    }
    g_free (filename);
    teatime->cup_full[i] = gdk_pixbuf_scale_simple (tmp2,
                                                    teatime->size, teatime->size,
                                                    GDK_INTERP_HYPER);
    g_object_unref (G_OBJECT (tmp2));
    tmp2 = gdk_pixbuf_copy (teatime->cup_full[i]);     
    if (tmp1)
      g_object_unref (G_OBJECT (tmp1));
      
    tmp1 = teatime->cup_full_hl[i];
    lighten (teatime->cup_full[i],
             tmp2,
             teatime->properties.cup,
             teatime->size);
    teatime->cup_full_hl[i] = tmp2;
    if (tmp1)
      g_object_unref (G_OBJECT (tmp1));
  }

  if (GTK_WIDGET_REALIZED (teatime->area))
    redraw (teatime); 
  return TRUE;
}

static gboolean
load_popup_images (Teatime *teatime)
{    
  GError *error = NULL;
  gint i;
  gchar *filename;
  const gchar *path;

  for (i=0; i<ROT_FRAMES; i++)
  {
    gchar *filename;
    GdkPixbuf *pixbuf;
    GdkPixmap *pixmap;
      
    if (i<10)

  /* FIXME   cup%2i.png ??? */
      filename = g_strdup_printf ("teatime/cup0%i.png", i);
    else
      filename = g_strdup_printf ("teatime/cup%i.png", i);
    path = gnome_program_locate_file (NULL,
                                      GNOME_FILE_DOMAIN_PIXMAP,
                                      filename,
                                      FALSE,
                                      NULL);
    pixbuf = gdk_pixbuf_new_from_file (path, &error);
    if (error)
    {
      g_warning (G_STRLOC ": cannot open %s: %s", path, error);
      g_error_free (error);
      return FALSE;
    }
    g_free (filename);
    gdk_pixbuf_render_pixmap_and_mask (pixbuf,
                                       &pixmap,
                                       &teatime->popup.bitmap[i],
                                       128);
    g_object_unref (G_OBJECT (pixbuf));
    teatime->popup.image[i] = gtk_image_new_from_pixmap (pixmap,
                                                         teatime->popup.bitmap[i]);
  }
  return TRUE;
}

static gboolean
load_pref_images (Teatime *teatime)
{
  GError *error = NULL;
  gint i;
  gchar *filename;
  gchar *path;

  for (i=0; i<TEACUPS; i++)
  {
    gchar *filename;
    GdkPixbuf *pixbuf;
    GdkPixmap *pixmap;
      
    if (i<10)
      filename = g_strdup_printf ("teatime/%s_full_0.png", teacups[i]);
    else
      filename = g_strdup_printf ("teatime/%s_full_0.png", teatime[i]);

    path = gnome_program_locate_file (NULL,
                                      GNOME_FILE_DOMAIN_PIXMAP,
                                      filename,
                                      FALSE,
                                      NULL);
    teatime->properties.teacups[i] = gdk_pixbuf_new_from_file (path, &error);
    g_free (filename);
    if (error)
    {
      g_warning (G_STRLOC ": cannot open %s: %s", path, error);
      g_free (path);
      g_error_free (error);
      return FALSE;
    }
  g_free (path);
  }
  return TRUE;
}

static void
load_properties (Teatime *teatime)
{
  gint i;
  PanelApplet *applet = teatime->applet;
  
  teatime->properties.sound = NULL;
  for (i=0; i<6; i++)
  {
    teatime->properties.drawing_time[i] = panel_applet_gconf_get_int (
                                          applet, teas_pref[i], NULL);
    if (teatime->properties.drawing_time[i] == 0)
        teatime->properties.drawing_time[i] = default_drawing_times[i];
  }
  teatime->properties.last_tea = panel_applet_gconf_get_int (
                                 applet, TEATIME_PREFS_LAST_TEA, NULL);
  teatime->properties.cup = panel_applet_gconf_get_int (
                            applet, TEATIME_PREFS_CUP, NULL);
  teatime->properties.popup = panel_applet_gconf_get_bool (
                              applet, TEATIME_PREFS_POPUP, NULL);
  teatime->properties.play_sound = panel_applet_gconf_get_bool (
                                   applet, TEATIME_PREFS_PLAY_SOUND, NULL);

  /* this is realy ugly, but without the (gchar*) i got warnings and don't know why... */
  teatime->properties.sound = (gchar*) panel_applet_gconf_get_string (
                              applet, TEATIME_PREFS_SOUND, NULL );
  teatime->properties.show_clock = panel_applet_gconf_get_bool (
                                   applet, TEATIME_PREFS_SHOW_CLOCK, NULL); 
}

static void
disable_sound_entry (GtkToggleButton *button, GtkWidget *entry)
{
  gtk_widget_set_sensitive (entry, button->active);
}


static void
properties_cup_changed (GtkWidget *toggle_button,
                        Teatime *teatime)
{
  gint i;

  for (i=0; i<TEACUPS; i++)
  {
    if (GTK_TOGGLE_BUTTON (teatime->pref_cup_button[i])->active)
    {
      panel_applet_gconf_set_int (PANEL_APPLET (teatime->applet),
                                  TEATIME_PREFS_CUP,
                                  i,
                                  NULL);
      if (teatime->properties.cup != i)
      {
        teatime->properties.cup = i;
        load_applet_images (teatime);
      }
      break;
    }
  }
}

static void
properties_sound_changed (GtkWidget *widget,
                          Teatime *teatime)
{
  GtkWidget *sound_entry;
  GtkWidget *entry;

  sound_entry = gtk_object_get_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_SOUND_ENTRY);
  entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (sound_entry));  

  if (teatime->properties.sound)
    g_free (teatime->properties.sound);
  teatime->properties.sound = gtk_editable_get_chars (
                              GTK_EDITABLE (entry), 0, -1);
  panel_applet_gconf_set_string (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_SOUND,
                                 teatime->properties.sound,
                                 NULL);
}

static void
properties_alarm_changed (GtkWidget *widget,
                          Teatime *teatime)
{
  GtkWidget *popup;
  GtkWidget *play_sound;
  gchar *sound;

  popup = gtk_object_get_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_POPUP);
  play_sound = gtk_object_get_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_PLAY_SOUND);

  if (teatime->properties.popup != GTK_TOGGLE_BUTTON (popup)->active)
  {
    panel_applet_gconf_set_bool (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_POPUP,
                                 GTK_TOGGLE_BUTTON (popup)->active,
                                 NULL);
    teatime->properties.popup = GTK_TOGGLE_BUTTON (popup)->active;
  }

  if (teatime->properties.play_sound != GTK_TOGGLE_BUTTON (play_sound)->active)
  {  
    panel_applet_gconf_set_bool (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_PLAY_SOUND,
                                 GTK_TOGGLE_BUTTON (play_sound)->active,
                                 NULL);
    teatime->properties.play_sound = GTK_TOGGLE_BUTTON (play_sound)->active;
  }
}

static void
properties_clock_changed (GtkWidget *widget,
			  Teatime *teatime)
{
  GtkWidget *clock;

  clock = gtk_object_get_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_SHOW_CLOCK);

  if (teatime->properties.show_clock != GTK_TOGGLE_BUTTON (clock)->active)
  {
    panel_applet_gconf_set_bool (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_SHOW_CLOCK,
                                 GTK_TOGGLE_BUTTON (clock)->active,
                                 NULL);
    teatime->properties.show_clock = GTK_TOGGLE_BUTTON (clock)->active;
  }
}

static void
properties_drawingtime_changed (GtkWidget *widget,
				Teatime *teatime)
{
  gint i, val;
  
  for (i=0; i<6; i++)
  {
    val = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (teatime->pref_time_spin[i]));
    if (val != teatime->properties.drawing_time[i])
    {  
      panel_applet_gconf_set_int (PANEL_APPLET (teatime->applet),
                                  teas_pref[i],
                                  val,
                                  NULL);
      teatime->properties.drawing_time[i] = val;
      set_menu (teatime);	
    }
  }
}

static void
reset_drawing_times (GtkWidget *widget, Teatime *teatime)
{
  GtkWidget *dialog;
  gint i;

  dialog = gtk_message_dialog_new (GTK_WINDOW (teatime->pb),
                                   GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_QUESTION,
                                   GTK_BUTTONS_YES_NO,
                                   _("Do you realy want to reset your tea's\n"
                                     "drawing times to the default values?"),
                                   NULL);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_YES)
  {
    gtk_widget_destroy (dialog);
    return;
  }
  
  for (i=0; i<6; i++)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON(teatime->pref_time_spin[i]),
                               (gdouble)default_drawing_times[i]);
  properties_drawingtime_changed (NULL, teatime);
  gtk_widget_destroy (dialog);
}

static void
display_properties_dialog(BonoboUIComponent *uic,
                          Teatime *teatime,
                          const gchar *verbname)
{

  PanelApplet *applet;

  GtkWidget *tab_label[3];
  gint i;
  gint drawing_times[6];
  GtkWidget *time_table;
  GtkWidget *tea_label[6];
  GtkWidget *seconds_label;
  GtkWidget *time_reset;
  
  GtkWidget *cup_vbox;
  GtkWidget *cup_hbox;
  GtkWidget *cup_label;
  GSList *group = NULL;
  
  GtkWidget *alarm_box;
  GtkWidget *alarm_popup_button;
  GtkWidget *alarm_play_sound_button;
  GtkWidget *alarm_sound_entry;
  GValue alarm_val = {0,};
  
  GtkWidget *entry;
  GtkWidget *show_clock_button;


  
  if (teatime->pb)
  {
    gtk_widget_show (teatime->pb);
    gdk_window_raise (teatime->pb->window);
    return;
  }

  applet = PANEL_APPLET (teatime->applet);
    
  teatime->pb = gnome_property_box_new ();
  gnome_window_icon_set_from_file (GTK_WINDOW (teatime->pb),
                                   teatime->icon_path);
    
  gtk_window_set_title (GTK_WINDOW (teatime->pb), _("Teatime Properties"));
  tab_label[0] = gtk_label_new (_("drawing time"));
  tab_label[1] = gtk_label_new (_("tea cup"));
  tab_label[2] = gtk_label_new (_("alarm"));


  /* drawing times page */
  
  time_table = gtk_table_new (7, 3, FALSE);
  for (i=0; i<6; i++)
  {
    gchar *label = g_strdup_printf (_("%s drawing time"), _(teas[i]));
    tea_label[i] = gtk_label_new (label);
    gtk_misc_set_alignment (GTK_MISC (tea_label[i]), 1.0, 0.5);
    g_free (label);
    gtk_table_attach (GTK_TABLE (time_table), tea_label[i], 0, 1, i, i+1,
                      GTK_EXPAND | GTK_FILL,
                      GTK_EXPAND | GTK_FILL,
                      4, 0);
      
    teatime->pref_time_adj[i] = (GtkAdjustment *)
                                gtk_adjustment_new (10.0, 10.0, 990.0, 10.0, 10.0, 10.0);
    teatime->pref_time_spin[i] = gtk_spin_button_new (teatime->pref_time_adj[i], 10, 0);
    gtk_spin_button_set_digits (GTK_SPIN_BUTTON (teatime->pref_time_spin[i]), 0);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (teatime->pref_time_spin[i]), TRUE);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (teatime->pref_time_spin[i]),
                               (gdouble)teatime->properties.drawing_time[i]);
    gtk_table_attach (GTK_TABLE (time_table), teatime->pref_time_spin[i],
                      1, 2, i, i+1,
                      GTK_SHRINK | GTK_FILL,
                      GTK_SHRINK | GTK_FILL, 0, 0);

    seconds_label = gtk_label_new (_("seconds"));
    gtk_misc_set_alignment (GTK_MISC (seconds_label), 0.0, 0.5);
    gtk_table_attach (GTK_TABLE (time_table), seconds_label, 2, 3, i, i+1,
                      GTK_EXPAND | GTK_FILL,
                      GTK_EXPAND | GTK_FILL,
                      4, 0);
    g_signal_connect (G_OBJECT (teatime->pref_time_spin[i]), "changed",
                      G_CALLBACK (properties_drawingtime_changed), teatime);
  }
  time_reset = gtk_button_new_with_label (_("reset to defaults"));
  g_signal_connect (G_OBJECT (time_reset), "clicked",
                    G_CALLBACK (reset_drawing_times), teatime);
  gtk_table_attach (GTK_TABLE (time_table), time_reset, 0, 3, 6, 7,
                    GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 6, 6);
    
  gnome_property_box_append_page ( GNOME_PROPERTY_BOX (teatime->pb),
                                  time_table,
                                  tab_label[0]);

  /* cup selection page */

  cup_vbox = gtk_vbox_new (FALSE, 6);
  cup_hbox = gtk_hbox_new (TRUE, 2);
  cup_label = gtk_label_new (_("Chose your teacup"));
  
  gtk_box_pack_start (GTK_BOX (cup_vbox), cup_label, FALSE, FALSE, 2);
  gtk_box_pack_start (GTK_BOX (cup_vbox), cup_hbox, FALSE, FALSE, 2);
  for (i=0; i<TEACUPS; i++)
  {
    gchar *string;
      
    teatime->pref_cup[i] = gtk_image_new_from_pixbuf (teatime->properties.teacups[i]);
    teatime->pref_cup_button[i] = gtk_radio_button_new(group);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (teatime->pref_cup_button[i]));
    gtk_container_add (GTK_CONTAINER (teatime->pref_cup_button[i]),
                       teatime->pref_cup[i]);
    gtk_container_set_border_width (GTK_CONTAINER ( teatime->pref_cup_button[i]), 0);
    gtk_box_pack_start (GTK_BOX (cup_hbox), teatime->pref_cup_button[i],
                        FALSE, FALSE, 2);
    g_signal_connect (G_OBJECT (teatime->pref_cup_button[i]), "toggled",
                      G_CALLBACK (properties_cup_changed), teatime);
  }
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
                                teatime->pref_cup_button[teatime->properties.cup]), TRUE);
  gnome_property_box_append_page (GNOME_PROPERTY_BOX (teatime->pb),
                                  cup_vbox,
                                  tab_label[1]);

  /* actions page */
    
  alarm_box = gtk_vbox_new (FALSE, 0);
  alarm_popup_button = gtk_check_button_new_with_label (_("Open window if tea is ready"));
  alarm_play_sound_button = gtk_check_button_new_with_label (_("Play sound if tea is ready"));

  alarm_sound_entry = gnome_file_entry_new (NULL, NULL);
  g_value_init (&alarm_val, G_TYPE_BOOLEAN);
  g_value_set_boolean (&alarm_val, TRUE);
  g_object_set_property (G_OBJECT (alarm_sound_entry), "use_filechooser", &alarm_val);

  entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (alarm_sound_entry));

  show_clock_button = gtk_check_button_new_with_label (_("Show clock while drawing"));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (alarm_popup_button),
                                teatime->properties.popup);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (alarm_play_sound_button),
                                teatime->properties.play_sound);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_clock_button),
                                teatime->properties.show_clock); 
  if (!teatime->properties.play_sound)
    disable_sound_entry (GTK_TOGGLE_BUTTON (alarm_play_sound_button), alarm_sound_entry);
  if (teatime->properties.sound)
    gtk_entry_set_text (GTK_ENTRY (entry),teatime->properties.sound);
  gtk_object_set_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_POPUP, alarm_popup_button);
  gtk_object_set_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_PLAY_SOUND, alarm_play_sound_button);
  gtk_object_set_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_SOUND_ENTRY, alarm_sound_entry);
  gtk_object_set_data (GTK_OBJECT (teatime->pb), TEATIME_PREFS_SHOW_CLOCK, show_clock_button);
  
  gtk_box_pack_start (GTK_BOX (alarm_box), alarm_popup_button, FALSE, FALSE, 6);
  gtk_box_pack_start (GTK_BOX (alarm_box), alarm_play_sound_button, FALSE, FALSE, 6);
  gtk_box_pack_start (GTK_BOX (alarm_box), alarm_sound_entry, FALSE, FALSE, 2);
  gtk_box_pack_start (GTK_BOX (alarm_box), show_clock_button, FALSE, FALSE, 12);
    
  g_signal_connect (G_OBJECT (alarm_play_sound_button), "toggled",
                    G_CALLBACK (disable_sound_entry), alarm_sound_entry);

  g_signal_connect (G_OBJECT (alarm_popup_button), "toggled",
                    G_CALLBACK (properties_alarm_changed), teatime);
  g_signal_connect (G_OBJECT (alarm_play_sound_button), "toggled",
                    G_CALLBACK (properties_alarm_changed), teatime);
  g_signal_connect (G_OBJECT (entry), "changed",
                    G_CALLBACK (properties_sound_changed), teatime);
  g_signal_connect (G_OBJECT (show_clock_button), "toggled",
                    G_CALLBACK (properties_clock_changed), teatime);
  
  gnome_property_box_append_page (GNOME_PROPERTY_BOX (teatime->pb),
                                  alarm_box,
                                  tab_label[2]);

  g_signal_connect (G_OBJECT (teatime->pb), "destroy",
                    G_CALLBACK (gtk_widget_destroyed), &teatime->pb);
		  
  gtk_widget_show_all (GNOME_PROPERTY_BOX (teatime->pb)->notebook); 
  gtk_widget_hide (GNOME_PROPERTY_BOX (teatime->pb)->ok_button); 
  gtk_widget_hide (GNOME_PROPERTY_BOX (teatime->pb)->apply_button);
  gtk_widget_hide (GNOME_PROPERTY_BOX (teatime->pb)->help_button);
  gtk_widget_set_usize (GTK_WIDGET (teatime->pb), 600, -1);
  gtk_widget_show (teatime->pb);
}

/* NO HELP FOR NOW
static void
display_help_dialog()
{
  g_message("help");
}
*/

static void
display_about_dialog(BonoboUIComponent *uic,
                     Teatime *teatime,
                     const gchar *verbname){

  GdkPixbuf *logo;
  GError *error = NULL;
  static GtkWidget *about = NULL;
  const gchar *authors [] = {N_("Detlef Reichl\n\tdetlef ! reichl () gmx ! org"), NULL};

  if (about != NULL)
  {
    gdk_window_show(about->window);
    gdk_window_raise(about->window);
    return;
  }
  logo = gdk_pixbuf_new_from_file (teatime->icon_path, &error);
  about = gnome_about_new (_("Teatime"), VERSION,
                           _("Copyright (C) 1998-2004 Detlef Reichl"),
                           _("A\ntea-drawing-time-observer.\n"
                             "The most usefull GNOME program ;-)"),
                           authors,
                           NULL,
                           NULL,
                           logo);
  g_object_unref (logo);
  gnome_window_icon_set_from_file (GTK_WINDOW (about),
                                   teatime->icon_path);
  g_signal_connect (G_OBJECT(about), "destroy",
                    G_CALLBACK(gtk_widget_destroyed), &about);
  gtk_widget_show (about);
}

static void
restart (Teatime *teatime)
{
  if (teatime->full)
  {
    GtkWidget *dialog;
      
    dialog = gtk_message_dialog_new (NULL,
                                     0,
                                     GTK_MESSAGE_QUESTION,
                                     GTK_BUTTONS_YES_NO,
                                     _("Restart drawing?"),
                                     NULL);
    if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_YES)
    {
      gtk_widget_destroy (dialog);
      return;
    }
    gtk_widget_destroy (dialog);
  }
  if (teatime->timer_handle)
    gtk_timeout_remove (teatime->timer_handle);
  if (teatime->popup.visible)
    popup_hide (NULL, NULL, &teatime->popup);
  teatime->full = TRUE;
  teatime->time_left = teatime->drawing_time * FRAMES;
  teatime->timer_handle = gtk_timeout_add (333, (GtkFunction)timer, teatime);
  redraw (teatime);
}

static gboolean
clicked_event (GtkWidget *widget, GdkEventButton *event, Teatime *teatime)
{
  if (event->button != 1 || teatime->popup.visible)
    return FALSE;
  
  teatime->drawing_time = teatime->properties.drawing_time[teatime->last_tea];
  restart (teatime);
  gtk_tooltips_force_window (teatime->tooltip);
  return TRUE;
}

static void
change_orient (GtkWidget *widget, PanelAppletOrient orient, gpointer data)
{
  Teatime *teatime = (Teatime*)data;

  teatime->orient_vert = (orient == PANEL_APPLET_ORIENT_UP
                          || orient == PANEL_APPLET_ORIENT_DOWN) ? FALSE : TRUE;
}

void
size_allocate (PanelApplet *applet,
               GtkAllocation *allocation,
               Teatime *teatime)
{
  if (teatime->orient_vert)
  {
    if (teatime->size == allocation->width)
      return;
    teatime->size = allocation->width;
  }
  else
  {
    if (teatime->size == allocation->height)
      return;
    teatime->size = allocation->height;
  }
 
  /* in the begining the applets are built with a size of 1 */
  if (teatime->size <= 1)
    return;
  gtk_widget_set_size_request (teatime->area, teatime->size, teatime->size);
  load_applet_images (teatime); 
}

static void
real_start_drawing (Teatime *teatime, gint tea)
{
  if (teatime->popup.visible)
    return;

  teatime->drawing_time = teatime->properties.drawing_time[tea];
  teatime->last_tea = tea;
  restart (teatime);
}

static void
start_drawing (BonoboUIComponent *uic,
              Teatime *teatime,
              const gchar *verbname)
{
  gshort tea = 0;
  switch (verbname[0])
  {
    case 'A':
      tea = 0;
    break;
    case 'D':
      tea = 1;
    break;
    case 'F':
      tea = 2;
    break;
    case 'G':
      tea = 3;
    break;
    case 'H':
      tea = 4;
    break;
    case 'M':
      tea = 5;
    break;
  }
  real_start_drawing (teatime, tea);
}

static void
destroy (GtkWidget *applet, Teatime *teatime)
{
  gint i;
  
  if (teatime->timer_handle)
    gtk_timeout_remove(teatime->timer_handle);

  if (teatime->popup.timer_handle)
      gtk_timeout_remove(teatime->popup.timer_handle);
  
  if (teatime->popup.label_window)
    gtk_widget_destroy (teatime->popup.label_window);
  
  if (teatime->popup.cup_window)
    gtk_widget_destroy (teatime->popup.cup_window);
  
  for (i=0; i<ROT_FRAMES; i++)
    if (teatime->popup.bitmap[i])
      gdk_pixmap_unref (teatime->popup.bitmap[i]);
  
  for (i=0; i<TEACUPS; i++)
    if (teatime->properties.teacups[i])
      g_object_unref (G_OBJECT (teatime->properties.teacups[i]));
  
  if (teatime->cup_empty)
    g_object_unref (G_OBJECT (teatime->cup_empty));
  
  if (teatime->cup_empty_hl)
    g_object_unref (G_OBJECT (teatime->cup_empty_hl));
  
  for (i=0; i<FRAMES; i++)
  {
    if (teatime->cup_full[i])
      g_object_unref (G_OBJECT (teatime->cup_full[i]));
    if (teatime->cup_full_hl[i])
      g_object_unref (G_OBJECT (teatime->cup_full_hl[i]));
  }
  
  g_free (teatime);
}

static const BonoboUIVerb teatime_menu_verbs [] = {
  BONOBO_UI_UNSAFE_VERB ("Prop",      display_properties_dialog),
//  BONOBO_UI_VERB ("Help",	display_help_dialog),
  BONOBO_UI_UNSAFE_VERB ("About",     display_about_dialog),
  BONOBO_UI_UNSAFE_VERB ("Assam",     start_drawing),
  BONOBO_UI_UNSAFE_VERB ("Darjeling", start_drawing), 
  BONOBO_UI_UNSAFE_VERB ("Fruit",     start_drawing),
  BONOBO_UI_UNSAFE_VERB ("Green",     start_drawing),
  BONOBO_UI_UNSAFE_VERB ("Herbes",    start_drawing),
  BONOBO_UI_UNSAFE_VERB ("Mint",      start_drawing),
  BONOBO_UI_VERB_END
};

static void
set_menu (Teatime *teatime)
{
  static gchar *string = NULL;

  if (string)
    g_free (string);
		 
  string = g_strdup_printf (
  "<popup name=\"button3\">\n"
  "  <menuitem name=\"Item 1\" verb=\"Prop\" _label=\"%s\"\n"
  "     pixtype=\"stock\" pixname=\"gtk-properties\"/>\n"
//  "    <menuitem name=\"Item 2\" verb=\"Help\" _label=\"%s\"\n"
//  "              pixtype=\"stock\" pixname=\"gtk-help\"/>\n"
  "  <menuitem name=\"Item 3\" verb=\"About\" _label=\"%s\"\n"
  "    pixtype=\"stock\" pixname=\"gnome-stock-about\"/>\n"
  "  <separator/>\n"
  "  <menuitem name=\"Item 4\" verb=\"Assam\" _label=\"%s %is\"/>\n"
  "  <menuitem name=\"Item 5\" verb=\"Darjeling\" _label=\"%s %is\"/>\n"
  "  <menuitem name=\"Item 6\" verb=\"Fruit\" _label=\"%s %is\"/>\n"
  "  <menuitem name=\"Item 7\" verb=\"Green\" _label=\"%s %is\"/>\n"
  "  <menuitem name=\"Item 8\" verb=\"Herbes\" _label=\"%s %is\"/>\n"
  "  <menuitem name=\"Item 9\" verb=\"Mint\" _label=\"%s %is\"/>\n"
  "</popup>\n",
    
    _("Properties ..."),
//    _("Help"),
    _("About"),
  
    _(teas[0]), teatime->properties.drawing_time[0],
    _(teas[1]), teatime->properties.drawing_time[1],
    _(teas[2]), teatime->properties.drawing_time[2],
    _(teas[3]), teatime->properties.drawing_time[3],
    _(teas[4]), teatime->properties.drawing_time[4],
    _(teas[5]), teatime->properties.drawing_time[5]);

  panel_applet_setup_menu (PANEL_APPLET (teatime->applet),
                           (const gchar *)string,
                           teatime_menu_verbs,
                           teatime);
}


static gboolean
teatime_applet_new (PanelApplet *applet)
{
  Teatime *teatime = g_new0 (Teatime, 1);
  gint i;
  gchar *string;
  PanelAppletOrient orient;
  
  teatime->applet = applet;
  teatime->full = FALSE;
  teatime->highlight = FALSE;
  teatime->frame = 0;
  teatime->popup.cup_window = NULL;
  teatime->clock_gc = NULL;
  
  panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);

  panel_applet_add_preferences (applet, "/schemas/apps/TeatimeApplet/prefs", NULL);
  
  gdk_rgb_init();

  gst_init(NULL, NULL);
  
  load_properties (teatime);
  load_popup_images (teatime);
  load_pref_images (teatime);

  teatime->border = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (teatime->border), GTK_SHADOW_IN);
  teatime->area = gtk_drawing_area_new ();

  gtk_container_add (GTK_CONTAINER (teatime->border), teatime->area);
  gtk_container_add (GTK_CONTAINER (teatime->applet), teatime->border);
  
  teatime->tooltip = gtk_tooltips_new ();
  gtk_tooltips_set_delay (teatime->tooltip, 50);

  g_signal_connect_data (G_OBJECT (teatime->area), "expose_event",
                         G_CALLBACK (redraw), teatime,
                         NULL, G_CONNECT_SWAPPED);
   g_signal_connect (G_OBJECT (teatime->area), "size_allocate",
                     G_CALLBACK (size_allocate), teatime);
 
  g_signal_connect (G_OBJECT (applet), "destroy",
                    G_CALLBACK (destroy), teatime);
  g_signal_connect (G_OBJECT (applet), "change_orient",
                    G_CALLBACK (change_orient), teatime);

  gtk_widget_add_events (teatime->area, GDK_BUTTON_PRESS_MASK
                         | GDK_ENTER_NOTIFY_MASK
                         | GDK_LEAVE_NOTIFY_MASK);
  g_signal_connect (G_OBJECT (teatime->area), "enter_notify_event",
                    G_CALLBACK (enter_event), teatime);
  g_signal_connect (G_OBJECT (teatime->area), "leave_notify_event",
                    G_CALLBACK (leave_event), teatime);
  g_signal_connect_data (G_OBJECT (teatime->area), "button_press_event",
                         G_CALLBACK (clicked_event), teatime,
                         NULL, G_CONNECT_AFTER);
  
  orient = panel_applet_get_orient (applet);
  teatime->orient_vert = (orient == PANEL_APPLET_ORIENT_UP
                          || orient == PANEL_APPLET_ORIENT_DOWN) ? FALSE : TRUE;

  gtk_widget_show_all(GTK_WIDGET(applet));
  set_menu (teatime);
  return TRUE;
}

static gboolean
teatime_factory (PanelApplet *applet,
		 const gchar *iid,
		 gpointer data)
{
  gboolean retval = FALSE;
  if (strcmp (iid, "OAFIID:GNOME_TeatimeApplet") == 0)
    retval = teatime_applet_new (applet);
  return retval;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_TeatimeApplet_Factory",
                             PANEL_TYPE_APPLET,
                             "teatime",
                             "0",
                             teatime_factory,
                             NULL)

