/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.mergedoc.pleiades.Pleiades;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.Files;
import jp.sourceforge.mergedoc.pleiades.resource.PropertySet;

import org.apache.commons.io.FileUtils;

/**
 * metadata を設定するためのクラスです。
 * <p>
 * @author cypher256
 */
public class Metadata {

	/** ロガー */
	private static Logger log = Logger.getLogger(Metadata.class);

	/** 除外リスト */
	@SuppressWarnings("serial")
	private static final List<String> EXCLUDES = Collections.unmodifiableList(new LinkedList<String>() {

		{
			// Ant デフォルト除外
			add(".*~");
			add("#.*#");
			add(".#.*");
			add("%.*%");
			add("\\._.*");
			add("CVS");
			add("\\.cvsignore");
			add("SCCS");
			add("vssver\\.scc");
			add("\\.svn");
			add("\\.DS_Store");

			// その他 [#13834]
			add("\\.bzr");			// Bazaar
			add("\\.cdv");			// Codeville
			add("\\.git");			// Git
			add("\\.hg");			// Mercurial
			add("\\.pc");			// quilt
			add("RCS");				// RCS
			add("_darcs");			// darcs
			add("_sgbak");			// Vault/Fortress

			// その他 [#13834] (miau)
			add("vssver2\\.scc");	// VSS2005
			add("_svn");			// SVN オプション
		}

		@Override
		public boolean contains(Object name) {
			for (String pattern : this) {
				if (((String) name).matches(pattern)) {
					return true;
				}
			}
			return false;
		}
	});

	/** Eclipse ホーム親ディレクトリー */
	private String eclipseHomeParent;

	/** metadata コピー元ディレクトリー */
	private File metadataSrcDir;

	/** metadata コピー先ディレクトリー */
	private File metadataDstDir;

	/** コピー数 */
	private int copyCount;

	/**
	 * コンストラクタ。
	 */
	public Metadata() {

		File eclipseHomeParentDir = Pleiades.getEclipseHome().getParentFile();
		if (eclipseHomeParentDir == null) {
			return;
		}
		eclipseHomeParent = eclipseHomeParentDir.getPath().replace("\\", "/");
		metadataSrcDir = new File(eclipseHomeParent, ".metadata.default/.plugins");
		if (!metadataSrcDir.exists()) {
			return;
		}
	}

	/**
	 * 新規ワークスペースのデフォルト自動設定エンコーディングを取得します。
	 * @return 新規ワークスペースでないか、デフォルト自動設定がない場合は null
	 */
	public String getNewWorkspaceEncoding() {

		if (copyCount == 0) {
			return null;
		}
		// Eclipse エンコーディング設定ファイル
		File encodingPrefs = new File(metadataSrcDir,
			"org.eclipse.core.runtime/.settings/org.eclipse.core.resources.prefs");

		if (!encodingPrefs.exists()) {
			return null;
		}
		PropertySet p = new PropertySet(encodingPrefs);
		return p.get("encoding");
	}

	/**
	 * 新規ワークスペースの metadata を作成します。
	 * @param workspace ワークスペース・ロケーション
	 * @throws URISyntaxException
	 * @throws IOException
	 */
	public void createNewWorkspaceMetadata(File workspace) throws URISyntaxException, IOException {

		if (eclipseHomeParent == null) {
			return;
		}
		if (!metadataSrcDir.exists()) {
			return;
		}
		metadataDstDir = new File(workspace, ".metadata/.plugins");

		// metadata がコピー済みか判断するファイル
		File copiedFile = new File(metadataDstDir.getParentFile(), ".copied.metadata.default");
		if (copiedFile.exists()) {
			return;
		}

		copyMetadata(metadataSrcDir);

		FileUtils.writeStringToFile(copiedFile, "");
		log.info(".metadata.default から " + copyCount +
			" 個のファイルをワークスペースにコピーしました。" + metadataDstDir.getParentFile());
	}

	/**
	 * metadata を再帰的にコピーします。存在する場合は上書きしません。
	 * @param srcDir コピー元ディレクトリー
	 * @throws IOException
	 */
	private void copyMetadata(File srcDir) throws IOException {

		final String HOME_STRING = "%ECLIPSE_HOME_PARENT%";

		for (File srcFile : srcDir.listFiles()) {

			if (EXCLUDES.contains(srcFile.getName())) {
				continue;
			}
			if (srcFile.isDirectory()) {
				copyMetadata(srcFile); // 再帰
			} else {
				File dstFile = new File(metadataDstDir, Files.relativePath(metadataSrcDir, srcFile));
				if (dstFile.exists()) {
					continue;
				}
				String s = FileUtils.readFileToString(srcFile, "UTF-8");
				if (s.contains(HOME_STRING)) {
					s = s.replace(HOME_STRING, eclipseHomeParent);
					FileUtils.writeStringToFile(dstFile, s, "UTF-8");
				} else {
					FileUtils.copyFile(srcFile, dstFile);
				}
				copyCount++;
			}
		}
	}
}
