#!/usr/bin/python2.1
# Copyright (C) 2000-2001 The OpenRPG Project
#
#        openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: mplay_server.py
# Author: Chris Davis
# Maintainer:
# Version:
#   $Id: mplay_server.py,v 1.78 2003/12/09 00:59:45 snowdog_ Exp $
#
# Description: This file contains the code for the server of the multiplayer
# features in the orpg project.
#

__version__ = "$Id: mplay_server.py,v 1.78 2003/12/09 00:59:45 snowdog_ Exp $"

#!/usr/bin/env python
"""
<msg to='' from='' group_id='' />
<player id='' ip='' group_id='' name='' action='new,del,group,update' status="" version=""/>
<group id='' name='' pwd='' players='' action='new,del,update' />
<create_group from='' pwd='' name='' />
<join_group from='' pwd='' group_id='' />
<role action='set,get,display' player='' group_id='' boot_pwd='' role=''/>
"""

from mplay_client import *
from mplay_client import MPLAY_LENSIZE
import orpg.dirpath
import orpg.tools.config_files
import gc
import cgi
import sys
import string
import time
from orpg.mapper.map_msg import *
from threading import Lock, RLock
from struct import pack, unpack, calcsize
from meta_server_lib import *
import traceback

# Import the minidom XML module
from xml.dom import minidom

# Snag the version number
from orpg.orpg_version import *

# To track the server's name
myname = ''

OPENRPG_PORT = 6774

class game_group:
    def __init__( self, id, name, pwd, desc="", boot_pwd="", minVersion="", mapFile=None, messageFile=None ):
        self.id = id
        self.name = name
        self.desc = desc
        self.minVersion = minVersion
        self.messageFile = messageFile
        self.players = []
        self.pwd = pwd
        self.boot_pwd = boot_pwd
        self.game_map = map_msg()
        self.lock = Lock()
        self.moderated = 0
        self.voice = {}

        if mapFile != None:
            f = open( mapFile )
            tree = f.read()
            f.close()

        else:
            f = open(orpg.dirpath.dir_struct["template"]+"default_map.xml")
            tree = f.read()
            f.close()

        self.game_map.init_from_xml(tree)


    def add_player(self,id):
        self.lock.acquire()
        self.players.append(id)
        self.lock.release()

    def remove_player(self,id):
        self.lock.acquire()
        if self.voice.has_key(id):
            del self.voice[id]
        self.players.remove(id)
        self.lock.release()

    def get_num_players(self):
        self.lock.acquire()
        num =  len(self.players)
        self.lock.release()
        return num

    def get_player_ids(self):
        self.lock.acquire()
        tmp = self.players
        self.lock.release()
        return tmp


    def check_pwd(self,pwd):
        return (pwd==self.pwd)

    def check_boot_pwd(self,pwd):
        return (pwd==self.boot_pwd)

    def check_version(self,ver):
        if (self.minVersion == ""):
            return 1
        minVersion=self.minVersion.split('.')
        version=ver.split('.')
        for i in range(min(len(minVersion),len(version))):
            w=max(len(minVersion[i]),len(version[i]))
            v1=minVersion[i].rjust(w);
            v2=version[i].rjust(w);
            if v1<v2:
                return 1
            if v1>v2:
                return 0

        if len(minVersion)>len(version):
            return 0
        return 1


    def toxml(self,act="new"):
        #  Please don't add the boot_pwd to the xml, as this will give it away to players watching their console
        xml_data = "<group id=\""+self.id
        xml_data += "\" name=\""+self.name
        xml_data += "\" pwd=\""+str(self.pwd!="")
        xml_data += "\" players=\""+str(self.get_num_players())
        xml_data += "\" action=\""+act+"\" />"
        return xml_data



class client_stub(client_base):
    def __init__(self,inbox,sock,props,log):
        client_base.__init__(self)
        self.ip = props['ip']
        self.role = props['role']
        self.id = props['id']
        self.group_id = props['group_id']
        self.name = props['name']
        self.version = props['version']
        self.protocol_version = props['protocol_version']
        self.client_string = props['client_string']
        self.inbox = inbox
        self.sock = sock
        self.timeout_time = None
        self.log_console = log
        self.ignorelist = {}



    # implement from our base class
    def isServer( self ):
        return 1


    def clear_timeout(self):
        self.timeout_time = None



    def check_time_out(self):
        if self.timeout_time==None:
            self.timeout_time = time.time()
        curtime = time.time()
        diff = curtime - self.timeout_time
#        print "Diff =",diff
        if diff > 60:
#            print "client timed out"
            return 1
        else:
#            print "not timed out"
            return 0



    def send(self,msg,player,group):
        if self.get_status() == MPLAY_CONNECTED:
            ##print "Sending to player " + player + " in group " + group + ":"
            ##print msg
            self.outbox.put("<msg to='"+player+"' from='0' group_id='"+group+"' />"+msg)




    def change_group(self,group_id,groups):
        old_group_id = self.group_id
        groups[group_id].add_player(self.id)
        groups[old_group_id].remove_player(self.id)
        self.group_id = group_id
        self.outbox.put(self.toxml('group'))
        msg = groups[group_id].game_map.get_all_xml()
        self.send(msg,self.id,group_id)

        return old_group_id



    def self_message(self,act):
        self.send(act,self.id,self.group_id)
#        self.outbox.put(self.toxml(act))



    def take_dom(self,xml_dom):
        self.name = xml_dom.getAttribute("name")
        self.text_status = xml_dom.getAttribute("status")
#        try:
#            self.role = xml_dom.getAttribute("role")
#        except:  pass






######################################################################
######################################################################
##
##
##   MPLAY SERVER
##
##
######################################################################
######################################################################

class mplay_server:
    def __init__(self, log_console=None, name='' ):
        global myname
        self.log_to_console = 1
        self.log_console = log_console
        self.alive = 1
        self.players = {}
        self.listen_event = Event()
        self.incoming_event = Event()
        self.incoming = Queue.Queue(0)
        self.p_lock = RLock()
        self.next_player_id = 1
        self.next_group_id = 1
        self.metas = {}              #  This holds the registerThread objects for each meta
        self.be_registered = 0       #  Status flag for whether we want to be registered.
        self.name = None            #  Name of this server in the metas
        self.server_address = None # IP or Name of server to post to the meta. None means the meta will auto-detect it.
        self.defaultMessageFile = None
        orpg.tools.config_files.validate_config_file( "Lobby_map.xml", "default_Lobby_map.xml" )
        orpg.tools.config_files.validate_config_file( "LobbyMessage.html", "default_LobbyMessage.html" )
        self.server_start_time = time.time()
        self.show_meta_messages = 0
        self.log_network_messages = 0
        self.allow_room_passwords = 1

        # Since the server is just starting here, we read in the XML configuration
        # file.  Notice the lobby is still created here by default.
        self.groups = { '0': game_group( '0','Lobby','','The game lobby', '', '', orpg.dirpath.dir_struct["user"]+'Lobby_map.xml',
                                         orpg.dirpath.dir_struct["user"]+'LobbyMessage.html' ) }
        self.initServerConfig()

        # Make sure the server's name gets set, in case we are being started from
        # elsewhere.  Basically, if it's passed in, we'll over ride what we were
        # prompted for.  This should never really happen at any rate.
        if len(name):
            myname = name

        self.listen_sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.listen_sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.listen_thread = thread.start_new_thread(self.listenAcceptThread, (0,))
        self.in_thread = thread.start_new_thread(self.incoming_handler,(0,))

        #  Starts the player reaper thread.  See self.player_reaper_thread_func() for more explanation
        self.player_reaper_thread = thread.start_new_thread(self.player_reaper_thread_func,(0,))




    # This method reads in the server's configuration file and reconfigs the server
    # as needed, over-riding any default values as requested.
    def initServerConfig( self ):

        print ("Processing Server Configuration File...")

        # make sure the server_ini.xml exists!
        orpg.tools.config_files.validate_config_file( "server_ini.xml", "default_server_ini.xml" )

        # try to use it.
        try:
            self.configDom = minidom.parse( orpg.dirpath.dir_struct["user"]+'server_ini.xml' )
            self.configDom.normalize()
            self.configDoc = self.configDom.documentElement

            # Obtain the lobby/server password if it's been specified
            admin_pwd = self.configDoc.getAttribute( "admin" )
            boot_pwd = self.configDoc.getAttribute( "boot" )

            # Update the lobby with the passwords if they've been specified
            if( len(admin_pwd) or len(boot_pwd) ):
                self.groups = { '0': game_group( '0', 'Lobby', "", 'The game lobby', boot_pwd, "",
                                                 orpg.dirpath.dir_struct["user"]+'Lobby_map.xml',
                                                 orpg.dirpath.dir_struct["user"]+'LobbyMessage.html' )
                                }

            # set ip or dns name to send to meta server
            service_node = self.configDoc.getElementsByTagName( "service" )[0]


            address = service_node.getAttribute( "address" )
            address = address.lower()
            if address == "" or address == "hostname/address" or address == "localhost":
                self.server_address = None
            else:
                self.server_address = address


            #finds out if the user wants a default room message, and what file the user wants --akoman
            #edit: jan10/03 - I assumed there would always be a newroom setting. Modified code to make it optional
            tags = self.configDoc.getElementsByTagName( "newrooms" )

            self.defaultMessageFile = ""
# This try/except bit is to allow older versions of python to continue without a list error.

            try:
                if tags.length > 0:
                    newroom_node = tags[0]
                    dmf = newroom_node.getAttribute( "file" )
                    if dmf != "":
                        self.defaultMessageFile = dmf
            except:
                try:
                    if tags > 0:
                        newroom_node = tags[0]
                        dmf = newroom_node.getAttribute("file")
                        if dmf != "":
                            self.defaultMessageFile = dmf
                except:
                    pass

            #-------------------------------[ START <ROOM_DEFAULT> TAG PROCESSING ]--------------------
            #
            # New room_defaults configuration option used to set various defaults
            # for all user created rooms on the server. Incorporates akomans older
            # default room message code (from above)      --Snowdog 11/03
            #
            # option syntax
            # <room_defaults passwords="yes" map="myfiles/LobbyMap.xml" message="myfiles/LobbyMessage.html" />

            #default settings for tag options...
            roomdefault_msg = str(self.defaultMessageFile) #no message is the default
            roomdefault_map = "" #use lobby map as default
            roomdefault_pass = 1 #allow passwords


            #pull information from config file DOM
            try:
                roomdefaults = self.configDom.getElementsByTagName( "room_defaults" )[0]
                #rd.normalize()
                #roomdefaults = self.rd.documentElement
                try:
                    setting = roomdefaults.getElementsByTagName( 'passwords' )[0]
                    rpw = setting.getAttribute('allow')
                    if rpw == "no" or rpw=="0":
                        roomdefault_pass = 0
                        print("Room Defaults: Disallowing Passworded Rooms" )
                    else:
                        print("Room Defaults: Allowing Passworded Rooms" )
                except:
                    print ("Room Defaults: [Warning] Allowing Passworded Rooms")
                try:
                    setting = roomdefaults.getElementsByTagName( 'map' )[0]
                    map = setting.getAttribute( 'file' )
                    if map != "":
                        roomdefault_map = map
                        print ("Room Defaults: Using "+str(map)+" for room map")
                except:
                    print ("Room Defaults: [Warning] Using Default Map")

                try:
                    setting = roomdefaults.getElementsByTagName( 'message' )[0]
                    msg = setting.getAttribute( 'file' )
                    if msg != "":
                        roomdefault_msg = msg
                        print ("Room Defaults: Using "+str(msg)+" for room messages")
                except:
                    print ("Room Defaults: [Warning] Using Default Message")
            except:
                traceback.print_exc()
                print ("**WARNING** Error loading default room settings from configuration file. Using internal defaults.")


            #set the defaults
            if roomdefault_msg != "" or roomdefault_msg != None:
                self.defaultMessageFile = roomdefault_msg  #<room_defaults> tag superceeds older <newrooms> tag
            else:
                self.defaultMessageFile = None

            ##### room default map not handled yet. SETTING IGNORED
            if roomdefault_pass == 0: self.allow_room_passwords = 0
            else: self.allow_room_passwords = 1

            #-------------------------------[ END <ROOM_DEFAULT> TAG PROCESSING ]--------------------


            ###Server Cheat message
            try:
		cheat_node = self.configDoc.getElementsByTagName( "cheat" )[0]
                self.cheat_msg = cheat_node.getAttribute( "text" )
            except:
                self.cheat_msg = ""
	        print "**WARNING** <cheat txt=\"\"> tag missing from server configuration file. Using empty string."
            


            # should validate protocal
            validate_protocol_node = self.configDom.getElementsByTagName("validate_protocol ")

            self.validate_protocol = 1

            if(validate_protocol_node):
                self.validate_protocol = (validate_protocol_node[0].getAttribute("value") == "true")
            if(self.validate_protocol != 1):
                print "Protocol Validation: OFF"
            self.makePersistentRooms()

            print ("Server Configuration File: Processing Completed.")

        except Exception, e:
            print "Unable to read server configuration file.  Ignoring."
            print "The error was: ", e


    def makePersistentRooms( self ):
        'Creates rooms on the server as defined in the server config file.'

        # Creat a default value in case there are no persistent rooms
        self.persistRoomIdThreshold = -1
        for element in self.configDom.getElementsByTagName( 'room' ):
            roomName = element.getAttribute( 'name' )
            roomPassword = element.getAttribute( 'password' )
            bootPassword = element.getAttribute( 'boot' )

            # Conditionally check for minVersion attribute
            if element.hasAttribute( 'minVersion' ):
                minVersion = element.getAttribute( 'minVersion' )
            else:
                minVersion = ""

            # Extract the map filename attribute from the map node
            # we only care about the first map element found -- others are ignored
            mapElement = element.getElementsByTagName( 'map' )[0]
            mapFile = mapElement.getAttribute( 'file' )

            messageElement = element.getElementsByTagName( 'message' )[0]
            messageFile = messageElement.getAttribute( 'file' )

            # Make sure we have a message to even mess with
            if( len(messageFile) == 0 ):
                messageFile = self.defaultMessageFile

            # Update the threshold id with the most current room id
            self.persistRoomIdThreshold = self.new_group( roomName, roomPassword, bootPassword, minVersion, mapFile, messageFile )



    def isPersistentRoom( self, id ):
        'Returns true if the id is a persistent room (other than the lobby), otherwise, false.'
        if id > 0 and id <= self.persistRoomIdThreshold:
            return 1

        else:
            return 0


#-----------------------------------------------------
#  Toggle Meta Logging  -- Added by Snowdog 4/03
#-----------------------------------------------------
    def toggleMetaLogging(self):
        if self.show_meta_messages != 0:
            print "Meta Server Logging: OFF"
            self.show_meta_messages = 0
        else:
            print "Meta Server Logging: ON"
            self.show_meta_messages = 1


#-----------------------------------------------------
#  Start/Stop Network Logging to File  -- Added by Snowdog 4/03
#-----------------------------------------------------
    def NetworkLogging(self, mode = 0):
        if mode == 0:
            print "Network Logging: OFF"
            self.log_network_messages = 0
        elif mode == 1:
            print "Network Logging: ON (composite logfile)"
            self.log_network_messages = 1
        elif mode == 2:
            print "Network Logging: ON (split logfiles)"
            self.log_network_messages = 2
        else: return
        #when log mode changes update all connection stubs
        for n in self.players:
            try:
                self.players[n].EnableMessageLogging = mode
            except:
                print "Error changing Message Logging Mode for client #"+str(self.players[n].id)
    def NetworkLoggingStatus(self):
        if self.log_network_messages == 0: return "Network Traffic Log: Off"
        elif self.log_network_messages == 1: return "Network Traffic Log: Logging (composite file)"
        elif self.log_network_messages == 2: return "Network Traffic Log: Logging (inbound/outbound files)"
        else: print "Network Traffic Log: [Unknown]"




    def register_callback(instance,xml_dom = None,source=None):
        if xml_dom:    # if we get something
            if source == getMetaServerBaseURL():    # if the source of this DOM is the authoritative meta
                try:
                    metacache_lock.acquire()
                    curlist = getRawMetaList()      #  read the raw meta cache lines into a list
                    updateMetaCache(xml_dom)        #  update the cache from the xml
                    newlist = getRawMetaList()      #  read it into a second list
                finally:
                    metacache_lock.release()

                if newlist != curlist:          #  If the two lists aren't identical
                                               #  then something has changed.
                    #print "\nmetaserver data changed!  updating threads.\n"
                    instance.register()             #  Call self.register()
                                                #  which will force a re-read of the meta cache and
                                                #  redo the registerThreads

                # Eventually, reset the MetaServerBaseURL here


    def register(self,name_given=None):
        if name_given == None:
            name = self.name
        else:
            self.name = name = name_given

        #  Set up the value for num_users
        if self.players:
            num_players = len(self.players)
        else:
            num_players = 0

        #  request only Meta servers compatible with version 2
        metalist = getMetaServers(versions=["2"])
        if self.show_meta_messages != 0: self.log_msg("Found these valid metas:")
        for meta in metalist:
            if self.show_meta_messages != 0: self.log_msg("Meta:" + meta)

        #  Go through the list and see if there is already a running register
        #  thread for the meta.
        #  If so, call it's register() method
        #  If not, start one, implicitly calling the new thread's register() method


        #  iterate through the currently running metas and prune any
        #  not currently listed in the Meta Server list.
        if self.show_meta_messages != 0: self.log_msg( "Checking running register threads for outdated metas.")
        for meta in self.metas.keys():
            if self.show_meta_messages != 0: self.log_msg("meta:" + meta + ": ")
            if not meta in metalist:  # if the meta entry running is not in the list
                if self.show_meta_messages != 0: self.log_msg( "Outdated.  Unregistering and removing")
                self.metas[meta].unregister()
                del self.metas[meta]
            else:
                if self.show_meta_messages != 0: self.log_msg( "Found in current meta list.  Leaving intact.")

        #  Now call register() for alive metas or start one if we need one
        for meta in metalist:
            if self.metas.has_key(meta) and self.metas[meta] and self.metas[meta].isAlive():
                self.metas[meta].register(name=name,realHostName=self.server_address,num_users=num_players)
            else:
                self.metas[meta] = registerThread(name=name,realHostName=self.server_address,num_users=num_players,MetaPath=meta,port=OPENRPG_PORT,register_callback=self.register_callback)
                self.metas[meta].start()

        self.be_registered = 1



    def unregister(self):

        #  loop through all existing meta entries
        #  Don't rely on getMetaServers(), as a server may have been
        #  removed since it was started.  In that case, then the meta
        #  would never get unregistered.
        #
        #  Instead, loop through all existing meta threads and unregister them

        for meta in self.metas.values():
            if meta and meta.isAlive():
                meta.unregister()

        self.be_registered = 0




    #  This method runs as it's own thread and does the group_member_check every
    #    sixty seconds.  This should eliminate zombies that linger when no one is
    #    around to spook them.  GC: Frequency has been reduced as I question how valid
    #    the implementation is as it will only catch a very small segment of lingering
    #    connections.
    def player_reaper_thread_func(self,arg):
        while self.alive:
            time.sleep(60)

            self.p_lock.acquire()
            for group in self.groups.keys():
                self.check_group_members(group)
            self.p_lock.release()





    def sendMsg( self, sock, msg ):
        """Very simple function that will properly encode and send a message to te
        remote on the specified socket."""

        # Calculate our message length
        length = len( msg )

        ##print "\nserver sendMsg>" + msg + "\n\n"

        # Encode the message length into network byte order
        lp = pack( 'i', socket.htonl( length ) )

        try:
            # Send the encoded length
            sentl = sock.send( lp )

            # Now, send the message the the length was describing
            sentm = sock.send( msg )
            #self.log_msg(("data_sent", sentl+sentm))
            return sentm

        except socket.error, e:
            self.log_msg( "Socket Error: sendMsg(): "+ e )

        except Exception, e:
            self.log_msg("Exception: sendMsg(): "+ e )


    def recvData( self, sock, readSize ):
        """Simple socket receive method.  This method will only return when the exact
        byte count has been read from the connection, if remote terminates our
        connection or we get some other socket exception."""

        data = ""
        offset = 0
        try:
            while offset != readSize:
                frag = sock.recv( readSize - offset )

                # See if we've been disconnected
                rs = len( frag )
                if rs <= 0:
                    # Loudly raise an exception because we've been disconnected!
                    raise IOError, "Remote closed the connection!"

                else:
                    # Continue to build complete message
                    offset += rs
                    data += frag

        except socket.error, e:
            self.log_msg("Socket Error: recvData(): "+ e )
            data = ""

        return data



    def recvMsg( self, sock ):
        """This method now expects to receive a message having a 4-byte prefix length.  It will ONLY read
        completed messages.  In the event that the remote's connection is terminated, it will throw an
        exception which should allow for the caller to more gracefully handles this exception event.

        Because we use strictly reading ONLY based on the length that is told to use, we no longer have to
        worry about partially adjusting for fragmented buffers starting somewhere within a buffer that we've
        read.  Rather, it will get ONLY a whole message and nothing more.  Everything else will remain buffered
        with the OS until we attempt to read the next complete message."""

        msgData = ""
        try:
            lenData = self.recvData( sock, MPLAY_LENSIZE )

            # Now, convert to a usable form
            (length,) = unpack( 'i', lenData )
            length = socket.ntohl( length )

            # Read exactly the remaining amount of data
            msgData = self.recvData( sock, length )
            #self.log_msg(("data_recv", length+4))

        except Exception, e:
            self.log_msg( "Exception: recvMsg(): "+e )


        ##print "\nserver recvMsg>" + msgData + "\n\n"
        return msgData



    def kill_server(self):
        self.alive = 0
        self.log_msg("Server stopping...")
        self.unregister()                    # unregister from the Meta
        keys = self.players.keys()
        for k in keys:
            self.players[k].disconnect()
        self.incoming.put("<system/>")
        try:
            ip = socket.gethostbyname(socket.gethostname())
            kill = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            kill.connect((ip,OPENRPG_PORT))

            # Now, send the "system" command using the correct protocol format
            self.sendMsg( kill, "<system/>" )
            kill.close()
        except:
            pass

        self.listen_sock.close()
        self.listen_event.wait(10)
        self.incoming_event.wait(10)
        self.log_msg("Server stopped!")



    def log_msg(self,msg):
        if self.log_to_console:
            if self.log_console:
                self.log_console(msg)
            else:
                print str(msg)


    def print_help(self):
        print
        print "Commands: "
        print "'kill' or 'quit' - to stop the server"
        print "'broadcast' - broadcast a message to all players"
        print "'list' - list players and groups"
        print "'dump' - to dump player data"
        print "'dump groups' - to list the group names and ids only"
        print "'group n' - to list details about one group only"
        print "'register' - To register the server as name.  Also used to change the server's name if registered."
        print "'unregister' - To remove this server from the list of servers"
        print "'get lobby boot password' - to show the Lobby's boot password"
        print "'set lobby boot password' - to set the Lobby's boot password"
        print "'log' - toggles logging to the console off or on"
        print "'log meta' - toggles logging of meta server messages on or off"
        print "'logfile [off|on|split]' - timestamped network traffic log"
        print "'remove room' - to remove a room from the server"
        print "'kick' - kick a player from the server"
        print "'uptime' - reports how long server has been running"
        print "'roompasswords' - allow/disallow room passwords (toggle)"
        print "'search - will prompt for pattern and display results"
        print "'help' or '?' or 'h' - for this help message"
        print


    def broadcast(self,msg):
        self.send_to_all("0","<msg to='all' from='0' group_id='1'><font color='#FF0000'>"+msg+"</font>")


    def console_log(self):
        if self.log_to_console == 1:
            print "console logging now off"
            self.log_to_console = 0
        else:
            print "console logging now on"
            self.log_to_console = 1


    def groups_list(self):
        self.p_lock.acquire()
        try:
            keys = self.groups.keys()
            for k in keys:
                pw = "-"
                pr = " -"
                if self.groups[k].pwd != "":
                    pw = "P"
                if self.isPersistentRoom( int(k) ):
                    pr = " S" #using S for static (P for persistant conflicts with passowrd)
                print "Group: " + k + pr+pw+'  Name: ' + self.groups[k].name
            print

        except Exception, e:
            self.log_msg(e)

        self.p_lock.release()

    def search(self,patern):
        keys = self.groups.keys()
        print "Search results:"
        for k in keys:
            ids = self.groups[k].get_player_ids()
            for id in ids:
                if self.players[id].id.find(patern)>-1:
                    self.print_player_info(self.players[id])

                elif self.players[id].name.find(patern)>-1:
                    self.print_player_info(self.players[id])

                elif self.players[id].ip.find(patern)>-1:
                    self.print_player_info(self.players[id])

                elif self.players[id].group_id.find(patern)>-1:
                    self.print_player_info(self.players[id])

                elif self.players[id].role.find(patern)>-1:
                    self.print_player_info(self.players[id])

                elif self.players[id].version.find(patern)>-1:
                    self.print_player_info(self.players[id])

                elif self.players[id].protocol_version.find(patern)>-1:
                    self.print_player_info(self.players[id])

                elif self.players[id].client_string.find(patern)>-1:
                    self.print_player_info(self.players[id])


    def print_player_info(self,player):
        print player.id,player.name,player.ip,player.group_id, player.role,player.version,player.protocol_version,player.client_string

#----------------------------------------------------------------
#  Uptime Function  -- Added by snowdog 4/03
#----------------------------------------------------------------
    def uptime(self , mode = 0):
        "returns string containing how long server has been in operation"
        ut = time.time() - self.server_start_time
        d = int(ut/86400)
        h = int( (ut-(86400*d))/3600 )
        m = int( (ut-(86400*d)-(3600*h))/60)
        s = int( (ut-(86400*d)-(3600*h)-(60*m)) )
	uts =  str( "This server has been running for:\n "+str(d)+" days  "+str(h)+" hours  "+str(m)+" min. "+str(s)+" sec.  ["+str(int(ut))+" seconds]")
	if mode == 0: print uts
	else: return uts

#-----------------------------------------------------
#  Toggle Room Password Allow  -- Added by Snowdog 11/03
#-----------------------------------------------------
    def RoomPasswords(self):
        if self.allow_room_passwords != 0:
            self.allow_room_passwords = 0
            return "Client Created Room Passwords: Disallowed"
        else:
            self.allow_room_passwords = 1
            return "Client Created Room Passwords: Allowed"




    def group_dump(self,k):
        self.p_lock.acquire()
        try:
            print "Group: " + k
            print "    Name:  %s" % self.groups[k].name
            print "    Desc:  %s" % self.groups[k].desc
            print "    Pass:  %s" % self.groups[k].pwd
            print "    Boot:  %s" % self.groups[k].boot_pwd
            print "    Map:  %s" % self.groups[k].game_map.get_all_xml()
            print
        except Exception, e:
            self.log_msg(e)
        self.p_lock.release()

#----------------------------------------------------------------
#  Player List  -- Added by snowdog 4/03
#----------------------------------------------------------------
    def player_list(self):
        "display a condensed list of players on the server"
        self.p_lock.acquire()
        try:
            print "------------[ PLAYER LIST ]------------"
            keys = self.groups.keys()
            for k in keys:
                groupstring = "Group " + str(k) +": "+ self.groups[k].name
                if self.groups[k].pwd != "":
                    groupstring += " (Pass: \""+self.groups[k].pwd+"\" )"
                print groupstring
                ids = self.groups[k].get_player_ids()
                for id in ids:
                    if self.players.has_key(id):
                        print "  (%s)%s [IP: %s]" % ((self.players[id]).id, (self.players[id]).name, (self.players[id]).ip)
                    else:
                        self.groups[k].remove_player(id)
                        print "Bad Player Ref (#"+id+") in group"
                if len(ids) > 0: print ""
            print "--------------------------------------"
            print "\nStatistics: groups: "+str(len(self.groups))+"  players: "+ str(len(self.players))
        except Exception, e:
            self.log_msg(e)
        self.p_lock.release()


    def player_dump(self):
        self.p_lock.acquire()
        try:
            keys = self.groups.keys()
            for k in keys:
                print "Group: %s  %s (pass: \"%s\")" % (str(k),self.groups[k].name, self.groups[k].pwd)

                ids = self.groups[k].get_player_ids()
                for id in ids:
                    if self.players.has_key(id):
                        print str(self.players[id])
                    else:
                        self.groups[k].remove_player(id)
                        print "Bad Player Ref (#"+id+") in group"
        except Exception, e:
            self.log_msg(e)

        self.p_lock.release()



    def update_request(self,newsock,xml_dom):
        # handle reconnects

        self.log_msg( "update_request() has been called." )

        # get player id
        id = xml_dom.getAttribute("id")
        group_id = xml_dom.getAttribute("group_id")

        self.p_lock.acquire()
        if self.players.has_key(id):
            self.sendMsg( newsock, self.players[id].toxml("update") )
            self.players[id].reset(newsock)
            self.players[id].clear_timeout()
            need_new = 0
        else:
            need_new = 1
        self.p_lock.release()

        if need_new:
            self.new_request(newsock,xml_dom)
        else:
            msg = self.groups[group_id].game_map.get_all_xml()
            self.send(msg,id,group_id)


    def new_request(self,newsock,xml_dom,group_id='0'):
        # handle new request
        global myname

        #build client stub
        props = {}
        # Don't trust what the client tells us...trust what they connected as!
        props['ip'] = socket.gethostbyname( newsock.getpeername()[0] )
        props['role'] = "Player"

        try:
            props['role'] = xml_dom.getAttribute("role")

        except:
            props['role'] = "GM"

        props['name'] = xml_dom.getAttribute("name")
        props['group_id'] = group_id
        props['id'] = str(self.next_player_id)
        props['version'] = xml_dom.getAttribute("version")
        props['protocol_version'] = xml_dom.getAttribute("protocol_version")
        props['client_string'] = xml_dom.getAttribute("client_string")
        self.next_player_id += 1
        new_stub = client_stub(self.incoming,newsock,props,self.log_console)

        #update newly create client stub with network logging state
        new_stub.EnableMessageLogging = self.log_network_messages

        self.sendMsg( newsock, new_stub.toxml("new") )

        #  try to remove circular refs
        if xml_dom:
            xml_dom.unlink()

        # send confirmation
        data = self.recvMsg( newsock )
        try:
            xml_dom = parseXml(data)
            xml_dom = xml_dom._get_documentElement()
        except Exception, e:
            print e
            (remote_host,remote_port) = newsock.getpeername()
            bad_xml_string = "Your client sent an illegal message to the server and will be disconnected.  Please report this bug to the development team at:<br> "
            bad_xml_string += '<a href="http://sourceforge.net/tracker/?group_id=2237&atid=102237">OpenRPG bugs (http://sourceforge.net/tracker/?group_id=2237&atid=102237)</a><br>'
            self.sendMsg( newsock, "<msg to='"+props['id']+"' from='" + props['id'] +"' group_id='0' />" + bad_xml_string )

            time.sleep(2)
            newsock.close()
            print "Error in parse found from " + str(remote_host) + ".  Disconnected."
            print "  Offending data(" + str(len(data)) + "bytes)=" + data
            print "Exception=" + str(e)

            if xml_dom:
                xml_dom.unlink()
            return

        #start threads and store player

        allowed = 1
        ##print "Incoming Protocal: " + props['protocol_version']
        if ((props['protocol_version'] != PROTOCOL_VERSION) and self.validate_protocol):
            allowed = 0

        ##print "Allowing: " + str(allowed)
        ##print "validate_protocol: " +str(self.validate_protocol)

        if not allowed:
            version_string = "Sorry, this server can't handle your client version.<br>"

            version_string += '  Please go to <a href="http://www.openrpg.com">http://www.openrpg.com</a> to find a compatible client.<br>'
            version_string += "If you can't find a compatible client on the website, chances are that the server is running an unreleased development version for testing purposes.<br>"

            self.sendMsg( newsock, "<msg to='"+props['id']+"' from='0' group_id='0' />" + version_string )
            #  Give messages time to flow
            time.sleep(1)
            self.log_msg("Connection terminating due to version incompatibility with client protocol version " + props['protocol_version'] )
            newsock.close()
            if xml_dom:
                xml_dom.unlink()
            return None

        #  Display the lobby message
        self.SendLobbyMessage(newsock,props['id'])


        if xml_dom.getAttribute("id") == props['id']:
            new_stub.initialize_threads()
            self.p_lock.acquire()
            self.players[props['id']] = new_stub
            self.groups[group_id].add_player(props['id'])
            data = new_stub.toxml("new")
            self.p_lock.release()
            self.incoming.put(data)

            msg = self.groups[group_id].game_map.get_all_xml()
            self.send(msg,props["id"],group_id)
            self.return_room_roles(props["id"],group_id)

            # Re-initialize the role for this player incase they came from a different server
            self.handle_role("set",props['id'], "GM",
                             self.groups[group_id].boot_pwd, group_id)

            cmsg = "Client Connect: ("+str(props['id'])+") "+str(props['name'])+" ["+str(props['ip'])+"]"
            self.log_msg(cmsg)

            #  If already registered then re-register, thereby updating the Meta
            #    on the number of players
            if self.be_registered:
                self.register()

        if xml_dom:
            xml_dom.unlink()

    def SendLobbyMessage(self, socket, player_id):
        #######################################################################
        #  Display the lobby message
        #  prepend this server's version string to the the lobby message
        try:
            lobbyMsg = "You have connected to an <a href=\"http://www.openrpg.com\">OpenRPG</a> server, version '" + VERSION + "'"

            # See if we have a server name to report!
            #print "my name @ lobby message generation = ", myname
            if len(myname):
                lobbyMsg += ", named '" + myname + "'."

            else:
                lobbyMsg += "."

            # Add extra line spacing
            lobbyMsg += "\n\n"

            try:
                orpg.tools.config_files.validate_config_file("LobbyMessage.html","default_LobbyMessage.html")
            except:
                #print "No lobby message has been found...ignoring..."
                pass
            else:
                open_msg = open( orpg.dirpath.dir_struct["user"]+"LobbyMessage.html", "r" )
                lobbyMsg += open_msg.read()
                open_msg.close()

            # Send the server's lobby message to the client no matter what
            self.sendMsg( socket, "<msg to='"+player_id+"' from='0' group_id='0' />" + lobbyMsg )
            return
        except:
            traceback.print_exc()
        #  End of lobby message code
        #######################################################################


    def listenAcceptThread(self,arg):
        #  Set up the socket to listen on.
        try:
            self.log_msg("listen thread running...")
            self.listen_sock.bind( ("", OPENRPG_PORT) )
            self.listen_sock.listen( 5 )

        except Exception, e:
            self.log_msg( ("Error binding request socket!", e) )
            self.alive = 0


        while self.alive:

            #  Block on the socket waiting for a new connection
            try:
                (newsock, addr) = self.listen_sock.accept()
                ## self.log_msg("New connection from " + str(addr)+ ". Interfacing with server...")

                # Now that we've accepted a new connection, we must immediately spawn a new
                # thread to handle it...otherwise we run the risk of having a DoS shoved into
                # our face!  :O  After words, this thread is dead ready for another connection
                # accept to come in.
                #print "spawning new thread to hande incoming connection request..."
                thread.start_new_thread( self.acceptedNewConnectionThread, ( newsock, addr ) )

            except:
                print "The following exception caught accepting new connection:"
                traceback.print_exc()

        #  At this point, we're done and cleaning up.
        self.log_msg("server socket listening thread exiting...")
        self.listen_event.set()



    def acceptedNewConnectionThread( self, newsock, addr ):
        """Once a new connection comes in and is accepted, this thread starts up to handle it."""

        # Initialize xml_dom
        xml_dom = None

        # get client info and send othe client info
        # If this receive fails, this thread should exit without even attempting to process it
        print "Connection from " + str(addr) + " has been accepted.  Waiting for data..."
        try:
            data = self.recvMsg( newsock )

        except:
            print "The following exception caught accepting new connection from " + str(addr) + ":"

            traceback.print_exc()

            try:
                newsock.close()

            except Exception, e:
                self.log_msg( e )
                print e
                pass


        if data != None:
            if data == "<system/>":
                try:
                    newsock.close()

                except:
                    pass

            #  Clear out the xml_dom in preparation for new stuff, if necessary
            try:
                if xml_dom:
                    xml_dom.unlink()

            except:
                self.log_msg( "The following exception caught unlinking xml_dom:")
#                traceback.print_exc()
                self.log_msg("Continuing")

                try:
                    newsock.close()

                except:
                    pass

            #  Parse the XML received from the connecting client
            try:
                xml_dom = parseXml(data)
                xml_dom = xml_dom._get_documentElement()

            except:
                try:
                    newsock.close()

                except:
                    pass

                self.log_msg( "Error in parse found from " + str(addr) + ".  Disconnected.")
                self.log_msg("  Offending data(" + str(len(data)) + "bytes)=" + data)
                self.log_msg( "Exception:")
                traceback.print_exc()

            #  Determine the correct action and execute it
            try:
                # get action
                action = xml_dom.getAttribute("action")

                # Figure out what type of connection we have going on now
                if action == "new":
                    self.new_request(newsock,xml_dom)

                elif action == "update":
                    self.update_request(newsock,xml_dom)

                else:
                    self.log_msg("Unknown Join Request!")

            except Exception, e:
                print "The following  message: " + str(data)
                print "from " + str(addr) + " created the following exception: "
                traceback.print_exc()

            #  Again attempt to clean out DOM stuff
            try:
                if xml_dom:
                    xml_dom.unlink()
            except:
                print "The following exception caught unlinking xml_dom:"
                traceback.print_exc()



    def incoming_handler(self,arg):
        xml_dom = None
        self.log_msg( "incoming thread running..." )
        while self.alive:
            bad = 0
            num_garbage = gc.collect()
            self.log_msg( "number of uncollectables = " + str(num_garbage))
            data = self.incoming.get(1)
            bytes = len(data)
            if bytes <= 0:
                continue
            self.log_msg("handling: " + str(bytes))
            self.p_lock.acquire()
            try:
                end = data.find(">")
                head = data[:end+1]
                self.log_msg(head)
                if xml_dom:
                    xml_dom.unlink()

                try:
                    xml_dom = parseXml(head)
                    xml_dom = xml_dom._get_documentElement()

                except Exception, e:
                    print "Error in parse of message found in inbox. Ignoring."
                    print "  Offending data(" + str(len(data)) + "bytes)=" + data
                    print "Exception=" + str(e)
                    if xml_dom:
                        xml_dom.unlink()
                    bad = 1

                if not bad:
                    tag_name = xml_dom._get_tagName()
                    if tag_name == "msg":
                        self.incoming_msg_handler(xml_dom,data)

                    elif tag_name == "player":
                        self.incoming_player_handler(xml_dom,data)

                    #---------------------------------------------------------
                    # Remote Server Control 6/03 (snowdog)
                    #---------------------------------------------------------
                    elif tag_name == "admin":
                        self.remote_admin_handler(xml_dom,data)



#---------------------------------------------------------
# [START] Snowdog Password/Room Name altering code 12/02
#---------------------------------------------------------
                    ## Allows room admins to change room names and passwords on the fly
                    ## Note: boot passwords should NOT be changable. It could lead to
                    ##       room piracy.
                    ## Should re-configure the join password to accept the boot pass as well.
                    elif tag_name == "alter":
                        target = xml_dom.getAttribute("key")
                        value = xml_dom.getAttribute("val")
                        player = xml_dom.getAttribute("plr")
                        group_id = xml_dom.getAttribute("gid")
                        boot_pwd = xml_dom.getAttribute("bpw")
                        actual_boot_pwd = self.groups[group_id].boot_pwd

                        if self.allow_room_passwords == 0:
                            msg ="<msg to='"+player+"' from='0' group_id='0' /> Room passwords have been disabled by the server administrator."
                            self.players[player].outbox.put(msg)
                            return
                        elif boot_pwd == actual_boot_pwd:
                            if target == "pwd":
                                lmessage = "Room password changed to from \"" + self.groups[group_id].pwd + "\" to \"" + value +"\" by " + player
                                self.groups[group_id].pwd = value
                                msg ="<msg to='"+player+"' from='0' group_id='0' /> Room password changed to \""+ value + "\"."
                                self.players[player].outbox.put(msg)
                                self.log_msg(lmessage)
                                self.send_to_all('0',self.groups[group_id].toxml('update'))
                            elif target == "name":
                                # Check for & in name.  We want to allow this because of its common
                                # use in d&d games
				result = self.change_group_name(group_id,value,player)
                                msg ="<msg to='"+player+"' from='0' group_id='0' />"+result
                                self.players[player].outbox.put(msg)
                        else:
                            msg ="<msg to='"+player+"' from='0' group_id='0'>Invalid Administrator Password."
                            self.players[player].outbox.put(msg)

#---------------------------------------------------------
# [END] Snowdog Password/Room Name altering code 12/02
#---------------------------------------------------------

                    elif tag_name == "role":
                        role = ""
                        boot_pwd = ""
                        act = xml_dom.getAttribute("action")
                        player = xml_dom.getAttribute("player")
                        group_id = xml_dom.getAttribute("group_id")
                        if act == "set":
                            role = xml_dom.getAttribute("role")
                            boot_pwd = xml_dom.getAttribute("boot_pwd")
                        xml_dom.unlink()
                        if group_id <> "0":
                            self.handle_role(act, player, role, boot_pwd, group_id)
                            self.log_msg(("role", (player, role)))

                    # Server side Ping command altered by Snowdog 8/03
                    # Now returns a time component if it is supplied by the client.
                    # This offers a data loop back for clients to compute message travel time with
                    # by sending the time (in seconds) and comparing the returned time
                    # value with the current time on the client. The difference is the 'ping time'
                    # if a ping time is sent to the server then a ping response is sent
                    # back to the user instead of the 'pong' message
                    # Note: this is not a true ping. The resulting time includes network
                    #       travel time as well the XML parsing on the client and server
                    #       and any in-queue time within the network layers.
                    elif tag_name == "ping":
                        player = xml_dom.getAttribute("player")
                        group_id = xml_dom.getAttribute("group_id")
                        sent_time = ""
                        msg = ""
                        try:
                            sent_time = xml_dom.getAttribute("time")
                        except:
                            pass

                        if sent_time != "":
                            #because a time was sent return a ping response
                            msg ="<ping time='"+str(sent_time)+"' />"
                        else:
                            msg ="<msg to='"+player+"' from='"+player+"' group_id='"+group_id+"'><font color='#FF0000'>PONG!?!</font>"


                        self.players[player].outbox.put(msg)
                        xml_dom.unlink()

                    elif tag_name == "system":
                        pass

                    elif tag_name == "join_group":
                        self.join_group(xml_dom,data)

                    elif tag_name == "create_group":
                        self.create_group(xml_dom,data)

                    elif tag_name == "moderate":
                        self.moderate_group(xml_dom,data)

                    else:
                        raise Exception, "Not a valid header!"

            except Exception, e:
                self.log_msg(e)
                if xml_dom:
                    xml_dom.unlink()
            self.p_lock.release()
        if xml_dom:
            xml_dom.unlink()
        self.log_msg("incoming thread exiting...")
        self.incoming_event.set()

    def moderate_group(self,xml_dom,data):
        try:
            action = xml_dom.getAttribute("action")
            from_id = xml_dom.getAttribute("from")
            if xml_dom.hasAttribute("pwd"):
                pwd=xml_dom.getAttribute("pwd")
            else:
                pwd=""
            group_id=self.players[from_id].group_id
#            if group_id=="0":
#                return   #No moderation permitted in the lobby
            if action == "list":
                if (self.groups[group_id].moderated):
                    msg = ""
                    for i in self.groups[group_id].voice.keys():
                        if msg != "":
                            msg +=", "
                        if self.players.has_key(i):
                            msg += '('+i+') '+self.players[i].name
                        else:
                            del self.groups[group_id].voice[i]
                    if (msg <> ""):
                        msg = "The following users may speak in this room: "+msg
                    else:
                        msg = "No people are currently in this room with the ability to chat"
                    self.players[from_id].self_message(msg)
                else:
                    self.players[from_id].self_message("This room is currently unmoderated")
            elif action == "enable":
                if not self.groups[group_id].check_boot_pwd(pwd):
                    self.players[from_id].self_message("Failed - incorrect admin password")
                    return
                self.groups[group_id].moderated = 1
                self.players[from_id].self_message("This channel is now moderated")
            elif action == "disable":
                if not self.groups[group_id].check_boot_pwd(pwd):
                    self.players[from_id].self_message("Failed - incorrect admin password")
                    return
                self.groups[group_id].moderated = 0
                self.players[from_id].self_message("This channel is now unmoderated")
            elif action == "addvoice":
                if not self.groups[group_id].check_boot_pwd(pwd):
                    self.players[from_id].self_message("Failed - incorrect admin password")
                    return
                users = xml_dom.getAttribute("users").split(',')
                for i in users:
                    self.groups[group_id].voice[i.strip()]=1
            elif action == "delvoice":
                if not self.groups[group_id].check_boot_pwd(pwd):
                    self.players[from_id].self_message("Failed - incorrect admin password")
                    return
                users = xml_dom.getAttribute("users").split(',')
                for i in users:
                    if self.groups[group_id].voice.has_key(i.strip()):
                        del self.groups[group_id].voice[i.strip()]
            else:
                print "Bad input: "+data

        except Exception,e:
            self.log_msg(e)




    def join_group(self,xml_dom,data):
        try:
            from_id = xml_dom.getAttribute("from")
            pwd = xml_dom.getAttribute("pwd")
            group_id = xml_dom.getAttribute("group_id")
            ver = self.players[from_id].version
            allowed = 1

            if not self.groups[group_id].check_version(ver):
                allowed = 0
                msg = 'failed - invalid client version ('+self.groups[group_id].minVersion+' or later required)'

            if not self.groups[group_id].check_pwd(pwd):
                allowed = 0

		#tell the clients password manager the password failed -- SD 8/03
		pm = "<password signal=\"fail\" type=\"room\" id=\""+ group_id +"\" data=\"\"/>"
		self.players[from_id].outbox.put(pm)

                msg = 'failed - incorrect room password'

            if not allowed:
                self.players[from_id].self_message(msg)
                #the following line makes sure that their role is reset to normal,
                #since it is briefly set to lurker when they even TRY to change
                #rooms
                msg = "<role action=\"update\" id=\"" + from_id +"\" role=\"" + self.players[from_id].role + "\" />"
                self.players[from_id].outbox.put(msg)
                return

            #move the player into their new group.
            self.move_player(from_id, group_id)

        except Exception, e:
            self.log_msg(e)




    #----------------------------------------------------------------------------
    # move_player function -- added by Snowdog 4/03
    #
    # Split join_group function in half. separating the player validation checks
    # from the actual group changing code. Done primarily to impliment
    # boot-from-room-to-lobby behavior in the server.

    def move_player(self, from_id, group_id ):
        "move a player from one group to another"
        try:
            try:
                if group_id == "0":
                    self.players[from_id].role = "GM"
                else:
                    self.players[from_id].role = "Lurker"
            except Exception, e:
                print "exception in move_player() "
                traceback.print_exc()

            old_group_id = self.players[from_id].change_group(group_id,self.groups)
            self.send_to_group(from_id,old_group_id,self.players[from_id].toxml('del'))
            self.send_to_group(from_id,group_id,self.players[from_id].toxml('new'))
            self.check_group(from_id, old_group_id)

            # Here, if we have a group specific lobby message to send, push it on
            # out the door!  Make it put the message then announce the player...just
            # like in the lobby during a new connection.
            # -- only do this check if the room id is within range of known persistent id thresholds
            #also goes ahead if there is a defaultRoomMessage --akoman

            if self.isPersistentRoom( int(group_id) ) or self.defaultMessageFile != None:
                try:
                    roomMsgFile = open( self.groups[group_id].messageFile, "r" )
                    roomMsg = roomMsgFile.read()
                    roomMsgFile.close()

                except Exception, e:
                    roomMsg = ""
                    self.log_msg(e)

                # Spit that darn message out now!
                self.players[from_id].outbox.put( "<msg to='" + from_id + "' from='0' group_id='" + group_id + "' />"
                                                  + roomMsg )

            # Now, tell everyone that we've arrived
            self.send_to_all('0',self.groups[group_id].toxml('update'))

            # this line sends a handle role message to change the players role
            self.send_player_list(from_id,group_id)

            #notify user about others in the room
            self.return_room_roles(from_id,group_id)
            self.log_msg(("join_group", (from_id, group_id)))
            self.handle_role("set", from_id, self.players[from_id].role,
                            self.groups[group_id].boot_pwd, group_id)

        except Exception, e:
            self.log_msg(e)

    def return_room_roles(self,from_id,group_id):
        for m in self.players.keys():
            if self.players[m].group_id == group_id:
                msg = "<role action=\"update\" id=\"" + self.players[m].id +"\" role=\"" + self.players[m].role + "\" />"
                self.players[from_id].outbox.put(msg)


    # This is pretty much the same thing as the create_group method, however,
    # it's much more generic whereas the create_group method is tied to a specific
    # xml message.  Ack!  This version simply creates the groups, it does not
    # send them to players.  Also note, both these methods have race
    # conditions written all over them.  Ack! Ack!
    def new_group( self, name, pwd, boot, minVersion, mapFile, messageFile ):
        group_id = str( self.next_group_id )
        self.next_group_id += 1

        self.groups[group_id] = game_group( group_id, name, pwd, "", boot, minVersion, mapFile, messageFile )
        lmsg = "Creating Group... ("+str(group_id)+") "+str(name)
        self.log_msg( lmsg )
#This change required because this return value becomes the idThreshold. This changes length to the last id of persistent rooms. --akoman
        return len(self.groups) - 1

    def change_group_name(self,gid,name,pid):
        "Change the name of a group"
        # Check for & in name.  We want to allow this because of its common
        # use in d&d games.
        try:
	    loc = name.find("&")
            oldloc = 0
            while loc >0:
                loc = name.find("&",oldloc)
                if loc >0:
                    b = name[:loc]
                    e = name[loc+1:]
                    value = b + "&amp;" + e
                    oldloc = loc+1
	    oldroomname = self.groups[gid].name
            self.groups[gid].name = str(name)
            lmessage = "Room name changed to from \""+oldroomname+"\" to \""+name+"\""
            self.log_msg(lmessage +" by "+str(pid) )
            self.send_to_all('0',self.groups[gid].toxml('update'))
            return lmessage
	except:
	    return "An error occured during rename of room!"



    def create_group(self,xml_dom,data):
        try:
            from_id = xml_dom.getAttribute("from")
            pwd = xml_dom.getAttribute("pwd")
            name = xml_dom.getAttribute("name")
            boot_pwd = xml_dom.getAttribute("boot_pwd")
            minVersion = xml_dom.getAttribute("min_version")
            #added var reassign -- akoman
            messageFile = self.defaultMessageFile

            # see if passwords are allowed on this server and null password if not
            if self.allow_room_passwords != 1: pwd = ""


            #
            # Check for & in name.  We want to allow this because of its common
            # use in d&d games.

            loc = name.find("&")
            oldloc = 0
            while loc >0:
                loc = name.find("&",oldloc)
                if loc >0:
                    b = name[:loc]
                    e = name[loc+1:]
                    name = b + "&amp;" + e
                    oldloc = loc+1

            group_id = str(self.next_group_id)
            self.next_group_id += 1
            self.groups[group_id] = game_group(group_id,name,pwd,"",boot_pwd, minVersion, None, messageFile )
            #self.groups[group_id] = game_group(group_id,name,pwd,"",boot_pwd, minVersion )
            self.groups[group_id].voice[from_id]=1
            self.players[from_id].outbox.put(self.groups[group_id].toxml('new'))
            old_group_id = self.players[from_id].change_group(group_id,self.groups)
            self.send_to_group(from_id,old_group_id,self.players[from_id].toxml('del'))
            self.check_group(from_id, old_group_id)
            self.send_to_all(from_id,self.groups[group_id].toxml('new'))
            self.send_to_all('0',self.groups[group_id].toxml('update'))
            self.handle_role("set",from_id,"GM",boot_pwd, group_id)
            lmsg = "Creating Group... ("+str(group_id)+") "+str(name)
            self.log_msg( lmsg )
            jmsg = "moving to room "+str(group_id)+"."
            self.log_msg( jmsg )
            #even creators of the room should see the HTML --akoman
            #edit: jan10/03 - was placed in the except statement. Silly me.
            if self.defaultMessageFile != None:
                open_msg = open( self.defaultMessageFile, "r" )
                roomMsg = open_msg.read()
                open_msg.close()
                # Send the rooms message to the client no matter what
                self.players[from_id].outbox.put( "<msg to='" + from_id + "' from='0' group_id='" + group_id + "' />" + roomMsg )

        except Exception, e:
            self.log_msg( "Exception: create_group(): "+ e)


    def check_group(self, from_id, group_id):
        try:
            if int(group_id) > self.persistRoomIdThreshold and self.groups[group_id].get_num_players() == 0:
#                print "delete group"
                self.send_to_all("0",self.groups[group_id].toxml('del'))
                del self.groups[group_id]
                self.log_msg(("delete_group", (from_id, group_id)))

            else:
#                print "don't delete group"
                self.send_to_all("0",self.groups[group_id].toxml('update'))

        except Exception, e:
            self.log_msg(e)

    def del_player(self,id,group_id):
        try:
            dmsg = "Client Disconnect: ("+str(id)+") "+str(self.players[id].name)
            self.players[id].disconnect()
            del self.players[id]
            self.groups[group_id].remove_player(id)
            self.log_msg(dmsg)
            

            #  If already registered then re-register, thereby updating the Meta
            #    on the number of players
            #  Note:  Upon server shutdown, the server is first unregistered, so
            #           this code won't be repeated for each player being deleted.
            if self.be_registered:
                self.register()


        except Exception, e:
            self.log_msg(e)

        self.log_msg("Explicit garbage collection shows %s undeletable items." % str(gc.collect()))



    def incoming_player_handler(self,xml_dom,data):
        id = xml_dom.getAttribute("id")
        act = xml_dom.getAttribute("action")
        #group_id = xml_dom.getAttribute("group_id")
        group_id = self.players[id].group_id
        self.send_to_group(id,group_id,data)
        if act=="new":
            self.send_player_list(id,group_id)
            self.send_group_list(id)
        elif act=="del":
            #print "del player"
            self.del_player(id,group_id)
            self.check_group(id, group_id)
        elif act=="update":
            self.players[id].take_dom(xml_dom)
            self.log_msg(("update", {"id": id,
                                     "name": xml_dom.getAttribute("name"),
                                     "status": xml_dom.getAttribute("status"),
                                     "role": xml_dom.getAttribute("role") \
                                     }))

    def incoming_msg_handler(self,xml_dom,data):
        #try:
            to_id = xml_dom.getAttribute("to")
            from_id = xml_dom.getAttribute("from")
            group_id = xml_dom.getAttribute("group_id")
            end = data.find(">")
            msg = data[end+1:]

            #
            # check for < body to prevent someone from changing the background
            #

            tmp =data.lower()
            location = tmp.find("body")
            location -= 1
            while location > 0:
                if (tmp[location] != "&") and (tmp[location] != "l") \
                        and (tmp[location] != "t") and (tmp[location] != ";" )\
                        and (tmp[location] != " "):
                    location = 0
                elif (tmp[location:location+3] == "&lt"):
#                    print "found lt"
                    b = data[:location]
                    e = data[location+3:]
                    newmsg = b + e
                    data = newmsg
                    location = 0
                else:
#                    print "msg part",tmp[location:location+3],"<<"
                    location -= 1

            #
            # check for &#91 and &#93  codes which are often used to cheat with dice.
            #
            cheat_detected = 0
            location = data.find("91")
            cheat_point = location
            if (location > 0):
                if (data[location+2] == ";"):
                    semicolon = 1
                else:
                    semicolon = 0
                previous_character = data[cheat_point-1]
                if (previous_character == "#"):
                    if (data[cheat_point-5:cheat_point-1] == "amp;"):
                        cheat_detected = 1
                elif (previous_character == "0"):
                    while data[cheat_point-1] == "0":
                        cheat_point -=1
                    previous_character = data[cheat_point-1]
                    if (previous_character == "#"):
                        if (data[cheat_point-5:cheat_point-1] == "amp;"):
                            cheat_detected =1
            if cheat_detected:
                cheat_detected = 0
                if self.players[from_id].role != "GM":
                    new_string=data[:cheat_point-6] + "&lt;font color='#FF0000' &gt;" + self.cheat_msg + data[cheat_point-6:location+2+semicolon]+ "&lt;/font&gt;" + data[location+2+semicolon:]
                    data = new_string
            location = data.find("93")
            cheat_point = location
            if (location > 0):
                if (data[location+2] == ";"):
                    semicolon=1
                else:
                    semicolon=0
                previous_character = data[cheat_point-1]
                if (previous_character == "#"):
                    if (data[cheat_point-5:cheat_point-1] == "amp;"):
                        cheat_detected = 1
                elif (previous_character == "0"):
                    while data[cheat_point-1] == "0":
                        cheat_point -=1
                    previous_character = data[cheat_point-1]
                    if (previous_character == "#"):
                        if (data[cheat_point-5:cheat_point-1] == "amp;"):
                            cheat_detected = 1
            if cheat_detected:
                if self.players[from_id].role != "GM":
                    new_string=data[:cheat_point-6] + "&lt;font color='#FF0000' &gt;" + self.cheat_msg + data[cheat_point-6:location+2+semicolon] +"&lt;/font&gt;" +data[location+2+semicolon:]
                    data=new_string
            if from_id == "0" or len(from_id) == 0:
                print "WARNING!! Message received with an invalid from_id.  Message dropped."
                return None

            if to_id == "all":
                if msg[:4] == "<map":
##                    print "incoming map message from player " + from_id + " in group " + group_id + ":"
##                    print msg
                    if group_id == "0":
                        print "Attempt to change Lobby map ignored."
                    elif (self.players[from_id].role.lower() == "gm" or
                          self.players[from_id].role.lower() == "player"):
                        self.send_to_group(from_id,group_id,data)
                        self.groups[group_id].game_map.init_from_xml(msg)
                else:
                    if self.groups[group_id].moderated and not self.groups[group_id].voice.has_key(from_id):
                        self.players[from_id].self_message('This room is moderated - message not sent to others')
                        return
                    self.send_to_group(from_id,group_id,data)
            else:
                if msg[:4] == "<map":
                  print "Attempt to send map to a single player.  Ignoring."

                elif msg[:6] == "<boot ":
                    print "boot message found...", from_id, to_id, msg
                    self.handle_boot(from_id,to_id,group_id,msg)

                elif msg == BOOT_MSG:                                     #  if this is the magic boot message
                    if self.groups[group_id].players[0] == from_id:     #  from the "oldest" player
                        if group_id <> "0":                             #  and we're not in the Lobby

                            #  Send delete player event to all
                            self.send_to_group("0",group_id,self.players[to_id].toxml("del"))
                            #  Remove the player from local data structures
                            self.del_player(to_id, group_id)
                            #  Refresh the group data
                            self.check_group(to_id, group_id)
                else:
                    self.players[to_id].outbox.put(data)
            self.check_group_members(group_id)
        #except Exception, e:
            #self.log_msg(e)

    def handle_role(self, act, player, role, given_boot_pwd, group_id):
#        if group_id == "0":
#            return # don't do anything if we're in the lobby!
        if act == "display":
            msg = "<msg to=\"" + player + "\" from=\"0\" group_id=\"" + group_id + "\" />"
            msg += "Displaying Roles<br><br><u>Role</u>&nbsp&nbsp&nbsp<u>Player</u><br>"
            keys = self.players.keys()
            for m in keys:
                if self.players[m].group_id == group_id:
                    msg += self.players[m].role + " " +self.players[m].name + "<br>"
            self.send(msg,player,group_id)
        elif act == "set":
            try:
                actual_boot_pwd = self.groups[group_id].boot_pwd
                if self.players[player].group_id == group_id:
                    if actual_boot_pwd == given_boot_pwd:
                        self.log_msg( "Administrator passwords match -- changing role")
                        # Send a message to everyone in the room, letting them know someone has been booted
                        # keep data on server to prevent spoofing of roles.
                        # Send nice message to the room
#                        self.send_to_group("0",group_id, boot_msg)
                        #  Send update role event to all
                        msg = "<role action=\"update\" id=\"" + player +"\" role=\"" + role + "\" />"
                        self.send_to_group("0", group_id, msg)
                        self.players[player].role = role
                        if (role.lower() == "gm" or role.lower() == "player"):
                            self.groups[group_id].voice[player]=1
                    else:
                	#tell the clients password manager the password failed -- SD 8/03
                	pm = "<password signal=\"fail\" type=\"admin\" id=\""+ group_id +"\" data=\"\"/>"
                	self.players[player].outbox.put(pm)
                        self.log_msg( "Administrator passwords did not match")
            except Exception, e:
                print e
                print "Error executing the role change"
                print "due to the following exception:"
                traceback.print_exc()
                print "Ignoring boot message"

    def handle_boot(self,from_id,to_id,group_id,msg):

        xml_dom = None
        try:
            given_boot_pwd = None
            try:
                xml_dom = parseXml(msg)
                xml_dom = xml_dom._get_documentElement()
                given_boot_pwd = xml_dom.getAttribute("boot_pwd")

            except:
                print "Error in parse of boot message, Ignoring."
                print "Exception: "
                traceback.print_exc()

            try:
                actual_boot_pwd = self.groups[group_id].boot_pwd
                server_admin_pwd = self.groups["0"].boot_pwd
                print "Actual boot pwd = " + actual_boot_pwd
                print "Given boot pwd = " + given_boot_pwd
                if self.players[to_id].group_id == group_id:

                    ### ---CHANGES BY SNOWDOG 4/03 ---
                    ### added boot to lobby code.
                    ### if boot comes from lobby dump player from the server
                    ### any user in-room boot will dump to lobby instead
                    if given_boot_pwd == server_admin_pwd:
                        # Send a message to everyone in the room, letting them know someone has been booted
                        boot_msg = "<msg to='all' from='%s' group_id='%s'/><font color='#FF0000'>Booting '(%s) %s' from server...</font>" % (from_id, group_id, to_id, self.players[to_id].name)
                        print "boot_msg:" + boot_msg
                        self.send_to_group( "0", group_id, boot_msg )
                        time.sleep( 1 )

                        print "Booting player "+str(to_id)+" from server."
                        #  Send delete player event to all
                        self.send_to_group("0",group_id,self.players[to_id].toxml("del"))

                        #  Remove the player from local data structures
                        self.del_player(to_id,group_id)

                        #  Refresh the group data
                        self.check_group(to_id, group_id)

                    elif actual_boot_pwd == given_boot_pwd:
                        # Send a message to everyone in the room, letting them know someone has been booted
                        boot_msg = "<msg to='all' from='%s' group_id='%s'/><font color='#FF0000'>Booting '(%s) %s' from room...</font>" % (from_id, group_id, to_id, self.players[to_id].name)
                        print "boot_msg:" + boot_msg
                        self.send_to_group( "0", group_id, boot_msg )
                        time.sleep( 1 )

                        #dump player into the lobby
                        self.move_player(to_id,"0")

                        #  Refresh the group data
                        self.check_group(to_id, group_id)
                    else:
			#tell the clients password manager the password failed -- SD 8/03
                	pm = "<password signal=\"fail\" type=\"admin\" id=\""+ group_id +"\" data=\"\"/>"
                	self.players[from_id].outbox.put(pm)
                        print "boot passwords did not match"
            except:
                print "Error executing the boot"
                print "due to the following exception:"
                traceback.print_exc()
                print "Ignoring boot message"

        finally:
            try:
                 if xml_dom:
                    xml_dom.unlink()
            except:
                print "The following exception caught unlinking xml_dom:"
                traceback.print_exc()
                print "Continuing"


    #---------------------------------------------------------------
    # admin_kick function -- by Snowdog 4/03
    #---------------------------------------------------------------
    def admin_kick(self, id, message="" ):
        "Kick a player from a server from the console"

        try:
            group_id = self.players[id].group_id
            # Send a message to everyone in the victim's room, letting them know someone has been booted
            boot_msg = "<msg to='all' from='0' group_id='%s'/><font color='#FF0000'>Kicking '(%s) %s' from server... %s</font>" % ( group_id, id, self.players[id].name, str(message))
            print "boot_msg:" + boot_msg
            self.send_to_group( "0", group_id, boot_msg )
            time.sleep( 1 )

            print "kicking player "+str(id)+" from server."
            #  Send delete player event to all
            self.send_to_group("0",group_id,self.players[id].toxml("del"))

            #  Remove the player from local data structures
            self.del_player(id,group_id)

            #  Refresh the group data
            self.check_group(id, group_id)

        except:
            print "Error executing the kick due to the following exception:"
            traceback.print_exc()
            print "Ignoring boot message"



    def remove_room(self,group):
        try:
            keys = self.groups[group].get_player_ids()
            for k in keys:
                self.del_player(k,str(group))
            self.check_group("0", str(group))
        except:
            pass

    def send(self,msg,player,group):
        self.players[player].send(msg,player,group)


    def send_to_all(self,from_id,data):
        try:
            self.p_lock.acquire()
            keys = self.players.keys()
            self.p_lock.release()
            for k in keys:
                if k != from_id:
                    self.players[k].outbox.put(data)
        except Exception, e:
            self.log_msg("Exception: send_to_all(): "+e)



    def send_to_group(self,from_id,group_id,data):
        try:
            keys = self.groups[group_id].get_player_ids()
            for k in keys:
                if k != from_id:
                    self.players[k].outbox.put(data)
        except Exception, e:
            self.log_msg("Exception: send_to_group(): "+e)

    def send_player_list(self,to_id,group_id):
        try:
            keys = self.groups[group_id].get_player_ids()
            for k in keys:
                if k != to_id:
                    data = self.players[k].toxml('new')
                    self.players[to_id].outbox.put(data)
        except Exception, e:
            self.log_msg("Exception: send_player_list(): "+e)

    def send_group_list(self,to_id):
        try:
            keys = self.groups.keys()
            for k in keys:
                data = self.groups[k].toxml('new')
                self.players[to_id].outbox.put(data)
        except Exception, e:
            self.log_msg("Exception: send_group_list(): "+e)


    # This really has little value as it will only catch people that are hung
    # on a disconnect which didn't complete.  Other idle connections which are
    # really dead go undeterred.
    def check_group_members(self,group_id):
        try:
            keys = self.groups[group_id].get_player_ids()
            for k in keys:
                if self.players[k].get_status() != MPLAY_CONNECTED:
                    if self.players[k].check_time_out():
                        self.incoming.put(self.players[k].toxml('del'))
                        self.log_msg("Player #"+k+" Lost connection!")
        except Exception, e:
            self.log_msg("Exception: check_group_members(): "+e)


    def remote_admin_handler(self,xml_dom,data):
        # handle incoming remove server admin messages
        # (allows basic administration of server from a remote client)
        # base message format: <admin id="" pwd="" cmd="" [data for command]>

        try:
            pid = xml_dom.getAttribute("id")
            gid = ""
            given_pwd = xml_dom.getAttribute("pwd")
            cmd = xml_dom.getAttribute("cmd")
            server_admin_pwd = self.groups["0"].boot_pwd
            p_id = ""
            p_name= ""
            p_ip = ""

            #verify that the message came from the proper ID/Socket and get IP address for logging
            if self.players.has_key(pid):
                p_name=(self.players[pid]).name
                p_ip=(self.players[pid]).ip
		gid=(self.players[pid]).group_id
            else:
                #invalid ID.. report fraud and log
                m = "Invalid Remote Server Control Message (invalid id) #"+str(pid)+" does not exist."
                self.log_msg( m )
                return

            #check the admin password(boot password) against the supplied one in message
            #dump and log any attempts to control server remotely with invalid password
            if server_admin_pwd != given_pwd:
		#tell the clients password manager the password failed -- SD 8/03
                pm = "<password signal=\"fail\" type=\"server\" id=\""+ str(self.players[pid].group_id) +"\" data=\"\"/>"
                self.players[pid].outbox.put(pm)
                m = "Invalid Remote Server Control Message (bad password) from #"+str(pid)+" ("+str(p_name)+") "+str(p_ip)
                self.log_msg( m )
		return

            #message now deemed 'authentic'
            #determine action to take based on command (cmd)

            if cmd == "list":
                #return player list to this user.
                msg ="<msg to='"+pid+"' from='0' group_id='"+gid+"'>"+self.player_list_remote()
                self.players[pid].outbox.put(msg)

	    elif cmd == "killgroup":
		ugid = xml_dom.getAttribute("gid")
		if ugid == "0":
			m = "<msg to='"+pid+"' from='0' group_id='"+gid+"'>Cannot Remove Lobby! Remote administrator request denied!"
			self.players[pid].outbox.put(m)
		else:
		    result = self.prune_room(ugid)
		    msg = "<msg to='"+pid+"' from='0' group_id='"+gid+"'>"+str(result)
                    self.players[pid].outbox.put(msg)

	    elif cmd == "message":
		tuid = xml_dom.getAttribute("to_id")
		msg = xml_dom.getAttribute("msg")
		pmsg = "<msg to='"+tuid+"' from='0' group_id='"+self.players[tuid].group_id+"' >"+msg
		try: self.players[tuid].outbox.put(pmsg)
		except:
			msg = "<msg to='"+pid+"' from='0' group_id='"+gid+">Unknown Player ID: No message sent."
			self.players[pid].outbox.put(msg)

 	    elif cmd == "broadcast":
		bmsg = xml_dom.getAttribute("msg")
		self.broadcast(bmsg)

	    elif cmd == "killserver":
		#dangerous command..once server stopped it must be restarted manually
		self.kill_server()

	    elif cmd == "uptime":
	        msg ="<msg to='"+pid+"' from='0' group_id='"+gid+"'>"+self.uptime(1)
                self.players[pid].outbox.put(msg)

	    elif cmd == "help":
		msg = "<msg to='"+pid+"' from='0' group_id='"+gid+"'>"
		msg += self.AdminHelpMessage()
		self.players[pid].outbox.put( msg)

	    elif cmd == "roompasswords":
                # Toggle if room passwords are allowed on this server
                msg = "<msg to='"+pid+"' from='0' group_id='"+gid+"'>"
		msg += self.RoomPasswords()
		self.players[pid].outbox.put( msg)

	    elif cmd == "createroom":
		rm_name = xml_dom.getAttribute("name")
		rm_pass = xml_dom.getAttribute("pass")
		rm_boot = xml_dom.getAttribute("boot")
		result = self.create_temporary_persistant_room(rm_name, rm_boot, rm_pass)
		msg = "<msg to='"+pid+"' from='0' group_id='"+gid+"'>"+result
		self.players[pid].outbox.put(msg)

	    elif cmd == "nameroom":
		rm_id   = xml_dom.getAttribute("rmid")
                rm_name = xml_dom.getAttribute("name")
                result = self.change_group_name(rm_id,rm_name,pid)
                msg ="<msg to='"+pid+"' from='0' group_id='"+gid+"'/>"+result
                self.players[pid].outbox.put(msg)

	    elif cmd == "passwd":
		tgid = xml_dom.getAttribute("gid")
		npwd = xml_dom.getAttribute("pass")
		if tgid == "0":
		    msg ="<msg to='"+pid+"' from='0' group_id='"+gid+"'>Server password may not be changed remotely!"
                    self.players[pid].outbox.put(msg)
		else:
		    try:
		        self.groups[tgid].boot_pwd = npwd
		        msg ="<msg to='"+pid+"' from='0' group_id='"+gid+"'>Password changed for room "+tgid
                        self.players[pid].outbox.put(msg)
		    except: pass

	    else:
 		msg ="<msg to='"+pid+"' from='0' group_id='"+gid+"'><i>[Unknown Remote Administration Command]</i>"
        	self.players[pid].outbox.put(msg)




        except Exception, e:
            self.log_msg("Exception: Remote Admin Handler Error: "+ e)
            traceback.print_exc()


#-----------------------------------------------------------------
# Remote Administrator Help (returns from server not client)
#-----------------------------------------------------------------
    def AdminHelpMessage(self):
	"returns a string to be sent as a message to a remote admin"
	help = "<hr><B>REMOTE ADMINISTRATOR COMMANDS SUPPORTED</b><br><br>"
	help += "<table border='1' cellpadding='2'>"
	help += "<tr><td width='15%'><b>Command</b></td><td width='25%'><b>Format</b></td><td width='60%'><b>Description</b></td></tr>"
	help += "<tr><td>list</td><td>/admin list</td><td>Displays information about rooms and players on the server</td></tr>"
	help += "<tr><td>uptime</td><td>/admin uptime</td><td>Information on how long server has been running</td></tr>"
        help += "<tr><td>help</td><td>/admin help</td><td>This help message</td></tr>"
	help += "<tr><td>passwd</td><td>/admin passwd &lt;group id&gt; &lt;new password&gt;</td><td>changes a rooms bootpassword. Server(lobby) password may not be changed</td></tr>"
        help += "<tr><td>roompasswords</td><td>/admin roompasswords</td><td>Allow/Disallow Room Passwords on the server (toggles)</td></tr>"
        help += "<tr><td>message</td><td>/admin message &lt;user id&gt; &lt;message&gt;</td><td>Send a message to a specific user on the server</td></tr>"
        help += "<tr><td>broadcast</td><td>/admin broadcast &lt;message&gt;</td><td>Broadcast message to all players</td></tr>"
	help += "<tr><td>createroom</td><td>/admin createroom &lt;room name&gt; &lt;boot password&gt; [password]</td><td>Creates a temporary persistant room if possible.<i>Rooms created this way are lost on server restarts</i></td></tr>"
	help += "<tr><td>nameroom</td><td>/admin nameroom &lt;group id&gt; &lt;new name&gt;</td><td>Rename a room</td></tr>"
        help += "<tr><td>killgroup</td><td>/admin killgroup &lt;room id&gt;</td><td>Remove a room from the server and kick everyone in it.</td></tr>"
	help += "<tr><td>killserver</td><td>/admin killserver</td><td>Shuts down the server. <b>WARNING: Server cannot be restarted remotely via OpenRPG</b></td></tr>"
	help += "</table><br>"
	return help


#----------------------------------------------------------------
# Create Persistant Group -- Added by Snowdog 6/03
#
# Allows persistant groups to be created on the fly.
# These persistant groups are not added to the server.ini file
# however and are lost on server restarts
#----------------------------------------------------------------

    def create_temporary_persistant_room(self, roomname, bootpass, password=""):
	# if the room id just above the persistant room limit is available (not in use)
 	# then it will be assigned as a persistant room on the server
	"create a temporary persistant room"

	next_persist = self.persistRoomIdThreshold + 1

	try:
	    a = self.groups[str(next_persist)]
	    #if the above operation succeeds there is a room in the way. Cannot create persistant room!
	    return "Cannot create room. Next persistant room id in use."
	except:
	   pass
        group_id = str( next_persist )

        self.groups[group_id] = game_group( group_id, roomname, password, "", bootpass )
	self.persistRoomIdThreshold = next_persist
	cgmsg = "Create Temporary Persistant Group: ("+str(group_id)+") "+str(roomname)
        self.log_msg( cgmsg )
	self.send_to_all('0',self.groups[group_id].toxml('new'))
	self.send_to_all('0',self.groups[group_id].toxml('update'))
        return str("Persistant room created (group "+group_id+").")

#----------------------------------------------------------------
# Prune Room  -- Added by Snowdog 6/03
#
# similar to remove_room() except rooms are removed regardless
# of them being persistant or not
#----------------------------------------------------------------

    def prune_room(self,group):
        try:
            keys = self.groups[group].get_player_ids()
            for k in keys:
                self.move_player(k,'0')

	    if int(group) > self.persistRoomIdThreshold:
                self.send_to_all("0",self.groups[group].toxml('del'))
                del self.groups[group]
                self.log_msg(("delete_group", ('0',group)))
		return "Room Removed"

	    else:
		#removing a persistant room
		if int(group) == self.persistRoomIdThreshold:
		    self.persistRoomIdThreshold -= 1

		self.send_to_all("0",self.groups[group].toxml('del'))
                del self.groups[group]
                self.log_msg(("delete_group", ('0',group)))
                return "Persistant Room Removed"
	except:
            traceback.print_exc()
	    return "An Error occured on the server during room removal!"


#----------------------------------------------------------------
#  Remote Player List  -- Added by snowdog 6/03
#
#  Similar to console listing except formated for web display
#  in chat window on remote client
#----------------------------------------------------------------
    def player_list_remote(self):
        "display a condensed list of players on the server"
        self.p_lock.acquire()
        pl = "<br><table border=\"1\"><tr><td bgcolor=\"#DDDDDD\"><b>GROUP &amp; PLAYER LIST</b></td></tr>"
        try:

            keys = self.groups.keys()
            for k in keys:
                groupstring = "<tr><td><b>Group " + str(k) +": "+ self.groups[k].name +"</b>"
                groupstring += " <i>(Pass: \""+self.groups[k].pwd+"\"  Boot: \""+self.groups[k].boot_pwd+"\")</i>"
		pl += groupstring
                ids = self.groups[k].get_player_ids()
                for id in ids:
                    if self.players.has_key(id):
                        pl += "<br>&nbsp;&nbsp;("+ (self.players[id]).id
                        pl += ") "+(self.players[id]).name
                        pl += " [IP: "+(self.players[id]).ip+"]"
                    else:
                        self.groups[k].remove_player(id)
                        pl +="Bad Player Ref (#"+id+") in group"
                pl+="</td></tr>"
            pl += "<tr><td bgcolor=\"#DDDDDD\"><b><i>Statistics: groups: "+str(len(self.groups))+"  players: "+ str(len(self.players))+"</i></b></td></tr></table>"
        except Exception, e:
            self.log_msg(e)
        self.p_lock.release()
        return pl
