#!/usr/bin/perl
#
# $Id: IPv4Packet.pm,v 1.4 2001/10/06 22:19:14 levine Exp $
#
# Copyright (C) 2001  James D. Levine (jdl@vinecorp.com)
#
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
# 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
# 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
#   02111-1307, USA.
#
####################################################################

use NWatch::Packet;
use strict;


####################################################################
#
# ipv4_packet
#
# Decodes IPv4 fields
#
####################################################################


package NWatch::ipv4_packet;

@NWatch::ipv4_packet::ISA = qw( NWatch::packet );

sub protocol_name { "ipv4"; }

sub new
{
    my( $type, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $type, $data );

    my $ihlbyte = unpack( "C", $data );
    my $ihl = $ihlbyte & 15;
    my $version = $ihlbyte >> 4;

    # unpack fields
    $self->set( "version",      $version );
    $self->set( "ihl",          $ihl );
    $self->set( "tos",          unpack "x1C", $data );
    $self->set( "total_length", unpack "x2n", $data );
    $self->set( "identification", unpack "x4n", $data );
    
    $self->set( "flags",        ( unpack "x6C", $data ) >> 5 );
    $self->set( "fragment_offset", ( unpack "x6n", $data ) & 16383 );

    $self->set( "ttl",          unpack( "x8C", $data ) );
    $self->set( "protocol",     unpack( "x9C" , $data ) );
    $self->set( "header_checksum", unpack "x11n", $data );

    my $source = substr( $data, 12, 4 );
    $self->set( "source_address", $source );
    $self->set( "source_address_text" , &format_addr( $source ) );

    my $destination = substr( $data, 16, 4 );
    $self->set( "destination_address", $destination );
    $self->set( "destination_address_text", &format_addr( $destination ) );

    my $begin = $ihl * 4;
    my $payload = substr( $data, $begin, length( $data ) - $begin );

    # for any known protocols, stack a packet instance on top
    # check proto ID
    # if known, set next to a new instance of proto with payload

#    print "ipv4::new (ihl is $ihl, ver $version) " . format_addr( $self->get( "source_address" ) ) 
#	    . " -> " . format_addr( $self->get( "destination_address" ) )
#		. "\n";

#    print "ipv4::new data length is " . length( $data ) . " proto is " . $self->get( "protocol" ) . "\n";

    my $protocol = $self->get( "protocol" );

    if( $protocol == 6 )	# TCP
    {
	$self->next( new NWatch::tcp_packet( $payload ) );
    }
    elsif( $protocol == 17 )
    {
	$self->next( new NWatch::udp_packet( $payload ) );
    }
    elsif( $protocol == 1 )	# ICMPv4
    {
	$self->next( new NWatch::icmp4_packet( $payload ) );
    }

    $self;
}


sub format_addr
{
    my( $a ) = @_;
    return sprintf "%d.%d.%d.%d", unpack( "CCCC", $a ); 
}





####################################################################
#
# icmp4_packet
#
# Decodes ICMPv4 fields
#
####################################################################




package NWatch::icmp4_packet;

@NWatch::icmp4_packet::ISA = qw( NWatch::packet );

sub protocol_name { "icmp4"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );


    # unpack fields

    my( $type, $code, $checksum ) = unpack "CCN", $data;

    $self->set( "type",       $type );
    $self->set( "code",  $code );
    $self->set( "checksum",        $checksum );

    my $payload = substr( $data, 4, length( $data ) - 4 );

    # for any known message types, stack a packet instance on top
    # check proto ID
    # if known, set next to a new instance of proto with payload
    # for now, hardcoded

#    print "icmp type $type \n";
	    
    if( $type == 0 )
    {
	$self->next( new NWatch::icmp4_echo_reply( $payload ) );
    }
    elsif( $type == 8 )
    {
	$self->next( new NWatch::icmp4_echo( $payload ) );
    }
    elsif( $type == 3 )
    {
	$self->next( new NWatch::icmp4_destination_unreachable( $payload ) );
    }
    elsif( $type == 4 )
    {
	$self->next( new NWatch::icmp4_source_quench( $payload ) );
    }
    elsif( $type == 5 )
    {
	$self->next( new NWatch::icmp4_redirect( $payload ) );
    }
    elsif( $type == 11 )
    {
	$self->next( new NWatch::icmp4_time_exceeded( $payload ) );
    }
    elsif( $type == 12 )
    {
	$self->next( new NWatch::icmp4_parameter_problem( $payload ) );
    }
    elsif( $type == 13 )
    {
	$self->next( new NWatch::icmp4_timestamp_request( $payload ) );
    }
    elsif( $type == 14 )
    {
	$self->next( new NWatch::icmp4_timestamp_reply( $payload ) );
    }
    elsif( $type == 15 )
    {
	$self->next( new NWatch::icmp4_info_request( $payload ) );
    }
    elsif( $type == 16 )
    {
	$self->next( new NWatch::icmp4_info_reply( $payload ) );
    }

    $self;
}



####################################################################
#
# icmp4_echo
#
# Decodes ICMPv4 echo message.
#
####################################################################


package NWatch::icmp4_echo;

@NWatch::icmp4_echo::ISA = qw( NWatch::packet );

sub protocol_name { "echo"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );


    # unpack fields

    my( $identifier, $sequence_number ) = unpack "nn", $data;

    $self->set( "identifier",       $identifier );
    $self->set( "sequence_number",  $sequence_number );

    my $payload = substr( $data, 4, length( $data ) - 4 );
    $self->set( "payload", $payload );

#    print "icmp type echo (8) \n";
	    
    $self;
}


####################################################################
#
# icmp4_echo_reply
#
# Decodes ICMPv4 echo reply message.
#
####################################################################

package NWatch::icmp4_echo_reply;

@NWatch::icmp4_echo_reply::ISA = qw( NWatch::packet );

sub protocol_name { "echo_reply"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );


    # unpack fields

    my( $identifier, $sequence_number ) = unpack "nn", $data;

    $self->set( "identifier",       $identifier );
    $self->set( "sequence_number",  $sequence_number );

    my $payload = substr( $data, 4, length( $data ) - 4 );
    $self->set( "payload", $payload );

#    print "icmp type echo reply (0) \n";
	    
    $self;
}



####################################################################
#
# icmp4_destination_unreachable
#
# Decodes ICMPv4 destination unreachable message.
#
####################################################################


package NWatch::icmp4_destination_unreachable;

@NWatch::icmp4_destination_unreachable::ISA = qw( NWatch::packet );

sub protocol_name { "destination_unreachable"; }

sub new
{
    my( $classtype, $data ) = @_;

#    print "destination_unreachable: entry\n";
    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );

    my $payload = substr( $data, 4, length( $data ) - 4 );

    $self->set( "header_plus_64", $payload );

    my $packet = new NWatch::ipv4_packet( $payload );
    $self->set( "packet", $packet );

#    print "destination_unreachable: exit\n";
    $self;
}



####################################################################
#
# icmp4_time_exceeded
#
# Decodes ICMPv4 time exceeded message.
#
####################################################################



package NWatch::icmp4_time_exceeded;

@NWatch::icmp4_time_exceeded::ISA = qw( NWatch::packet );

sub protocol_name { "time_exceeded"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );

    my $payload = substr( $data, 4, length( $data ) - 4 );

    $self->set( "header_plus_64", $payload );
    my $packet = new NWatch::ipv4_packet( $payload );

    $self->set( "packet", $packet );

#    print "icmp type time exceeded \n";
	    
    $self;
}


####################################################################
#
# icmp4_parameter_problem
#
# Decodes ICMPv4 parameter problem message.
#
####################################################################

package NWatch::icmp4_parameter_problem;

@NWatch::icmp4_parameter_problem::ISA = qw( NWatch::packet );

sub protocol_name { "parameter_problem"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );

    $self->set( "pointer", unpack( "C", $data ) );
    my $payload = substr( $data, 4, length( $data ) - 4 );

    $self->set( "header_plus_64", $payload );

    my $packet = new NWatch::ipv4_packet( $payload );

    $self->set( "packet", $packet );
#    print "icmp type parameter problem \n";
	    
    $self;
}


####################################################################
#
# icmp4_redirect
#
# Decodes ICMPv4 redirect message.
#
####################################################################

package NWatch::icmp4_redirect;

@NWatch::icmp4_redirect::ISA = qw( NWatch::packet );

sub protocol_name { "redirect"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );

    $self->set( "gateway_address", substr( $data, 0, 4 ) );


    my $payload = substr( $data, 4, length( $data ) - 4 );
    $self->set( "header_plus_64", $payload );

    my $packet = new NWatch::ipv4_packet( $payload );
    $self->set( "packet", $packet );


#    print "icmp type redirect \n";
	    
    $self;
}


####################################################################
#
# icmp4_source_quench
#
# Decodes ICMPv4 source quench message.
#
####################################################################



package NWatch::icmp4_source_quench;

@NWatch::icmp4_source_quench::ISA = qw( NWatch::packet );

sub protocol_name { "source_quench"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );

    my $payload = substr( $data, 4, length( $data ) - 4 );
    $self->set( "header_plus_64", $payload );

    my $packet = new NWatch::ipv4_packet( $payload );
    $self->set( "packet", $packet );

#    print "icmp source quench exceeded \n";
	    
    $self;
}





####################################################################
#
# icmp4_info_request
#
# Decodes ICMPv4 info request message.
#
####################################################################



package NWatch::icmp4_info_request;

@NWatch::icmp4_info_request::ISA = qw( NWatch::packet );

sub protocol_name { "info_request"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );


    # unpack fields

    my( $identifier, $sequence_number ) = unpack "nn", $data;

    $self->set( "identifier",       $identifier );
    $self->set( "sequence_number",  $sequence_number );

#    print "icmp4 type info request\n";
	    
    $self;
}


####################################################################
#
# icmp4_info_reply
#
# Decodes ICMPv4 info reply message.
#
####################################################################


package NWatch::icmp4_info_reply;

@NWatch::icmp4_info_reply::ISA = qw( NWatch::packet );

sub protocol_name { "info_reply"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );


    # unpack fields

    my( $identifier, $sequence_number ) = unpack "nn", $data;

    $self->set( "identifier",       $identifier );
    $self->set( "sequence_number",  $sequence_number );

#    print "icmp type info reply\n";
	    
    $self;
}



####################################################################
#
# icmp4_timestamp_request
#
# Decodes ICMPv4 timestamp request message.
#
####################################################################


package NWatch::icmp4_timestamp_request;

@NWatch::icmp4_timestamp_request::ISA = qw( NWatch::packet );

sub protocol_name { "timestamp_request"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );


    # unpack fields

    my( $identifier,
	$sequence_number,
	$originate_timestamp,
	$receive_timestamp,
	$transmit_timestamp
	) = unpack "nnNNN", $data;


    $self->set( "identifier",       $identifier );
    $self->set( "sequence_number",  $sequence_number );


    $self->set( "originate_timestamp",  $originate_timestamp );
    $self->set( "receive_timestamp",    $receive_timestamp );
    $self->set( "transmit_timestamp",   $transmit_timestamp );


#    print "icmp type timestamp request\n";
	    
    $self;
}


####################################################################
#
# icmp4_timestamp_reply
#
# Decodes ICMPv4 timestamp reply message.
#
####################################################################

package NWatch::icmp4_timestamp_reply;

@NWatch::icmp4_timestamp_reply::ISA = qw( NWatch::packet );

sub protocol_name { "timestamp_reply"; }

sub new
{
    my( $classtype, $data ) = @_;

    # init from base class
    my $self = NWatch::packet::new( $classtype, $data );


    # unpack fields

    my( $identifier,
	$sequence_number,
	$originate_timestamp,
	$receive_timestamp,
	$transmit_timestamp
	) = unpack "nnNNN", $data;


    $self->set( "identifier",       $identifier );
    $self->set( "sequence_number",  $sequence_number );


    $self->set( "originate_timestamp",  $originate_timestamp );
    $self->set( "receive_timestamp",    $receive_timestamp );
    $self->set( "transmit_timestamp",   $transmit_timestamp );


#    print "icmp type timestamp reply\n";
	    
    $self;
}


1;






























