/**
 * @file threads.c
 *
 * This is the threads object. It handles thread ops in NJAMD
 * @TODO Write a pthreads wrapper so we don't have to many _THREAD_SAFE
 * ifdefs, and we don't have to worry about non-setup threads.
 */

#include <lib/threads.h>
#include <config.h>

/**
 * Bootstrap init for threads
 * 
 * @param threads The thread symbols
 * @param syms The symbol source
 */
void __nj_threads_bootstrap_init(struct nj_threads *threads, struct nj_libc_syms *syms)
{
#ifdef _THREAD_SAFE
# ifdef HAVE_SANE_DLOPEN
	threads->libc_pthread_exit = __nj_libc_syms_resolve_pthread(syms, "pthread_exit");
	threads->libc_pthread_create = __nj_libc_syms_resolve_pthread(syms, "pthread_create");
	pthread_mutex_init(&threads->launch_lock, NULL);
# endif
#endif
}

/**
 * Does nothing
 */
void __nj_threads_user_init(struct nj_threads *threads)
{
	return;
}

/**
 * Does nothing
 */
void __nj_threads_fini(struct nj_threads *threads)
{
#ifdef _THREAD_SAFE
	pthread_mutex_destroy(&threads->launch_lock);
#endif	
}

#ifdef _THREAD_SAFE
# ifdef HAVE_SANE_DLOPEN
/** 
 * Our own thread creation function
 *
 * Creates a pthread, used so we know when a thread starts, both callstack 
 * wise, and globally 
 */
int __nj_threads_create_thread(struct nj_threads *threads,
		pthread_t  *thread, pthread_attr_t *attr, 
		void * (*start_routine)(void *), void * arg)
{
	static struct pthread_arg_wrapper warg;

	/* we need this lock to protect the static variable, which must be static
	 * because this function most likely will return before pthread_launch is
	 * called. The lock is released by __nj_threads_launch */
	pthread_mutex_lock(&threads->launch_lock);

	warg.start = start_routine;
	warg.real_arg = arg;
	warg.threads = threads;
	
	return threads->libc_pthread_create(thread, attr, __nj_threads_launch, &warg);
}

/**
 * Launch a new thread. Using this function we can detect the end of a call
 * stack trace, so we don't fly off the improperly terminated stacks of some
 * pthread implementations.
 *
 * Isn't this fantastic? I'm pretty proud of this little hack actually :)
 *
 * @param arg A "wrapper" argument contianing the function to call as well as
 * 		      it's argument.
 * @returns Whatever the actual pthread function returns.
 * @sideeffects Releases the __nj_pthread_lock created in pthread_create() to
 * 			    protect it's argument.
 * @see pthread_create() next_address_valid() __nj_userspace_ret()
 */
void *__nj_threads_launch(void *arg)
{
	struct pthread_arg_wrapper warg = *(struct pthread_arg_wrapper *)arg;

	/* This lock initiated in our wrapped version of pthread_create, to
	 * protect the static variable passed to us */
	pthread_mutex_unlock(&warg.threads->launch_lock);
	
	return warg.start(warg.real_arg);
}

/** Placeholder to detrmine the size of __nj_pthread_launch */
void __nj_threads_launch_end(void)
{
	return;
}
# endif
#endif
// vim:ts=4
