
void __nj_stats_init(struct nj_stats *stats)
{
	memset(stats, 0, sizeof(struct nj_stats));
#ifdef _THREAD_SAFE
	pthread_mutex_init(&stats->lock);
#endif
}

void __nj_stats_fini(struct nj_stats *stats, void *dest)
{
	*(struct nj_stats_light *)dest = *(struct nj_stats_light *)stats;
#ifdef _THREAD_SAFE
	pthread_mutex_destroy(&stats->lock);
#endif
}

/** 
 * Update NJAMD statistics.
 * This function updates the statistics when amt memory is alloced
 * (> 0) or freed (< 0)
 * @param user The amount of user mem to add to stats (can be negative for free)
 * @param faulted The amount of total faulted pages (also negative for free)
 * @param as The amount of address space used (only negative if free and
 * 			 NO_CHK_FREE)
 */
void __nj_stats_update(struct nj_stats *stats, int user, int faulted, int as)
{
	/* No stats for the system stuff, please. Also, don't bother keeping stats
	 * if no one'll ever see them. */
	if((!NJ_ISSET(__nj_flags, NJ_DUMP_STATS)
			   	&& !NJ_ISSET(__nj_flags, NJ_PERSIST_HEAP))
			|| !NJ_ISSET(__nj_flags, NJ_ENTERED_MAIN)) 
			return;

#ifdef _THREAD_SAFE
	pthread_mutex_lock(&stats->lock);
#endif
	
#if 0
	if(NJ_ISSET(__nj_flags, NJ_ENTERED_DESTRUCT))
		__nj_eprintf("Warning: free called in destructor. Please mail mikepery@fscked.org\n");
#endif

	stats->leaked_user += user;
	stats->leaked_faulted += faulted;
	stats->as_used += as;

	if(user > 0)
	{
		stats->total_user += user;
		stats->total_faulted += faulted;
		stats->total_allocs++;
		stats->leaked_allocs++;

		if(__nj_leaked_user > stats->peak_user)
		{
			stats->peak_user = __nj_leaked_user;
			stats->peak_faulted = __nj_leaked_faulted;
		}
		
		if(__nj_leaked_faulted - __nj_leaked_user > stats->peak_overhead)
			stats->peak_overhead = __nj_leaked_faulted - __nj_leaked_user;
	}
	else
		stats->leaked_allocs--;
#ifdef _THREAD_SAFE
	pthread_mutex_unlock(&stats->lock);
#endif
}

char *__nj_byte_to_string(char *buf, int len, int bytes)
{
	if(bytes > 1<<SHIFT_MEGABYTE)
		snprintf(buf, len, "%.3f mB", bytes/(1024*1024.0));
	else if(bytes > 1024)
		snprintf(buf, len, "%.3f kB", bytes/1024.0);
	else
		snprintf(buf, len, "%d bytes", bytes);

	return buf;
}

/**
 * Print usage statistics.
 * 
 * This is just a quick little stat function so users can get statistics on
 * NJAMD performance as well as their program. Hopefully eventually we will
 * have some REALLY neat memory profiling cababilites that make this look like
 * trash.
 */
void __nj_stats_print(struct nj_stats *stats)
{
	/* Kinda silly, but what the hell. eprintf is much more expensive than
	 * stack space allocation. */
	char buf1[80], buf2[80], buf3[80], buf4[80], buf5[80], buf6[80], buf7[80];

#ifdef _THREAD_SAFE
	pthread_mutex_lock(&stats->lock);
#endif

	if(!stats->total_allocs)
	{
		__nj_eprintf("\nNJAMD Totals:\n\tmalloc() was never called!\n");

#ifdef _THREAD_SAFE
		pthread_mutex_unlock(&stats->lock);
#endif
		return;
	}

	__nj_eprintf(
			"\nNJAMD totals:\n\n"
			"\tAllocation totals:\t\t%d total, %d leaked\n"
			"\tLeaked User Memory:\t\t%s\n"
			"\tPeak User Memory:\t\t%s\n"
			"\tNJAMD Overhead at peak:\t\t%s\n" 
			"\tPeak NJAMD Overhead:\t\t%s\n" 
			"\tAverage NJAMD Overhead:\t\t%s per alloc\n" 
			"\tAddress space used:\t\t%s\n"
			"\tNJAMD Overhead at exit:\t\t%s\n\n",
			stats->total_allocs, stats->leaked_allocs, 
			__nj_byte_to_string(buf1, sizeof(buf1), __nj_leaked_user), 
			__nj_byte_to_string(buf2, sizeof(buf2), stats->peak_user), 
			__nj_byte_to_string(buf3, sizeof(buf3), stats->peak_faulted-stats->peak_user),
			__nj_byte_to_string(buf4, sizeof(buf5), stats->peak_overhead),
			__nj_byte_to_string(buf5, sizeof(buf5), (stats->total_faulted-stats->total_user)/stats->total_allocs),
			__nj_byte_to_string(buf6, sizeof(buf6), stats->as_used), 
			__nj_byte_to_string(buf7, sizeof(buf7), __nj_leaked_faulted-__nj_leaked_user)
				);
#ifdef _THREAD_SAFE
	pthread_mutex_unlock(&stats->lock);
#endif
}

